package com.biz.crm.tpm.business.audit.fee.local.service.internal.track;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.biz.crm.mn.common.base.dto.CommonSelectDto;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.util.ObjectConvertStringUtil;
import com.biz.crm.mn.common.base.vo.CommonSelectVo;
import com.biz.crm.tpm.business.activities.template.config.sdk.dto.ActivitiesTemplateConfigDto;
import com.biz.crm.tpm.business.activities.template.config.sdk.enums.ActivitiesTemplateConfigTypeEnum;
import com.biz.crm.tpm.business.activities.template.config.sdk.service.ActivitiesTemplateSdkService;
import com.biz.crm.tpm.business.activities.template.config.sdk.vo.ActivitiesTemplateConfigDetailVo;
import com.biz.crm.tpm.business.activities.template.config.sdk.vo.ActivitiesTemplateConfigVo;
import com.biz.crm.tpm.business.audit.fee.local.entity.track.AuditFeeDiffTrack;
import com.biz.crm.tpm.business.audit.fee.local.repository.track.AuditFeeDiffTrackDetailRepository;
import com.biz.crm.tpm.business.audit.fee.local.repository.track.AuditFeeDiffTrackRepository;
import com.biz.crm.tpm.business.audit.fee.local.service.third.DiffTrackPushCowManager;
import com.biz.crm.tpm.business.audit.fee.sdk.constants.AuditFeeConstants;
import com.biz.crm.tpm.business.audit.fee.sdk.constants.AuditFeeDiffTrackConstants;
import com.biz.crm.tpm.business.audit.fee.sdk.dto.track.AuditFeeDiffTrackApproveSubmitDto;
import com.biz.crm.tpm.business.audit.fee.sdk.dto.track.AuditFeeDiffTrackDetailDto;
import com.biz.crm.tpm.business.audit.fee.sdk.dto.track.AuditFeeDiffTrackDetailLedgerDto;
import com.biz.crm.tpm.business.audit.fee.sdk.dto.track.AuditFeeDiffTrackDto;
import com.biz.crm.tpm.business.audit.fee.sdk.dto.track.AuditFeeDiffTrackForecastDto;
import com.biz.crm.tpm.business.audit.fee.sdk.enumeration.AuditFeeDiffTrackDataSourceEnum;
import com.biz.crm.tpm.business.audit.fee.sdk.service.track.AuditFeeDiffTrackDetailVoService;
import com.biz.crm.tpm.business.audit.fee.sdk.service.track.AuditFeeDiffTrackVoService;
import com.biz.crm.tpm.business.audit.fee.sdk.vo.track.AuditFeeDiffTrackDetailLedgerVo;
import com.biz.crm.tpm.business.audit.fee.sdk.vo.track.AuditFeeDiffTrackDetailVo;
import com.biz.crm.tpm.business.audit.fee.sdk.vo.track.AuditFeeDiffTrackSummaryVo;
import com.biz.crm.tpm.business.audit.fee.sdk.vo.track.AuditFeeDiffTrackVo;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.dto.ProcessStatusDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBatchBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

/**
 * 差异费用追踪接口实现
 *
 * @author liyang
 * @date 2023/09/28
 */
@Slf4j
@Service
public class AuditFeeDiffTrackVoServiceImpl implements AuditFeeDiffTrackVoService {
  @Autowired
  private AuditFeeDiffTrackRepository auditFeeDiffTrackRepository;
  @Autowired
  private AuditFeeDiffTrackDetailRepository auditFeeDiffTrackDetailRepository;
  @Autowired
  private AuditFeeDiffTrackDetailVoService auditFeeDiffTrackDetailVoService;
  /**
   * 活动配置模板服务
   */
  @Autowired(required = false)
  private ActivitiesTemplateSdkService activitiesTemplateSdkService;
  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private ProcessBatchBusinessService processBatchBusinessService;

  @Autowired(required = false)
  private RedisTemplate<String, Object> redisTemplate;

  @Autowired(required = false)
  private AuditFeeDiffTrackDetailPageCacheHelper helper;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  private DiffTrackPushCowManager diffTrackPushCowManager;

  @Autowired(required = false)
  private CustomerVoService customerVoService;

  @Autowired(required = false)
  private DictDataVoService dictDataVoService;

  @Autowired(required = false)
  private TerminalVoService terminalVoService;

  @Override
  public Page<AuditFeeDiffTrackVo> findByConditions(Pageable pageable, AuditFeeDiffTrackDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(1, 50));
    dto = Optional.ofNullable(dto).orElse(new AuditFeeDiffTrackDto());
    Page<AuditFeeDiffTrackVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.auditFeeDiffTrackRepository.findByConditions(page, dto);
  }

  @Override
  public AuditFeeDiffTrackVo findById(String id) {
    if (StringUtils.isBlank(id)) return null;
    AuditFeeDiffTrack entity = auditFeeDiffTrackRepository.findById(id);
    if (entity == null) return null;
    AuditFeeDiffTrackVo vo = nebulaToolkitService.copyObjectByWhiteList(entity, AuditFeeDiffTrackVo.class, HashSet.class, ArrayList.class);
    // 查询含有明细数据的模板编码
    List<String> templateCodeList = auditFeeDiffTrackDetailVoService.findTemplateCodeByPlanCode(vo.getPlanCode());
    vo.setTemplateCodeList(templateCodeList);
    return vo;
  }

  /**
   * 详情-流程表单
   * @param processNo
   * @return
   */
  @Override
  public AuditFeeDiffTrackVo findDetailByProcessNo(String processNo) {
    if (StringUtils.isBlank(processNo)) return null;
    AuditFeeDiffTrack entity = auditFeeDiffTrackRepository.findByProcessNo(processNo);
    if (entity == null) return null;
    AuditFeeDiffTrackVo vo = nebulaToolkitService.copyObjectByWhiteList(entity, AuditFeeDiffTrackVo.class, HashSet.class, ArrayList.class);
    // 查询含有明细数据的模板编码
    List<String> templateCodeList = auditFeeDiffTrackDetailVoService.findTemplateCodeByPlanCode(vo.getPlanCode());
    vo.setTemplateCodeList(templateCodeList);
    return vo;
  }

  @Override
  public void pushCowManagerManual() {
    // 查询明细数据
    List<AuditFeeDiffTrackDetailVo> detailVos = auditFeeDiffTrackDetailRepository.pushCowManagerManual();
    if (CollectionUtils.isEmpty(detailVos)) {
      return;
    }
    // 推送牛人管家
    diffTrackPushCowManager.passItemPush(detailVos);
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void create(AuditFeeDiffTrackDto dto) {
    log.info("差异费用追踪新增{}", dto);
    this.createValidate(dto);

    // 开始保存
    AuditFeeDiffTrack auditFeeDiffTrack = new AuditFeeDiffTrack();
    BeanUtils.copyProperties(dto, auditFeeDiffTrack);
    auditFeeDiffTrackRepository.initInfo(auditFeeDiffTrack);
    auditFeeDiffTrack.setPlanCode(generateCodeService.generateCode(AuditFeeConstants.AUDIT_FEE_DIFF_TRACK_CODE, 1).get(0));
    auditFeeDiffTrack.setProcessStatus(ProcessStatusEnum.PREPARE.getDictCode());
    auditFeeDiffTrack.setIsClose(BooleanEnum.FALSE.getCapital());
    auditFeeDiffTrack.setDatasource(AuditFeeDiffTrackDataSourceEnum.MANUAL.getCode());
    auditFeeDiffTrack.setWholeAudit(BooleanEnum.FALSE.getSure());
    auditFeeDiffTrackRepository.save(auditFeeDiffTrack);

    // 将缓存中的费用关联数据 覆盖到数据库
    String cacheKey = dto.getCacheKey();
    if (StringUtils.isNotBlank(cacheKey)) {
      // 删除旧数据
      this.auditFeeDiffTrackDetailVoService.removeBatchByPlanCodeList(Lists.newArrayList(auditFeeDiffTrack.getPlanCode()));
      List<AuditFeeDiffTrackDetailDto> detailDtoList = this.auditFeeDiffTrackDetailVoService.findCacheList(dto.getCacheKey());
      this.createDetail(auditFeeDiffTrack, detailDtoList);
      log.info("追踪明细数据,{}",detailDtoList.size());
      this.auditFeeDiffTrackDetailVoService.clearCache(dto.getCacheKey());
      auditFeeDiffTrackDetailVoService.clearCache(cacheKey);
    }
  }

  private void createValidate(AuditFeeDiffTrackDto dto) {
    Validate.notNull(dto.getPlanName(),"差异费用处理名称为空");
    Validate.isTrue(dto.getPlanName().length() < 255, "差异费用处理名称超长");

    // 校验差异费用台账金额是否足够
    List<AuditFeeDiffTrackSummaryVo> summaryVos = this.collectDiffAmount(dto);
    List<String> notEnoughList = summaryVos.stream()
            .filter(e -> e.getBeRecoveredAmountTotal().compareTo(e.getUsedAmountTotal()) < 0)
            .map(AuditFeeDiffTrackSummaryVo::getFeeDiffLedgerCode).collect(Collectors.toList());
    Validate.isTrue(CollectionUtils.isEmpty(notEnoughList), String.join(",", notEnoughList) + "金额不足");
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void edit(AuditFeeDiffTrackDto dto) {
    log.info("差异费用追踪编辑{}", dto);

    this.updateValidate(dto);

    // 开始更新
    AuditFeeDiffTrack auditFeeDiffTrack = new AuditFeeDiffTrack();
    BeanUtils.copyProperties(dto, auditFeeDiffTrack);
    if (StringUtil.isEmpty(auditFeeDiffTrack.getWholeAudit())) {
      auditFeeDiffTrack.setWholeAudit(BooleanEnum.FALSE.getSure());
    }
    this.auditFeeDiffTrackRepository.updateById(auditFeeDiffTrack);

    // 将缓存中的费用关联数据 覆盖到数据库
    if (StringUtils.isNotBlank(dto.getCacheKey())) {
      // 删除旧数据
      this.auditFeeDiffTrackDetailVoService.removeBatchByPlanCodeList(Lists.newArrayList(auditFeeDiffTrack.getPlanCode()));
      List<AuditFeeDiffTrackDetailDto> detailDtoList = this.auditFeeDiffTrackDetailVoService.findCacheList(dto.getCacheKey());
      this.createDetail(auditFeeDiffTrack, detailDtoList);
      log.info("追踪明细数据,{}",detailDtoList.size());
      this.auditFeeDiffTrackDetailVoService.clearCache(dto.getCacheKey());
    }
  }

  private void updateValidate(AuditFeeDiffTrackDto dto) {
    Validate.notNull(dto.getPlanName(),"差异费用处理名称为空");
    Validate.isTrue(dto.getPlanName().length() < 255, "差异费用处理名称超长");

    // 校验差异费用台账金额是否足够
    List<AuditFeeDiffTrackSummaryVo> summaryVos = this.collectDiffAmount(dto);
    List<String> notEnoughList = summaryVos.stream()
            .filter(e -> e.getBeRecoveredAmountTotal().compareTo(e.getUsedAmountTotal()) < 0)
            .map(AuditFeeDiffTrackSummaryVo::getFeeDiffLedgerCode).collect(Collectors.toList());
    Validate.isTrue(CollectionUtils.isEmpty(notEnoughList), String.join(",", notEnoughList) + "金额不足");
  }

  private void createDetail(AuditFeeDiffTrack auditFeeDiffTrack, List<AuditFeeDiffTrackDetailDto> detailDtoList) {
    if (CollectionUtils.isEmpty(detailDtoList)) {
      return;
    }

    //是否必填从模板上获取，这里再做下主要字段的必填校验
    Set<String> templateConfigCodeSet = Sets.newHashSet();
    Set<String> terminalCodes = Sets.newHashSet();
    for (AuditFeeDiffTrackDetailDto detail : detailDtoList) {
      Validate.isTrue(detail.getActivityBeginDateStr().substring(0, 7).equals(detail.getActivityEndDateStr().substring(0, 7)),"活动开始/结束不允许跨月！");
      Validate.notBlank(detail.getTemplateConfigCode(),"活动明细模板不能为空！");
      templateConfigCodeSet.add(detail.getTemplateConfigCode());
      if (detail.getTerminalCode() != null) {
        terminalCodes.add(detail.getTerminalCode());
      }
      if (StringUtil.isEmpty(detail.getWholeAudit())){
        detail.setWholeAudit(BooleanEnum.FALSE.getSure());
      }
    }
    List<ActivitiesTemplateConfigVo> templateList = activitiesTemplateSdkService.findByCodeList(Lists.newArrayList(templateConfigCodeSet));
    if (templateList.size() != templateConfigCodeSet.size()){
      throw new RuntimeException("活动明细模板数据有误！");
    }

    // 字典
    final String firstChannel = "firstChannelName";
    final String secondChannel = "secondChannelCode";
    Map<String, List<DictDataVo>> dictTypeCodeMap = dictDataVoService.findByDictTypeCodeList(Lists.newArrayList(firstChannel, secondChannel));
    Map<String, String> firstChannelMap = CollectionUtils.isEmpty(dictTypeCodeMap.get(firstChannel)) ? new HashMap<>(0) :
        dictTypeCodeMap.get(firstChannel).stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictValue));
    Map<String, String> secondChannelMap = CollectionUtils.isEmpty(dictTypeCodeMap.get(secondChannel)) ? new HashMap<>(0) :
        dictTypeCodeMap.get(secondChannel).stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictValue));

    //查询门店数据
    Map<String, TerminalVo> terminalMap = new HashMap<>();
    List<TerminalVo> terminalList = terminalVoService.findBaseByTerminalCodes(Lists.newArrayList(terminalCodes));
    if (CollectionUtils.isNotEmpty(terminalList)) {
      terminalMap.putAll(terminalList.stream().collect(Collectors.toMap(TerminalVo::getTerminalCode, Function.identity())));
    }

    Map<String, ActivitiesTemplateConfigVo> templateMap = templateList.stream().collect(Collectors.toMap(ActivitiesTemplateConfigVo::getConfigCode, Function.identity()));
    // 按模板给明细分组
    Map<String,List<AuditFeeDiffTrackDetailDto>> groupBy = detailDtoList.stream().collect(Collectors.groupingBy(AuditFeeDiffTrackDetailDto::getTemplateConfigCode));

    //我方承担金额
    AtomicReference<BigDecimal> feeAmount = new AtomicReference<>(BigDecimal.ZERO);
    //差异费用金额
    AtomicReference<BigDecimal> useAmount = new AtomicReference<>(BigDecimal.ZERO);

    groupBy.forEach((k,v) -> {
      ActivitiesTemplateConfigVo templateConfigVo = templateMap.get(k);
      //根据模板区分一下
      Map<String, String> templateDetailTitleMap = templateConfigVo.getDetails().stream().collect(Collectors.toMap(ActivitiesTemplateConfigDetailVo::getField, ActivitiesTemplateConfigDetailVo::getTitle));
      // 根据模板字段设置值
      ObjectConvertStringUtil.convertObjectListStrProperties(v, AuditFeeDiffTrackDetailDto.class,null,true, templateDetailTitleMap);
      v.forEach(e -> {
        e.setPlanCode(auditFeeDiffTrack.getPlanCode());
        BigDecimal theFeeAmount = StringUtils.isBlank(e.getFeeAmountStr()) ? BigDecimal.ZERO : new BigDecimal(e.getFeeAmountStr());
        feeAmount.set(feeAmount.get().add(theFeeAmount));
        if(CollectionUtils.isNotEmpty(e.getAuditFeeDiffTrackDetailLedgerList())){
          useAmount.set(useAmount.get().add(e.getAuditFeeDiffTrackDetailLedgerList().stream().filter(diff -> Objects.nonNull(diff.getUsedAmount())).map(AuditFeeDiffTrackDetailLedgerDto::getUsedAmount).reduce(BigDecimal.ZERO, BigDecimal::add)));
        }
        if(StringUtils.isNotEmpty(e.getFirstChannelCode())){
          e.setFirstChannelName(firstChannelMap.get(e.getFirstChannelCode()));
        }
        if(StringUtils.isNotEmpty(e.getSecondChannelCode())){
          e.setSecondChannelName(secondChannelMap.get(e.getSecondChannelCode()));
        }
        if(StringUtils.isNotEmpty(e.getTerminalCode())){
          TerminalVo terminalVo = terminalMap.get(e.getTerminalCode());
          if (terminalVo != null) {
            e.setCustomerCode(terminalVo.getSellerCode());
            e.setCustomerName(terminalVo.getSellerName());
          }
        }
        /*if(StringUtils.isNotBlank(e.getActivityBeginDateStr())) {
            e.setActivityBeginDate(DateUtil.parseDate(e.getActivityBeginDateStr(), DateUtil.DEFAULT_YEAR_MONTH_DAY));
        }
        if(StringUtils.isNotBlank(e.getActivityEndDateStr())) {
            e.setActivityEndDate(DateUtil.parseDate(e.getActivityEndDateStr(),DateUtil.DEFAULT_YEAR_MONTH_DAY));
        }
        if(StringUtils.isNotBlank(e.getScheduleBeginDateStr())) {
            e.setScheduleBeginDate(DateUtil.parseDate(e.getScheduleBeginDateStr(),DateUtil.DEFAULT_YEAR_MONTH_DAY));
        }
        if(StringUtils.isNotBlank(e.getScheduleEndDateStr())) {
            e.setScheduleEndDate(DateUtil.parseDate(e.getScheduleEndDateStr(),DateUtil.DEFAULT_YEAR_MONTH_DAY));
        }*/
      });
      log.info("插入明细数据,{}", JSON.toJSONString(v));
      auditFeeDiffTrackDetailVoService.createBatch(v);
    });
    Validate.isTrue(feeAmount.get().compareTo(useAmount.get()) == 0,"关联预算金额需等于我方承担金额之和");
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void delete(List<String> ids) {
    Validate.notEmpty(ids, "id集合不能为空");
    List<AuditFeeDiffTrack> auditFeeDiffTrackList = auditFeeDiffTrackRepository.findByIds(ids);
    Validate.notEmpty(auditFeeDiffTrackList, "选择的数据不存在");
    List<String> notPrepareList = auditFeeDiffTrackList.stream().map(AuditFeeDiffTrack::getProcessStatus).filter(e -> !ProcessStatusEnum.PREPARE.getDictCode().equals(e)).collect(Collectors.toList());
    Validate.isTrue(CollectionUtils.isEmpty(notPrepareList), "审核状态不是待审核，只有待审核状态才能删除");
    log.info("差异追踪逻辑删除-开始");

    auditFeeDiffTrackList = auditFeeDiffTrackList.stream().peek(e -> e.setDelFlag(DelFlagStatusEnum.DELETE.getCode())).collect(Collectors.toList());
    auditFeeDiffTrackRepository.updateBatchById(auditFeeDiffTrackList);

    List<String> planCodeList = auditFeeDiffTrackList.stream().map(AuditFeeDiffTrack::getPlanCode).filter(Objects::nonNull).collect(Collectors.toList());
    if (CollectionUtils.isNotEmpty(planCodeList)) {
      //  移除活动明细，包含对差异费用台账的操作
      auditFeeDiffTrackDetailVoService.removeBatchByPlanCodeList(planCodeList);
    }
  }

  @Override
  public List<CommonSelectVo> findActivitiesTemplateConfigSelectList(ActivitiesTemplateConfigDto configDto, CommonSelectDto dto) {
    Validate.notBlank(configDto.getBusinessFormatCode(),"业态不能为空");
    Validate.notBlank(configDto.getBusinessUnitCode(),"业务单元不能为空");
    //如果是小业务单元的话就查主体的模板
    if (BusinessUnitEnum.isDefaultBusinessUnit(configDto.getBusinessUnitCode())){
      configDto.setBusinessUnitCode(BusinessUnitEnum.HEADQUARTERS.getCode());
    }
    configDto.setType(ActivitiesTemplateConfigTypeEnum.DIFF_FEE_DISPOSE.getCode());
    return activitiesTemplateSdkService.findActivitiesTemplateConfigSelectList(dto, configDto);
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void submitApproval(List<String> ids, AuditFeeDiffTrackApproveSubmitDto dto) {
    log.info("差异费用追踪流程提交id{} 对象{}", ids, dto);
    List<AuditFeeDiffTrack> auditFeeDiffTracks = auditFeeDiffTrackRepository.listByIds(ids);

    for (AuditFeeDiffTrack auditFeeDiffTrack : auditFeeDiffTracks) {
      if (Boolean.TRUE.equals(auditFeeDiffTrack.getTempSave())) {
        throw new RuntimeException("追踪编码[" + auditFeeDiffTrack.getPlanCode() + "]为暂存状态，不能提交审批！");
      }
      //待提交、驳回、追回的数据才能提交
      if (!ProcessStatusEnum.PREPARE.getDictCode().equals(auditFeeDiffTrack.getProcessStatus()) &&
          !ProcessStatusEnum.RECOVER.getDictCode().equals(auditFeeDiffTrack.getProcessStatus()) &&
          !ProcessStatusEnum.REJECT.getDictCode().equals(auditFeeDiffTrack.getProcessStatus())) {
        throw new RuntimeException("追踪编码[" + auditFeeDiffTrack.getPlanCode() + "]不处于待提交、驳回、追回状态，不能提交审批！");
      }
    }
    List<String> planCodeList = auditFeeDiffTracks.stream().map(AuditFeeDiffTrack::getPlanCode).collect(Collectors.toList());

    //扣减预算
    this.auditFeeDiffTrackDetailVoService.useMonthBudgetByPlanCodeList(planCodeList,dto.getOperateId());

//    ActivityPlanProcessBusinessForm processBusinessForm = activityPlanRepository.getActivityPlanProcessBusinessForm(planCodeList);
//    List<ActivityPlanBudget> isOverBudgetList = getIsOverBudget(planCodeList);
//    String isOverBudget = org.springframework.util.CollectionUtils.isEmpty(isOverBudgetList) ? BooleanEnum.FALSE.getCapital() : BooleanEnum.TRUE.getCapital();
//    processBusinessForm.setIsOverBudget(isOverBudget);
//    //查询管控力度
//    List<MonthBudgetVo> monthBudgetVoList = monthBudgetService.findByCodes(budgetDtoList.stream().map(ActivityPlanBudgetDto::getMonthBudgetCode).filter(
//        Objects::nonNull).collect(Collectors.toList()), null);
//    Set<String> controlSituationList = monthBudgetVoList.stream().map(MonthBudgetVo::getControlSituation).collect(Collectors.toSet());
//    String isSpecialGrantProcess = controlSituationList.contains(ControlSituationEnum.SPECIAL_GRANT_PROCESS.getCode()) ? BooleanEnum.TRUE.getCapital() : BooleanEnum.FALSE.getCapital();
//    String isSpecialProcess = BooleanEnum.TRUE.getCapital().equals(isSpecialGrantProcess)
//        && (BooleanEnum.TRUE.getCapital().equals(isOverBudget) || BooleanEnum.TRUE.getCapital().equals(processBusinessForm.getIsBreakPrice()))
//        ? BooleanEnum.TRUE.getCapital() : BooleanEnum.FALSE.getCapital();
//    processBusinessForm.setIsSpecialGrantProcess(isSpecialGrantProcess);
//    processBusinessForm.setIsSpecialProcess(isSpecialProcess);
//    log.error("方案提交审批:{}", processBusinessForm);

    ProcessBusinessDto processBusinessDto = dto.getProcessBusiness();
    processBusinessDto.setBusinessNoList(planCodeList);
    processBusinessDto.setBusinessCode(AuditFeeDiffTrackConstants.AUDIT_FEE_DIFF_TRACK);
    //批量提交-业务编码设置为uuid
    String businessNo = UUID.randomUUID().toString().replace("-", "");
    processBusinessDto.setBusinessNo(businessNo);
//    processBusinessDto.setBusinessFormJson(JSONObject.toJSONString(processBusinessForm));
    ProcessBusinessVo processBusinessVo = processBatchBusinessService.processStart(processBusinessDto);
    //提交成功
    auditFeeDiffTrackRepository.updateProcessStatusAndProcessNo(planCodeList, ProcessStatusEnum.COMMIT.getDictCode(), processBusinessVo.getProcessNo());
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void processPass(ProcessStatusDto dto) {
    auditFeeDiffTrackRepository.updateProcessStatus(dto.getBusinessNoList(), dto.getProcessStatus());
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void processRejectAndRecover(ProcessStatusDto dto,String operateId) {
    // 回退差异费用扣减
    auditFeeDiffTrackDetailVoService.returnMonthBudgetByPlanCodeList(dto.getBusinessNoList(),operateId);
    // 改状态
    auditFeeDiffTrackRepository.updateProcessStatus(dto.getBusinessNoList(),dto.getProcessStatus());
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void close(List<String> ids,String operateId) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    List<AuditFeeDiffTrack> entities = auditFeeDiffTrackRepository.findByIds(ids);
    if (CollectionUtils.isEmpty(entities)) {
      return;
    }
    // 审批通过的才能关闭
    List<String> notPassEntityPlanNameList = entities.stream().filter(e -> ProcessStatusEnum.PASS.getDictCode().equals(e.getProcessStatus())).map(AuditFeeDiffTrack::getPlanName).collect(Collectors.toList());
    Assert.isTrue(CollectionUtils.isNotEmpty(notPassEntityPlanNameList), "差异费用处理名称["+ String.join(",", notPassEntityPlanNameList) +"]未通过审核，无法关闭");
    // 判断结案
    List<String> wholeAuditedPlanNameList = entities.stream().filter(e -> BooleanEnum.TRUE.getCapital().equals(e.getWholeAudit())).map(AuditFeeDiffTrack::getPlanName).collect(Collectors.toList());
    Assert.isTrue(CollectionUtils.isNotEmpty(notPassEntityPlanNameList), "差异费用处理名称["+ String.join(",", wholeAuditedPlanNameList) +"]已完全结案，无法关闭");
    // 根据时间判断结案
    /*Date now = new Date();
    List<String> timeWholeAuditedPlanNameList = entities.stream().filter(e -> now.after(e.getEndDate())).map(AuditFeeDiffTrack::getPlanName).collect(Collectors.toList());
    Assert.isTrue(CollectionUtils.isNotEmpty(notPassEntityPlanNameList), "差异费用处理名称["+ String.join(",", timeWholeAuditedPlanNameList) +"]已大于方案结束时间，无法关闭");
*/
    entities = entities.stream().peek(e -> e.setIsClose(BooleanEnum.TRUE.getCapital())).collect(Collectors.toList());
    auditFeeDiffTrackRepository.saveOrUpdateBatch(entities);

    // 回退差异费用扣减
    List<String> planCodeList = entities.stream().map(AuditFeeDiffTrack::getPlanCode).distinct().collect(Collectors.toList());
    auditFeeDiffTrackDetailVoService.returnMonthBudgetByPlanCodeList(planCodeList,operateId);

    // 推送牛人管家
    diffTrackPushCowManager.closePush(planCodeList);
  }

  @Override
  public List<AuditFeeDiffTrackForecastDto> buildAuditFeeDiffTrackPlanParams(List<String> businessNoList) {
    List<AuditFeeDiffTrackForecastDto> list = new ArrayList<>();

    if(!CollectionUtils.isEmpty(businessNoList)) {
      List<AuditFeeDiffTrack> entityList = this.auditFeeDiffTrackRepository.lambdaQuery().in(AuditFeeDiffTrack::getPlanCode, businessNoList).list();

      List<AuditFeeDiffTrackDetailVo> detailList = auditFeeDiffTrackDetailVoService.findDetailByPlanCodeList(businessNoList);
      Map<String, List<AuditFeeDiffTrackDetailVo>> trackPlanDetailsMap = detailList.stream().collect(Collectors.groupingBy(AuditFeeDiffTrackDetailVo::getPlanCode));

      List<AuditFeeDiffTrackDetailLedgerVo> diffTrackDetailLedgerVoList = new ArrayList<>();
      for (AuditFeeDiffTrackDetailVo detailVo : detailList) {
        diffTrackDetailLedgerVoList.addAll(detailVo.getAuditFeeDiffTrackDetailLedgerList());
      }
      Map<String, List<AuditFeeDiffTrackDetailLedgerVo>> ledgerMap = new HashMap<>();
      if(CollectionUtils.isNotEmpty(diffTrackDetailLedgerVoList)) {
        ledgerMap = diffTrackDetailLedgerVoList.stream().collect(Collectors.groupingBy(AuditFeeDiffTrackDetailLedgerVo::getDetailCode));
      }

      for (AuditFeeDiffTrack entity : entityList) {
        String businessUnitCode = entity.getBusinessUnitCode();
        String businessFormatCode = entity.getBusinessFormatCode();
        AuditFeeDiffTrackForecastDto trackForecastDto = this.nebulaToolkitService.copyObjectByWhiteList(entity, AuditFeeDiffTrackForecastDto.class, HashSet.class, ArrayList.class);
        List<AuditFeeDiffTrackDetailVo> detailItemList = trackPlanDetailsMap.get(entity.getPlanCode());
        if(CollectionUtils.isNotEmpty(detailItemList)){
          List<AuditFeeDiffTrackDetailDto> detailItemDtoList = (List<AuditFeeDiffTrackDetailDto>)this.nebulaToolkitService.copyCollectionByBlankList(detailItemList, AuditFeeDiffTrackDetailVo.class, AuditFeeDiffTrackDetailDto.class, LinkedHashSet.class, ArrayList.class);
          for (AuditFeeDiffTrackDetailDto detailItemDto : detailItemDtoList) {
            detailItemDto.setBusinessUnitCode(businessUnitCode);
            detailItemDto.setBusinessFormatCode(businessFormatCode);
            List<AuditFeeDiffTrackDetailLedgerVo> ledgerVoList = ledgerMap.get(detailItemDto.getDetailCode());
            if(CollectionUtils.isNotEmpty(ledgerVoList)){
              Collection<AuditFeeDiffTrackDetailLedgerDto> activityPlanBudgetDtos = this.nebulaToolkitService.copyCollectionByBlankList(ledgerVoList, AuditFeeDiffTrackDetailLedgerVo.class, AuditFeeDiffTrackDetailLedgerDto.class, LinkedHashSet.class, ArrayList.class);
              detailItemDto.setAuditFeeDiffTrackDetailLedgerList((List<AuditFeeDiffTrackDetailLedgerDto>) activityPlanBudgetDtos);
            }
          }
          trackForecastDto.setItemList(detailItemDtoList);
        }
        list.add(trackForecastDto);
      }
    }
    return list;
  }

  @Override
  public List<AuditFeeDiffTrackSummaryVo> collectDiffAmount(AuditFeeDiffTrackDto dto) {
    log.info("汇总差异费用信息{}", dto);
    Validate.notNull(dto, "数据空");
    Validate.notNull(dto.getCacheKey(), "数据缓存KEY空");
//    List<String> templateCodeList = dto.getTemplateCodeList();
//    Validate.notEmpty(templateCodeList, "数据模板编码空");

    // 查询缓存中存在的数据
    List<AuditFeeDiffTrackDetailLedgerDto> cacheDiffList = new ArrayList<>();
    List<AuditFeeDiffTrackDetailDto> detailDtoList = auditFeeDiffTrackDetailVoService.findCacheList(dto.getCacheKey());
    if (CollectionUtils.isNotEmpty(detailDtoList)) {
      detailDtoList.forEach(e -> {
        if (CollectionUtils.isNotEmpty(e.getAuditFeeDiffTrackDetailLedgerList())) {
          cacheDiffList.addAll(e.getAuditFeeDiffTrackDetailLedgerList());
        }
      });
    }
    if (CollectionUtils.isEmpty(cacheDiffList)) {
      return new ArrayList<>(0);
    }
    Collection<AuditFeeDiffTrackDetailLedgerVo> diffVoList = this.nebulaToolkitService.copyCollectionByBlankList(cacheDiffList, AuditFeeDiffTrackDetailLedgerDto.class, AuditFeeDiffTrackDetailLedgerVo.class, LinkedHashSet.class, ArrayList.class);

    // 汇总数据
    Map<String, List<AuditFeeDiffTrackDetailLedgerVo>> diffCodeMap = diffVoList.stream().collect(Collectors.groupingBy(AuditFeeDiffTrackDetailLedgerVo::getFeeDiffLedgerCode));
    List<AuditFeeDiffTrackSummaryVo> diffTrackSummaryVos = new ArrayList<>(diffCodeMap.size());
    diffCodeMap.forEach((k,v) -> {
      AuditFeeDiffTrackSummaryVo trackSummaryVo = new AuditFeeDiffTrackSummaryVo();
      trackSummaryVo.setFeeDiffLedgerCode(k);
      AuditFeeDiffTrackDetailLedgerVo detailLedgerVo = v.get(0);
      trackSummaryVo.setYearMonth(detailLedgerVo.getFeeYearMonth());

      BigDecimal usedAmountTotal = BigDecimal.ZERO;
      for (AuditFeeDiffTrackDetailLedgerVo ledgerVo : v) {
        usedAmountTotal = usedAmountTotal.add(Optional.ofNullable(ledgerVo.getUsedAmount()).orElse(BigDecimal.ZERO));
      }
      trackSummaryVo.setBeRecoveredAmountTotal(detailLedgerVo.getBeRecoveredAmount());
      trackSummaryVo.setUsedAmountTotal(usedAmountTotal);
      diffTrackSummaryVos.add(trackSummaryVo);
    });
    return diffTrackSummaryVos;
  }

  @Override
  public BigDecimal collectFeeAmount(AuditFeeDiffTrackDto dto) {
    log.info("申请金额汇总参数{}", dto);
    Validate.notNull(dto, "数据空");
    Validate.notNull(dto.getCacheKey(), "数据缓存KEY空");
    // 查询缓存中存在的数据
    List<AuditFeeDiffTrackDetailDto> detailDtoList = auditFeeDiffTrackDetailVoService.findCacheList(dto.getCacheKey());
    BigDecimal feeAmountTotal = BigDecimal.ZERO;
    if (CollectionUtils.isNotEmpty(detailDtoList)) {
      for (AuditFeeDiffTrackDetailDto trackDetailDto : detailDtoList) {
        String feeAmount = trackDetailDto.getFeeAmountStr();
        feeAmountTotal = feeAmountTotal.add(StringUtils.isBlank(feeAmount) ? BigDecimal.ZERO : new BigDecimal(feeAmount));
      }
    }
    return feeAmountTotal;
  }
}
