package com.biz.crm.tpm.business.audit.fee.local.service.settlement.check.internal;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.kms.business.audit.fee.sdk.dto.AuditFeeMatchedDetailRespDto;
import com.biz.crm.kms.business.audit.fee.sdk.dto.AuditFeeMatchedRespDto;
import com.biz.crm.kms.business.audit.fee.sdk.dto.AuditFeePushMatchedKmsDataEventDto;
import com.biz.crm.kms.business.audit.fee.sdk.enums.DataSourceEnum;
import com.biz.crm.kms.business.audit.fee.sdk.event.PushMatchedKmsDataToStatementEvent;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.audit.fee.local.entity.settlement.check.AuditFeeSettlementCheckUpdateMatched;
import com.biz.crm.tpm.business.audit.fee.local.repository.settlement.check.AuditFeeSettlementCheckUpdateMatchedRepository;
import com.biz.crm.tpm.business.audit.fee.local.service.settlement.check.AuditFeeSettlementCheckService;
import com.biz.crm.tpm.business.audit.fee.sdk.enumeration.MatchStatusEnum;
import com.biz.crm.tpm.business.audit.fee.sdk.template.enums.PullKmsDataSourceEnum;
import com.biz.crm.tpm.business.audit.fee.sdk.vo.settlement.check.AuditFeeSettlementCheckFeeVo;
import com.biz.crm.tpm.business.audit.fee.sdk.vo.settlement.check.AuditFeeSettlementCheckSettlementVo;
import com.biz.crm.tpm.business.audit.fee.sdk.vo.settlement.check.AuditFeeSettlementCheckVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * kms推送数据事件实现
 * @Description
 * @Author zhouYang
 * @Date 2023/10/12
 */
@Slf4j
@Component
public class PushMatchedKmsDataToStatementEventImpl implements PushMatchedKmsDataToStatementEvent {

    @Autowired
    private AuditFeeSettlementCheckService service;

    @Autowired(required = false)
    private AuditFeeSettlementCheckUpdateMatchedRepository updateMatchedRepository;

    @Override
    public void pushMatchedKmsData(AuditFeePushMatchedKmsDataEventDto dto) {
        log.info("kms推送结算单数据事件,接受参数：{}", JSONObject.toJSONString(dto));
        if (Objects.isNull(dto)) {
            return;
        }
        if (!dto.isSuccess()) {
            log.info("kms推送数据错误：{}", dto.getMsg());
            return;
        }
        List<AuditFeeSettlementCheckUpdateMatched> updateMatchedList = new ArrayList<>();
        List<AuditFeeSettlementCheckVo> list = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(dto.getResultList())) {
            dto.getResultList().forEach(e -> {
                AuditFeeSettlementCheckVo vo = this.buildCheckData(e);
                list.add(vo);
                List<AuditFeeSettlementCheckUpdateMatched> updateMatcheds = this.buildUpdateMatched(e.getDataSource(),e.getDetailList());
                if (CollectionUtils.isNotEmpty(updateMatcheds)) {
                    updateMatchedList.addAll(updateMatcheds);
                }
            });
        }
        log.info("kms推送结算单数据事件,数据整理完成：{}", JSONObject.toJSONString(list));
        // 有确认数据时，同一维度拉取到新的结算单，如果有增量，形成新的一条结算核对数据，费用单数据只能关联新拉取到的费用单
        this.service.saveBatchForSettlementAndFee(list);
        if (CollectionUtils.isNotEmpty(updateMatchedList)) {
            updateMatchedRepository.saveBatch(updateMatchedList);
        }
    }

    private List<AuditFeeSettlementCheckUpdateMatched> buildUpdateMatched(String dataSource, List<AuditFeeMatchedDetailRespDto> detailList) {
        if (CollectionUtils.isEmpty(detailList) || StringUtils.isBlank(dataSource)) {
            return Lists.newArrayList();
        }
        List<AuditFeeSettlementCheckUpdateMatched> list = new ArrayList<>();
        detailList.forEach(d -> {
            DataSourceEnum datasource = DataSourceEnum.getTypeByCode(d.getDataSource());
            if (Objects.isNull(d)) {
                return;
            }
            AuditFeeSettlementCheckUpdateMatched updateMatched = new AuditFeeSettlementCheckUpdateMatched();

            switch (datasource) {
                case EXPENSE:
                    //费用明细
                    if (StringUtils.isNotBlank(d.getInvoiceExpenseSheetCode())) {
                        if (StringUtils.isBlank(d.getItemIndex())){
                            updateMatched.setTpmDeductionCode(d.getInvoiceExpenseSheetCode());
                        }else {
                            updateMatched.setTpmDeductionCode(d.getInvoiceExpenseSheetCode() + "-" + d.getItemIndex());
                        }
                    }

                case ACCEPTANCE:
                    //验收明细
                    if (StringUtils.isNotBlank(d.getOrderNumber())) {
                        if (StringUtils.isBlank(d.getItemIndex())){
                            updateMatched.setTpmDeductionCode(d.getOrderNumber());
                        }else {
                            updateMatched.setTpmDeductionCode(d.getOrderNumber() + "-" + d.getItemIndex());
                        }
                    }

                case RETURN:
                    //退货明细
                    if (StringUtils.isNotBlank(d.getOrderNumber())) {
                        if (StringUtils.isBlank(d.getItemIndex())){
                            updateMatched.setTpmDeductionCode(d.getOrderNumber());
                        }else {
                            updateMatched.setTpmDeductionCode(d.getOrderNumber() + "-" + d.getItemIndex());
                        }
                    }
                case STATEMENT_RETURN:
                    //结算单退货明细
                    if (StringUtils.isNotBlank(d.getStatementCode())) {
                        if (StringUtils.isBlank(d.getDocumentCode())){
                            updateMatched.setTpmDeductionCode(d.getStatementCode());
                        }else {
                            updateMatched.setTpmDeductionCode(d.getStatementCode() + "-" + d.getDocumentCode());
                        }
                    }
                case STATEMENT_ACCEPTANCE:
                    //结算单验收明细
                    if (StringUtils.isNotBlank(d.getStatementCode())) {
                        if (StringUtils.isBlank(d.getDocumentCode())){
                            updateMatched.setTpmDeductionCode(d.getStatementCode());

                        }else {
                            updateMatched.setTpmDeductionCode(d.getStatementCode() + "-" + d.getDocumentCode());
                        }
                    }
                case STATEMENT_FEE:
                    //结算单扣费明细
                    if (StringUtils.isNotBlank(d.getStatementCode())) {
                        if (StringUtils.isBlank(d.getDocumentCode())){
                            updateMatched.setTpmDeductionCode(d.getStatementCode());

                        }else {
                            updateMatched.setTpmDeductionCode(d.getStatementCode() + "-" + d.getDocumentCode());
                        }
                    }
                default:
            }
            if (StringUtils.isBlank(updateMatched.getTpmDeductionCode())) {
                return;
            }
            updateMatched.setDataSource(dataSource);
            updateMatched.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            updateMatched.setTenantCode(TenantUtils.getTenantCode());
            list.add(updateMatched);

        });
        return list;
    }

    @Override
    public void pushStatementMatchedFeeResult(AuditFeePushMatchedKmsDataEventDto dto) {
        // 推送来的即是匹配上的费用单
        log.info("kms推送费用单数据事件,接受参数：{}", JSONObject.toJSONString(dto));
        if (Objects.isNull(dto)) {
            return;
        }
        if (!dto.isSuccess()) {
            log.info("kms推送数据错误：{}", dto.getMsg());
            return;
        }
        List<AuditFeeSettlementCheckVo> list = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(dto.getResultList())) {
            dto.getResultList().forEach(e -> {
                AuditFeeSettlementCheckVo vo = this.buildDataForFee(e);
                list.add(vo);
            });
        }
        // 有确认数据时，同一维度拉取到新的费用单，如果有新的结算单，形成新的结算核对数据A，将新费用单与A关联；如未形成新结算核对数据，放弃这部分新的费用单
        this.service.saveBatchForFee(list);
    }

    /**
     * 组装头表信息
     * @param respDto
     * @return
     */
    private AuditFeeSettlementCheckVo buildCheckData(AuditFeeMatchedRespDto respDto) {
        AuditFeeSettlementCheckVo vo = new AuditFeeSettlementCheckVo();
        vo.setBusinessFormatCode(respDto.getBusinessFormatCode());
        vo.setBusinessUnitCode(respDto.getBusinessUnitCode());
        vo.setMatchTemplateCode(respDto.getMatchTemplateCode());
        vo.setMatchTemplateName(respDto.getMatchTemplateName());
        vo.setSystemCode(respDto.getCustomerRetailerCode());
        vo.setSystemName(respDto.getCustomerRetailerName());
        vo.setCustomerCode(respDto.getSoldToPartyCode());
        vo.setCustomerName(respDto.getSoldToPartyName());
        vo.setRegion(respDto.getBusinessArea());
        // 结算单一定有结算单总金额
        vo.setSettlementAmount(Optional.ofNullable(respDto.getStatementAmountTotal()).orElse(BigDecimal.ZERO));
        vo.setFeeAmount(BigDecimal.ZERO);
        vo.setDetailPlanAmount(BigDecimal.ZERO);
        vo.setSettlementFeeDiff(BigDecimal.ZERO);
        vo.setSettlementDetailPlanDiff(BigDecimal.ZERO);
        vo.setDiffConfirmAmount(BigDecimal.ZERO);
        vo.setMd5UniqueKey(respDto.getOnlyKey());
        vo.setMatchActivity(Boolean.FALSE);
        vo.setMatchStatus(MatchStatusEnum.WAIT_MATCH.getCode());
        vo.setSalesOrgCode(respDto.getSalesOrgCode());
        vo.setSalesOrgName(respDto.getSalesOrgName());
        vo.setQueryStartDate(respDto.getOrderDateBegin());
        vo.setQueryEndDate(respDto.getOrderDateEnd());

        if (CollectionUtils.isNotEmpty(respDto.getDetailList())) {
            List<AuditFeeSettlementCheckSettlementVo> settlementVoList = Lists.newArrayList();
            if (Objects.nonNull(respDto.getStatementAmountTotal()) && respDto.getStatementAmountTotal().compareTo(BigDecimal.ZERO) != 0) {
                respDto.getDetailList().forEach(e -> {
                    if (Objects.nonNull(e.getAmount()) && e.getAmount().compareTo(BigDecimal.ZERO) != 0) {
                        AuditFeeSettlementCheckSettlementVo settlementVo = this.buildSettlementInfo(respDto,e);
                        if (settlementVo.getSettlementDetailCode() == null) log.info("推送结算数据缺失，数据：{}", JSONObject.toJSONString(e));
                        settlementVoList.add(settlementVo);
                    }
                });
            }
            vo.setSettlementList(settlementVoList);

            if (CollectionUtils.isEmpty(vo.getSettlementList())) vo.setRemark("推送数据中，无有效结算单明细");
            else {
                AuditFeeSettlementCheckSettlementVo settlementVo = vo.getSettlementList().get(0);
                vo.setDataSource(settlementVo.getDataSource());
                vo.setActualYearMonth(settlementVo.getActualYearMonth());
                vo.setOrderYearMonthStr(settlementVo.getOrderYearMonth());
                vo.setOrderDate(settlementVo.getOrderDate());
            }
        }
        return vo;
    }

    /**
     * 组装结算单信息
     * @param respDto
     * @param e
     * @return
     */
    private AuditFeeSettlementCheckSettlementVo buildSettlementInfo(AuditFeeMatchedRespDto respDto, AuditFeeMatchedDetailRespDto e) {
        AuditFeeSettlementCheckSettlementVo settlementVo = new AuditFeeSettlementCheckSettlementVo();
        // 不同单据，编码方式不同
        if (PullKmsDataSourceEnum.FY.getCode().equals(e.getDataSource())) {
            settlementVo.setSettlementDetailCode(e.getItemIndex() == null ? e.getInvoiceExpenseSheetCode() : e.getInvoiceExpenseSheetCode()+"-"+e.getItemIndex());
        }
        if (PullKmsDataSourceEnum.YS.getCode().equals(e.getDataSource())) {
            settlementVo.setSettlementDetailCode(e.getItemIndex() == null ? e.getOrderNumber() : e.getOrderNumber()+"-"+e.getItemIndex());
        }
        if (PullKmsDataSourceEnum.TH.getCode().equals(e.getDataSource())) {
            settlementVo.setSettlementDetailCode(e.getItemIndex() == null ? e.getOrderNumber() : e.getOrderNumber()+"-"+e.getItemIndex());
        }
        if (StringUtils.equalsAny(e.getDataSource(), PullKmsDataSourceEnum.JS_TH.getCode(), PullKmsDataSourceEnum.JS_YS.getCode(), PullKmsDataSourceEnum.JS_KF.getCode())) {
            settlementVo.setSettlementDetailCode(e.getDocumentCode() == null ? e.getStatementCode() : e.getStatementCode()+"-"+e.getDocumentCode());
        }
        settlementVo.setBusinessArea(e.getBusinessArea());
        settlementVo.setDirectCode(respDto.getCustomerRetailerCode());
        settlementVo.setSupermarketName(respDto.getCustomerRetailerName());
        settlementVo.setProvinceCode(e.getProvinceCode());
        settlementVo.setProvinceName(e.getProvinceName());
        settlementVo.setDeliveryPartyCode(respDto.getDeliveryPartyCode());
        settlementVo.setDeliveryPartyName(respDto.getDeliveryPartyName());
        settlementVo.setProductCode(e.getProductCode());
        settlementVo.setProductName(e.getProductName());
        settlementVo.setSlotDateName(e.getSlotDateName());
        settlementVo.setActualYearMonth(DateUtil.format(e.getCreateTime(), "yyyy-MM"));
        settlementVo.setOrderDate(e.getOrderDate());
        if (StringUtils.isNotEmpty(e.getOrderDate())) {
            String[] str = e.getOrderDate().split("-");
            if (str.length >= 2) {
                settlementVo.setOrderYearMonth(str[0]+"-"+str[1]);
            }
        }
        if (DataSourceEnum.EXPENSE.getCode().equals(e.getDataSource())) {
            if (StringUtils.isNotEmpty(e.getItemIndex())) {
                settlementVo.setDeductionCode(e.getInvoiceExpenseSheetCode()+"-"+e.getItemIndex());
            }
        } else {
            if (StringUtils.isNotEmpty(e.getDocumentCode())) {
                settlementVo.setDeductionCode(e.getStatementCode()+"-"+e.getDocumentCode());
            } else {
                settlementVo.setDeductionCode(e.getStatementCode());
            }
        }
        settlementVo.setDeductionName(e.getDeductionName());
        settlementVo.setAuditWay(e.getAuditWay());
        settlementVo.setCashingType(e.getCashingType());
        settlementVo.setAmount(e.getAmount());
        settlementVo.setAmountNot(e.getAmountNot());
        settlementVo.setDataSource(e.getDataSource());
        return settlementVo;
    }

    /**
     * 组装费用单信息
     * @param respDto
     * @return
     */
    private AuditFeeSettlementCheckVo buildDataForFee(AuditFeeMatchedRespDto respDto) {
      AuditFeeSettlementCheckVo vo = new AuditFeeSettlementCheckVo();
      vo.setMd5UniqueKey(respDto.getOnlyKey());
      vo.setFeeAmount(Optional.ofNullable(respDto.getFeeAmountTotal()).orElse(BigDecimal.ZERO));
      if (Objects.nonNull(respDto.getFeeAmountTotal()) && respDto.getFeeAmountTotal().compareTo(BigDecimal.ZERO) != 0) {
        if (CollectionUtils.isNotEmpty(respDto.getMatchedFeeList())) {
          List<AuditFeeSettlementCheckFeeVo> feeVoList = Lists.newArrayList();
          List<String> incompleteList = Lists.newArrayList();
          if (CollectionUtils.isNotEmpty(respDto.getMatchedFeeList())) {
            respDto.getMatchedFeeList().forEach(e -> {
              if (Objects.nonNull(e.getAmount()) && e.getAmount().compareTo(BigDecimal.ZERO) != 0) {
                AuditFeeSettlementCheckFeeVo feeVo = this.buildFeeInfo(respDto,e);
                if (feeVo.getFeeDetailCode() == null) incompleteList.add(e.getDataSource()+"_" + e.getStatementCode() + "_" + e.getOrderDate());
                else feeVoList.add(feeVo);
              }
            });
            if (CollectionUtils.isNotEmpty(incompleteList)) log.info("推送费用数据缺失，数据：{}", String.join(",", JSONObject.toJSONString(incompleteList)));
          }
          vo.setFeeList(feeVoList);
        }
      }
      return vo;
    }

    private AuditFeeSettlementCheckFeeVo buildFeeInfo(AuditFeeMatchedRespDto respDto, AuditFeeMatchedDetailRespDto e) {
        AuditFeeSettlementCheckFeeVo feeVo = new AuditFeeSettlementCheckFeeVo();
        // 不同单据，编码方式不同
        if (PullKmsDataSourceEnum.FY.getCode().equals(e.getDataSource())) {
            feeVo.setFeeDetailCode(e.getItemIndex() == null ? e.getInvoiceExpenseSheetCode() : e.getInvoiceExpenseSheetCode()+"-"+e.getItemIndex());
        }
        if (PullKmsDataSourceEnum.YS.getCode().equals(e.getDataSource())) {
            feeVo.setFeeDetailCode(e.getItemIndex() == null ? e.getOrderNumber() : e.getOrderNumber()+"-"+e.getItemIndex());
        }
        if (PullKmsDataSourceEnum.TH.getCode().equals(e.getDataSource())) {
            feeVo.setFeeDetailCode(e.getItemIndex() == null ? e.getOrderNumber() : e.getOrderNumber()+"-"+e.getItemIndex());
        }
        if (StringUtils.equalsAny(e.getDataSource(), PullKmsDataSourceEnum.JS_TH.getCode(), PullKmsDataSourceEnum.JS_YS.getCode(), PullKmsDataSourceEnum.JS_KF.getCode())) {
            feeVo.setFeeDetailCode(e.getDocumentCode() == null ? e.getStatementCode() : e.getStatementCode()+"-"+e.getDocumentCode());
        }
        feeVo.setBusinessArea(e.getBusinessArea());
        feeVo.setDirectCode(respDto.getCustomerRetailerCode());
        feeVo.setSupermarketName(respDto.getCustomerRetailerName());
        feeVo.setProvinceCode(e.getProvinceCode());
        feeVo.setProvinceName(e.getProvinceName());
        feeVo.setDeliveryPartyCode(respDto.getDeliveryPartyCode());
        feeVo.setDeliveryPartyName(respDto.getDeliveryPartyName());
        feeVo.setProductCode(e.getProductCode());
        feeVo.setProductName(e.getProductName());
        feeVo.setSlotDateName(e.getSlotDateName());
        feeVo.setActualYearMonth(DateUtil.format(e.getCreateTime(), "yyyy-MM"));
        feeVo.setOrderDate(e.getOrderDate());
        if (StringUtils.isNotEmpty(e.getOrderDate())) {
            String[] str = e.getOrderDate().split("-");
            if (str.length >= 2) {
                feeVo.setOrderYearMonth(str[0]+"-"+str[1]);
            }
        }
        if (StringUtils.isNotEmpty(e.getItemIndex())) {
            feeVo.setDeductionCode(e.getInvoiceExpenseSheetCode()+"-"+e.getItemIndex());
        }
        feeVo.setDeductionName(e.getDeductionName());
        feeVo.setAuditWay(e.getAuditWay());
        feeVo.setCashingType(e.getCashingType());
        feeVo.setAmount(e.getAmount());
        feeVo.setDataSource(e.getDataSource());
        return feeVo;
    }
}
