package com.biz.crm.tpm.business.audit.fee.local.service.third;

import com.alibaba.fastjson.JSON;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mn.third.system.cow.manager.sdk.dto.CowManagerCellActivityDto;
import com.biz.crm.mn.third.system.cow.manager.sdk.dto.CowManagerDisplayActivityDto;
import com.biz.crm.mn.third.system.cow.manager.sdk.dto.CowManagerDisplayActivityItemDto;
import com.biz.crm.mn.third.system.cow.manager.sdk.dto.CowManagerLongTermPersonnelActivityDto;
import com.biz.crm.mn.third.system.cow.manager.sdk.dto.CowManagerLongTermPersonnelActivityItemDto;
import com.biz.crm.mn.third.system.cow.manager.sdk.dto.CowManagerPromotionActivityDto;
import com.biz.crm.mn.third.system.cow.manager.sdk.dto.CowManagerPromotionActivityItemDto;
import com.biz.crm.mn.third.system.cow.manager.sdk.dto.CowManagerTemporaryPersonnelActivityDto;
import com.biz.crm.mn.third.system.cow.manager.sdk.dto.CowManagerTemporaryPersonnelActivityItemDto;
import com.biz.crm.mn.third.system.cow.manager.sdk.service.CowManagerService;
import com.biz.crm.mn.third.system.cow.manager.sdk.vo.CowManagerVo;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.enums.InterfacePushStateEnum;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.enums.VerticalActivityTypeEnum;
import com.biz.crm.tpm.business.activity.form.sdk.service.ActivityFormService;
import com.biz.crm.tpm.business.audit.fee.sdk.service.track.AuditFeeDiffTrackDetailVoService;
import com.biz.crm.tpm.business.audit.fee.sdk.vo.track.AuditFeeDiffTrackDetailVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

/**
 * 差异费用追踪推送牛人管家
 * @author liyang
 * @date 2023/11/13
 */
@Slf4j
@Component
public class DiffTrackPushCowManager {

  @Autowired(required = false)
  private AuditFeeDiffTrackDetailVoService auditFeeDiffTrackDetailVoService;

  @Autowired(required = false)
  private ActivityFormService activityFormService;

  @Autowired(required = false)
  private DictDataVoService dictDataVoService;


  /**
   * 审批通过推送牛人管家
   * @param planCodes 活动细案编码集合
   */
  public void passPush(List<String> planCodes) {
    Validate.notEmpty(planCodes,"传入编码不能为空！");
    //获取审批通过的活动细案的明细数据
    List<AuditFeeDiffTrackDetailVo> detailPlanItemVoList = auditFeeDiffTrackDetailVoService.findDetailByPlanCodeList(planCodes);
    if (CollectionUtils.isEmpty(detailPlanItemVoList)) {
      return;
    }
    this.buildAndPush(detailPlanItemVoList, false);
  }

  /**
   * 审批通过推送牛人管家
   * @param detailPlanItemVoList 活动细案编码集合
   */
  public void passItemPush(List<AuditFeeDiffTrackDetailVo> detailPlanItemVoList) {
    if (CollectionUtils.isEmpty(detailPlanItemVoList)) {
      return;
    }
    this.buildAndPush(detailPlanItemVoList, false);
  }

  /**
   * 关闭推送牛人管家
   * @param planCodes 活动编码集合
   */
  public void closePush(List<String> planCodes) {
    Validate.notEmpty(planCodes,"传入编码不能为空！");
    List<AuditFeeDiffTrackDetailVo> detailPlanItemVoList = auditFeeDiffTrackDetailVoService.findDetailByPlanCodeList(planCodes);
    if (CollectionUtils.isEmpty(detailPlanItemVoList)) {
      return;
    }
    this.buildAndPush(detailPlanItemVoList, true);
  }

  /**
   * 关闭推送牛人管家
   * @param detailPlanItemVoList 活动细案集合
   */
  public void closeItemPush(List<AuditFeeDiffTrackDetailVo> detailPlanItemVoList) {
    if (CollectionUtils.isEmpty(detailPlanItemVoList)) {
      return;
    }
    this.buildAndPush(detailPlanItemVoList, true);
  }

  private void buildAndPush(List<AuditFeeDiffTrackDetailVo> detailPlanItemVoList, boolean close) {
    // 需要套推送牛人管家的活动形式
    Map<String, String> formMap = activityFormService.findPushCowManager();

    log.info("差异费用追踪审批通过推送牛人管家_buildAndPush: {}", JSON.toJSONString(formMap));
    if (CollectionUtils.isEmpty(formMap)) {
      return;
    }

    // 过滤不需要推送牛人管家的明细
    Set<String> activityFormCodeSet = formMap.keySet();
    detailPlanItemVoList.removeIf(e -> !activityFormCodeSet.contains(e.getActivityFormCode()));

    // 区分推送的活动类型
    Map<String, List<AuditFeeDiffTrackDetailVo>> planItemMap = this.categorizeDetailPlanItem(formMap, detailPlanItemVoList);
    log.info("差异费用追踪审批通过推送牛人管家_分类完成");

    // 组装数据
    List<CowManagerCellActivityDto> marketingDataList = this.buildCellActivityInterfaceData(planItemMap.get(VerticalActivityTypeEnum.MARKETING.getCode()), close);
    List<CowManagerDisplayActivityDto> displayDataList = this.buildDisplayActivityInterfaceData(planItemMap.get(VerticalActivityTypeEnum.DISPLAY.getCode()), close);
    List<CowManagerPromotionActivityDto> promotionDataList = this.buildPromotionActivityInterfaceData(planItemMap.get(VerticalActivityTypeEnum.PROMOTION.getCode()), close);
    List<CowManagerTemporaryPersonnelActivityDto> temporaryDataList = this.buildTemporaryPersonnelActivity(planItemMap.get(VerticalActivityTypeEnum.TEMPORARY.getCode()), close);
    List<CowManagerLongTermPersonnelActivityDto> longPersonDataList = this.buildLongTermPersonnelActivity(planItemMap.get(VerticalActivityTypeEnum.LONG_PROMOTION.getCode()), close);
    // 推送
    List<List<String>> pair = this.pushData(marketingDataList, displayDataList, promotionDataList, temporaryDataList, longPersonDataList);

    List<String> successItemCodes = pair.get(0);
    List<String> failItemCodes = pair.get(1);
    log.info("推送完毕成功明细编码_{}", successItemCodes);
    log.info("推送完毕失败明细编码_{}", failItemCodes);

    // 打已推送牛人管家标记
    auditFeeDiffTrackDetailVoService.updateDetailPlanCowManagerState(successItemCodes, InterfacePushStateEnum.SUCCESS);
    auditFeeDiffTrackDetailVoService.updateDetailPlanCowManagerState(failItemCodes, InterfacePushStateEnum.FAIL);
  }

  /**
   * 将明细按垂直活动类型分类
   * @param fromMap 活动形式编码对应的垂直活动类型map
   * @param detailPlanItemVoList 明细集合
   */
  private Map<String,List<AuditFeeDiffTrackDetailVo>> categorizeDetailPlanItem(Map<String, String> fromMap, List<AuditFeeDiffTrackDetailVo> detailPlanItemVoList) {
    if (CollectionUtils.isEmpty(detailPlanItemVoList) || CollectionUtils.isEmpty(fromMap)){
      return Maps.newHashMap();
    }
    Map<String,List<AuditFeeDiffTrackDetailVo>> map = new HashMap<>(5);
    //推送行销活动接口的明细集合
    map.put(VerticalActivityTypeEnum.MARKETING.getCode(),new ArrayList<>());
    //推送陈列接口的明细集合
    map.put(VerticalActivityTypeEnum.DISPLAY.getCode(),new ArrayList<>());
    //推送促销接口的明细集合
    map.put(VerticalActivityTypeEnum.PROMOTION.getCode(),new ArrayList<>());
    //推送临促人员活动的明细集合
    map.put(VerticalActivityTypeEnum.TEMPORARY.getCode(),new ArrayList<>());
    //推送长促人员活动的明细集合
    map.put(VerticalActivityTypeEnum.LONG_PROMOTION.getCode(),new ArrayList<>());

    detailPlanItemVoList.forEach(detailPlanItem -> {
      String activityType = fromMap.get(detailPlanItem.getActivityFormCode());
      if(StringUtil.isNotBlank(activityType)) {
        if(VerticalActivityTypeEnum.MARKETING.getCode().equals(activityType)){
          map.get(VerticalActivityTypeEnum.MARKETING.getCode()).add(detailPlanItem);

        }else if(VerticalActivityTypeEnum.DISPLAY.getCode().equals(activityType)){
          map.get(VerticalActivityTypeEnum.DISPLAY.getCode()).add(detailPlanItem);

        }else if(VerticalActivityTypeEnum.PROMOTION.getCode().equals(activityType)){
          map.get(VerticalActivityTypeEnum.PROMOTION.getCode()).add(detailPlanItem);

        }else if(VerticalActivityTypeEnum.TEMPORARY.getCode().equals(activityType)){
          map.get(VerticalActivityTypeEnum.TEMPORARY.getCode()).add(detailPlanItem);

        }else if(VerticalActivityTypeEnum.LONG_PROMOTION.getCode().equals(activityType)){
          map.get(VerticalActivityTypeEnum.LONG_PROMOTION.getCode()).add(detailPlanItem);
        }
      }
    });
    return map;
  }

  @Autowired(required = false)
  private ProductVoService productVoService;

  @Autowired(required = false)
  private CowManagerService cowManagerService;

  /**
   * 组装行销活动接口数据
   * @param ItemList 明细集合
   * @param close 是否关闭 true：关闭
   * @return 待推送行销活动接口数据集合
   */
  public List<CowManagerCellActivityDto> buildCellActivityInterfaceData(List<AuditFeeDiffTrackDetailVo> ItemList,boolean close) {
    if(CollectionUtils.isEmpty(ItemList)){
      return Lists.newArrayListWithCapacity(0);
    }
    List<CowManagerCellActivityDto> cowManagerCellActivityDtoList = new ArrayList<>();
    SimpleDateFormat sif = new SimpleDateFormat("yyyy-MM");
    SimpleDateFormat sif2 = new SimpleDateFormat("yyyy-MM-dd");
    ItemList.forEach(item -> {
      CowManagerCellActivityDto cowManagerCellActivity = new CowManagerCellActivityDto();
      if(close){
        cowManagerCellActivity.setActivityStatus("已关闭");
        Date currentDate = new Date();
        cowManagerCellActivity.setActivityEndTime(sif2.format(currentDate));
      }else {
        cowManagerCellActivity.setActivityStatus("审批通过");
        cowManagerCellActivity.setActivityEndTime(item.getActivityEndDate() != null ? sif2.format(item.getActivityEndDate()) : "");
      }
      cowManagerCellActivity.setActivityNo(item.getPlanCode());
      cowManagerCellActivity.setActiveNumber(item.getDetailCode());
      cowManagerCellActivity.setCostYearMonth(item.getFeeYearMonth() != null ? sif.format(item.getFeeYearMonth()) : "");
      cowManagerCellActivity.setRegionInfo(item.getRegion());
      cowManagerCellActivity.setSystemInfo(item.getSystemCode());
      cowManagerCellActivity.setStoreId(item.getTerminalCode());
      cowManagerCellActivity.setStoreName(item.getTerminalName());
      cowManagerCellActivity.setChannelFirst(item.getFirstChannelName());
      cowManagerCellActivity.setChannelSecond(item.getSecondChannelName());
      cowManagerCellActivity.setActivityBrand(item.getProductBrandName());
      cowManagerCellActivity.setActivityStartTime(item.getActivityBeginDate() != null ? sif2.format(item.getActivityBeginDate()) : "");
      cowManagerCellActivity.setActivityTypeCode(item.getActivityTypeCode());
      cowManagerCellActivity.setActivityTypeName(item.getActivityTypeName());
      cowManagerCellActivity.setActivityCode(item.getActivityFormCode());
      cowManagerCellActivity.setActivityName(item.getActivityFormName());
      cowManagerCellActivity.setActivityDesc(item.getActivityDesc());
      cowManagerCellActivity.setActivityIndicators(item.getQuantity() != null ? item.getQuantity().toString() : "");//关键指标
      cowManagerCellActivity.setApplyTotalCost(item.getTotalFeeAmount() != null ? String.format("%.2f", item.getTotalFeeAmount()) : "0.00");
      cowManagerCellActivityDtoList.add(cowManagerCellActivity);
    });
    return cowManagerCellActivityDtoList;
  }

  /**
   * 组装陈列活动接口数据
   * @param ItemList 明细集合
   * @param close 是否关闭 true：关闭
   * @return  待推送的陈列活动接口数据集合
   */
  public List<CowManagerDisplayActivityDto> buildDisplayActivityInterfaceData(List<AuditFeeDiffTrackDetailVo> ItemList,boolean close) {
    if(CollectionUtils.isEmpty(ItemList)){
      return Lists.newArrayListWithCapacity(0);
    }
    final String BUY_WAY = "Purchase method";//核销方式
    List<DictDataVo> dictList = dictDataVoService.findByDictTypeCode(BUY_WAY);
    log.info("推送牛人管家购买方式字典:{}", dictList);
    Map<String, String> dictMap = dictList.stream().collect(Collectors.toMap(DictDataVo::getDictCode, DictDataVo::getDictValue));

    List<CowManagerDisplayActivityDto> cowManagerDisplayActivityDtoList = new ArrayList<>();
    SimpleDateFormat sif = new SimpleDateFormat("yyyy-MM");
    SimpleDateFormat sif2 = new SimpleDateFormat("yyyy-MM-dd");
    ItemList.forEach(detailPlanItem -> {
      CowManagerDisplayActivityDto cowManagerDisplayActivity = new CowManagerDisplayActivityDto();
      if(close){
        cowManagerDisplayActivity.setActivityStatus("已关闭");
        Date currentDate = new Date();
        cowManagerDisplayActivity.setActivityEndTime(sif2.format(currentDate));
      }else {
        cowManagerDisplayActivity.setActivityStatus("审批通过");
        cowManagerDisplayActivity.setActivityEndTime(detailPlanItem.getActivityEndDate() != null ? sif2.format(detailPlanItem.getActivityEndDate()) : "");
      }
      cowManagerDisplayActivity.setActivityNo(detailPlanItem.getPlanCode());
      cowManagerDisplayActivity.setActiveNumber(detailPlanItem.getDetailCode());
      cowManagerDisplayActivity.setCostYearMonth(detailPlanItem.getFeeYearMonth() != null ? sif.format(detailPlanItem.getFeeYearMonth()) : "");
      cowManagerDisplayActivity.setActivityStartTime(detailPlanItem.getActivityBeginDate() != null ? sif2.format(detailPlanItem.getActivityBeginDate()) : "");
      cowManagerDisplayActivity.setActivityTypeCode(detailPlanItem.getActivityTypeCode());
      cowManagerDisplayActivity.setActivityTypeName(detailPlanItem.getActivityTypeName());
      cowManagerDisplayActivity.setActivityCode(detailPlanItem.getActivityFormCode());
      cowManagerDisplayActivity.setActivityName(detailPlanItem.getActivityFormName());
      cowManagerDisplayActivity.setStoreId(detailPlanItem.getTerminalCode());
      cowManagerDisplayActivity.setStoreName(detailPlanItem.getTerminalName());
      cowManagerDisplayActivity.setActivityDesc(detailPlanItem.getActivityDesc());
      cowManagerDisplayActivity.setProcurementMethod(dictMap.getOrDefault(detailPlanItem.getBuyWay(), null));
      cowManagerDisplayActivity.setScheduleStartTime(detailPlanItem.getScheduleBeginDate() != null ? sif2.format(detailPlanItem.getScheduleBeginDate()) : "");
      cowManagerDisplayActivity.setScheduleEndTime(detailPlanItem.getScheduleEndDate() != null ? sif2.format(detailPlanItem.getScheduleEndDate()) : "");
      cowManagerDisplayActivity.setDisplayApplyNum(detailPlanItem.getDisplayQuantity() != null ? String.valueOf(detailPlanItem.getDisplayQuantity()) : "0.00");//陈列申请数量
      cowManagerDisplayActivity.setDisplayApplyCost(detailPlanItem.getTotalFeeAmount() != null ? String.format("%.2f", detailPlanItem.getTotalFeeAmount()) : "0.00");//陈列申请费用
//      if(!com.aliyun.tea.utils.StringUtils.isEmpty(detailPlanItem.getProductCode())){
        List<CowManagerDisplayActivityItemDto> cowManagerDisplayActivityItemList = new ArrayList<>();
        cowManagerDisplayActivity.setProductLists(cowManagerDisplayActivityItemList);
        CowManagerDisplayActivityItemDto cowManagerDisplayActivityItem = new CowManagerDisplayActivityItemDto();
        cowManagerDisplayActivityItem.setBranCode(detailPlanItem.getProductBrandCode());
        cowManagerDisplayActivityItem.setBranName(detailPlanItem.getProductBrandName());
        cowManagerDisplayActivityItem.setBrandItemCode(detailPlanItem.getProductItemCode());
        cowManagerDisplayActivityItem.setBrandItemName(detailPlanItem.getProductItemName());
        cowManagerDisplayActivityItem.setBrandTypeCode(detailPlanItem.getProductCategoryCode());
        cowManagerDisplayActivityItem.setBrandTypeName(detailPlanItem.getProductCategoryName());
        cowManagerDisplayActivityItem.setProductNo(detailPlanItem.getProductCode());
        cowManagerDisplayActivityItem.setProductName(detailPlanItem.getProductName());
        cowManagerDisplayActivityItemList.add(cowManagerDisplayActivityItem);

//      };
      cowManagerDisplayActivityDtoList.add(cowManagerDisplayActivity);

    });
    return cowManagerDisplayActivityDtoList;
  }


  /**
   * 组装促销活动接口数据
   * @param itemList 明细集合
   * @param close 是否关闭 true：关闭
   * @return 待推送的促销活动接口数据
   */
  public List<CowManagerPromotionActivityDto> buildPromotionActivityInterfaceData(List<AuditFeeDiffTrackDetailVo> itemList,boolean close) {
    if(CollectionUtils.isEmpty(itemList)){
      return Lists.newArrayListWithCapacity(0);
    }
    Set<String> productCodes = itemList.stream().map(AuditFeeDiffTrackDetailVo::getProductCode).filter(
        Objects::nonNull).collect(
        Collectors.toSet());
    Set<String> giftCodes = itemList.stream().map(AuditFeeDiffTrackDetailVo::getGiftCode).filter(Objects::nonNull).collect(Collectors.toSet());
    productCodes.addAll(giftCodes);
    ArrayList<String> productCodeList = new ArrayList<>(productCodes);
    //根据产品赠品编码从产品主数据中查询产品详细信息
    List<ProductVo> productMainList = productVoService.findDetailsByIdsOrProductCodes(null, productCodeList);
    List<CowManagerPromotionActivityDto> cowManagerPromotionActivityDtoList = new ArrayList<>();
    SimpleDateFormat sif = new SimpleDateFormat("yyyy-MM");
    SimpleDateFormat sif2 = new SimpleDateFormat("yyyy-MM-dd");
    itemList.forEach(item -> {
      CowManagerPromotionActivityDto cowManagerPromotionActivity = new CowManagerPromotionActivityDto();
      if(close){
        cowManagerPromotionActivity.setActivityStatus("已关闭");
        Date currentDate = new Date();
        cowManagerPromotionActivity.setActivityEndTime(sif2.format(currentDate));
      }else {
        cowManagerPromotionActivity.setActivityStatus("审批通过");
        cowManagerPromotionActivity.setActivityEndTime(item.getActivityEndDate() != null ? sif2.format(item.getActivityEndDate()) : "");
      }
      cowManagerPromotionActivity.setActivityNo(item.getPlanCode());
      cowManagerPromotionActivity.setActiveNumber(item.getDetailCode());
      cowManagerPromotionActivity.setCostYearMonth(item.getFeeYearMonth() != null ? sif.format(item.getFeeYearMonth()) : "");
      cowManagerPromotionActivity.setRegionInfo(item.getRegion());
      cowManagerPromotionActivity.setSystemInfo(item.getSystemCode());
      cowManagerPromotionActivity.setStoreId(item.getTerminalCode());
      cowManagerPromotionActivity.setStoreName(item.getTerminalName());
//            cowManagerPromotionActivity.setStoreType(item.getTerminalType());
      cowManagerPromotionActivity.setChannelFirst(item.getFirstChannelName());
      cowManagerPromotionActivity.setChannelSecond(item.getSecondChannelName());
      cowManagerPromotionActivity.setActivityStartTime(item.getActivityBeginDate() != null ? sif2.format(item.getActivityBeginDate()) : "");
      cowManagerPromotionActivity.setActivityTypeCode(item.getActivityTypeCode());
      cowManagerPromotionActivity.setActivityTypeName(item.getActivityTypeName());
      cowManagerPromotionActivity.setActivityCode(item.getActivityFormCode());
      cowManagerPromotionActivity.setActivityName(item.getActivityFormName());
      cowManagerPromotionActivity.setActivityDesc(item.getActivityDesc());
      cowManagerPromotionActivity.setOriginalPrice(item.getRetailPrice() != null ? String.format("%.2f", item.getRetailPrice()) : "0.00");
      cowManagerPromotionActivity.setPromotionalPrice(item.getPromotionalPrice() != null ? String.format("%.2f", item.getPromotionalPrice()) : "0.00");
      if(!com.aliyun.tea.utils.StringUtils.isEmpty(item.getProductCode())){
        List<CowManagerPromotionActivityItemDto> promotionActivityItemDtoList = new ArrayList<>();
        cowManagerPromotionActivity.setSalesPromotionProductList(promotionActivityItemDtoList);
        ProductVo productVo = productMainList.stream().filter(k -> k.getProductCode().equals(item.getProductCode())).findAny().orElse(null);
        CowManagerPromotionActivityItemDto productDto = new CowManagerPromotionActivityItemDto();
        productDto.setProductType("BP");
        productDto.setProductTypeName("本品");
        productDto.setActivityBrand(productVo.getProductBrandName());
                /* 1031007
                productDto.setPackageNo(productVo.getBarCode());
                 */
        productDto.setPackageNo(productVo.getCartonBarCode());
        productDto.setSapCode(productVo.getProductCode());
        productDto.setSapName(productVo.getProductName());
        productDto.setProdDate(item.getThisProductProductionDate() != null ? sif2.format(item.getThisProductProductionDate()) : null);
        promotionActivityItemDtoList.add(productDto);
        if(!com.aliyun.tea.utils.StringUtils.isEmpty(item.getGiftCode())){
          ProductVo giftVo = productMainList.stream().filter(k -> k.getProductCode().equals(item.getGiftCode())).findAny().orElse(null);
          CowManagerPromotionActivityItemDto giftDto = new CowManagerPromotionActivityItemDto();
          giftDto.setProductType("ZP");
          giftDto.setProductTypeName("赠品");
          giftDto.setActivityBrand(giftVo.getProductBrandName());
          /*
           *giftDto.setPackageNo(giftVo.getBarCode());
           */
          giftDto.setPackageNo(giftVo.getCartonBarCode());
          giftDto.setSapCode(giftVo.getProductCode());
          giftDto.setSapName(giftVo.getProductName());
          giftDto.setProdDate(item.getGiftProductionDate() != null ? sif2.format(item.getGiftProductionDate()) : null);
          promotionActivityItemDtoList.add(giftDto);
        }
      }
      cowManagerPromotionActivityDtoList.add(cowManagerPromotionActivity);
    });
    return cowManagerPromotionActivityDtoList;
  }

  /**
   * 组装临促人员活动接口数据
   * @param itemList 明细集合
   * @param close 是否关闭 true：关闭
   * @return 待推送的临促人员活动接口数据集合
   */
  public List<CowManagerTemporaryPersonnelActivityDto> buildTemporaryPersonnelActivity(List<AuditFeeDiffTrackDetailVo> itemList,boolean close) {
    if(CollectionUtils.isEmpty(itemList)){
      return Lists.newArrayListWithCapacity(0);
    }
    List<CowManagerTemporaryPersonnelActivityDto> temporaryPersonnelActivityDtoList = new ArrayList<>();
    SimpleDateFormat sif = new SimpleDateFormat("yyyy-MM");
    SimpleDateFormat sif2 = new SimpleDateFormat("yyyy-MM-dd");
    itemList.forEach(item -> {
      CowManagerTemporaryPersonnelActivityDto temporaryPersonnelActivity = new CowManagerTemporaryPersonnelActivityDto();
      if(close){
        temporaryPersonnelActivity.setActivityStatus("已关闭");
        Date currentDate = new Date();
        temporaryPersonnelActivity.setActivityEndTime(sif2.format(currentDate));
      }else {
        temporaryPersonnelActivity.setActivityStatus("审批通过");
        temporaryPersonnelActivity.setActivityEndTime(item.getActivityEndDate() != null ? sif2.format(item.getActivityEndDate()) : "");
      }
      temporaryPersonnelActivity.setActivityNo(item.getPlanCode());
      temporaryPersonnelActivity.setActiveNumber(item.getDetailCode());
      temporaryPersonnelActivity.setCostYearMonth(item.getFeeYearMonth() != null ? sif.format(item.getFeeYearMonth()) : "");
      temporaryPersonnelActivity.setSystemInfo(item.getSystemCode());
      temporaryPersonnelActivity.setStoreId(item.getTerminalCode());
      temporaryPersonnelActivity.setStoreName(item.getTerminalName());
      temporaryPersonnelActivity.setActivityStartTime(item.getActivityBeginDate() != null ? sif2.format(item.getActivityBeginDate()) : "");
      temporaryPersonnelActivity.setActivityTypeCode(item.getActivityTypeCode());
      temporaryPersonnelActivity.setActivityTypeName(item.getActivityTypeName());
      temporaryPersonnelActivity.setActivityCode(item.getActivityFormCode());
      temporaryPersonnelActivity.setActivityName(item.getProductCategoryName());
      temporaryPersonnelActivity.setUserNum(item.getQuantity() != null ? item.getQuantity().toString() : "0.00");
      temporaryPersonnelActivity.setActivityDesc(item.getActivityDesc());
      if(!com.aliyun.tea.utils.StringUtils.isEmpty(item.getProductCode())){
        List<CowManagerTemporaryPersonnelActivityItemDto> productLists = new ArrayList<>();
        temporaryPersonnelActivity.setProductLists(productLists);
        CowManagerTemporaryPersonnelActivityItemDto tpProductDto = new CowManagerTemporaryPersonnelActivityItemDto();
        tpProductDto.setBranCode(item.getProductBrandCode());
        tpProductDto.setBranName(item.getProductBrandName());
        tpProductDto.setBrandItemCode(item.getProductItemCode());
        tpProductDto.setBrandItemName(item.getProductItemName());
        tpProductDto.setBrandTypeCode(item.getProductCategoryCode());
        tpProductDto.setBrandTypeName(item.getProductCategoryName());
        tpProductDto.setProductNo(item.getProductCode());
        tpProductDto.setProductName(item.getProductCode());
        productLists.add(tpProductDto);

      };
      temporaryPersonnelActivityDtoList.add(temporaryPersonnelActivity);
    });
    return temporaryPersonnelActivityDtoList;
  }

  /**
   * 组装长促人员活动接口数据
   * @param itemList 明细集合
   * @param close 是否关闭 true：关闭
   * @return  待推送的长促人员活动接口数据集合
   */
  public List<CowManagerLongTermPersonnelActivityDto> buildLongTermPersonnelActivity(List<AuditFeeDiffTrackDetailVo> itemList,boolean close) {
    if(CollectionUtils.isEmpty(itemList)){
      return Lists.newArrayListWithCapacity(0);
    }
    List<CowManagerLongTermPersonnelActivityDto> longTermPersonnelActivityDtoList = new ArrayList<>();
    SimpleDateFormat sif = new SimpleDateFormat("yyyy-MM");
    SimpleDateFormat sif2 = new SimpleDateFormat("yyyy-MM-dd");
    itemList.forEach( item -> {
      CowManagerLongTermPersonnelActivityDto longTermPersonnelActivity = new CowManagerLongTermPersonnelActivityDto();
      if(close){
        longTermPersonnelActivity.setActivityStatus("已关闭");
        Date currentDate = new Date();
        longTermPersonnelActivity.setActivityEndTime(sif2.format(currentDate));
      }else {
        longTermPersonnelActivity.setActivityStatus("审批通过");
        longTermPersonnelActivity.setActivityEndTime(item.getActivityEndDate() != null ? sif2.format(item.getActivityEndDate()) : "");
      }
      longTermPersonnelActivity.setActivityNo(item.getPlanCode());
      longTermPersonnelActivity.setActiveNumber(item.getDetailCode());
      longTermPersonnelActivity.setCostYearMonth(item.getFeeYearMonth() != null ? sif.format(item.getFeeYearMonth()) : "");
      longTermPersonnelActivity.setRegionInfo(item.getRegion());
      longTermPersonnelActivity.setSystemInfo(item.getSystemCode());
      longTermPersonnelActivity.setStoreId(item.getTerminalCode());
      longTermPersonnelActivity.setStoreName(item.getTerminalName());
      longTermPersonnelActivity.setActivityStartTime(item.getActivityBeginDate() != null ? sif2.format(item.getActivityBeginDate()) : "");
      longTermPersonnelActivity.setActivityTypeCode(item.getActivityTypeCode());
      longTermPersonnelActivity.setActivityTypeName(item.getActivityTypeName());
      longTermPersonnelActivity.setActivityCode(item.getActivityFormCode());
      longTermPersonnelActivity.setActivityName(item.getProductCategoryName());
      longTermPersonnelActivity.setUserName(item.getNameOfShoppingGuide());
      longTermPersonnelActivity.setUserId(item.getEmployeeId());
      longTermPersonnelActivity.setIdNumber(item.getPersonIdCard());
      longTermPersonnelActivity.setPhone(item.getTelephone());
      longTermPersonnelActivity.setActivityDesc(item.getActivityDesc());
      if(!com.aliyun.tea.utils.StringUtils.isEmpty(item.getProductCode())){
        List<CowManagerLongTermPersonnelActivityItemDto> productLists = new ArrayList<>();
        longTermPersonnelActivity.setProductLists(productLists);
        CowManagerLongTermPersonnelActivityItemDto ltProductDto = new CowManagerLongTermPersonnelActivityItemDto();
        ltProductDto.setBranCode(item.getProductBrandCode());
        ltProductDto.setBranName(item.getProductBrandName());
        ltProductDto.setBrandItemCode(item.getProductItemCode());
        ltProductDto.setBrandItemName(item.getProductItemName());
        ltProductDto.setBrandTypeCode(item.getProductCategoryCode());
        ltProductDto.setBrandTypeName(item.getProductCategoryName());
        ltProductDto.setProductNo(item.getProductCode());
        ltProductDto.setProductName(item.getProductName());
        productLists.add(ltProductDto);
      }
      longTermPersonnelActivityDtoList.add(longTermPersonnelActivity);
    });
    return longTermPersonnelActivityDtoList;
  }



  /**
   * 推送活动数据至牛人管家
   * @param marketingDataList 行销活动集合
   * @param displayDataList 陈列活动集合
   * @param promotionDataList 促销活动集合
   * @param temporaryDataList 临促活动集合
   * @param longPersonDataList 长促活动集合
   * @return 推送成功的明细编码
   */
  private List<List<String>> pushData(List<CowManagerCellActivityDto> marketingDataList,
      List<CowManagerDisplayActivityDto> displayDataList,
      List<CowManagerPromotionActivityDto> promotionDataList,
      List<CowManagerTemporaryPersonnelActivityDto> temporaryDataList,
      List<CowManagerLongTermPersonnelActivityDto> longPersonDataList) {
    log.info("marketingDataList_{}_displayDataList_{}_promotionDataList_{}_temporaryDataList_{}_longPersonDataList_{}", marketingDataList.size(), displayDataList.size(), promotionDataList.size(), temporaryDataList.size(), longPersonDataList.size());
    log.info("差异费用追踪审批通过推送牛人管家_开始推送");

    List<String> successCodes = new ArrayList<>();
    List<String> failCodes = new ArrayList<>();
    //行销活动
    if(!CollectionUtils.isEmpty(marketingDataList)){
      CowManagerVo mar = this.pushCellActivity(marketingDataList);
      List<String> itemCodes = marketingDataList.stream().map(CowManagerCellActivityDto::getActiveNumber).filter(Objects::nonNull).collect(Collectors.toList());
      try {
        if(mar.isSuccess() && (boolean)mar.getData()){
          successCodes.addAll(itemCodes);
        } else throw new IllegalArgumentException("返回数据异常");
      } catch (Exception e) {
        log.warn("行销活动推送失败! 返回数据：{}",JSON.toJSONString(mar));
        log.warn("行销活动推送失败! cause by：{}",e.getMessage());
        failCodes.addAll(itemCodes);
      }
    }
    //陈列活动
    if(!CollectionUtils.isEmpty(displayDataList)){
      CowManagerVo dis = this.pushDisplayActivity(displayDataList);
      List<String> itemCodes = displayDataList.stream().map(CowManagerDisplayActivityDto::getActiveNumber).filter(Objects::nonNull).collect(Collectors.toList());
      try {
        if(dis.isSuccess() && (boolean)dis.getData()){
          successCodes.addAll(itemCodes);
        } else throw new IllegalArgumentException("返回数据异常");
      } catch (Exception e) {
        log.warn("陈列活动推送失败! 返回数据：{}",JSON.toJSONString(dis));
        log.warn("陈列活动推送失败! cause by：{}",e.getMessage());
        failCodes.addAll(itemCodes);
      }
    }
    //促销活动
    if(!CollectionUtils.isEmpty(promotionDataList)){
      CowManagerVo pro = this.pushPromotionActivity(promotionDataList);
      List<String> itemCodes = promotionDataList.stream().map(CowManagerPromotionActivityDto::getActiveNumber).filter(Objects::nonNull).collect(Collectors.toList());
      try {
        if(pro.isSuccess() && (boolean)pro.getData()){
          successCodes.addAll(itemCodes);
        } else throw new IllegalArgumentException("返回数据异常");
      } catch (Exception e) {
        log.warn("促销活动推送失败! 返回数据：{}",JSON.toJSONString(pro));
        log.warn("促销活动推送失败! cause by：{}",e.getMessage());
        failCodes.addAll(itemCodes);
      }
    }
    //临促活动
    if(!CollectionUtils.isEmpty(temporaryDataList)){
      CowManagerVo tem = this.pushTemporaryPersonnelActivity(temporaryDataList);
      List<String> itemCodes = temporaryDataList.stream().map(CowManagerTemporaryPersonnelActivityDto::getActiveNumber).filter(Objects::nonNull).collect(Collectors.toList());
      try {
        if(tem.isSuccess() && (boolean)tem.getData()){
          successCodes.addAll(itemCodes);
        } else throw new IllegalArgumentException("返回数据异常");
      } catch (Exception e) {
        log.warn("临促活动推送失败! 返回数据：{}",JSON.toJSONString(tem));
        log.warn("临促活动推送失败! cause by：{}",e.getMessage());
        failCodes.addAll(itemCodes);
      }
    }
    //长促活动
    if(!CollectionUtils.isEmpty(longPersonDataList)){
      CowManagerVo lon = this.pushLongTermPersonnelActivity(longPersonDataList);
      List<String> itemCodes = longPersonDataList.stream().map(CowManagerLongTermPersonnelActivityDto::getActiveNumber).filter(Objects::nonNull).collect(Collectors.toList());
      try {
        if(lon.isSuccess() && (boolean)lon.getData()){
          successCodes.addAll(itemCodes);
        } else throw new IllegalArgumentException("返回数据异常");
      } catch (Exception e) {
        log.warn("长促活动推送失败! 返回数据：{}",JSON.toJSONString(lon));
        log.warn("长促活动推送失败! cause by：{}",e.getMessage());
        failCodes.addAll(itemCodes);
      }
    }

    return Lists.newArrayList(successCodes, failCodes);
  }

  /**
   * 推送行销活动
   * @param marketingDataList 行销活动集合
   */
  private CowManagerVo pushCellActivity(List<CowManagerCellActivityDto> marketingDataList) {
    try {
      CowManagerVo result = cowManagerService.pushCellActivity(marketingDataList);
      return result;
    } catch (Exception e) {
      e.printStackTrace();
    }
    return new CowManagerVo("2001",false,"false","失败");
  }

  /**
   * 推送陈列活动
   * @param displayDataList 陈列活动集合
   */
  private CowManagerVo pushDisplayActivity(List<CowManagerDisplayActivityDto> displayDataList) {
    try {
      CowManagerVo result = cowManagerService.pushDisplayActivity(displayDataList);
      return result;
    } catch (Exception e) {
      e.printStackTrace();
    }
    return new CowManagerVo("2001",false,"false","失败");
  }

  /**
   * 推送促销活动
   * @param promotionDataList 促销活动集合
   */
  private CowManagerVo pushPromotionActivity(List<CowManagerPromotionActivityDto> promotionDataList) {
    try {
      CowManagerVo result = cowManagerService.pushPromotionActivity(promotionDataList);
      return result;
    } catch (Exception e) {
      e.printStackTrace();
    }
    return new CowManagerVo("2001",false,"false","失败");
  }

  /**
   * 推送临促活动
   * @param temporaryDataList 临促活动集合
   */
  private CowManagerVo pushTemporaryPersonnelActivity(List<CowManagerTemporaryPersonnelActivityDto> temporaryDataList) {
    try {
      CowManagerVo result = cowManagerService.pushTemporaryPersonnelActivity(temporaryDataList);
      return result;
    } catch (Exception e) {
      e.printStackTrace();
    }
    return new CowManagerVo("2001",false,"false","失败");
  }

  /**
   * 推送长促活动
   * @param longPersonDataList 长促活动集合
   */
  private CowManagerVo pushLongTermPersonnelActivity(List<CowManagerLongTermPersonnelActivityDto> longPersonDataList) {
    try {
      CowManagerVo result = cowManagerService.pushLongTermPersonnelActivity(longPersonDataList);
      return result;
    } catch (Exception e) {
      e.printStackTrace();
    }
    return new CowManagerVo("2001",false,"false","失败");
  }
}
