package com.biz.crm.tpm.business.audit.fee.local.template.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.tpm.business.audit.fee.local.template.entity.TpmDeductionMatchingTemplate;
import com.biz.crm.tpm.business.audit.fee.local.template.entity.TpmDeductionMatchingTemplateAllowance;
import com.biz.crm.tpm.business.audit.fee.local.template.entity.TpmDeductionMatchingTemplateRules;
import com.biz.crm.tpm.business.audit.fee.local.template.repository.TpmDeductionMatchingTemplateAllowanceRepository;
import com.biz.crm.tpm.business.audit.fee.local.template.repository.TpmDeductionMatchingTemplateRepository;
import com.biz.crm.tpm.business.audit.fee.local.template.repository.TpmDeductionMatchingTemplateRulesRepository;
import com.biz.crm.tpm.business.audit.fee.sdk.template.constant.TpmDeductionMatchingTemplateConstant;
import com.biz.crm.tpm.business.audit.fee.sdk.template.dto.TpmDeductionMatchingTemplateAllowanceDto;
import com.biz.crm.tpm.business.audit.fee.sdk.template.dto.TpmDeductionMatchingTemplateDto;
import com.biz.crm.tpm.business.audit.fee.sdk.template.dto.TpmDeductionMatchingTemplateRulesDto;
import com.biz.crm.tpm.business.audit.fee.sdk.template.dto.TpmDeductionMatchingTemplateSelectDto;
import com.biz.crm.tpm.business.audit.fee.sdk.template.dto.log.TpmDeductionMatchingTemplateLogEventDto;
import com.biz.crm.tpm.business.audit.fee.sdk.template.enums.AllowanceTypeEnum;
import com.biz.crm.tpm.business.audit.fee.sdk.template.enums.DeductionMatchingTemplateConditionEnum;
import com.biz.crm.tpm.business.audit.fee.sdk.template.enums.DeductionMatchingTemplateTypeEnum;
import com.biz.crm.tpm.business.audit.fee.sdk.template.enums.TpmDeductionMatchingTemplateEnums;
import com.biz.crm.tpm.business.audit.fee.sdk.template.log.TpmDeductionMatchingTemplateLogEventListener;
import com.biz.crm.tpm.business.audit.fee.sdk.template.service.TpmDeductionMatchingTemplateService;
import com.biz.crm.tpm.business.audit.fee.sdk.template.vo.TpmDeductionMatchingTemplateAllowanceVo;
import com.biz.crm.tpm.business.audit.fee.sdk.template.vo.TpmDeductionMatchingTemplateRulesVo;
import com.biz.crm.tpm.business.audit.fee.sdk.template.vo.TpmDeductionMatchingTemplateVo;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.service.TpmDeductionDetailMappingService;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.vo.TpmDeductionDetailMappingVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * TPM-扣费匹配模板(TpmDeductionMatchingTemplate)表服务实现类
 *
 * @author duyiran
 * @since 2022-11-11 09:57:04
 */
@Service("tpmDeductionMatchingTemplateService")
public class TpmDeductionMatchingTemplateServiceImpl implements TpmDeductionMatchingTemplateService {

    @Autowired(required = false)
    private TpmDeductionMatchingTemplateRepository tpmDeductionMatchingTemplateRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    @Autowired(required = false)
    private TpmDeductionMatchingTemplateAllowanceRepository tpmDeductionMatchingTemplateAllowanceRepository;

    @Autowired(required = false)
    private TpmDeductionMatchingTemplateRulesRepository tpmDeductionMatchingTemplateRulesRepository;

    @Autowired(required = false)
    private TpmDeductionDetailMappingService tpmDeductionDetailMappingService;

    /**
     * 分页查询数据
     *
     * @param pageable 分页对象
     * @param dto      实体对象
     * @return
     */
    @Override
    public Page<TpmDeductionMatchingTemplateVo> findByConditions(Pageable pageable, TpmDeductionMatchingTemplateDto dto) {
        Page<TpmDeductionMatchingTemplateVo> result = this.tpmDeductionMatchingTemplateRepository.findByConditions(pageable, dto);
        if (Objects.isNull(result) || CollectionUtils.isEmpty(result.getRecords())) {
            return result;
        }
        List<TpmDeductionMatchingTemplateVo> templateVoList = result.getRecords();
        List<String> codes = templateVoList.stream().map(TpmDeductionMatchingTemplateVo::getCode).distinct().collect(Collectors.toList());
        //查使用范围
        TpmDeductionMatchingTemplateAllowanceDto allowanceDto = new TpmDeductionMatchingTemplateAllowanceDto();
        allowanceDto.setCodeList(codes);
        allowanceDto.setAllowanceType(dto.getAllowanceType());
        List<TpmDeductionMatchingTemplateAllowanceVo> allowanceVos = tpmDeductionMatchingTemplateAllowanceRepository.findListAnywayByCodeOrCodes(allowanceDto);
        if (CollectionUtils.isEmpty(allowanceVos)) {
            return result;
        }
        //主表编码+适用范围类型分组
        Map<String, List<TpmDeductionMatchingTemplateAllowanceVo>> allowanceMap = allowanceVos.stream().collect(Collectors.groupingBy(a -> a.getCode()+"-"+a.getAllowanceType()));
        for (TpmDeductionMatchingTemplateVo templateVo : templateVoList) {

            List<TpmDeductionMatchingTemplateAllowanceVo> posAllowanceList = allowanceMap.get(templateVo.getCode() + "-" + AllowanceTypeEnum.POS.getCode());
            List<TpmDeductionMatchingTemplateAllowanceVo> feeAllowanceList = allowanceMap.get(templateVo.getCode() + "-" + AllowanceTypeEnum.FEE.getCode());
            List<TpmDeductionMatchingTemplateAllowanceVo> statementAllowanceList = allowanceMap.get(templateVo.getCode() + "-" + AllowanceTypeEnum.STATEMENT.getCode());
            if (CollectionUtils.isNotEmpty(posAllowanceList)) {
                templateVo.setPosAllowances(posAllowanceList);
            }
            if (CollectionUtils.isNotEmpty(feeAllowanceList)) {
                templateVo.setFeeAllowances(feeAllowanceList);
            }
            if (CollectionUtils.isNotEmpty(statementAllowanceList)) {
                templateVo.setStatementAllowances(statementAllowanceList);
            }
        }

        return result;
    }

    /**
     * 根据编码查询数据(不分页)
     *
     * @param dto dto
     * @return
     */
    @Override
    public List<TpmDeductionMatchingTemplateVo> findAllListByConditions(TpmDeductionMatchingTemplateDto dto) {
        if (Objects.isNull(dto)) {
            return Lists.newArrayList();
        }
        String tenantCode = TenantUtils.getTenantCode();
        dto.setTenantCode(tenantCode);
        List<TpmDeductionMatchingTemplateVo> voList = this.tpmDeductionMatchingTemplateRepository.findAllListByConditions(dto);
        if (CollectionUtils.isEmpty(voList)) {
            return voList;
        }
        List<TpmDeductionMatchingTemplateAllowanceVo> allowances = tpmDeductionMatchingTemplateAllowanceRepository.findByCodeList(voList.stream().map(TpmDeductionMatchingTemplateVo::getCode).collect(Collectors.toList()));
        if (CollectionUtils.isEmpty(allowances)) {
            return voList;
        }
        //主表编码+适用范围类型分组
        Map<String, List<TpmDeductionMatchingTemplateAllowanceVo>> allowanceMap = allowances.stream().collect(Collectors.groupingBy(a -> a.getCode() + "-" + a.getAllowanceType()));
        for (TpmDeductionMatchingTemplateVo templateVo : voList) {

            List<TpmDeductionMatchingTemplateAllowanceVo> posAllowanceList = allowanceMap.get(templateVo.getCode() + "-" + AllowanceTypeEnum.POS.getCode());
            List<TpmDeductionMatchingTemplateAllowanceVo> feeAllowanceList = allowanceMap.get(templateVo.getCode() + "-" + AllowanceTypeEnum.FEE.getCode());
            List<TpmDeductionMatchingTemplateAllowanceVo> statementAllowanceList = allowanceMap.get(templateVo.getCode() + "-" + AllowanceTypeEnum.STATEMENT.getCode());
            if (CollectionUtils.isNotEmpty(posAllowanceList)) {
                templateVo.setPosAllowances(posAllowanceList);
            }
            if (CollectionUtils.isNotEmpty(feeAllowanceList)) {
                templateVo.setFeeAllowances(feeAllowanceList);
            }
            if (CollectionUtils.isNotEmpty(statementAllowanceList)) {
                templateVo.setStatementAllowances(statementAllowanceList);
            }
        }
        return voList;
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public TpmDeductionMatchingTemplateVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        TpmDeductionMatchingTemplate entity = this.tpmDeductionMatchingTemplateRepository.getById(id);
        Validate.notNull(entity, "数据不存在，请刷新后重试！");
        Validate.isTrue(DelFlagStatusEnum.NORMAL.getCode().equals(entity.getDelFlag()), "数据不存在，请刷新后重试！");
        String tenantCode = TenantUtils.getTenantCode();
        Validate.isTrue(tenantCode.equals(entity.getTenantCode()), "数据不存在，请刷新后重试！");
        TpmDeductionMatchingTemplateVo vo = this.nebulaToolkitService.copyObjectByWhiteList(entity, TpmDeductionMatchingTemplateVo.class, LinkedHashSet.class, ArrayList.class);
        List<TpmDeductionMatchingTemplateAllowanceVo> allowances = tpmDeductionMatchingTemplateAllowanceRepository.findByCode(entity.getCode());
        if (CollectionUtils.isNotEmpty(allowances)) {
            Map<String, List<TpmDeductionMatchingTemplateAllowanceVo>> allowancesMap = allowances.stream().collect(Collectors.groupingBy(TpmDeductionMatchingTemplateAllowanceVo::getAllowanceType));
            for (String key : allowancesMap.keySet()) {
                if (AllowanceTypeEnum.POS.getCode().equals(key)) {
                    vo.setPosAllowances(allowancesMap.get(key));
                }
                if (AllowanceTypeEnum.FEE.getCode().equals(key)) {
                    vo.setFeeAllowances(allowancesMap.get(key));
                }
                if (AllowanceTypeEnum.STATEMENT.getCode().equals(key)) {
                    vo.setStatementAllowances(allowancesMap.get(key));
                }
            }
        }
        List<TpmDeductionMatchingTemplateRules> rulesList = tpmDeductionMatchingTemplateRulesRepository.findByCode(entity.getCode());
        if (CollectionUtils.isNotEmpty(rulesList)) {
            List<TpmDeductionMatchingTemplateRulesVo> rulesVoList = (List<TpmDeductionMatchingTemplateRulesVo>) nebulaToolkitService.copyCollectionByWhiteList(rulesList, TpmDeductionMatchingTemplateRules.class, TpmDeductionMatchingTemplateRulesVo.class, HashSet.class, ArrayList.class);
            vo.setRulesList(rulesVoList);
        }
        return vo;
    }

    /**
     * 新增数据
     *
     * @param dto 实体对象
     * @return 新增结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public TpmDeductionMatchingTemplateVo create(TpmDeductionMatchingTemplateDto dto) {
        // 根据配置校验必填字段、同纬度数据库验重
        this.createValidate(dto);
        Validate.notBlank(dto.getName(),"扣费模板名称不能为空");
        Validate.isTrue(dto.getName().length() <= 255,"扣费模板名称最大长度为255");
        TpmDeductionDetailMappingVo mappingVo = tpmDeductionDetailMappingService.findByCode(dto.getApplyMappingCode());
        Validate.notNull(mappingVo,"未找到扣费映射【"+dto.getApplyMappingCode()+"】");
        dto.setSalesInstitutionCode(mappingVo.getSalesInstitutionCode());
        dto.setSalesInstitutionName(mappingVo.getSalesInstitutionName());
        dto.setSalesInstitutionErpCode(mappingVo.getSalesInstitutionErpCode());
        dto.setResaleCommercialCode(mappingVo.getResaleCommercialCode());
        dto.setResaleCommercialName(mappingVo.getResaleCommercialName());
        //保存验重，根据业态+业务单元+模板类型+适用映射进行保存验重
        List<TpmDeductionMatchingTemplate> list = this.tpmDeductionMatchingTemplateRepository.lambdaQuery()
                //.eq(TpmDeductionMatchingTemplate::getCode,dto.getApplyMappingCode())
                .eq(StringUtils.isNotBlank(dto.getBusinessFormatCode()),TpmDeductionMatchingTemplate::getBusinessFormatCode, dto.getBusinessFormatCode())
                .eq(StringUtils.isNotBlank(dto.getBusinessUnitCode()),TpmDeductionMatchingTemplate::getBusinessUnitCode, dto.getBusinessUnitCode())
                .eq(StringUtils.isNotBlank(dto.getDeductionMatchingTemplateType()),TpmDeductionMatchingTemplate::getDeductionMatchingTemplateType, dto.getDeductionMatchingTemplateType())
                .eq(StringUtils.isNotBlank(dto.getSalesInstitutionCode()),TpmDeductionMatchingTemplate::getSalesInstitutionCode, dto.getSalesInstitutionCode())
                .eq(StringUtils.isNotBlank(dto.getApplyMappingCode()),TpmDeductionMatchingTemplate::getApplyMappingCode,dto.getApplyMappingCode())
                .eq(TpmDeductionMatchingTemplate::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
                .eq(TpmDeductionMatchingTemplate::getTenantCode,TenantUtils.getTenantCode())
                .list();
        Validate.isTrue(CollectionUtils.isEmpty(list),"业态+业务单元+销售机构+模板类型+适用映射重复新增");

        String tenantCode = TenantUtils.getTenantCode();
        dto.setTenantCode(tenantCode);
        //新建的默认启用状态。
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
//      String ruleCode = StringUtils.join(TpmDeductionMatchingTemplateConstant.TPM_DEDUCTION_MATCHING_TEMPLATE_FORMULA_PREFIX, DateFormatUtils.format(new Date(), "yyyyMMdd"));
        String code = this.generateCodeService.generateCode(TpmDeductionMatchingTemplateConstant.TPM_DEDUCTION_MATCHING_TEMPLATE_FORMULA_PREFIX, 1, 10, 2, TimeUnit.DAYS).get(0);
        dto.setCode(code);
        List<TpmDeductionMatchingTemplateAllowance> allowances = new ArrayList<>();
        if (!CollectionUtils.isEmpty(dto.getPosAllowances())) {
            allowances.addAll(this.buildAllowanceData(dto,dto.getPosAllowances(), AllowanceTypeEnum.POS));
        }
        if (!CollectionUtils.isEmpty(dto.getFeeAllowances())) {
            allowances.addAll(this.buildAllowanceData(dto,dto.getFeeAllowances(), AllowanceTypeEnum.FEE));
        }
        if (!CollectionUtils.isEmpty(dto.getStatementAllowances())) {
            allowances.addAll(this.buildAllowanceData(dto,dto.getStatementAllowances(), AllowanceTypeEnum.STATEMENT));
        }

        Collection<TpmDeductionMatchingTemplateRules> rulesList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(dto.getRulesList())) {
            dto.getRulesList().forEach(rules -> {
                rules.setTenantCode(TenantUtils.getTenantCode());
                rules.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                rules.setCode(dto.getCode());
            });
            rulesList = nebulaToolkitService.copyCollectionByWhiteList(dto.getRulesList(), TpmDeductionMatchingTemplateRulesDto.class, TpmDeductionMatchingTemplateRules.class, HashSet.class, ArrayList.class);
        }
        TpmDeductionMatchingTemplate entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, TpmDeductionMatchingTemplate.class, LinkedHashSet.class, ArrayList.class);
        this.saveOrUpdateTpmDeductionMatchingTemplate(true,dto,null,entity,rulesList,allowances);
        return null;
    }

    /**
     *
     * @param flag true 新增；false 更新
     * @param dto
     * @param old
     * @param entity
     * @param rulesList
     * @param allowances
     */
    @Transactional(rollbackFor = Exception.class)
    public void saveOrUpdateTpmDeductionMatchingTemplate(boolean flag,TpmDeductionMatchingTemplateDto dto,TpmDeductionMatchingTemplateDto old, TpmDeductionMatchingTemplate entity, Collection<TpmDeductionMatchingTemplateRules> rulesList, List<TpmDeductionMatchingTemplateAllowance> allowances) {
        if (!Objects.isNull(dto) || !Objects.isNull(entity)) {
            //主表
            this.tpmDeductionMatchingTemplateRepository.saveOrUpdate(entity);

            //取值公式表
            if (CollectionUtils.isNotEmpty(rulesList)) {

                if (!flag) {
                    //更新、 物理删除字表数据
                    this.tpmDeductionMatchingTemplateRulesRepository.removeByCode(entity.getCode());
                }
                this.tpmDeductionMatchingTemplateRulesRepository.saveBatch(rulesList);
            }
            //适用范围表
            if (CollectionUtils.isNotEmpty(allowances)) {
                if (!flag) {
                    //更新、 物理删除子表数据
                    this.tpmDeductionMatchingTemplateAllowanceRepository.removeByCode(entity.getCode());
                }
                this.tpmDeductionMatchingTemplateAllowanceRepository.saveBatch(allowances);
            }

            if (flag) {
                //新增日志
                TpmDeductionMatchingTemplateLogEventDto logEventDto = new TpmDeductionMatchingTemplateLogEventDto();
                logEventDto.setOriginal(null);
                logEventDto.setNewest(dto);
                SerializableBiConsumer<TpmDeductionMatchingTemplateLogEventListener, TpmDeductionMatchingTemplateLogEventDto> onCreate =
                        TpmDeductionMatchingTemplateLogEventListener::onCreate;
                this.nebulaNetEventClient.publish(logEventDto, TpmDeductionMatchingTemplateLogEventListener.class, onCreate);
            }else {
                //编辑日志
                TpmDeductionMatchingTemplateLogEventDto logEventDto = new TpmDeductionMatchingTemplateLogEventDto();
                logEventDto.setOriginal(old);
                logEventDto.setNewest(dto);
                SerializableBiConsumer<TpmDeductionMatchingTemplateLogEventListener, TpmDeductionMatchingTemplateLogEventDto> onUpdate =
                        TpmDeductionMatchingTemplateLogEventListener::onUpdate;
                this.nebulaNetEventClient.publish(logEventDto, TpmDeductionMatchingTemplateLogEventListener.class, onUpdate);
            }
        }
    }

    private List<TpmDeductionMatchingTemplateAllowance> buildAllowanceData(TpmDeductionMatchingTemplateDto dto,List<TpmDeductionMatchingTemplateAllowanceDto> allowanceDto, AllowanceTypeEnum allowanceType) {
        if (CollectionUtils.isEmpty(allowanceDto) || Objects.isNull(allowanceType)) {
            return Lists.newArrayList();
        }
        List<TpmDeductionMatchingTemplateAllowance> allowances = (List<TpmDeductionMatchingTemplateAllowance>) nebulaToolkitService.copyCollectionByWhiteList(allowanceDto, TpmDeductionMatchingTemplateAllowanceDto.class,
                TpmDeductionMatchingTemplateAllowance.class, HashSet.class, ArrayList.class);

        Map<String,Integer> applyBusinessAreaCodeMap = new HashMap<>();
        for (TpmDeductionMatchingTemplateAllowance allowance : allowances) {
             //唯一键（适用模板编码+适用映射+模板类型+业务区域编码）
            StringBuilder uniqueKeyBuilder = new StringBuilder();
            uniqueKeyBuilder.append(dto.getCode())
                    .append("_")
                    .append(StringUtils.isNotBlank(allowance.getApplyMappingCode()) ? allowance.getApplyMappingCode() : "")
                    .append("_")
                    .append(allowanceType)
                    .append("_")
                    .append(allowance.getApplyBusinessAreaCode());
            allowance.setDeductionUniqueKey(uniqueKeyBuilder.toString());

            if (StringUtils.isNotBlank(allowance.getApplyBusinessAreaCode())) {
                String[] split = allowance.getApplyBusinessAreaCode().split(",");
                for (int i = 0 ; i < split.length; i++){
                    if (Objects.isNull(applyBusinessAreaCodeMap.get(split[i]))) {
                        applyBusinessAreaCodeMap.put(split[i],0);
                    }
                    Integer num = applyBusinessAreaCodeMap.get(split[i]);
                    num = num + 1;
                    applyBusinessAreaCodeMap.put(split[i],num);
                }
            }

            allowance.setTenantCode(TenantUtils.getTenantCode());
            allowance.setCode(dto.getCode());
            allowance.setAllowanceType(allowanceType.getCode());
        }

        applyBusinessAreaCodeMap.forEach( (k,v) -> {
            Validate.isTrue( (!Objects.isNull(v) && v == 1),allowanceType.getDesc()+"适用范围中，业务区域存在重复区域，请检查");
        });


        return allowances;
    }

    /**
     * 修改新据
     *
     * @param dto 实体对象
     * @return 修改结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public TpmDeductionMatchingTemplateVo update(TpmDeductionMatchingTemplateDto dto) {
        this.updateValidate(dto);
        Validate.notBlank(dto.getName(),"扣费模板名称不能为空");
        Validate.isTrue(dto.getName().length() <= 255,"扣费模板名称最大长度为255");
        //取出旧的记录
        TpmDeductionMatchingTemplate oldEntity = tpmDeductionMatchingTemplateRepository.getById(dto.getId());
        Validate.notNull(oldEntity, "数据不存在，请刷新后重试！");
        Validate.isTrue(DelFlagStatusEnum.NORMAL.getCode().equals(oldEntity.getDelFlag()), "数据不存在，请刷新后重试！");
        String tenantCode = TenantUtils.getTenantCode();
        Validate.isTrue(tenantCode.equals(oldEntity.getTenantCode()), "数据不存在，请刷新后重试！");
        TpmDeductionMatchingTemplateDto oldDto = this.nebulaToolkitService.copyObjectByWhiteList(oldEntity, TpmDeductionMatchingTemplateDto.class, LinkedHashSet.class, ArrayList.class);
        //List<TpmDeductionMatchingTemplateAllowanceVo> oldAllowances = tpmDeductionMatchingTemplateAllowanceRepository.findByCode(oldEntity.getCode());
        //oldDto.setPosAllowances(Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(oldAllowances, TpmDeductionMatchingTemplateAllowanceVo.class, TpmDeductionMatchingTemplateAllowanceDto.class, HashSet.class, ArrayList.class)));

        //数据库验重
        //保存验重，根据业态+业务单元+销售组织+模板类型+映射进行保存验重
        List<TpmDeductionMatchingTemplate> list = this.tpmDeductionMatchingTemplateRepository.lambdaQuery()
                .eq(StringUtils.isNotBlank(dto.getBusinessFormatCode()),TpmDeductionMatchingTemplate::getBusinessFormatCode, dto.getBusinessFormatCode())
                .eq(StringUtils.isNotBlank(dto.getBusinessUnitCode()),TpmDeductionMatchingTemplate::getBusinessUnitCode, dto.getBusinessUnitCode())
                .eq(StringUtils.isNotBlank(dto.getSalesInstitutionCode()),TpmDeductionMatchingTemplate::getSalesInstitutionCode, dto.getSalesInstitutionCode())
                .eq(StringUtils.isNotBlank(dto.getApplyMappingCode()),TpmDeductionMatchingTemplate::getApplyMappingCode,dto.getApplyMappingCode())
                .eq(StringUtils.isNotBlank(dto.getDeductionMatchingTemplateType()),TpmDeductionMatchingTemplate::getDeductionMatchingTemplateType, dto.getDeductionMatchingTemplateType())
                .eq(TpmDeductionMatchingTemplate::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
                .eq(TpmDeductionMatchingTemplate::getTenantCode,TenantUtils.getTenantCode())
                .ne(TpmDeductionMatchingTemplate::getId,dto.getId())
                .list();
        Validate.isTrue(CollectionUtils.isEmpty(list),"业态+业务单元+销售机构+模板类型+适用映射重复新增");
        /*if (!CollectionUtils.isEmpty(list)){
            dto.getPosAllowances().forEach(allowance -> {
                List<String> codes = list.stream().map(o -> o.getCode()).collect(Collectors.toList());
                List<TpmDeductionMatchingTemplateAllowance> templateAllowanceList = this.tpmDeductionMatchingTemplateAllowanceRepository.lambdaQuery()
                        .in(TpmDeductionMatchingTemplateAllowance::getCode, codes)
                        .eq(StringUtils.isNotBlank(allowance.getResaleCommercialCode()), TpmDeductionMatchingTemplateAllowance::getResaleCommercialCode, allowance.getResaleCommercialCode())
                        .eq(StringUtils.isNotBlank(allowance.getApplyBusinessCustomerCode()), TpmDeductionMatchingTemplateAllowance::getApplyBusinessCustomerCode, allowance.getApplyBusinessCustomerCode())
                        .eq(TpmDeductionMatchingTemplateAllowance::getApplyMappingCode, allowance.getApplyMappingCode())
                        .list();
                Validate.isTrue(CollectionUtils.isEmpty(templateAllowanceList),"该模板内容重复");
            });
        }*/
        TpmDeductionDetailMappingVo mappingVo = tpmDeductionDetailMappingService.findByCode(dto.getApplyMappingCode());
        Validate.notNull(mappingVo,"未找到扣费映射【"+dto.getApplyMappingCode()+"】");
        dto.setSalesInstitutionCode(mappingVo.getSalesInstitutionCode());
        dto.setSalesInstitutionName(mappingVo.getSalesInstitutionName());
        dto.setSalesInstitutionErpCode(mappingVo.getSalesInstitutionErpCode());
        dto.setResaleCommercialCode(mappingVo.getResaleCommercialCode());
        dto.setResaleCommercialName(mappingVo.getResaleCommercialName());

        //新的实体,租户编号赋值
        TpmDeductionMatchingTemplate newEntity = nebulaToolkitService.copyObjectByWhiteList(dto, TpmDeductionMatchingTemplate.class, HashSet.class, ArrayList.class);
        String code = newEntity.getCode();
        List<TpmDeductionMatchingTemplateAllowance> allowances = new ArrayList<>();
        if (!CollectionUtils.isEmpty(dto.getPosAllowances())) {
            allowances.addAll(this.buildAllowanceData(dto,dto.getPosAllowances(), AllowanceTypeEnum.POS));
        }
        if (!CollectionUtils.isEmpty(dto.getFeeAllowances())) {
            allowances.addAll(this.buildAllowanceData(dto,dto.getFeeAllowances(), AllowanceTypeEnum.FEE));
        }
        if (!CollectionUtils.isEmpty(dto.getStatementAllowances())) {
            allowances.addAll(this.buildAllowanceData(dto,dto.getStatementAllowances(), AllowanceTypeEnum.STATEMENT));
        }

        Collection<TpmDeductionMatchingTemplateRules> rulesList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(dto.getRulesList())) {
            dto.getRulesList().forEach(rules -> {
                rules.setTenantCode(TenantUtils.getTenantCode());
                rules.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                rules.setCode(dto.getCode());
            });
            rulesList = nebulaToolkitService.copyCollectionByWhiteList(dto.getRulesList(), TpmDeductionMatchingTemplateRulesDto.class, TpmDeductionMatchingTemplateRules.class, HashSet.class, ArrayList.class);
        }

        this.saveOrUpdateTpmDeductionMatchingTemplate(false,dto,oldDto,newEntity,rulesList,allowances);
        return null;
    }

    /**
     * 删除数据
     *
     * @param idList 主键结合
     * @return 删除结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        String tenantCode = TenantUtils.getTenantCode();
        List<TpmDeductionMatchingTemplate> list = tpmDeductionMatchingTemplateRepository.lambdaQuery()
                .select(TpmDeductionMatchingTemplate::getCode,TpmDeductionMatchingTemplate::getEnableStatus)
                .in(TpmDeductionMatchingTemplate::getId, idList)
                .eq(TpmDeductionMatchingTemplate::getTenantCode, tenantCode)
                .list();
        Validate.notEmpty(list, "数据不存在，请刷新后重试！");
        list.forEach(e -> {
           if (EnableStatusEnum.ENABLE.getCode().equals(e.getEnableStatus())) {
               throw new RuntimeException("启用中的模板不可删除！");
           }
        });
        List<String> codeList = list.stream().map(TpmDeductionMatchingTemplate::getCode).collect(Collectors.toList());
        tpmDeductionMatchingTemplateRepository.lambdaUpdate().in(TpmDeductionMatchingTemplate::getId, idList)
                .set(TpmDeductionMatchingTemplate::getDelFlag, DelFlagStatusEnum.DELETE.getCode()).update();
        tpmDeductionMatchingTemplateAllowanceRepository.lambdaUpdate().in(TpmDeductionMatchingTemplateAllowance::getCode, codeList)
                .set(TpmDeductionMatchingTemplateAllowance::getDelFlag, DelFlagStatusEnum.DELETE.getCode()).update();
    }

    /**
     * 启、禁用数据
     *
     * @param idList 主键结合
     * @param code
     * @return 启、禁用结果
     */
    @Override
    public void disableOrEnable(List<String> idList, String code) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        List<TpmDeductionMatchingTemplate> list = tpmDeductionMatchingTemplateRepository.lambdaQuery()
                .in(TpmDeductionMatchingTemplate::getId, idList).list();
        Validate.noNullElements(list, "数据不存在，请刷新后重试！");
        String tenantCode = TenantUtils.getTenantCode();
        List<String> codeList = list.stream().map(TpmDeductionMatchingTemplate::getCode).collect(Collectors.toList());
        tpmDeductionMatchingTemplateRepository.lambdaUpdate()
                .in(TpmDeductionMatchingTemplate::getId, idList)
                .eq(TpmDeductionMatchingTemplate::getTenantCode, tenantCode)
                .set(TpmDeductionMatchingTemplate::getEnableStatus, code).update();
        tpmDeductionMatchingTemplateAllowanceRepository.lambdaUpdate()
                .in(TpmDeductionMatchingTemplateAllowance::getCode, codeList)
                .eq(TpmDeductionMatchingTemplateAllowance::getTenantCode, tenantCode)
                .set(TpmDeductionMatchingTemplateAllowance::getEnableStatus, code).update();
        //日志
        List<TpmDeductionMatchingTemplateDto> oldDtos = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(list, TpmDeductionMatchingTemplate.class, TpmDeductionMatchingTemplateDto.class, HashSet.class, ArrayList.class));
        oldDtos.forEach(oldDto -> {
            TpmDeductionMatchingTemplateLogEventDto logEventDto = new TpmDeductionMatchingTemplateLogEventDto();
            logEventDto.setOriginal(oldDto);
            TpmDeductionMatchingTemplateDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(oldDto, TpmDeductionMatchingTemplateDto.class, LinkedHashSet.class, ArrayList.class);
            newDto.setEnableStatus(code);
            logEventDto.setNewest(newDto);
            SerializableBiConsumer<TpmDeductionMatchingTemplateLogEventListener, TpmDeductionMatchingTemplateLogEventDto> onUpdateEnable =
                    TpmDeductionMatchingTemplateLogEventListener::onUpdateEnable;
            this.nebulaNetEventClient.publish(logEventDto, TpmDeductionMatchingTemplateLogEventListener.class, onUpdateEnable);
        });
    }

    @Override
    public TpmDeductionMatchingTemplateVo findByCode(String matchTemplateCode) {
        TpmDeductionMatchingTemplate template = this.tpmDeductionMatchingTemplateRepository.findByCode(matchTemplateCode);
        if (Objects.isNull(template)) {
            return null;
        }
        TpmDeductionMatchingTemplateVo templateVo = this.nebulaToolkitService.copyObjectByWhiteList(template, TpmDeductionMatchingTemplateVo.class, LinkedHashSet.class, ArrayList.class);
        List<TpmDeductionMatchingTemplateAllowanceVo> allowances = tpmDeductionMatchingTemplateAllowanceRepository.findByCode(templateVo.getCode());
        if (CollectionUtils.isNotEmpty(allowances)) {
            Map<String, List<TpmDeductionMatchingTemplateAllowanceVo>> allowancesMap = allowances.stream().collect(Collectors.groupingBy(TpmDeductionMatchingTemplateAllowanceVo::getAllowanceType));
            for (String key : allowancesMap.keySet()) {
                if (AllowanceTypeEnum.POS.getCode().equals(key)) {
                    templateVo.setPosAllowances(allowancesMap.get(key));
                }
                if (AllowanceTypeEnum.FEE.getCode().equals(key)) {
                    templateVo.setFeeAllowances(allowancesMap.get(key));
                }
                if (AllowanceTypeEnum.STATEMENT.getCode().equals(key)) {
                    templateVo.setStatementAllowances(allowancesMap.get(key));
                }
            }
        }
        return templateVo;
    }

    @Override
    public List<TpmDeductionMatchingTemplateVo> findByCodes(List<String> matchTemplateCodes) {
        if (CollectionUtils.isEmpty(matchTemplateCodes)) {
            return Lists.newArrayList();
        }
        List<TpmDeductionMatchingTemplate> templates = this.tpmDeductionMatchingTemplateRepository.findByCodes(matchTemplateCodes);
        if (CollectionUtils.isEmpty(templates)) {
            return Lists.newArrayList();
        }
        List<TpmDeductionMatchingTemplateVo> templateVoList = (List<TpmDeductionMatchingTemplateVo>) nebulaToolkitService
                .copyCollectionByWhiteList(templates,TpmDeductionMatchingTemplate.class,TpmDeductionMatchingTemplateVo.class,HashSet.class,ArrayList.class);
        List<TpmDeductionMatchingTemplateRules> rulesList = this.tpmDeductionMatchingTemplateRulesRepository.findByCodes(matchTemplateCodes);
        if (CollectionUtils.isNotEmpty(rulesList)) {
            List<TpmDeductionMatchingTemplateRulesVo> rulesVoList =(List<TpmDeductionMatchingTemplateRulesVo>) nebulaToolkitService
                    .copyCollectionByWhiteList(rulesList,TpmDeductionMatchingTemplateRules.class,TpmDeductionMatchingTemplateRulesVo.class,HashSet.class,ArrayList.class);
            Map<String,List<TpmDeductionMatchingTemplateRulesVo>> ruleVoMap = rulesVoList.stream().collect(Collectors.groupingBy(TpmDeductionMatchingTemplateRulesVo::getCode));
            templateVoList.forEach(e -> {
                if (ruleVoMap.containsKey(e.getCode())) {
                    e.setRulesList(ruleVoMap.get(e.getCode()));
                }
            });
        }
        return templateVoList;
    }

    /**
     * 创建验证
     *
     * @param dto
     */
    private void createValidate(TpmDeductionMatchingTemplateDto dto) {
        Validate.notNull(dto, "新增时，对象信息不能为空！");
        dto.setId(null);
        Validate.notBlank(dto.getBusinessFormatCode(), "新增数据时，业态不能为空！");
        Validate.notBlank(dto.getBusinessUnitCode(), "新增数据时，业务单元不能为空！");
        Validate.notBlank(dto.getDeductionMatchingTemplateType(), "新增数据时，扣费匹配模板类型不能为空！");
        Validate.notBlank(dto.getFeeMatchingCondition(), "新增数据时，费用单匹配条件设置不能为空！");
        Validate.notBlank(dto.getStatementMatchingCondition(), "新增数据时，结算单匹配条件设置不能为空！");
        //匹配条件校验
        ArrayList<String> feeMatchList = Lists.newArrayList(dto.getFeeMatchingCondition().split(","));
        ArrayList<String> statementMatchList = Lists.newArrayList(dto.getStatementMatchingCondition().split(","));
        if (feeMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH.getCode())
                && feeMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH_DAY.getCode())){
            throw new IllegalArgumentException("年月日和年月不可同时选择");
        }
        if (statementMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH.getCode())
                && statementMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH_DAY.getCode())){
            throw new IllegalArgumentException("年月日和年月不可同时选择");
        }
        if (feeMatchList.contains(DeductionMatchingTemplateConditionEnum.EMPTY.getCode())) {
            Validate.isTrue(feeMatchList.size() == 1,"费用单匹配条件配置，'无条件'只可单选");
        }
        if (statementMatchList.contains(DeductionMatchingTemplateConditionEnum.EMPTY.getCode())) {
            Validate.isTrue(statementMatchList.size() == 1,"结算单匹配条件配置，'无条件'只可单选");
        }
        //匹配条件包含年月时是否累计匹配默认为否
        if (feeMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH.getCode())){
            dto.setIsAddUpMapping(BooleanEnum.FALSE.getCapital());
        }

        if (feeMatchList.contains(DeductionMatchingTemplateConditionEnum.SETTLEMENT_CODE.getCode())
                || dto.getFeeMatchingCondition().contains(DeductionMatchingTemplateConditionEnum.FEE_CODE.getCode())){
            Validate.isTrue(feeMatchList.size() == 1,"选择费用单编码或结算单编码时，不可选择其他匹配条件");
        }



        if (DeductionMatchingTemplateTypeEnum.DETAIL_POS_FEE_DETAIL_FEE_SETATEMENT.getCode().equals(dto.getDeductionMatchingTemplateType())) {
            Validate.noNullElements(dto.getPosAllowances(), "POS单适用范围不能为空！");
            dto.getPosAllowances().forEach( a -> {
                Validate.notBlank(a.getDataSource(),"POS适用范围的数据来源不能为空！");
            });
        }
        Validate.noNullElements(dto.getFeeAllowances(), "费用单适用范围不能为空！");
        dto.getFeeAllowances().forEach( a -> {
            Validate.notBlank(a.getDataSource(),"费用单适用范围的数据来源不能为空！");
            //匹配条件包含年月日时，时间容差默认为上下浮动、时间容差类型为日、时间容值为0且不可编辑；
            if (feeMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH_DAY.getCode())){
                a.setTimeAllowanceType(TpmDeductionMatchingTemplateEnums.AllowanceType.INTEGER.getValue());
                a.setTimeAllowanceUnit(TpmDeductionMatchingTemplateEnums.AllowanceUnit.DAY.getValue());
                a.setTimeAllowanceValue(0);
            }
            //匹配条件包含年月时，时间容差类型默认为月，且不可编辑
            if (feeMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH.getCode())){
                Validate.notBlank(a.getTimeAllowanceType(), "费用单适用范围，时间容差不能为空！");
                if (TpmDeductionMatchingTemplateEnums.AllowanceType.APPOINT_DATE.getValue().equals(a.getTimeAllowanceType())) {
                    Validate.notBlank(a.getBeginDate(),"费用单适用范围，时间容差为'指定范围',日期范围-开始日期不能为空！");
                    Validate.notBlank(a.getEndDate(),"费用单适用范围，时间容差为'指定范围',日期范围-结束日期不能为空！");
                }else {
                    a.setTimeAllowanceUnit(TpmDeductionMatchingTemplateEnums.AllowanceUnit.MONTH.getValue());
                    Validate.notNull(a.getTimeAllowanceValue(), "费用单适用范围，时间容差值不能为空！");
                    Validate.isTrue(a.getTimeAllowanceValue() >= 0, "费用单适用范围，时间容差值不能小于0！");
                }
            }
            //匹配条件不包含年月日或年月时，时间容差、时间容差类型，时间容值为空且不可编辑；
            if (!feeMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH.getCode())
                    && !feeMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH_DAY.getCode())){
                a.setTimeAllowanceType(null);
                a.setTimeAllowanceUnit(null);
                a.setTimeAllowanceValue(null);
            }
        });
        Validate.noNullElements(dto.getStatementAllowances(), "结算单适用范围不能为空！");
        dto.getStatementAllowances().forEach( a -> {
            Validate.notBlank(a.getDataSource(),"结算单适用范围的数据来源不能为空！");
            //匹配条件包含年月日时，时间容差默认为上下浮动、时间容差类型为日、时间容值为0且不可编辑；
            if (statementMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH_DAY.getCode())){
                a.setTimeAllowanceType(TpmDeductionMatchingTemplateEnums.AllowanceType.INTEGER.getValue());
                a.setTimeAllowanceUnit(TpmDeductionMatchingTemplateEnums.AllowanceUnit.DAY.getValue());
                a.setTimeAllowanceValue(0);
            }
            //匹配条件包含年月时，时间容差类型默认为月，且不可编辑
            if (statementMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH.getCode())){
                Validate.notBlank(a.getTimeAllowanceType(), "结算单适用范围，时间容差不能为空！");
                if (TpmDeductionMatchingTemplateEnums.AllowanceType.APPOINT_DATE.getValue().equals(a.getTimeAllowanceType())) {
                    Validate.notBlank(a.getBeginDate(),"结算单适用范围，时间容差为'指定范围',日期范围-开始日期不能为空！");
                    Validate.notBlank(a.getEndDate(),"结算单适用范围，时间容差为'指定范围',日期范围-结束日期不能为空！");
                }else {
                    a.setTimeAllowanceUnit(TpmDeductionMatchingTemplateEnums.AllowanceUnit.MONTH.getValue());
                    Validate.notNull(a.getTimeAllowanceValue(), "结算单适用范围，时间容差值不能为空！");
                    Validate.isTrue(a.getTimeAllowanceValue() >= 0, "结算单适用范围，时间容差值不能小于0！");
                }
            }
            //匹配条件不包含年月日或年月时，时间容差、时间容差类型，时间容值为空且不可编辑；
            if (!statementMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH.getCode())
                    && !feeMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH_DAY.getCode())){
                a.setTimeAllowanceType(null);
                a.setTimeAllowanceUnit(null);
                a.setTimeAllowanceValue(null);
            }
        });

    }

    /**
     * 修改验证
     *
     * @param dto
     */
    private void updateValidate(TpmDeductionMatchingTemplateDto dto) {
        Validate.notNull(dto, "修改数据时，对象信息不能为空！");
        Validate.notBlank(dto.getId(), "修改数据时，ID不能为空！");
        Validate.notBlank(dto.getBusinessFormatCode(), "修改数据时，业态不能为空！");
        Validate.notBlank(dto.getBusinessUnitCode(), "修改数据时，业务单元不能为空！");
        Validate.notBlank(dto.getFeeMatchingCondition(), "修改数据时，费用单匹配条件设置不能为空！");
        Validate.notBlank(dto.getStatementMatchingCondition(), "修改数据时，结算单匹配条件设置不能为空！");
        //匹配条件校验
        ArrayList<String> feeMatchList = Lists.newArrayList(dto.getFeeMatchingCondition().split(","));
        ArrayList<String> statementMatchList = Lists.newArrayList(dto.getStatementMatchingCondition().split(","));
        if (feeMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH.getCode())
                && feeMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH_DAY.getCode())){
            throw new IllegalArgumentException("年月日和年月不可同时选择");
        }
        if (statementMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH.getCode())
                && statementMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH_DAY.getCode())){
            throw new IllegalArgumentException("年月日和年月不可同时选择");
        }
        if (feeMatchList.contains(DeductionMatchingTemplateConditionEnum.EMPTY.getCode())) {
            Validate.isTrue(feeMatchList.size() == 1,"费用单匹配条件配置，'无条件'只可单选");
        }
        if (statementMatchList.contains(DeductionMatchingTemplateConditionEnum.EMPTY.getCode())) {
            Validate.isTrue(statementMatchList.size() == 1,"结算单匹配条件配置，'无条件'只可单选");
        }
        //匹配条件包含年月时是否累计匹配默认为否
        if (feeMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH.getCode())){
            dto.setIsAddUpMapping(BooleanEnum.FALSE.getCapital());
        }

        if (feeMatchList.contains(DeductionMatchingTemplateConditionEnum.SETTLEMENT_CODE.getCode())
                || dto.getFeeMatchingCondition().contains(DeductionMatchingTemplateConditionEnum.FEE_CODE.getCode())){
            Validate.isTrue(feeMatchList.size() == 1,"选择费用单编码或结算单编码时，不可选择其他匹配条件");
        }

        if (DeductionMatchingTemplateTypeEnum.DETAIL_POS_FEE_DETAIL_FEE_SETATEMENT.getCode().equals(dto.getDeductionMatchingTemplateType())) {
            Validate.noNullElements(dto.getPosAllowances(), "POS单适用范围不能为空！");
            dto.getPosAllowances().forEach( a -> {
                Validate.notBlank(a.getDataSource(),"POS单适用范围的数据来源不能为空！");
            });
        }
        Validate.noNullElements(dto.getFeeAllowances(), "费用单适用范围不能为空！");
        dto.getFeeAllowances().forEach( a -> {
            Validate.notBlank(a.getDataSource(),"费用单适用范围的数据来源不能为空！");

            //匹配条件包含年月日时，时间容差默认为上下浮动、时间容差类型为日、时间容值为0且不可编辑；
            if (feeMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH_DAY.getCode())){
                a.setTimeAllowanceType(TpmDeductionMatchingTemplateEnums.AllowanceType.INTEGER.getValue());
                a.setTimeAllowanceUnit(TpmDeductionMatchingTemplateEnums.AllowanceUnit.DAY.getValue());
                a.setTimeAllowanceValue(0);
            }
            //匹配条件包含年月时，时间容差类型默认为月，且不可编辑
            if (feeMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH.getCode())){
                Validate.notBlank(a.getTimeAllowanceType(), "费用单适用范围，时间容差不能为空！");
                if (TpmDeductionMatchingTemplateEnums.AllowanceType.APPOINT_DATE.getValue().equals(a.getTimeAllowanceType())) {
                    Validate.notBlank(a.getBeginDate(),"费用单适用范围，时间容差为'指定范围',日期范围-开始日期不能为空！");
                    Validate.notBlank(a.getEndDate(),"费用单适用范围，时间容差为'指定范围',日期范围-结束日期不能为空！");
                }else {
                    a.setTimeAllowanceUnit(TpmDeductionMatchingTemplateEnums.AllowanceUnit.MONTH.getValue());
                    Validate.notNull(a.getTimeAllowanceValue(), "费用单适用范围，时间容差值不能为空！");
                    Validate.isTrue(a.getTimeAllowanceValue() >= 0, "费用单适用范围，时间容差值不能小于0！");
                }
            }
            //匹配条件不包含年月日或年月时，时间容差、时间容差类型，时间容值为空且不可编辑；
            if (!feeMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH.getCode())
                    && !feeMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH_DAY.getCode())){
                a.setTimeAllowanceType(null);
                a.setTimeAllowanceUnit(null);
                a.setTimeAllowanceValue(null);
            }
        });
        Validate.noNullElements(dto.getStatementAllowances(), "结算单适用范围不能为空！");
        dto.getStatementAllowances().forEach( a -> {
            Validate.notBlank(a.getDataSource(),"结算单适用范围的数据来源不能为空！");
            //匹配条件包含年月日时，时间容差默认为上下浮动、时间容差类型为日、时间容值为0且不可编辑；
            if (statementMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH_DAY.getCode())){
                a.setTimeAllowanceType(TpmDeductionMatchingTemplateEnums.AllowanceType.INTEGER.getValue());
                a.setTimeAllowanceUnit(TpmDeductionMatchingTemplateEnums.AllowanceUnit.DAY.getValue());
                a.setTimeAllowanceValue(0);
            }
            //匹配条件包含年月时，时间容差类型默认为月，且不可编辑
            if (statementMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH.getCode())){
                Validate.notBlank(a.getTimeAllowanceType(), "结算单适用范围，时间容差不能为空！");

                if (TpmDeductionMatchingTemplateEnums.AllowanceType.APPOINT_DATE.getValue().equals(a.getTimeAllowanceType())) {
                    Validate.notBlank(a.getBeginDate(),"结算单适用范围，时间容差为'指定范围',日期范围-开始日期不能为空！");
                    Validate.notBlank(a.getEndDate(),"结算单适用范围，时间容差为'指定范围',日期范围-结束日期不能为空！");
                }else {
                    a.setTimeAllowanceUnit(TpmDeductionMatchingTemplateEnums.AllowanceUnit.MONTH.getValue());
                    Validate.notNull(a.getTimeAllowanceValue(), "结算单适用范围，时间容差值不能为空！");
                    Validate.isTrue(a.getTimeAllowanceValue() >= 0, "结算单适用范围，时间容差值不能小于0！");
                }

            }
            //匹配条件不包含年月日或年月时，时间容差、时间容差类型，时间容值为空且不可编辑；
            if (!statementMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH.getCode())
                    && !feeMatchList.contains(DeductionMatchingTemplateConditionEnum.YEAR_MONTH_DAY.getCode())){
                a.setTimeAllowanceType(null);
                a.setTimeAllowanceUnit(null);
                a.setTimeAllowanceValue(null);
            }
        });

    }

    /**
     * 分页查询模板信息，更据适用的零售商（商超）、适用客户过滤
     *
     * @param pageable
     * @param dto
     * @return
     */
    @Override
    public Page<TpmDeductionMatchingTemplateVo> findByConditionsWithResaleCommercial(Pageable pageable, TpmDeductionMatchingTemplateSelectDto dto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(dto)) {
            dto = new TpmDeductionMatchingTemplateSelectDto();
        }
        if (StringUtils.isBlank(dto.getTenantCode())) {
            dto.setTenantCode(TenantUtils.getTenantCode());
        }
        return this.tpmDeductionMatchingTemplateRepository.findByConditionsWithResaleCommercial(pageable, dto);

    }

    /**
     * 根据dto查询
     * @param dto
     * @return
     */
    @Override
    public List<TpmDeductionMatchingTemplateVo> findByDto(TpmDeductionMatchingTemplateDto dto) {
        if (Objects.isNull(dto)) {
            dto = new TpmDeductionMatchingTemplateDto();
        }
        List<TpmDeductionMatchingTemplate> list = this.tpmDeductionMatchingTemplateRepository.findByDto(dto);
        if (CollectionUtils.isEmpty(list)) {
            return Lists.newArrayList();
        }
        List<TpmDeductionMatchingTemplateVo> voList = (List<TpmDeductionMatchingTemplateVo>) this.nebulaToolkitService.copyCollectionByWhiteList(list,TpmDeductionMatchingTemplate.class,TpmDeductionMatchingTemplateVo.class,HashSet.class,ArrayList.class);
        List<String> codes = voList.stream().map(TpmDeductionMatchingTemplateVo::getCode).collect(Collectors.toList());
        List<TpmDeductionMatchingTemplateRules> rulesList = this.tpmDeductionMatchingTemplateRulesRepository.findByCodes(codes);
        if (CollectionUtils.isNotEmpty(rulesList)) {
            List<TpmDeductionMatchingTemplateRulesVo> rulesVoList = (List<TpmDeductionMatchingTemplateRulesVo>) this.nebulaToolkitService.copyCollectionByWhiteList(rulesList,TpmDeductionMatchingTemplateRules.class,TpmDeductionMatchingTemplateRulesVo.class,HashSet.class,ArrayList.class);
            Map<String,List<TpmDeductionMatchingTemplateRulesVo>> rulesVoMap = rulesVoList.stream().collect(Collectors.groupingBy(TpmDeductionMatchingTemplateRulesVo::getCode));
            voList.forEach(e -> {
                if (rulesVoMap.containsKey(e.getCode())) {
                    e.setRulesList(rulesVoMap.get(e.getCode()));
                }
            });
        }
        return voList;
    }

    @Override
    public void updateByMappingUpdate(TpmDeductionMatchingTemplateDto dto) {
        if (Objects.isNull(dto) || StringUtils.isBlank(dto.getApplyMappingCode())) {
            return;
        }
        this.tpmDeductionMatchingTemplateRepository.updateByMappingUpdate(dto);
    }
}

