package com.biz.crm.tpm.business.audit.fee.validation.local.cache;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.mn.common.base.util.UuidCrmUtil;
import com.biz.crm.mn.common.page.cache.constant.MnPageCacheConstant;
import com.biz.crm.mn.common.page.cache.service.MnPageCacheHelper;
import com.biz.crm.tpm.business.audit.fee.validation.local.entity.AuditFeeValidationDetailEntity;
import com.biz.crm.tpm.business.audit.fee.validation.local.repository.AuditFeeValidationDetailRepository;
import com.biz.crm.tpm.business.audit.fee.validation.local.service.AuditFeeValidationInfoService;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.constant.AuditFeeValidationConstant;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.dto.AuditFeeValidationDetailDto;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.vo.AuditFeeValidationDetailVo;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.vo.AuditFeeValidationInfoVo;
import com.biz.crm.tpm.business.detailed.forecast.sdk.service.DetailedForecastService;
import com.biz.crm.tpm.business.detailed.forecast.sdk.vo.DetailedForecastVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Description 明细
 * @Author wei·yang
 * @Date Created in 2023/9/27 15:25
 */
@Component
@Slf4j
public class AuditFeeValidationDetailCacheHelper extends MnPageCacheHelper<AuditFeeValidationDetailVo, AuditFeeValidationDetailDto> {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private AuditFeeValidationDetailRepository auditFeeValidationDetailRepository;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private RedisTemplate<String, Object> redisTemplate;

    @Autowired(required = false)
    private AuditFeeValidationInfoService auditFeeValidationInfoService;

    @Autowired(required = false)
    private DetailedForecastService detailedForecastService;

    @Override
    public String getCacheKeyPrefix() {
        return AuditFeeValidationConstant.AUDIT_FEE_VALIDATION_DETAIL_CACHE_KEY;
    }

    @Override
    public Class<AuditFeeValidationDetailDto> getDtoClass() {
        return AuditFeeValidationDetailDto.class;
    }

    @Override
    public Class<AuditFeeValidationDetailVo> getVoClass() {
        return AuditFeeValidationDetailVo.class;
    }

    /**
     * 从数据库查询初始化数据
     *
     * @param dto
     * @param cacheKey
     * @return {@link List}<{@link AuditFeeValidationDetailDto}>
     */
    @Override
    public List<AuditFeeValidationDetailDto> findDtoListFromRepository(AuditFeeValidationDetailDto dto, String cacheKey) {
        if (!StringUtils.hasText(dto.getAuditCode())) {
            return Lists.newArrayList();
        }
        Assert.hasText(cacheKey, "cacheKey不能为空");
        //查询明细
        List<AuditFeeValidationDetailEntity> entities = this.auditFeeValidationDetailRepository.findByAuditCode(dto.getAuditCode());
        List<AuditFeeValidationDetailDto> detailDtos = (List<AuditFeeValidationDetailDto>) this.nebulaToolkitService
                .copyCollectionByWhiteList(entities, AuditFeeValidationDetailEntity.class,
                        AuditFeeValidationDetailDto.class, LinkedHashSet.class, ArrayList.class);
        //核销资料
        String auditCode = dto.getAuditCode();
        List<AuditFeeValidationInfoVo> auditFeeValidationInfoVos = this.auditFeeValidationInfoService.findByAuditCode(auditCode);
        //放入缓存
        this.auditFeeValidationInfoService.saveAuditInfoCache(cacheKey,auditFeeValidationInfoVos);

        return detailDtos;
    }

    /**
     * 是否初始化数据到缓存，默认为true
     *
     * @param
     * @return {@link boolean}
     */
    @Override
    public boolean initToCacheFromRepository() {
        return true;
    }

    @Override
    public void filterSaveItem(String cacheKey, List<Object> idList, List<AuditFeeValidationDetailDto> itemList, List<AuditFeeValidationDetailDto> updateList) {
        if (CollectionUtils.isEmpty(idList) || org.apache.commons.lang3.StringUtils.isBlank(cacheKey)) {
            return;
        }
        itemList.stream().distinct();
        String redisCacheIdKey = this.getRedisCacheIdKey(cacheKey);
        Set<String> idKeys = redisTemplate.keys(redisCacheIdKey + "*");
        List<AuditFeeValidationDetailDto> dataList = Lists.newArrayList();
        if (!CollectionUtils.isEmpty(idKeys)){
            for (String idKey : idKeys) {
                String dataKey = idKey.replace(MnPageCacheConstant.REDIS_KEY_ID,MnPageCacheConstant.REDIS_KEY_DATA);
                dataList.addAll((List)redisTemplate.opsForHash().multiGet(dataKey, idList));
            }
        }
        log.info("差异结案缓存dataList:{}",dataList.size());
        List<String> detailCodes = dataList.stream().map(AuditFeeValidationDetailDto::getDetailCode).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(detailCodes)) {
            return;
        }
        log.info("差异结案缓存detailCodes:{}", JSON.toJSONString(detailCodes));
        log.info("差异结案缓存itemList:{}",itemList.size());
        if (!CollectionUtils.isEmpty(itemList)) {
            itemList.removeIf(f -> detailCodes.contains(f.getDetailCode()));
        }
        log.info("差异结案缓存itemList:{}",itemList.size());

    }

    /**
     * 新增数据
     *
     *   里面根据追踪明细编码 查询了 细案预测。
     *
     * @param cacheKey
     * @param itemList
     * @return {@link List}<{@link AuditFeeValidationDetailDto}>
     */
    @Override
    public List<AuditFeeValidationDetailDto> newItem(String cacheKey, List<AuditFeeValidationDetailDto> itemList) {
        if(CollectionUtils.isEmpty(itemList)){
            return Lists.newArrayList();
        }else {
            List<AuditFeeValidationDetailDto> newItemDtoList = Lists.newArrayList();
            Map<String, List<AuditFeeValidationDetailDto>> itemMap = itemList.stream().collect(Collectors.groupingBy(AuditFeeValidationDetailDto::getDetailCode));
            for (Map.Entry<String, List<AuditFeeValidationDetailDto>> entry : itemMap.entrySet()) {
                if (CollectionUtils.isEmpty(entry.getValue())) {
                    continue;
                }
                if (entry.getValue().size() >= 2) {
                    newItemDtoList.add(entry.getValue().get(0));
                }else {
                    newItemDtoList.addAll(entry.getValue());
                }
            }

            Set<String> detailCodes = newItemDtoList.stream().map(AuditFeeValidationDetailDto::getDetailCode).collect(Collectors.toSet());
            List<DetailedForecastVo> detailedForecastVos = detailedForecastService.findByActivityDetailItemCode(detailCodes);
            Map<String, DetailedForecastVo> detailedForecastVoMap = detailedForecastVos.stream().collect(Collectors.toMap(DetailedForecastVo::getActivityDetailItemCode, Function.identity(), (v1, v2) -> v2));

            newItemDtoList.forEach(newItem->{
                newItem.setId(UUID.randomUUID().toString().replace("-", ""));
                newItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                newItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                newItem.setApplyAmount(newItem.getFeeAmount());
                //新选择的明细，默认是否完全结案设置为’是‘
                newItem.setWholeAudit(BooleanEnum.TRUE.getCapital());
                if (org.apache.commons.lang3.StringUtils.isNotBlank(newItem.getAuditDetailCode()) && !newItem.getAuditDetailCode().startsWith(AuditFeeValidationConstant.AUDIT_FEE_DIFF_VALIDATION_DETAIL)) {
                    newItem.setAuditDetailCode(null);
                }
                if (org.apache.commons.lang3.StringUtils.isNotBlank(newItem.getDetailCode()) && !CollectionUtils.isEmpty(detailedForecastVoMap)) {
                    DetailedForecastVo detailedForecastVo = detailedForecastVoMap.get(newItem.getDetailCode());
                    if (!Objects.isNull(detailedForecastVo)) {
                        newItem.setAuditAmount(detailedForecastVo.getEstimatedWriteOffAmount());
                    }
                }
            });
            return newItemDtoList;
        }
    }

    /**
     * 更新数据
     */
    @Override
    public void updateItem(String cacheKey, List<AuditFeeValidationDetailDto> itemList) {
    }

    /**
     * 复制数据
     *
     * @param cacheKey
     * @param itemList
     * @return {@link List}<{@link AuditFeeValidationDetailDto}>
     */
    @Override
    public List<AuditFeeValidationDetailDto> copyItem(String cacheKey, List<AuditFeeValidationDetailDto> itemList) {
        List<AuditFeeValidationDetailDto> newList = (List<AuditFeeValidationDetailDto>) this.nebulaToolkitService
                .copyCollectionByBlankList(itemList, AuditFeeValidationDetailDto.class,
                        AuditFeeValidationDetailDto.class, HashSet.class, ArrayList.class);
        for (AuditFeeValidationDetailDto newItem : newList) {
            newItem.setId(UuidCrmUtil.general());
            newItem.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            newItem.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            newItem.setTenantCode(TenantUtils.getTenantCode());
        }
        return newList;
    }

    /**
     * 获取唯一key
     *
     * @param dto
     * @return {@link Object}
     */
    @Override
    public Object getDtoKey(AuditFeeValidationDetailDto dto) {
        return dto.getId();
    }

    /**
     * 获取是否选中状态
     *
     * @param dto
     * @return {@link String}
     */
    @Override
    public String getCheckedStatus(AuditFeeValidationDetailDto dto) {
        return dto.getChecked();
    }


}
