package com.biz.crm.tpm.business.audit.fee.validation.local.controller;

import cn.hutool.core.lang.Snowflake;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.audit.fee.sdk.dto.track.AuditFeeDiffTrackDetailDto;
import com.biz.crm.tpm.business.audit.fee.sdk.service.ledger.AuditFeeDiffLedgerLockService;
import com.biz.crm.tpm.business.audit.fee.validation.local.service.AuditFeeValidationInfoService;
import com.biz.crm.tpm.business.audit.fee.validation.local.service.AuditFeeValidationService;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.constant.AuditFeeValidationConstant;
import com.biz.crm.tpm.business.audit.fee.validation.sdk.dto.AuditFeeValidationDto;
import com.bizunited.nebula.security.sdk.login.UserIdentity;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 商超费用结案核销主表
 *
 * @author weiÂ·yang
 * @date 2023-09-26 17:23:21
 */
@Slf4j
@RestController
@RequestMapping("/v1/auditFeeValidation/validation")
@Api(tags = "业务模块: auditFeeValidation: 商超费用结案核销主表")
public class AuditFeeValidationController {

    @Autowired(required = false)
    private AuditFeeValidationService auditFeeValidationService;

    @Autowired(required = false)
    private AuditFeeValidationInfoService auditFeeValidationInfoService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private AuditFeeDiffLedgerLockService auditFeeDiffLedgerLockService;

    /**
     * 新增
     *
     * @param dto
     * @return {@link Result}<{@link ?}>
     */
    @ApiOperation(value = "新增编辑")
    @PostMapping("")
    public Result<?> createOrUpdate(@RequestBody AuditFeeValidationDto dto) {
        String lockKey = AuditFeeValidationConstant.AUDIT_FEE_EDIT_LOCK + dto.getId();
        boolean locked = false;
        String currentOperateId = new Snowflake().nextIdStr();
        try {
            if (StringUtils.isNotEmpty(dto.getId())) {
                locked = redisLockService.tryLock(lockKey, TimeUnit.MINUTES, 30, 1);
                Validate.isTrue(locked, "数据正在编辑，请稍后重试");
            }
            dto.setCurrentOperateId(currentOperateId);
            this.auditFeeValidationService.createOrUpdate(dto);
            return Result.ok();
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (locked) {
                redisLockService.unlock(lockKey);
            }
            String redisKey = String.format(AuditFeeValidationConstant.AUDIT_FEE_VALIDATION_AMOUNT_LOCK, currentOperateId);
            if (this.redisService.hasKey(redisKey)) {
                List<String> obj = (List<String>) this.redisService.get(redisKey);
                this.auditFeeDiffLedgerLockService.unlock(obj);
                this.redisService.del(redisKey);
            }
        }
    }

    @ApiOperation("提交审批流")
    @PostMapping("submitApproval")
    public Result<?> submitApproval(@RequestBody AuditFeeValidationDto dto) {

        String lockKey = AuditFeeValidationConstant.AUDIT_FEE_SUBMIT_APPROVAL + dto.getAuditCode();
        boolean locked = false;
        String currentOperateId = new Snowflake().nextIdStr();
        try {
            locked = redisLockService.tryLock(lockKey, TimeUnit.MINUTES, 30, 1);
            Validate.isTrue(locked, "正在提交，请稍后");
            dto.setCurrentOperateId(currentOperateId);
            this.auditFeeValidationService.submitApproval(dto);
            return Result.ok();
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (locked) {
                redisLockService.unlock(lockKey);
            }
            String redisKey = String.format(AuditFeeValidationConstant.AUDIT_FEE_VALIDATION_AMOUNT_LOCK, currentOperateId);
            if (this.redisService.hasKey(redisKey)) {
                List<String> obj = (List<String>) this.redisService.get(redisKey);
                this.auditFeeDiffLedgerLockService.unlock(obj);
                this.redisService.del(redisKey);
            }
        }
    }

    @ApiOperation("批量关联")
    @GetMapping("batchRelevance")
    public Result<?> batchRelevance(AuditFeeDiffTrackDetailDto dto) {
        try {
            Validate.notNull(dto, "输入参数不能为空");
            Validate.notBlank(dto.getCacheKey(), "缓存key不能为空");
            UserIdentity loginUser = loginUserService.getLoginUser();
            this.auditFeeValidationService.batchRelevance(loginUser, dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 删除
     *
     * @param ids
     * @return {@link Result}<{@link ?}>
     */
    @ApiOperation(value = "删除")
    @DeleteMapping("/deleteBatch")
    public Result<?> deleteBatch(@RequestParam("ids") List<String> ids) {
        try {
            this.auditFeeValidationService.deleteBatch(ids);
            return Result.ok("删除成功");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 启用
     *
     * @param ids
     * @return {@link Result}<{@link ?}>
     */
    @ApiOperation(value = "启用")
    @PatchMapping("/enableBatch")
    public Result<?> enableBatch(@RequestBody List<String> ids) {
        try {
            this.auditFeeValidationService.enableBatch(ids);
            return Result.ok("启用成功");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 禁用
     *
     * @param ids
     * @return {@link Result}<{@link ?}>
     */
    @ApiOperation(value = "禁用")
    @PatchMapping("/disableBatch")
    public Result<?> disableBatch(@RequestBody List<String> ids) {
        try {
            this.auditFeeValidationService.disableBatch(ids);
            return Result.ok("禁用成功");
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("保存核销资料到缓存中")
    @PostMapping("saveAuditInfoCache")
    public Result<?> saveAuditInfoCache(@RequestBody AuditFeeValidationDto dto) {
        try {
            this.auditFeeValidationInfoService.saveAuditInfoCache(dto);
            return Result.ok();
        } catch (RuntimeException e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
}
