package com.biz.crm.tpm.business.budget.cal.config.local.cache;

import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.mn.common.page.cache.service.MnPageCacheHelper;
import com.biz.crm.tpm.business.budget.cal.config.local.entity.BudgetCalConfigProductRatioEntity;
import com.biz.crm.tpm.business.budget.cal.config.local.repository.BudgetCalConfigProductRatioRepository;
import com.biz.crm.tpm.business.budget.cal.config.sdk.dto.BudgetCalConfigProductRatioDto;
import com.biz.crm.tpm.business.budget.cal.config.sdk.vo.BudgetCalConfigProductRatioVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Maps;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 活动细案明细分页缓存
 * @author wanghaojia
 * @date 2022/12/12 20:11
 */
@Slf4j
@Component
public class BudgetCalConfigProductRatioCacheHelper extends MnPageCacheHelper<BudgetCalConfigProductRatioVo, BudgetCalConfigProductRatioDto> {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private BudgetCalConfigProductRatioRepository budgetCalConfigProductRatioRepository;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    @Autowired(required = false)
    private RedisTemplate<String, Object> redisTemplate;

    /**
     * 缓存键前缀
     * @param
     * @return {@link String}
     */
    @Override
    public String getCacheKeyPrefix() {
        return "tpm_budget_cal_config_product_ratio_key:";
    }

    /**
     * Dto Class对象
     * @param
     * @return {@link Class}<{@link BudgetCalConfigProductRatioDto}>
     */
    @Override
    public Class<BudgetCalConfigProductRatioDto> getDtoClass() {
        return BudgetCalConfigProductRatioDto.class;
    }

    /**
     * Vo Class对象
     * @param
     * @return {@link Class}<{@link BudgetCalConfigProductRatioVo}>
     */
    @Override
    public Class<BudgetCalConfigProductRatioVo> getVoClass() {
        return BudgetCalConfigProductRatioVo.class;
    }

    /**
     * 从数据库查询初始化数据
     * @param budgetCalConfigProductRatioDto
     * @param cacheKey
     * @return {@link List}<{@link BudgetCalConfigProductRatioDto}>
     */
    @Override
    public List<BudgetCalConfigProductRatioDto> findDtoListFromRepository(BudgetCalConfigProductRatioDto dto, String cacheKey) {
        //预算计算配置编码
        String budgetCalCode = dto.getBudgetCalCode();
        if(!StringUtils.hasText(budgetCalCode)){
            return null;
        }
        List<BudgetCalConfigProductRatioEntity> entities = this.budgetCalConfigProductRatioRepository
            .listByCalConfigCode(dto);
        List<BudgetCalConfigProductRatioDto> ratioDtos =
            (List<BudgetCalConfigProductRatioDto>) this.nebulaToolkitService
                .copyCollectionByWhiteList(entities, BudgetCalConfigProductRatioEntity.class,
            BudgetCalConfigProductRatioDto.class, LinkedHashSet.class, ArrayList.class);
        //不走筛选
        if(!CollectionUtils.isEmpty(ratioDtos)){
            //是否扣减默认为否
            ratioDtos.forEach(o -> {
                if (StringUtil.isBlank(o.getReduceTag())) {
                    o.setReduceTag(BooleanEnum.FALSE.getCapital());
                }
            });

            //获取缓存key
            String redisCacheIdKey = getRedisCacheIdKey(cacheKey);
            String redisCacheDataKey = getRedisCacheDataKey(cacheKey);
            //数据集合
            ratioDtos.forEach(o -> {
                o.setId(cacheKey + "-" + o.getProductCode());
            });
            Map<String, BudgetCalConfigProductRatioDto> ratioDtoMap = ratioDtos.stream()
                .collect(Collectors.toMap(BudgetCalConfigProductRatioDto::getId, Function.identity()));
            //保存到缓存中
            redisService.lPushAll(redisCacheIdKey, this.getExpireTime(), ratioDtoMap.keySet().toArray());
            redisTemplate.opsForHash().putAll(redisCacheDataKey, ratioDtoMap);
            redisService.expire(redisCacheDataKey, this.getExpireTime());
        }
        return ratioDtos;
    }

    /**
     * 是否初始化数据到缓存，默认为true
     * @param
     * @return {@link boolean}
     */
    @Override
    public boolean initToCacheFromRepository() {
        return false;
    }

    /**
     * 新增数据
     * @param cacheKey
     * @param itemList
     * @return {@link List}<{@link BudgetCalConfigProductRatioDto}>
     */
    @Override
    public List<BudgetCalConfigProductRatioDto> newItem(String cacheKey, List<BudgetCalConfigProductRatioDto> itemList) {
        String redisCacheIdKey = this.getRedisCacheIdKey(cacheKey);
        String redisCacheDataKey = this.getRedisCacheDataKey(cacheKey);
        //获取已缓存的数据
        List<Object> idList = redisService.lRange(redisCacheIdKey, 0, -1);
        //过滤掉已经存在的数据
        List<BudgetCalConfigProductRatioDto> newList = itemList.stream()
            .filter(item -> !idList.contains(cacheKey + "-" + item.getProductCode())).collect(Collectors.toList());
        Assert.notEmpty(newList,"添加的数据全部重复");
        //获取缓存key
        newList.forEach(dto -> {
            dto.setId(cacheKey + "-" + dto.getProductCode());
            dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            dto.setTenantCode(TenantUtils.getTenantCode());
            dto.setReduceTag(BooleanEnum.FALSE.getCapital());
        });
        return newList;
    }

    /**
     * 更新数据
     */
    public void updateItem(String cacheKey,List<BudgetCalConfigProductRatioDto> itemList){

    }

    /**
     * 复制数据
     * @param cacheKey
     * @param itemList
     * @return {@link List}<{@link BudgetCalConfigProductRatioDto}>
     */
    @Override
    public List<BudgetCalConfigProductRatioDto> copyItem(String cacheKey, List<BudgetCalConfigProductRatioDto> itemList) {
        return null;
    }

    /**
     * 获取Dto中的主键
     * @param budgetCalConfigProductRatioDto
     * @return {@link Object}
     */
    @Override
    public Object getDtoKey(BudgetCalConfigProductRatioDto budgetCalConfigProductRatioDto) {
        return budgetCalConfigProductRatioDto.getId();
    }

    /**
     * 获取是否选中状态
     * @param budgetCalConfigProductRatioDto
     * @return {@link String}
     */
    @Override
    public String getCheckedStatus(BudgetCalConfigProductRatioDto budgetCalConfigProductRatioDto) {
        return budgetCalConfigProductRatioDto.getChecked();
    }

    /**
     * 导入数据存入缓存
     * @param cacheKey
     * @param dto
     */
    public void importSave(String cacheKey, BudgetCalConfigProductRatioDto dto) {
        String redisCacheIdKey = this.getRedisCacheIdKey(cacheKey);
        String redisCacheDataKey = this.getRedisCacheDataKey(cacheKey);
        dto.setId(cacheKey + "-" + dto.getProductCode());
        //获取已缓存的数据
        List<Object> idList = redisService.lRange(redisCacheIdKey, 0, -1);
        //判断是否已经存在
        if(!idList.contains(dto.getId())){
            //缓存ID
            Object[] newIdArr = {dto.getId()};
            redisService.lPushAll(redisCacheIdKey, this.getExpireTime(),newIdArr );
        }

        //导入新增数据
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setTenantCode(TenantUtils.getTenantCode());

        //缓存数据
        Map<Object, BudgetCalConfigProductRatioDto> ratioDtoMap = Maps.newHashMap();
        ratioDtoMap.put(dto.getId(), dto);

        redisTemplate.opsForHash().putAll(redisCacheDataKey, ratioDtoMap);
        redisService.expire(redisCacheDataKey, this.getExpireTime());
    }
}
