package com.biz.crm.tpm.business.budget.cal.config.local.imports;

import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mn.common.base.util.ExceptionStackMsgUtil;
import com.biz.crm.tpm.business.budget.cal.config.local.cache.BudgetCalConfigProductRatioCacheHelper;
import com.biz.crm.tpm.business.budget.cal.config.sdk.dto.BudgetCalConfigProductRatioDto;
import com.biz.crm.tpm.business.budget.cal.config.sdk.vo.BudgetCalConfigProductRatioImportVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * @Description 月度预算导入
 * @Author YangWei
 * @Date 2023/2/2 下午4:17
 */
@Slf4j
@Component
public class BudgetCalConfigProductRatioImportProcess implements ImportProcess<BudgetCalConfigProductRatioImportVo> {

    @Autowired(required = false)
    @Qualifier("nebulaToolkitService")
    private NebulaToolkitService nebulaToolkitService;

    @Resource
    private DictDataVoService dictDataVoService;

    @Resource
    private ProductVoService productVoService;

    @Resource
    private BudgetCalConfigProductRatioCacheHelper budgetCalConfigProductRatioCacheHelper;

    /**
     * 数据字典
     */
    private Map<String, List<DictDataVo>> dictMap = Maps.newHashMap();

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Map<Integer, String> execute(LinkedHashMap<Integer, BudgetCalConfigProductRatioImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        final Optional<BudgetCalConfigProductRatioImportVo> first = data.values().stream().findFirst();
        if (!first.isPresent()) {
            return null;
        }
        BudgetCalConfigProductRatioImportVo vo = first.get();
        try {
            //初始数据字典
            //this.initDict();
            //校验
            BudgetCalConfigProductRatioDto dto = this.validate(vo);
            //保存
            String cacheKey = String.valueOf(params.get("cacheKey"));
            Assert.hasText(cacheKey,"cacheKey不能为空");
            this.budgetCalConfigProductRatioCacheHelper.importSave(cacheKey, dto);
        } catch (IllegalArgumentException e) {
            throw e;
        } catch (Exception e) {
            String error = ExceptionStackMsgUtil.stackMsg(e);
            log.error("产品计费比例导入异常:{}", error);
            throw new IllegalArgumentException("导入异常[" + error + "]");
        }
        return null;
    }

    /**
     * 初始数据字典
     * @param
     */
    private void initDict() {
        this.dictMap = this.dictDataVoService.findByDictTypeCodeList(Lists.newArrayList());
    }

    /**
     * 校验
     * @param vo
     * @return {@link MonthBudgetDto}
     */
    private BudgetCalConfigProductRatioDto validate(BudgetCalConfigProductRatioImportVo vo) {
        //非空校验
        this.verifyEmpty(vo);
        //查询商品信息
        String productCode = vo.getProductCode();
        List<ProductVo> productVos = this.productVoService.findByCodes(Lists.newArrayList(productCode));
        Assert.notEmpty(productVos,"未查询到商品信息");
        ProductVo productVo = productVos.get(0);
        if (StringUtils.isNotBlank(vo.getReduceTag())) {
            if (BooleanEnum.FALSE.getSure().equals(vo.getReduceTag())) {
                vo.setReduceTag(BooleanEnum.FALSE.getCapital());
            } else if (BooleanEnum.TRUE.getSure().equals(vo.getReduceTag())) {
                vo.setReduceTag(BooleanEnum.TRUE.getCapital());
            } else {
                Validate.isTrue(false, "是否扣减无法识别");
            }
        } else {
            vo.setReduceTag(BooleanEnum.FALSE.getCapital());
        }
        //结果
        BudgetCalConfigProductRatioDto dto = new BudgetCalConfigProductRatioDto();
        dto.setId(productVo.getId());
        dto.setChargedRatio(vo.getChargedRatio());
        dto.setProductCode(productVo.getProductCode());
        dto.setProductName(productVo.getProductName());
        dto.setProductBrandCode(productVo.getProductBrandCode());
        dto.setProductBrandName(productVo.getProductBrandName());
        dto.setProductCategoryCode(productVo.getProductCategoryCode());
        dto.setProductCategoryName(productVo.getProductCategoryName());
        dto.setProductItemCode(productVo.getProductLevelCode());
        dto.setProductItemName(productVo.getProductLevelName());
        dto.setReduceTag(vo.getReduceTag());
        return dto;
    }

    /**
     * 校验数据字典
     * @param vo
     */
    private void verifyDictCode(BudgetCalConfigProductRatioImportVo vo) {

    }

    /**
     * 非空校验
     * @param vo
     */
    private void verifyEmpty(BudgetCalConfigProductRatioImportVo vo) {
        Assert.hasText(vo.getProductCode(),"商品编码不能为空");
        Assert.notNull(vo.getChargedRatio(),"计费比例不能为空");
    }

    /**
     * 获取字典对应的value值
     *
     * @param dictValue
     * @param typeCode
     * @return
     */
    private String findDictValue(String dictValue, String typeCode) {
        DictDataVo dictDataVo = this.dictMap.get(typeCode).stream()
            .filter(a -> a.getDictValue().equals(dictValue)).findFirst().orElse(null);
        if (!ObjectUtils.isEmpty(dictDataVo)) {
            return dictDataVo.getDictCode();
        }
        return null;
    }

    /**
     * 获取字典对应的code值
     *
     * @param dictCode
     * @param typeCode
     * @return
     */
    private String findDictCode(String dictCode, String typeCode) {
        DictDataVo dictDataVo = this.dictMap.get(typeCode).stream()
            .filter(a -> a.getDictCode().equals(dictCode)).findFirst().orElse(null);
        if (!ObjectUtils.isEmpty(dictDataVo)) {
            return dictDataVo.getDictCode();
        }
        return null;
    }

    @Override
    public Integer getBatchCount() {
        //一次处理一条
        return 1;
    }

    @Override
    public Class<BudgetCalConfigProductRatioImportVo> findCrmExcelVoClass() {
        return BudgetCalConfigProductRatioImportVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "TPM_BUDGET_CAL_CONFIG_PRODUCT_RATIO_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "TPM-预算计算配置产品比例导入模版";
    }

    @Override
    public String getBusinessCode() {
        return "TPM_BUDGET_CAL_CONFIG_IMPORT";
    }

    @Override
    public String getBusinessName() {
        return "TPM-预算计算配置导入";
    }

}