package com.biz.crm.tpm.business.budget.discount.rate.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.budget.discount.rate.local.service.DiscountRateService;
import com.biz.crm.tpm.business.budget.discount.rate.local.service.SurplusFeePoolBalanceService;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.constant.DiscountRateConstant;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.*;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.service.DiscountRateSdkService;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.vo.DiscountRateBudgetVo;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.vo.DiscountRateVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @author: chenlong
 * @date: 2023/1/5 21:01
 * @description: 折扣率管理(DiscountRate)表相关的http接口
 */
@RestController
@RequestMapping("/v1/discountRate")
@Slf4j
@Api(tags = "折扣率管理")
public class DiscountRateController {

    @Autowired(required = false)
    private DiscountRateService discountRateService;
    @Autowired(required = false)
    private DiscountRateSdkService discountRateSdkService;
    @Autowired(required = false)
    private RedisLockService redisLockService;
    @Autowired(required = false)
    private SurplusFeePoolBalanceService surplusFeePoolBalanceService;

    /**
     * 分页查询所有数据
     *
     * @param pageable 分页对象
     * @param rateDto  查询dto
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findByConditions")
    public Result<Page<DiscountRateVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                         @ApiParam(name = "rateDto", value = "折扣率Dto") DiscountRateDto rateDto) {
        try {
            Page<DiscountRateVo> page = this.discountRateSdkService.findByConditions(pageable, rateDto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @ApiOperation(value = "通过主键查询单条数据")
    @GetMapping("{id}")
    public Result<DiscountRateVo> findById(@PathVariable @ApiParam(name = "id", value = "主键id") String id) {
        try {
            DiscountRateVo rateVo = this.discountRateService.findById(id);
            return Result.ok(rateVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

//    /**
//     * 新增数据
//     *
//     * @param configDto dto对象
//     * @return 新增结果
//     */
//    @ApiOperation(value = "新增数据")
//    @PostMapping
//    public Result<?> create(@ApiParam(name = "configDto", value = "折扣率配置主表Dto") @RequestBody DiscountRateConfigDto configDto) {
//        try {
//            this.discountRateConfigService.create(configDto);
//            return Result.ok();
//        } catch (Exception e) {
//            log.error(e.getMessage(), e);
//            return Result.error(e.getMessage());
//        }
//    }

//    /**
//     * 修改数据
//     *
//     * @param rateDto dto对象
//     * @return 修改结果
//     */
//    @ApiOperation(value = "修改数据")
//    @PatchMapping
//    public Result<?> update(@ApiParam(name = "rateDto", value = "折扣率Dto") @RequestBody DiscountRateDto rateDto) {
//        try {
//            this.discountRateService.update(rateDto);
//            return Result.ok();
//        } catch (Exception e) {
//            log.error(e.getMessage(), e);
//            return Result.error(e.getMessage());
//        }
//    }

    /**
     * 删除数据
     *
     * @param ids 主键结合
     * @return 删除结果
     */
    @ApiOperation(value = "删除数据")
    @DeleteMapping("/delete")
    public Result<?> delete(@RequestParam("ids") List<String> ids) {
        try {
            this.discountRateService.delete(ids);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 启用
     *
     * @param ids 主键列表
     * @return 启用结果
     */
    @ApiOperation(value = "启用", httpMethod = "PATCH")
    @PatchMapping("/enable")
    public Result<?> enable(@RequestBody List<String> ids) {
        try {
            this.discountRateService.enableBatch(ids);
            return Result.ok("启用成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 禁用
     *
     * @param ids 主键列表
     * @return 禁用结果
     */
    @ApiOperation(value = "禁用", httpMethod = "PATCH")
    @PatchMapping("/disable")
    public Result<?> disable(@RequestBody List<String> ids) {
        try {
            this.discountRateService.disableBatch(ids);
            return Result.ok("禁用成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 提交审批流
     *
     * @param promotionPlanSubmitDto
     * @return
     */
    @ApiOperation(value = "提交审批流")
    @PostMapping("submitApproval")
    public Result<?> submitApproval(@ApiParam(name = "processDto", value = "提交审批dto") @RequestBody DiscountRateProcessDto processDto) {
        String lockKey = "";
        boolean lockSuccess = false;
        try {
            Validate.notNull(processDto.getId(), "请选择数据！");
            //提交审批前加锁一个小时
            lockKey = DiscountRateConstant.DISCOUNT_RATE_LOCK_APPROVE + processDto.getId();
            lockSuccess = redisLockService.tryLock(lockKey, TimeUnit.HOURS, 1);
            Validate.isTrue(lockSuccess, "此折扣率正在提交审批，请勿重复提交");
            this.discountRateService.submitApproval(processDto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess) {
                redisLockService.unlock(lockKey);
            }
        }
    }

    /**
     * 更新计算
     *
     * @param promotionPlanSubmitDto
     * @return
     */
    @ApiOperation(value = "更新计算")
    @PostMapping("recal")
    public Result<?> recal(@ApiParam(name = "recalDto", value = "更新计算dto") @RequestBody DiscountRateRecalDto recalDto) {
        String lockKey = "";
        boolean lockSuccess = false;
        try {
//            Validate.notNull(processDto.getId(), "请选择数据！");
            //更新计算前加锁一个小时
            lockKey = DiscountRateConstant.DISCOUNT_RATE_LOCK_APPROVE;
            lockSuccess = redisLockService.tryLock(lockKey, TimeUnit.HOURS, 1);
            Validate.isTrue(lockSuccess, "已有更新计算任务在执行，请稍等");
            this.discountRateService.recal(recalDto);
            return Result.ok("更新计算任务已开始");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess) {
                redisLockService.unlock(lockKey);
            }
        }
    }

    /**
     * 垂直根据预算获取折扣率
     *
     * @param budgetDtoList 预算数据
     * @return Result<Map < String, DiscountRateBudgetVo>>
     */
    @ApiOperation(value = "垂直根据预算获取折扣率")
    @PostMapping("getDiscountRateByDimensionCode")
    public Result<Map<String, DiscountRateBudgetVo>> getDiscountRateByDimensionCode(@ApiParam(name = "budgetDtoList", value = "折扣率预算dto列表") @RequestBody List<DiscountRateBudgetDto> budgetDtoList) {
        try {
            Map<String, DiscountRateBudgetVo> map = this.discountRateSdkService.getDiscountRateByDimensionCode(budgetDtoList);
            return Result.ok(map);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 销售计划获取折扣率
     *
     * @param calDto 预算数据
     * @return Result<List < DiscountRateDto>>
     */
    @ApiOperation(value = "销售计划获取折扣率")
    @PostMapping("findRateByPlanConditions")
    public Result<List<DiscountRateDto>> findRateByPlanConditions(@RequestBody DiscountRatePlanCalDto calDto) {
        try {
            List<DiscountRateDto> list = this.discountRateSdkService.findRateByPlanConditions(calDto);
            return Result.ok(list);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 测试定时任务自动生成折扣率
     *
     * @return Result<?>
     */
    @ApiOperation(value = "测试定时任务自动生成折扣率")
    @PostMapping("textVerticalDiscountRateCal")
    public Result<?> textVerticalDiscountRateCal(@RequestParam("yearMonth")String yearMonth) {
        try {
            this.discountRateService.verticalDiscountRateCal(yearMonth);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 测试定时任务自动生成期初
     *
     * @return Result<?>
     */
    @ApiOperation(value = "测试定时任务自动生成期初")
    @GetMapping("textVerticalSurplusFeePoolBalanceCal")
    public Result<?> textVerticalSurplusFeePoolBalanceCal(@RequestParam("yearMonth") String yearMonth) {
        try {
            this.surplusFeePoolBalanceService.verticalSurplusFeePoolBalanceCal(yearMonth);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


}
