package com.biz.crm.tpm.business.budget.discount.rate.local.process;

import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerOrgChannelStoresVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.enums.YesOrNoEnum;
import com.biz.crm.tpm.business.budget.discount.rate.local.entity.DiscountRateConfig;
import com.biz.crm.tpm.business.budget.discount.rate.local.service.DiscountRateConfigService;
import com.biz.crm.tpm.business.budget.discount.rate.local.service.DiscountRateService;
import com.biz.crm.tpm.business.budget.discount.rate.local.service.DiscountRateVariableService;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.constant.DiscountRateConfigConstant;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.CalculateDto;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.DiscountRateDto;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.enums.DiscountRateDimensionEnum;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.vo.DiscountRateCustomerImportVo;
import com.biz.crm.tpm.business.sales.plan.sdk.dto.SalesPlanSummaryDto;
import com.biz.crm.tpm.business.sales.plan.sdk.enums.SummaryDimensionEunm;
import com.biz.crm.tpm.business.sales.plan.sdk.service.SalesPlanService;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.*;

/**
 * @author: chenlong
 * @date: 2022/11/15 15:29
 * @description: 折扣率表(DiscountRate)导入实现层-客户维度
 */
@Slf4j
@Component
public class DiscountRateCustomerImportProcess implements ImportProcess<DiscountRateCustomerImportVo> {

    @Autowired(required = false)
    private DiscountRateService discountRateService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;
    @Autowired(required = false)
    private CustomerVoService customerVoService;
    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;
    @Autowired(required = false)
    private DiscountRateVariableService discountRateVariableService;
    @Autowired(required = false)
    private DiscountRateConfigService discountRateConfigService;
    @Autowired(required = false)
    private SalesPlanService salesPlanService;

    /**
     * 数据分片
     *
     * @return 分片长度
     */
    @Override
    public Integer getBatchCount() {
        return 50001;
    }

    /**
     * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
     * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
     *
     * @return
     */
    @Override
    public boolean importBeforeValidationFlag() {
        return Boolean.TRUE;
    }

    /**
     * 数据校验
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> tryVerify(LinkedHashMap<Integer, DiscountRateCustomerImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        for (Map.Entry<Integer, DiscountRateCustomerImportVo> row : data.entrySet()) {
            int rowNum = row.getKey();
            DiscountRateCustomerImportVo vo = row.getValue();
            this.validateIsTrue(StringUtils.isNotBlank(vo.getBusinessFormatCode()), "业态不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getBusinessUnitCode()), "业务单元不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getRateVersion()), "版本不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getBusinessDepartment()), "业务部门不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getYearAndMonthStr()), "年月不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getCustomerCode()), "客户编码不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getSalesInstitutionCode()), "销售机构编码不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(vo.getChannelCode()), "渠道编码不能为空");
            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(rowNum, errInfo);
            }
        }
        return errMap;
    }

    /**
     * 数据处理
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, DiscountRateCustomerImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        Validate.notNull(data, "导入数据不能为空！");
        Validate.isTrue(data.values().size() <= 50000, "单次导入数据不能超过50000条");
        List<DiscountRateDto> importList = this.validate(data, errMap);
        if (errMap.isEmpty()) {
            log.error("开始执行大批量保存-----------------");
            //大批量保存
            discountRateService.bulkImportSave(importList);
            log.error("导入逻辑处理完毕-----------------");
        }
        return errMap;
    }

    /**
     * 校验数据
     *
     * @param data data
     **/
    private List<DiscountRateDto> validate(LinkedHashMap<Integer, DiscountRateCustomerImportVo> data, Map<Integer, String> errMap) {
        List<DiscountRateDto> list = new ArrayList<>();
//        //先获取销售机构的erpCode
//        Set<String> orgCodes = new HashSet<>();
//        data.forEach((k, v) -> {
//            Validate.isTrue(StringUtils.isNotBlank(v.getSalesInstitutionCode()), "[行%s]销售机构编码不能为空", k);
//            orgCodes.add(v.getSalesInstitutionCode());
//        });
//        Map<String, String> orgMap = salesOrgVoService.findSapCodesBySalesOrgCodes(new ArrayList<>(orgCodes));
        //获取业务单元字典
        Map<String, String> unitMap = dictToolkitService.findConvertMapByDictTypeCode(DiscountRateConfigConstant.DISCOUNT_RATE_BUSINESS_UNIT);
        //获取业态字典
        Map<String, String> formatMap = dictToolkitService.findConvertMapByDictTypeCode(DiscountRateConfigConstant.DISCOUNT_RATE_BUSINESS_FORMAT);
        //获取业务部门型字典
        Map<String, String> departMap = dictToolkitService.findConvertMapByDictTypeCode(DiscountRateConfigConstant.DISCOUNT_RATE_BUSINESS_DEPARTMENT);
        //获取版本字典
        Map<String, String> versionMap = dictToolkitService.findConvertMapByDictTypeCode(DiscountRateConfigConstant.DISCOUNT_RATE_RATE_VERSION);

        //客户维度
        String type = DiscountRateDimensionEnum.CUSTOMER.getCode();
        data.forEach((k, v) -> {
            list.add(this.validateData(k, v, type, formatMap, unitMap, departMap, versionMap, errMap));
        });

        if (!errMap.isEmpty()) {
            return new ArrayList<>();
        }

        Set<String> cusCodes = new HashSet<>();
        for (DiscountRateDto vo : list) {
            cusCodes.add(vo.getCustomerCode());
        }
        List<List<String>> cusList = Lists.partition(new ArrayList<>(cusCodes), 500);
        //获取客户绑定信息
        Map<String, CustomerOrgChannelStoresVo> cusMap = customerVoService.getCustomerByCodes(cusList);
        this.validateCorrect(list, cusMap, errMap);

        if (!errMap.isEmpty()) {
            return new ArrayList<>();
        }

        return new ArrayList<>(list);


    }


    /**
     * 校验数据
     *
     * @param index 行号
     * @param dto   行数据
     **/
    private DiscountRateDto validateData(Integer index, DiscountRateCustomerImportVo dto, String type,
                                         Map<String, String> formatMap, Map<String, String> unitMap,
                                         Map<String, String> departMap, Map<String, String> versionMap, Map<Integer, String> errMap) {
        DiscountRateDto rate = new DiscountRateDto();
        rate.setBusinessFormatCode(dto.getBusinessFormatCode());
        rate.setBusinessUnitCode(dto.getBusinessUnitCode());
        rate.setRateVersion(dto.getRateVersion());
        rate.setBusinessDepartment(dto.getBusinessDepartment());
        if (formatMap.containsKey(dto.getBusinessFormatCode())) {
            rate.setBusinessFormatCode(formatMap.get(dto.getBusinessFormatCode()));
        } else {
            this.validateIsTrue(false, "业态未能识别");
        }
        rate.setCustomerErpCode(dto.getCustomerCode());
        rate.setCustomerCode(dto.getCustomerCode() + dto.getSalesInstitutionCode() + dto.getChannelCode() + rate.getBusinessFormatCode());
        if (unitMap.containsKey(dto.getBusinessUnitCode())) {
            rate.setBusinessUnitCode(unitMap.get(dto.getBusinessUnitCode()));
        } else {
            this.validateIsTrue(false, "业务单元未能识别");
        }
        if (versionMap.containsKey(dto.getRateVersion())) {
            rate.setRateVersion(versionMap.get(dto.getRateVersion()));
        } else {
            this.validateIsTrue(false, "版本未能识别");
        }
        if (departMap.containsKey(dto.getBusinessDepartment())) {
            rate.setBusinessDepartment(departMap.get(dto.getBusinessDepartment()));
        } else {
            this.validateIsTrue(false, "业务部门未能识别");
        }
        try {
            rate.setYearAndMonth(DateUtil.date_yyyy_MM.parse(dto.getYearAndMonthStr()));
            rate.setYearAndMonthStr(dto.getYearAndMonthStr());
            if (dto.getYearAndMonthStr().trim().length() != 7) {
                this.validateIsTrue(false, "年月格式错误【yyyy-MM】");
            }
        } catch (Exception e) {
            String msg = "年月格式错误【yyyy-MM】";
            this.validateIsTrue(false, msg);
        }
        if (StringUtils.isNotBlank(dto.getSystemRate())) {
            try {
                rate.setSystemRate(BigDecimal.valueOf(Double.parseDouble(dto.getSystemRate())).setScale(6, BigDecimal.ROUND_DOWN));
                rate.setSystemRateStr(rate.getSystemRate().toPlainString() + "%");
            } catch (Exception e) {
                String msg = "系统折扣率(%)格式错误 例【11.78】";
                this.validateIsTrue(false, msg);
            }
        }
        if (StringUtils.isNotBlank(dto.getPlanRate())) {
            try {
                rate.setPlanRate(BigDecimal.valueOf(Double.parseDouble(dto.getPlanRate())).setScale(6, BigDecimal.ROUND_DOWN));
                rate.setPlanRateStr(rate.getPlanRate().toPlainString() + "%");
            } catch (Exception e) {
                String msg = "计划/修正折扣率(%)格式错误 例【11.78】";
                this.validateIsTrue(false, msg);
            }
        }
        if (StringUtils.isNotBlank(dto.getAdjustSystemRate())) {
            try {
                rate.setAdjustSystemRate(new BigDecimal(Double.parseDouble(dto.getAdjustSystemRate())).setScale(6, BigDecimal.ROUND_DOWN));
                rate.setAdjustSystemRateStr(rate.getAdjustSystemRate() + "%");
                rate.setSystemRate(rate.getAdjustSystemRate());
                rate.setSystemRateStr(rate.getAdjustSystemRateStr());
                if (new BigDecimal(20).compareTo(rate.getAdjustSystemRate()) < 0 && StringUtils.isBlank(dto.getAdjustReason())) {
                    String msg = "请填写调整后系统折扣率＞20%原因";
                    this.validateIsTrue(false, msg);
                } else {
                    rate.setAdjustReason(dto.getAdjustReason());
                }
            } catch (Exception e) {
                String msg = "调整后系统折扣率(%)格式错误 例【11.78】";
                this.validateIsTrue(false, msg);
            }
        }
        if (StringUtils.isNotBlank(dto.getAdjustPlanRate())) {
            try {
                rate.setAdjustPlanRate(BigDecimal.valueOf(Double.parseDouble(dto.getAdjustPlanRate())).setScale(6, BigDecimal.ROUND_DOWN));
                rate.setAdjustPlanRateStr(rate.getAdjustPlanRate() + "%");
                rate.setPlanRate(rate.getPlanRate());
                rate.setPlanRateStr(rate.getPlanRateStr());
            } catch (Exception e) {
                String msg = "调整后计划/修正折扣率(%)格式错误 例【11.78】";
                this.validateIsTrue(false, msg);
            }
        }
        if (null == rate.getSystemRate() || null == rate.getPlanRate()) {
            try {
                rate.setPredictDigestionFee(BigDecimal.valueOf(Double.parseDouble(dto.getPredictDigestionFee())));
            } catch (Exception e) {
                String msg = "预计消化费用池费用格式错误 例【11.78】";
                this.validateIsTrue(false, msg);
            }
            try {
                rate.setPredictCarFee(BigDecimal.valueOf(Double.parseDouble(dto.getPredictCarFee())));
            } catch (Exception e) {
                String msg = "预计随车折扣额格式错误 例【11.78】";
                this.validateIsTrue(false, msg);
            }
        }
        rate.setDimensionType(type);
        String onlyKey = rate.getBusinessFormatCode() + rate.getBusinessUnitCode() + rate.getRateVersion() + rate.getBusinessDepartment() + rate.getDimensionType() + rate.getCustomerCode();
        rate.setOnlyKey(onlyKey);
        rate.setLineIndex(index);

        return rate;
    }

    /**
     * 校验数据正确性
     *
     * @param list   导入数据
     * @param cusMap 客户信息
     **/
    private void validateCorrect(List<DiscountRateDto> list, Map<String, CustomerOrgChannelStoresVo> cusMap, Map<Integer, String> errMap) {
        Set<String> cusCodes = new HashSet<>();
        Set<String> formatCodes = new HashSet<>();
        Set<String> unitCodes = new HashSet<>();
        Set<String> onlyKeySet = new HashSet<>();
        Set<String> dateSet = new HashSet<>();
        Set<String> keys = new HashSet<>();
        for (DiscountRateDto dto : list) {
            if (cusMap.containsKey(dto.getCustomerCode())) {
                CustomerOrgChannelStoresVo storesVo = cusMap.get(dto.getCustomerCode());
                dto.setCustomerName(storesVo.getCustomerName());
                dto.setSalesInstitutionCode(storesVo.getSalesInstitutionCode());
                dto.setSalesInstitutionName(storesVo.getSalesInstitutionName());
                dto.setSalesRegionCode(storesVo.getSalesRegionCode());
                dto.setSalesRegionName(storesVo.getSalesRegionName());
                dto.setSalesOrgCode(storesVo.getSalesOrgCode());
                dto.setSalesOrgName(storesVo.getSalesOrgName());
                dto.setCustomerChannelCode(storesVo.getChannelCode());
                dto.setCustomerChannelName(storesVo.getChannelName());
                dto.setCustomerType(storesVo.getCustomerType());
            } else {
                this.validateIsTrue(false, "根据客户编码+销售机构编码+渠道编码+业态未能查找到客户信息");
            }
            if (null != dto.getSystemRate() && null != dto.getPlanRate()) {
                continue;
            }
            cusCodes.add(dto.getCustomerCode());
            dateSet.add(DateUtil.dateToStr(dto.getYearAndMonth(), DateUtil.date_yyyy_MM));
            formatCodes.add(dto.getBusinessFormatCode());
            unitCodes.add(dto.getBusinessUnitCode());
            keys.add(dto.getOnlyKey());
            String onlyKey = dto.getBusinessFormatCode() + dto.getBusinessUnitCode() + dto.getRateVersion() + dto.getBusinessDepartment() + dto.getDimensionType();
            onlyKeySet.add(onlyKey);
        }
        //历史数据id map
        Map<String, String> oldMap = discountRateService.getIdByKeys(new ArrayList<>(keys));

        //根据业态，业务单元，客户编码维度和月份获取预计上账费用
        Map<String, BigDecimal> accountMap = new HashMap<>();

        //根据业态，业务单元，客户维度和月份获取回复量
        SalesPlanSummaryDto summaryDto = new SalesPlanSummaryDto();
        summaryDto.setFormatLsit(new ArrayList<>(formatCodes));
        summaryDto.setUnitLsit(new ArrayList<>(unitCodes));
        summaryDto.setMonthLsit(new ArrayList<>(dateSet));
        List<List<String>> cusList = Lists.partition(new ArrayList<>(cusCodes), 500);
        summaryDto.setCusLsit(cusList);
        summaryDto.setDimen(SummaryDimensionEunm.CUSTOMER.getCode());
        Map<String, BigDecimal> replyMap = salesPlanService.summaryRecoveryAmount(summaryDto);

        //根据onlyKeySet查找折扣率配置
        Map<String, DiscountRateConfig> configMap = discountRateConfigService.getConfigByOnlyKeys(new ArrayList<>(onlyKeySet));

        log.error("开始执行折扣率计算-----------------");
        for (DiscountRateDto dto : list) {
            if (oldMap.containsKey(dto.getOnlyKey())) {
                dto.setId(oldMap.getOrDefault(dto.getOnlyKey(), null));
            }
            if (null != dto.getSystemRate() && null != dto.getPlanRate()) {
                continue;
            }
            String key = dto.getBusinessFormatCode() + dto.getBusinessUnitCode() + DateUtil.dateToStr(dto.getYearAndMonth(), DateUtil.date_yyyy_MM) + dto.getCustomerCode();
            if (!accountMap.containsKey(key)) {
                dto.setPredictAccountFee(BigDecimal.ZERO);
            }
            dto.setPredictAccountFee(accountMap.getOrDefault(key, BigDecimal.ZERO));

            if (!replyMap.containsKey(dto.getCustomerCode())) {
                dto.setRecoveryAmount(BigDecimal.ZERO);
            }
            dto.setRecoveryAmount(replyMap.getOrDefault(key, BigDecimal.ZERO));

            String onlyKey = dto.getBusinessFormatCode() + dto.getBusinessUnitCode() + dto.getRateVersion() + dto.getBusinessDepartment() + dto.getDimensionType();
            if (configMap.containsKey(onlyKey)) {
                DiscountRateConfig config = configMap.get(onlyKey);
                dto.setApproveTag(config.getApproveTag());
                if (YesOrNoEnum.YES.getCode().equals(config.getApproveTag())) {
                    dto.setProcessStatus(ProcessStatusEnum.PREPARE.getDictCode());
                }
                //判断是否需要走折扣率配置去计算折扣率
                if (null == dto.getSystemRate()) {
                    CalculateDto calDto = new CalculateDto();
                    calDto.setFormula(config.getSystemRateFormula());
                    calDto.setPredictAccountFee(dto.getPredictAccountFee());
                    calDto.setPredictDigestionFee(dto.getPredictDigestionFee());
                    calDto.setPredictCarFee(dto.getPredictCarFee());
                    calDto.setRecoveryAmount(dto.getRecoveryAmount());
                    try {
                        BigDecimal b = this.discountRateVariableService.singleCalculateExpression(calDto);
                        dto.setSystemRate(b.multiply(BigDecimal.valueOf(100)).setScale(6, BigDecimal.ROUND_HALF_UP));
                        dto.setSystemRateStr(dto.getSystemRate().toPlainString() + "%");
                    } catch (Exception e) {
                        String msg = "客户[" + dto.getMdgCode() + "]计算系统折扣率失败，折扣率配置编码[" + config.getConfigCode() + "]";
                        this.validateIsTrue(false, msg);
                    }
                }
                if (null == dto.getPlanRate()) {
                    CalculateDto calDto = new CalculateDto();
                    calDto.setFormula(config.getPlanRateFormula());
                    calDto.setPredictAccountFee(dto.getPredictAccountFee());
                    calDto.setPredictDigestionFee(dto.getPredictDigestionFee());
                    calDto.setPredictCarFee(dto.getPredictCarFee());
                    calDto.setRecoveryAmount(dto.getRecoveryAmount());
                    try {
                        BigDecimal b = this.discountRateVariableService.singleCalculateExpression(calDto);
                        dto.setPlanRate(b.multiply(BigDecimal.valueOf(100)).setScale(6, BigDecimal.ROUND_HALF_UP));
                        dto.setPlanRateStr(dto.getPlanRate().toPlainString() + "%");
                    } catch (Exception e) {
                        String msg = "客户[" + dto.getMdgCode() + "]计算计划/修正折扣率失败，折扣率配置编码[" + config.getConfigCode() + "]";
                        this.validateIsTrue(false, msg);
                    }
                }
            }

            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(dto.getLineIndex(), errInfo);
            }
        }
    }


    /**
     * 获取数据实体
     *
     * @return Class<AuditExecuteIndicatorImportVo>
     */
    @Override
    public Class<DiscountRateCustomerImportVo> findCrmExcelVoClass() {
        return DiscountRateCustomerImportVo.class;
    }

    /**
     * 获取业务编码
     *
     * @return String
     */
    @Override
    public String getBusinessCode() {
        return "TPM_DISCOUNT_RATE_IMPORT";
    }

    /**
     * 获取业务名称
     *
     * @return String
     */
    @Override
    public String getBusinessName() {
        return "TPM折扣率导入";
    }

    /**
     * 获取数据实体
     *
     * @return String
     */
    @Override
    public String getTemplateCode() {
        return "TPM_DISCOUNT_RATE_CUSTOMER_IMPORT";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return String
     */
    @Override
    public String getTemplateName() {
        return "TPM-折扣率客户维度导入模板";
    }
}
