package com.biz.crm.tpm.business.budget.discount.rate.local.process;

import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.enums.YesOrNoEnum;
import com.biz.crm.tpm.business.budget.discount.rate.local.entity.DiscountRateConfig;
import com.biz.crm.tpm.business.budget.discount.rate.local.service.DiscountRateConfigService;
import com.biz.crm.tpm.business.budget.discount.rate.local.service.DiscountRateService;
import com.biz.crm.tpm.business.budget.discount.rate.local.service.DiscountRateVariableService;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.constant.DiscountRateConfigConstant;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.CalculateDto;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.DiscountRateDto;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.enums.DiscountRateDimensionEnum;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.vo.DiscountRateProductImportVo;
import com.biz.crm.tpm.business.sales.plan.sdk.dto.SalesPlanSummaryDto;
import com.biz.crm.tpm.business.sales.plan.sdk.enums.SummaryDimensionEunm;
import com.biz.crm.tpm.business.sales.plan.sdk.service.SalesPlanService;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.*;

/**
 * @author: chenlong
 * @date: 2022/11/15 15:29
 * @description: 折扣率表(DiscountRate)导入实现层-客户+产品维度
 */
@Slf4j
@Component
public class DiscountRateProductImportProcess implements ImportProcess<DiscountRateProductImportVo> {

    @Autowired(required = false)
    private DiscountRateService discountRateService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;
    @Autowired(required = false)
    private CustomerVoService customerVoService;
    @Autowired(required = false)
    private ProductVoService productVoService;
    @Autowired(required = false)
    private DiscountRateVariableService discountRateVariableService;
    @Autowired(required = false)
    private DiscountRateConfigService discountRateConfigService;
    @Autowired(required = false)
    private SalesPlanService salesPlanService;

    /**
     * 数据分片
     *
     * @return 分片长度
     */
    @Override
    public Integer getBatchCount() {
        return 50001;
    }

    /**
     * 数据处理
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, DiscountRateProductImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Validate.notNull(data, "导入数据不能为空！");
        Validate.isTrue(data.values().size() <= 50000, "单次导入数据不能超过2000条");
        List<DiscountRateDto> importList = this.validate(data);
        //大批量保存
        discountRateService.bulkImportSave(importList);
        return null;
    }

    /**
     * 校验数据
     *
     * @param data data
     **/
    private List<DiscountRateDto> validate(LinkedHashMap<Integer, DiscountRateProductImportVo> data) {
        List<DiscountRateDto> list = new ArrayList<>();
        //获取业态字典
        Map<String, String> formatMap = dictToolkitService.findConvertMapByDictTypeCode(DiscountRateConfigConstant.DISCOUNT_RATE_BUSINESS_FORMAT);
        //获取业务单元字典
        Map<String, String> unitMap = dictToolkitService.findConvertMapByDictTypeCode(DiscountRateConfigConstant.DISCOUNT_RATE_BUSINESS_UNIT);
        //获取业务部门型字典
        Map<String, String> departMap = dictToolkitService.findConvertMapByDictTypeCode(DiscountRateConfigConstant.DISCOUNT_RATE_BUSINESS_DEPARTMENT);
        //获取版本字典
        Map<String, String> versionMap = dictToolkitService.findConvertMapByDictTypeCode(DiscountRateConfigConstant.DISCOUNT_RATE_RATE_VERSION);
        //产品维度
        String type = DiscountRateDimensionEnum.PRODUCT.getCode();
        data.forEach((k, v) -> {
            list.add(this.validateData(k + 2, v, type, formatMap, unitMap, departMap, versionMap));
        });

        Set<String> proCodes = new HashSet<>();
        for (DiscountRateDto vo : list) {
            proCodes.add(vo.getProductCode());
        }
        //获取产品信息
        List<List<String>> proList = Lists.partition(new ArrayList<>(proCodes), 500);
        Map<String, String> proMap = productVoService.getProductByCodes(proList);
        proCodes.clear();
        this.validateCorrect(list, proMap);

        return new ArrayList<>(list);


    }


    /**
     * 校验数据
     *
     * @param index 行号
     * @param dto   行数据
     **/
    private DiscountRateDto validateData(Integer index, DiscountRateProductImportVo dto, String type,
                                         Map<String, String> formatMap, Map<String, String> unitMap,
                                         Map<String, String> departMap, Map<String, String> versionMap) {
        DiscountRateDto rate = new DiscountRateDto();
        Validate.isTrue(StringUtils.isNotBlank(dto.getBusinessFormatCode()), "[行%s]业态不能为空", index);
        Validate.isTrue(StringUtils.isNotBlank(dto.getBusinessUnitCode()), "[行%s]业务单元不能为空", index);
        Validate.isTrue(StringUtils.isNotBlank(dto.getRateVersion()), "[行%s]版本不能为空", index);
        Validate.isTrue(StringUtils.isNotBlank(dto.getBusinessDepartment()), "[行%s]业务部门不能为空", index);
        Validate.isTrue(StringUtils.isNotBlank(dto.getProductCode()), "[行%s]产品编码不能为空", index);
        Validate.isTrue(StringUtils.isNotBlank(dto.getYearAndMonthStr()), "[行%s]年月不能为空", index);
//        Validate.isTrue(StringUtils.isNotBlank(dto.getPredictDigestionFee()), "[行%s]预计消化费用池费用不能为空", index);
//        Validate.isTrue(StringUtils.isNotBlank(dto.getPredictCarFee()), "[行%s]预计随车折扣额不能为空", index);
        rate.setBusinessFormatCode(dto.getBusinessFormatCode());
        rate.setBusinessUnitCode(dto.getBusinessUnitCode());
        rate.setRateVersion(dto.getRateVersion());
        rate.setBusinessDepartment(dto.getBusinessDepartment());
        rate.setProductCode(dto.getProductCode());
        if (formatMap.containsKey(dto.getBusinessFormatCode())) {
            rate.setBusinessFormatCode(formatMap.get(dto.getBusinessFormatCode()));
        } else {
            Validate.isTrue(false, "行号：" + index + ",业态未能识别");
        }
        if (unitMap.containsKey(dto.getBusinessUnitCode())) {
            rate.setBusinessUnitCode(unitMap.get(dto.getBusinessUnitCode()));
        } else {
            Validate.isTrue(false, "行号：" + index + ",业务单元未能识别");
        }
        if (versionMap.containsKey(dto.getRateVersion())) {
            rate.setRateVersion(versionMap.get(dto.getRateVersion()));
        } else {
            Validate.isTrue(false, "行号：" + index + ",版本未能识别");
        }
        if (departMap.containsKey(dto.getBusinessDepartment())) {
            rate.setBusinessDepartment(departMap.get(dto.getBusinessDepartment()));
        } else {
            Validate.isTrue(false, "行号：" + index + ",业务单元未能识别");
        }
        try {
            rate.setYearAndMonth(DateUtil.date_yyyy_MM.parse(dto.getYearAndMonthStr()));
            rate.setYearAndMonthStr(dto.getYearAndMonthStr());
        } catch (Exception e) {
            String msg = "[行" + index + "]年月格式错误【yyyy-MM】";
            throw new IllegalArgumentException(msg);
        }
        if (StringUtils.isNotBlank(dto.getSystemRate())) {
            try {
                rate.setSystemRate(BigDecimal.valueOf(Double.parseDouble(dto.getSystemRate())).setScale(6,BigDecimal.ROUND_DOWN));
                rate.setSystemRateStr(rate.getSystemRate().toPlainString() + "%");
            } catch (Exception e) {
                String msg = "[行" + index + "]系统折扣率(%)格式错误 例【11.78】";
                throw new IllegalArgumentException(msg);
            }
        }
        if (StringUtils.isNotBlank(dto.getPlanRate())) {
            try {
                rate.setPlanRate(BigDecimal.valueOf(Double.parseDouble(dto.getPlanRate())).setScale(6,BigDecimal.ROUND_DOWN));
                rate.setPlanRateStr(rate.getPlanRate().toPlainString() + "%");
            } catch (Exception e) {
                String msg = "[行" + index + "]计划/修正折扣率(%)格式错误 例【11.78】";
                throw new IllegalArgumentException(msg);
            }
        }
        if (StringUtils.isNotBlank(dto.getAdjustSystemRate())) {
            try {
                rate.setAdjustSystemRate(BigDecimal.valueOf(Double.parseDouble(dto.getAdjustSystemRate())).setScale(6,BigDecimal.ROUND_DOWN));
                rate.setAdjustSystemRateStr(rate.getAdjustSystemRate() + "%");
                rate.setSystemRate(rate.getAdjustSystemRate());
                rate.setSystemRateStr(rate.getAdjustSystemRateStr());
            } catch (Exception e) {
                String msg = "[行" + index + "]调整后系统折扣率(%)格式错误 例【11.78】";
                throw new IllegalArgumentException(msg);
            }
            if (new BigDecimal(20).compareTo(rate.getAdjustSystemRate()) < 0 && StringUtils.isBlank(dto.getAdjustReason())) {
                String msg = "[行" + index + "]请填写调整后系统折扣率＞20%原因";
                throw new IllegalArgumentException(msg);
            } else {
                rate.setAdjustReason(dto.getAdjustReason());
            }
        }
        if (StringUtils.isNotBlank(dto.getAdjustPlanRate())) {
            try {
                rate.setAdjustPlanRate(BigDecimal.valueOf(Double.parseDouble(dto.getAdjustPlanRate())).setScale(6,BigDecimal.ROUND_DOWN));
                rate.setAdjustPlanRateStr(rate.getAdjustPlanRate() + "%");
                rate.setPlanRate(rate.getPlanRate());
                rate.setPlanRateStr(rate.getPlanRateStr());
            } catch (Exception e) {
                String msg = "[行" + index + "]调整后计划/修正折扣率(%)格式错误 例【11.78】";
                throw new IllegalArgumentException(msg);
            }
        }
        if (null == rate.getSystemRate() || null == rate.getPlanRate()) {
            try {
                rate.setPredictDigestionFee(BigDecimal.valueOf(Double.parseDouble(dto.getPredictDigestionFee())));
            } catch (Exception e) {
                String msg = "[行" + index + "]预计消化费用池费用格式错误 例【11.78】";
                throw new IllegalArgumentException(msg);
            }
            try {
                rate.setPredictCarFee(BigDecimal.valueOf(Double.parseDouble(dto.getPredictCarFee())));
            } catch (Exception e) {
                String msg = "[行" + index + "]预计随车折扣额格式错误 例【11.78】";
                throw new IllegalArgumentException(msg);
            }
        }
        rate.setLineIndex(index);
        rate.setDimensionType(type);
        String onlyKey = rate.getBusinessFormatCode() + rate.getBusinessUnitCode() + rate.getRateVersion() + rate.getBusinessDepartment() + rate.getDimensionType() + rate.getCustomerCode();
        rate.setOnlyKey(onlyKey);
        return rate;
    }

    /**
     * 校验数据正确性
     *
     * @param list   导入数据
     * @param proMap 客户信息
     * @return List<AuditExecuteIndicatorDto>
     **/
    private void validateCorrect(List<DiscountRateDto> list, Map<String, String> proMap) {
        List<String> proCodes = new ArrayList<>();
        Set<String> formatCodes = new HashSet<>();
        Set<String> unitCodes = new HashSet<>();
        Set<String> onlyKeySet = new HashSet<>();
        Set<String> dateSet = new HashSet<>();
        Set<String> keys = new HashSet<>();
        for (DiscountRateDto dto : list) {
            if (proMap.containsKey(dto.getProductCode())) {
                dto.setProductName(dto.getProductCode());
            } else {
                Validate.isTrue(false, "行号：" + dto.getLineIndex() + ",根据产品编码未能查找到产品信息");
            }
            if (null != dto.getSystemRate() && null != dto.getPlanRate()) {
                continue;
            }
            dateSet.add(DateUtil.dateToStr(dto.getYearAndMonth(), DateUtil.date_yyyy_MM));
            proCodes.add(dto.getProductCode());
            formatCodes.add(dto.getBusinessFormatCode());
            unitCodes.add(dto.getBusinessUnitCode());
            keys.add(dto.getOnlyKey());
            String onlyKey = dto.getBusinessFormatCode() + dto.getBusinessUnitCode() + dto.getRateVersion() + dto.getBusinessDepartment() + dto.getDimensionType();
            onlyKeySet.add(onlyKey);
        }

        //历史数据id map
        Map<String, String> oldMap = discountRateService.getIdByKeys(new ArrayList<>(keys));
        //根据产品编码维度和月份获取预计上账费用
        Map<String, BigDecimal> accountMap = new HashMap<>();

        //根据产品编码和月份获取回复量
        SalesPlanSummaryDto summaryDto = new SalesPlanSummaryDto();
        summaryDto.setFormatLsit(new ArrayList<>(formatCodes));
        summaryDto.setUnitLsit(new ArrayList<>(unitCodes));
        summaryDto.setMonthLsit(new ArrayList<>(dateSet));
        List<List<String>> proList = Lists.partition(new ArrayList<>(proCodes), 500);
        summaryDto.setProLsit(proList);
        summaryDto.setDimen(SummaryDimensionEunm.PRODUCT.getCode());
        Map<String, BigDecimal> replyMap = salesPlanService.summaryRecoveryAmount(summaryDto);

        //根据onlyKeySet查找折扣率配置
        Map<String, DiscountRateConfig> configMap = discountRateConfigService.getConfigByOnlyKeys(new ArrayList<>(onlyKeySet));

        for (DiscountRateDto dto : list) {
            if (oldMap.containsKey(dto.getOnlyKey())) {
                dto.setId(oldMap.getOrDefault(dto.getOnlyKey(), null));
            }
            if (null != dto.getSystemRate() && null != dto.getPlanRate()) {
                continue;
            }
            String key = dto.getBusinessFormatCode() + dto.getBusinessUnitCode() + DateUtil.dateToStr(dto.getYearAndMonth(), DateUtil.date_yyyy_MM) + dto.getProductCode();
            if (!accountMap.containsKey(dto.getCustomerCode())) {
                dto.setPredictAccountFee(BigDecimal.ZERO);
            }
            dto.setPredictAccountFee(accountMap.getOrDefault(key, BigDecimal.ZERO));

            if (!replyMap.containsKey(dto.getCustomerCode())) {
                dto.setRecoveryAmount(BigDecimal.ZERO);
            }
            dto.setRecoveryAmount(replyMap.getOrDefault(key, BigDecimal.ZERO));

            String onlyKey = dto.getBusinessFormatCode() + dto.getBusinessUnitCode() + dto.getRateVersion() + dto.getBusinessDepartment() + dto.getDimensionType();
            if (configMap.containsKey(onlyKey)) {
                DiscountRateConfig config = configMap.get(onlyKey);
                dto.setApproveTag(config.getApproveTag());
                if (YesOrNoEnum.YES.getCode().equals(config.getApproveTag())) {
                    dto.setProcessStatus(ProcessStatusEnum.PREPARE.getDictCode());
                }
                //判断是否需要走折扣率配置去计算折扣率
                if (null == dto.getSystemRate()) {


                    CalculateDto calDto = new CalculateDto();
                    calDto.setFormula(config.getSystemRateFormula());
                    calDto.setPredictAccountFee(dto.getPredictAccountFee());
                    calDto.setPredictDigestionFee(dto.getPredictDigestionFee());
                    calDto.setPredictCarFee(dto.getPredictCarFee());
                    calDto.setRecoveryAmount(dto.getRecoveryAmount());
                    try {
                        BigDecimal b = this.discountRateVariableService.singleCalculateExpression(calDto);
                        dto.setSystemRate(b.multiply(BigDecimal.valueOf(100)).setScale(6, BigDecimal.ROUND_HALF_UP));
                        dto.setSystemRateStr(dto.getSystemRate().toPlainString() + "%");
                    } catch (Exception e) {
                        String msg = "客户[" + dto.getMdgCode() + "]计算系统折扣率失败，折扣率配置编码[" + config.getConfigCode() + "]";
                        throw new IllegalArgumentException(msg);
                    }
                }
                if (null == dto.getPlanRate()) {
                    CalculateDto calDto = new CalculateDto();
                    calDto.setFormula(config.getPlanRateFormula());
                    calDto.setPredictAccountFee(dto.getPredictAccountFee());
                    calDto.setPredictDigestionFee(dto.getPredictDigestionFee());
                    calDto.setPredictCarFee(dto.getPredictCarFee());
                    calDto.setRecoveryAmount(dto.getRecoveryAmount());
                    try {
                        BigDecimal b = this.discountRateVariableService.singleCalculateExpression(calDto);
                        dto.setPlanRate(b.multiply(BigDecimal.valueOf(100)).setScale(6, BigDecimal.ROUND_HALF_UP));
                        dto.setPlanRateStr(dto.getPlanRate().toPlainString() + "%");
                    } catch (Exception e) {
                        String msg = "客户[" + dto.getMdgCode() + "]计算计划/修正折扣率失败，折扣率配置编码[" + config.getConfigCode() + "]";
                        throw new IllegalArgumentException(msg);
                    }

                }
            }
        }
    }


    /**
     * 获取数据实体
     *
     * @return Class<AuditExecuteIndicatorImportVo>
     */
    @Override
    public Class<DiscountRateProductImportVo> findCrmExcelVoClass() {
        return DiscountRateProductImportVo.class;
    }

    /**
     * 获取业务编码
     *
     * @return String
     */
    @Override
    public String getBusinessCode() {
        return "TPM_DISCOUNT_RATE_IMPORT";
    }

    /**
     * 获取业务名称
     *
     * @return String
     */
    @Override
    public String getBusinessName() {
        return "TPM折扣率导入";
    }

    /**
     * 获取数据实体
     *
     * @return String
     */
    @Override
    public String getTemplateCode() {
        return "TPM_DISCOUNT_RATE_PRODUCT_IMPORT";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return String
     */
    @Override
    public String getTemplateName() {
        return "TPM-折扣率产品维度导入模板";
    }
}
