package com.biz.crm.tpm.business.budget.discount.rate.local.repository;

import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.tpm.business.budget.discount.rate.local.entity.DiscountRate;
import com.biz.crm.tpm.business.budget.discount.rate.local.mapper.DiscountRateMapper;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.DiscountRateDto;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.DiscountRatePlanCalDto;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.vo.DiscountRateVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.List;

/**
 * @author: chenlong
 * @date: 2023/1/5 19:31
 * @description: 折扣率(DiscountRate)表数据库访问层
 */
@Component
public class DiscountRateRepository extends ServiceImpl<DiscountRateMapper, DiscountRate> {

    @Autowired(required = false)
    private DiscountRateMapper discountRateMapper;

    /**
     * 分页查询数据
     *
     * @param pageable 分页对象
     * @param rateDto  查询Dto
     * @return Page<DiscountRateConfigVo>
     */
    public Page<DiscountRateVo> findByConditions(Pageable pageable, DiscountRateDto rateDto) {
        Page<DiscountRateVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.discountRateMapper.findByConditions(page, rateDto);
    }


    /**
     * 重构修改方法
     *
     * @param entity     实体类
     * @param tenantCode 租户编码
     */
    public void updateByIdAndTenantCode(DiscountRate entity, String tenantCode) {
        LambdaUpdateWrapper<DiscountRate> lambdaUpdateWrapper = Wrappers.lambdaUpdate();
        lambdaUpdateWrapper.eq(DiscountRate::getTenantCode, tenantCode);
        lambdaUpdateWrapper.eq(DiscountRate::getId, entity.getId());
        this.baseMapper.update(entity, lambdaUpdateWrapper);
    }


    /**
     * 重构批量修改方法
     *
     * @param entitys    实体类集合
     * @param tenantCode 租户编码
     */
    public void updateBatchByIdAndTenantCode(Collection<DiscountRate> entitys, String tenantCode) {
        if (CollectionUtils.isEmpty(entitys)) {
            return;
        }
        if (StringUtils.isEmpty(tenantCode)) {
            return;
        }
        entitys.forEach(item -> {
            if (StringUtils.isNotBlank(item.getId())) {
                LambdaUpdateWrapper<DiscountRate> lambdaUpdateWrapper = Wrappers.lambdaUpdate();
                lambdaUpdateWrapper.eq(DiscountRate::getTenantCode, tenantCode);
                lambdaUpdateWrapper.eq(DiscountRate::getId, item.getId());
                this.baseMapper.update(item, lambdaUpdateWrapper);
            }
        });
    }

    /**
     * 大批量保存
     *
     * @param entities 数据列表
     */
    public void bulkSave(List<DiscountRate> entities) {
        if (CollectionUtils.isEmpty(entities)) {
            return;
        }
        //保存
        this.baseMapper.bulkSave(entities);
    }

    /**
     * 修改审批状态方法
     *
     * @param code          付款单编码
     * @param processStatus 审批状态
     */
    public void updateProcessStatus(String code, String processStatus) {
        if (StringUtils.isBlank(code)) {
            return;
        }
        this.lambdaUpdate()
                .eq(DiscountRate::getDiscountRateCode, code)
                .eq(DiscountRate::getTenantCode, TenantUtils.getTenantCode())
                .set(DiscountRate::getProcessStatus, processStatus)
                .update();
    }

    /**
     * 根据业态+业务单元+客户+产品组装的key获取折扣率
     *
     * @param keys       分页对象
     * @param tenantCode 租户
     * @return List<DiscountRateDto>
     */
    public List<DiscountRateDto> getDiscountRateByKeys(List<List<String>> keys, String tenantCode) {
        return this.baseMapper.getDiscountRateByKeys(keys, tenantCode);
    }

    /**
     * 根据业态+业务单元+客户组装的key获取折扣率
     *
     * @param keys       分页对象
     * @param tenantCode 租户
     * @return List<DiscountRateDto>
     */
    public List<DiscountRateDto> getDiscountRateByKeysNoPro(List<List<String>> keys, String tenantCode) {
        return this.baseMapper.getDiscountRateByKeysNoPro(keys, tenantCode);
    }

    /**
     * 销售计划获取折扣率
     *
     * @param calDto 筛选数据
     * @return List < DiscountRateDto>
     */
    public List<DiscountRateDto> findRateByPlanConditions(DiscountRatePlanCalDto calDto) {
        return this.baseMapper.findRateByPlanConditions(calDto);
    }

    /**
     * 根据onlykeys获取id
     *
     * @param keyList onlyKey列表
     * @return List<DiscountRate>
     */
    public List<DiscountRate> getIdByKeys(List<List<String>> keyList, String tenantCode) {
        if (CollectionUtils.isEmpty(keyList) || StringUtils.isBlank(tenantCode)) {
            return Lists.newArrayList();
        }
        return this.baseMapper.getIdByKeys(keyList, tenantCode);
    }
}
