package com.biz.crm.tpm.business.budget.discount.rate.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.tpm.business.budget.discount.rate.local.entity.DiscountRateConfigDimension;
import com.biz.crm.tpm.business.budget.discount.rate.local.repository.DiscountRateConfigDimensionRepository;
import com.biz.crm.tpm.business.budget.discount.rate.local.service.DiscountRateConfigDimensionService;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.DiscountRateConfigDimensionDto;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.DiscountRateConfigDimensionPageDto;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.enums.*;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.vo.DiscountRateConfigDimensionVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import jodd.util.StringUtil;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author: chenlong
 * @date: 2023/1/6 16:10
 * @description: 折扣率配置维度管理(DiscountRateConfigDimension)表服务实现类
 */
@Service("discountRateConfigDimensionService")
public class DiscountRateConfigDimensionServiceImpl implements DiscountRateConfigDimensionService {

    @Autowired(required = false)
    private DiscountRateConfigDimensionRepository discountRateConfigDimensionRepository;
    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 批量保存维度信息
     *
     * @param dimensionList 维度数据列表
     * @param configCode    配置编码
     * @param tenantCode    租户编码
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void saveBatchDimension(List<DiscountRateConfigDimensionDto> dimensionList, String configCode, String tenantCode) {
        if (CollectionUtils.isEmpty(dimensionList)) {
            return;
        }
        Collection<DiscountRateConfigDimension> list = this.nebulaToolkitService.copyCollectionByBlankList(
                dimensionList, DiscountRateConfigDimensionDto.class, DiscountRateConfigDimension.class, HashSet.class, ArrayList.class);
        list.forEach(item -> {
            item.setConfigCode(configCode);
            item.setTenantCode(tenantCode);
            item.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            item.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        });
        this.discountRateConfigDimensionRepository.saveBatch(list);
    }

    /**
     * 批量修改维度信息
     *
     * @param dimensionList 维度数据列表
     * @param configCode    配置编码
     * @param tenantCode    租户编码
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void editBatchDimension(List<DiscountRateConfigDimensionDto> dimensionList, String configCode, String tenantCode) {
        if (CollectionUtils.isEmpty(dimensionList)) {
            return;
        }

        List<DiscountRateConfigDimensionDto> newList = new ArrayList<>();
        List<DiscountRateConfigDimensionDto> oldList = new ArrayList<>();
        List<String> ids = new ArrayList<>();
        for (DiscountRateConfigDimensionDto dto : dimensionList) {
            if (StringUtil.isNotBlank(dto.getId())) {
                ids.add(dto.getId());
                oldList.add(dto);
                continue;
            }
            dto.setConfigCode(configCode);
            dto.setTenantCode(tenantCode);
            dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            dto.setId(null);
            newList.add(dto);
        }
        //删除数据
        if (!CollectionUtils.isEmpty(ids)) {
            this.discountRateConfigDimensionRepository.lambdaUpdate()
                    .eq(DiscountRateConfigDimension::getTenantCode, tenantCode)
                    .eq(DiscountRateConfigDimension::getConfigCode, configCode)
                    .notIn(DiscountRateConfigDimension::getId, ids)
                    .set(DiscountRateConfigDimension::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                    .update();
        } else {
            this.discountRateConfigDimensionRepository.lambdaUpdate()
                    .eq(DiscountRateConfigDimension::getTenantCode, tenantCode)
                    .eq(DiscountRateConfigDimension::getConfigCode, configCode)
                    .set(DiscountRateConfigDimension::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                    .update();
        }
        //更新数据
        if (!CollectionUtils.isEmpty(oldList)) {
            Collection<DiscountRateConfigDimension> dimensions = this.nebulaToolkitService.copyCollectionByBlankList(
                    oldList, DiscountRateConfigDimensionDto.class, DiscountRateConfigDimension.class, HashSet.class, ArrayList.class);
            this.discountRateConfigDimensionRepository.updateBatchByIdAndTenantCode(dimensions, tenantCode);
        }
        //新增数据
        if (!CollectionUtils.isEmpty(newList)) {
            Collection<DiscountRateConfigDimension> dimensions = this.nebulaToolkitService.copyCollectionByBlankList(
                    newList, DiscountRateConfigDimensionDto.class, DiscountRateConfigDimension.class, HashSet.class, ArrayList.class);
            this.discountRateConfigDimensionRepository.saveBatch(dimensions);
        }
    }

    /**
     * 获取维度信息列表
     *
     * @return List<DiscountRateConfigDimensionVo>
     */
    @Override
    public Page<DiscountRateConfigDimensionVo> findDimension(Pageable pageable, DiscountRateConfigDimensionPageDto pageDto) {
        //定义一个集合
        if (pageable == null) {
            pageable = PageRequest.of(1, 50);
        }
        Validate.notNull(pageDto, "请求参数不能为空！");
        Validate.notEmpty(pageDto.getDimensionType(), "维度类型不能为空！");

        // 手动分页
        List<DiscountRateConfigDimensionVo> allList = new ArrayList<>();
        Map<String, String> fieldingMap = new HashMap<>(8);
        if (DiscountRateDimensionEnum.CUSTOMER.getCode().equals(pageDto.getDimensionType())) {
            fieldingMap = DiscountRateCustomerEnum.concertEnumToMap();
        } else if (DiscountRateDimensionEnum.CUSTOMER_PRODUCT.getCode().equals(pageDto.getDimensionType())) {
            fieldingMap = DiscountRateCustomerProductEnum.concertEnumToMap();
        } else if (DiscountRateDimensionEnum.PRODUCT.getCode().equals(pageDto.getDimensionType())) {
            fieldingMap = DiscountRateProductEnum.concertEnumToMap();
        }else if(DiscountRateDimensionEnum.RETAILER_REGION_BRAND.getCode().equals(pageDto.getDimensionType())){
            fieldingMap = DiscountRateRetailerRegionBrandEnum.concertEnumToMap();
        }else if(DiscountRateDimensionEnum.RETAILER_REGION.getCode().equals(pageDto.getDimensionType())){
            fieldingMap = DiscountRateRetailerRegionEnum.concertEnumToMap();
        }
        fieldingMap.forEach((code, name) -> {
            DiscountRateConfigDimensionVo dimensionVo = new DiscountRateConfigDimensionVo();
            dimensionVo.setFieldCode(code);
            dimensionVo.setFieldName(name);
            allList.add(dimensionVo);
        });
        List<DiscountRateConfigDimensionVo> pageList = allList.stream()
                .skip((long) (pageable.getPageNumber() - 1) * pageable.getPageSize())
                .limit(pageable.getPageSize()).collect(Collectors.toList());
        Page<DiscountRateConfigDimensionVo> page = new Page<>();
        page.setRecords(pageList);
        page.setSize(allList.size());
        page.setTotal(allList.size());
        return page;
    }
}
