package com.biz.crm.tpm.business.budget.discount.rate.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.budget.discount.rate.local.entity.DiscountRateConfig;
import com.biz.crm.tpm.business.budget.discount.rate.local.entity.DiscountRateConfigDimension;
import com.biz.crm.tpm.business.budget.discount.rate.local.repository.DiscountRateConfigDimensionRepository;
import com.biz.crm.tpm.business.budget.discount.rate.local.repository.DiscountRateConfigRepository;
import com.biz.crm.tpm.business.budget.discount.rate.local.service.DiscountRateAsyncService;
import com.biz.crm.tpm.business.budget.discount.rate.local.service.DiscountRateConfigDimensionService;
import com.biz.crm.tpm.business.budget.discount.rate.local.service.DiscountRateConfigService;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.constant.DiscountRateConfigConstant;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.DiscountRateConfigDto;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.DiscountRateConfigLogEventDto;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.enums.DiscountRateDimensionEnum;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.event.log.DiscountRateConfigEventListener;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.register.FormulaVariableRegister;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.vo.DiscountRateConfigDimensionVo;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.vo.DiscountRateConfigFormulaVariableVo;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.vo.DiscountRateConfigVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author: chenlong
 * @date: 2023/1/6 11:31
 * @description: 折扣率配置管理(DiscountRateConfig)表服务实现类
 */
@Service("discountRateConfigService")
public class DiscountRateConfigServiceImpl implements DiscountRateConfigService {

    @Autowired(required = false)
    private DiscountRateConfigRepository discountRateConfigRepository;
    @Autowired(required = false)
    private DiscountRateConfigDimensionRepository discountRateConfigDimensionRepository;
    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;
    @Autowired(required = false)
    private GenerateCodeService generateCodeService;
    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;
    @Autowired(required = false)
    private DiscountRateConfigDimensionService discountRateConfigDimensionService;
    @Autowired(required = false)
    private DiscountRateAsyncService discountRateAsyncService;
    @Autowired(required = false)
    private List<FormulaVariableRegister> formulaVariableRegisterList;

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public DiscountRateConfigVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        DiscountRateConfig config = this.discountRateConfigRepository.getById(id);
        if (null == config) {
            return null;
        }
        DiscountRateConfigVo configVo = this.nebulaToolkitService.copyObjectByWhiteList(config, DiscountRateConfigVo.class, null, null);
        if (StringUtils.isBlank(configVo.getConfigCode())) {
            return configVo;
        }
        //获取维度信息
        List<DiscountRateConfigDimension> list = this.discountRateConfigDimensionRepository.lambdaQuery()
                .eq(DiscountRateConfigDimension::getConfigCode, configVo.getConfigCode())
                .eq(DiscountRateConfigDimension::getTenantCode, TenantUtils.getTenantCode())
                .eq(DiscountRateConfigDimension::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .orderByAsc(DiscountRateConfigDimension::getSort)
                .list();
        if (CollectionUtils.isEmpty(list)) {
            return configVo;
        }
        Collection<DiscountRateConfigDimensionVo> dimensionVos = this.nebulaToolkitService.copyCollectionByBlankList(
                list, DiscountRateConfigDimension.class, DiscountRateConfigDimensionVo.class, HashSet.class, ArrayList.class);
        configVo.setDimensionList((List<DiscountRateConfigDimensionVo>) dimensionVos);
        return configVo;
    }

    /**
     * 新增数据
     *
     * @param configDto dto对象
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void create(DiscountRateConfigDto configDto) {
        this.createValidate(configDto);
        //判断数据重复
        int count = this.discountRateConfigRepository.lambdaQuery()
                .eq(DiscountRateConfig::getOnlyKey, configDto.getOnlyKey())
                .eq(DiscountRateConfig::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .count();
        Validate.isTrue(count == 0, "已存在相同维度的折扣率配置，请勿重复添加");

//        String ruleCode = StringUtils.join(DiscountRateConfigConstant.DISCOUNT_RATE_CONFIG_PREFIX, DateFormatUtils.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY_NO_CH));
        String code = this.generateCodeService.generateCode(DiscountRateConfigConstant.DISCOUNT_RATE_CONFIG_PREFIX, 1, 5, 2, TimeUnit.DAYS).get(0);
        configDto.setConfigCode(code);
        configDto.setTenantCode(TenantUtils.getTenantCode());
        configDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        configDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        DiscountRateConfig config = nebulaToolkitService.copyObjectByWhiteList(configDto, DiscountRateConfig.class, null, null);
        this.discountRateConfigRepository.save(config);
        this.discountRateConfigDimensionService.saveBatchDimension(configDto.getDimensionList(), configDto.getConfigCode(), configDto.getTenantCode());

        //新增业务日志
        DiscountRateConfigLogEventDto logEventDto = new DiscountRateConfigLogEventDto();
        logEventDto.setOriginal(null);
        logEventDto.setNewest(configDto);
        SerializableBiConsumer<DiscountRateConfigEventListener, DiscountRateConfigLogEventDto> onCreate =
                DiscountRateConfigEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, DiscountRateConfigEventListener.class, onCreate);
    }

    /**
     * 创建验证
     *
     * @param configDto dto对象
     */
    private void createValidate(DiscountRateConfigDto configDto) {
        Validate.notNull(configDto, "新增时，对象信息不能为空！");
        configDto.setId(null);
        Validate.notBlank(configDto.getBusinessFormatCode(), "新增时，业态不能为空！");
        Validate.notBlank(configDto.getBusinessUnitCode(), "新增时，业务单元不能为空！");
        Validate.isTrue(BusinessUnitEnum.VERTICAL.getCode().equals(configDto.getBusinessUnitCode())
                        || BusinessUnitEnum.isDefaultBusinessUnit(configDto.getBusinessUnitCode()),
                "折扣率配置目前只支持主体和垂直重客");

        Validate.notBlank(configDto.getRateVersion(), "新增时，版本不能为空！");
        Validate.notBlank(configDto.getDimensionType(), "新增时，维度类型不能为空！");
        Validate.notBlank(configDto.getApproveTag(), "新增时，是否审批不能为空！");
        if (!BusinessUnitEnum.VERTICAL.getCode().equals(configDto.getBusinessUnitCode())) {
            Validate.notBlank(configDto.getBusinessDepartment(), "新增时，业务部门不能为空！");
        }

        if (BusinessUnitEnum.isDefaultBusinessUnit(configDto.getBusinessUnitCode())) {
            Validate.notBlank(configDto.getSystemRateFormula(), "新增时，系统折扣率计算公式不能为空！");
            Validate.notBlank(configDto.getSystemRateFormulaName(), "新增时，系统折扣率计算公式（展示用）不能为空！");
        }

        Validate.notBlank(configDto.getPlanRateFormula(), "新增时，计划/修正折扣率计算公式不能为空！");
        Validate.notBlank(configDto.getPlanRateFormulaName(), "新增时，计划/修正折扣率计算公式（展示用）不能为空！");

        //组装onlyKey
        String builder = configDto.getBusinessFormatCode() +
                configDto.getBusinessUnitCode() +
                configDto.getRateVersion();
        if (StringUtils.isNotBlank(configDto.getBusinessDepartment())) {
            builder = builder + configDto.getBusinessDepartment();
        }
        builder = builder + configDto.getDimensionType();
        configDto.setOnlyKey(builder);
    }

    /**
     * 修改数据
     *
     * @param configDto dto对象
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(DiscountRateConfigDto configDto) {
        this.updateValidate(configDto);
        //判断数据重复
        int count = this.discountRateConfigRepository.lambdaQuery()
                .eq(DiscountRateConfig::getOnlyKey, configDto.getOnlyKey())
                .ne(DiscountRateConfig::getId, configDto.getId())
                .eq(DiscountRateConfig::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .count();
        Validate.isTrue(count == 0, "已存在相同维度的折扣率配置，请勿重复添加");
        //获取历史vo
        DiscountRateConfigVo oldVo = this.findById(configDto.getId());

        DiscountRateConfig config = nebulaToolkitService.copyObjectByWhiteList(configDto, DiscountRateConfig.class, null, null);
        String tenantCode = TenantUtils.getTenantCode();
        this.discountRateConfigRepository.updateByIdAndTenantCode(config, tenantCode);
        this.discountRateConfigDimensionService.editBatchDimension(configDto.getDimensionList(), oldVo.getConfigCode(), tenantCode);

        //修改业务日志
        DiscountRateConfigLogEventDto logEventDto = new DiscountRateConfigLogEventDto();
        logEventDto.setOriginal(oldVo);
        logEventDto.setNewest(configDto);
        SerializableBiConsumer<DiscountRateConfigEventListener, DiscountRateConfigLogEventDto> onUpdate =
                DiscountRateConfigEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, DiscountRateConfigEventListener.class, onUpdate);
    }

    /**
     * 编辑验证
     *
     * @param configDto dto对象
     */
    private void updateValidate(DiscountRateConfigDto configDto) {
        Validate.notNull(configDto, "编辑时，对象信息不能为空！");
        Validate.notBlank(configDto.getId(), "编辑时，id不能为空！");
        Validate.notBlank(configDto.getBusinessFormatCode(), "编辑时，业态不能为空！");
        Validate.notBlank(configDto.getBusinessUnitCode(), "编辑时，业务单元不能为空！");
        Validate.notBlank(configDto.getRateVersion(), "编辑时，版本不能为空！");
        Validate.notBlank(configDto.getDimensionType(), "编辑时，维度类型不能为空！");
        Validate.notBlank(configDto.getApproveTag(), "编辑时，是否审批不能为空！");
        if (!BusinessUnitEnum.VERTICAL.getCode().equals(configDto.getBusinessUnitCode())) {
            Validate.notBlank(configDto.getBusinessDepartment(), "编辑时，业务部门不能为空！");
        }

        if (BusinessUnitEnum.isDefaultBusinessUnit(configDto.getBusinessUnitCode())) {
            Validate.notBlank(configDto.getSystemRateFormula(), "编辑时，系统折扣率计算公式不能为空！");
            Validate.notBlank(configDto.getSystemRateFormulaName(), "编辑时，系统折扣率计算公式（展示用）不能为空！");
        }

        Validate.notBlank(configDto.getPlanRateFormula(), "编辑时，计划/修正折扣率计算公式不能为空！");
        Validate.notBlank(configDto.getPlanRateFormulaName(), "编辑时，计划/修正折扣率计算公式（展示用）不能为空！");

        //组装onlyKey
        String builder = configDto.getBusinessFormatCode() +
                configDto.getBusinessUnitCode() +
                configDto.getRateVersion();
        if (StringUtils.isNotBlank(configDto.getBusinessDepartment())) {
            builder = builder + configDto.getBusinessDepartment();
        }
        builder = builder + configDto.getDimensionType();
        configDto.setOnlyKey(builder);
    }

    /**
     * 删除数据
     *
     * @param ids 主键结合
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "删除数据时，主键集合不能为空！");
        String tenantCode = TenantUtils.getTenantCode();
        List<DiscountRateConfig> configList = this.discountRateConfigRepository.lambdaQuery()
                .eq(DiscountRateConfig::getTenantCode, tenantCode)
                .in(DiscountRateConfig::getId, ids)
                .eq(DiscountRateConfig::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
        if (CollectionUtils.isEmpty(configList)) {
            return;
        }
        ArrayList<DiscountRateConfig> configs = new ArrayList<>();
        configList.forEach(item -> {
            DiscountRateConfig config = new DiscountRateConfig();
            config.setId(item.getId());
            config.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
            configs.add(config);

            //删除业务日志
            DiscountRateConfigLogEventDto logEventDto = new DiscountRateConfigLogEventDto();
            DiscountRateConfigVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(item, DiscountRateConfigVo.class, null, null);
            logEventDto.setOriginal(oldVo);
            DiscountRateConfigDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(item, DiscountRateConfigDto.class, null, null);
            newDto.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
            logEventDto.setNewest(newDto);
            SerializableBiConsumer<DiscountRateConfigEventListener, DiscountRateConfigLogEventDto> onDelete =
                    DiscountRateConfigEventListener::onDelete;
            this.nebulaNetEventClient.publish(logEventDto, DiscountRateConfigEventListener.class, onDelete);
        });
        this.discountRateConfigRepository.updateBatchByIdAndTenantCode(configs, tenantCode);
    }

    /**
     * 启用
     *
     * @param ids 主键列表
     */
    @Override
    public void enableBatch(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "启用数据时，主键集合不能为空！");
        String tenantCode = TenantUtils.getTenantCode();
        List<DiscountRateConfig> configList = this.discountRateConfigRepository.lambdaQuery()
                .eq(DiscountRateConfig::getTenantCode, tenantCode)
                .in(DiscountRateConfig::getId, ids)
                .eq(DiscountRateConfig::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
        if (CollectionUtils.isEmpty(configList)) {
            return;
        }
        ArrayList<DiscountRateConfig> configs = new ArrayList<>();
        configList.forEach(item -> {
            DiscountRateConfig config = new DiscountRateConfig();
            config.setId(item.getId());
            config.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            configs.add(config);

            //删除业务日志
            DiscountRateConfigLogEventDto logEventDto = new DiscountRateConfigLogEventDto();
            DiscountRateConfigVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(item, DiscountRateConfigVo.class, null, null);
            logEventDto.setOriginal(oldVo);
            DiscountRateConfigDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(item, DiscountRateConfigDto.class, null, null);
            newDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            logEventDto.setNewest(newDto);
            SerializableBiConsumer<DiscountRateConfigEventListener, DiscountRateConfigLogEventDto> onEnable =
                    DiscountRateConfigEventListener::onEnable;
            this.nebulaNetEventClient.publish(logEventDto, DiscountRateConfigEventListener.class, onEnable);
        });
        this.discountRateConfigRepository.updateBatchByIdAndTenantCode(configs, tenantCode);
    }

    /**
     * 禁用
     *
     * @param ids 主键列表
     */
    @Override
    public void disableBatch(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "禁用数据时，主键集合不能为空！");
        String tenantCode = TenantUtils.getTenantCode();
        List<DiscountRateConfig> configList = this.discountRateConfigRepository.lambdaQuery()
                .eq(DiscountRateConfig::getTenantCode, tenantCode)
                .in(DiscountRateConfig::getId, ids)
                .eq(DiscountRateConfig::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
        if (CollectionUtils.isEmpty(configList)) {
            return;
        }
        ArrayList<DiscountRateConfig> configs = new ArrayList<>();
        configList.forEach(item -> {
            DiscountRateConfig config = new DiscountRateConfig();
            config.setId(item.getId());
            config.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
            configs.add(config);

            //删除业务日志
            DiscountRateConfigLogEventDto logEventDto = new DiscountRateConfigLogEventDto();
            DiscountRateConfigVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(item, DiscountRateConfigVo.class, null, null);
            logEventDto.setOriginal(oldVo);
            DiscountRateConfigDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(item, DiscountRateConfigDto.class, null, null);
            newDto.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
            logEventDto.setNewest(newDto);
            SerializableBiConsumer<DiscountRateConfigEventListener, DiscountRateConfigLogEventDto> onDisable =
                    DiscountRateConfigEventListener::onDisable;
            this.nebulaNetEventClient.publish(logEventDto, DiscountRateConfigEventListener.class, onDisable);
        });
        this.discountRateConfigRepository.updateBatchByIdAndTenantCode(configs, tenantCode);
    }

    /**
     * 获取折扣率配置公式变量
     *
     * @param businessUnitCode 业务单元
     * @return 列表数据
     */
    @Override
    public List<DiscountRateConfigFormulaVariableVo> getFormulaVariable(String businessUnitCode) {

        Validate.notBlank(businessUnitCode, "业务单元不能为空");
        List<DiscountRateConfigFormulaVariableVo> voList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(formulaVariableRegisterList)) {
            formulaVariableRegisterList.forEach(register -> {
                if (businessUnitCode.equals(register.getType())) {
                    DiscountRateConfigFormulaVariableVo vo = new DiscountRateConfigFormulaVariableVo();
                    vo.setCode(register.getVariableCode());
                    vo.setName(register.getVariableName());
                    vo.setSort(register.getSort());
                    voList.add(vo);
                }
            });
        }
        if (!CollectionUtils.isEmpty(voList)) {
            return voList.stream().sorted(Comparator.comparing(DiscountRateConfigFormulaVariableVo::getSort)).collect(Collectors.toList());
        }
        return Lists.newArrayList();
    }

    /**
     * 获取折扣率配置通过onlyKeys
     *
     * @param onlyKeys onlyKey列表
     * @return 列表数据
     */
    @Override
    public Map<String, DiscountRateConfig> getConfigByOnlyKeys(List<String> onlyKeys) {
        Map<String, DiscountRateConfig> map = new HashMap<>();
        if (CollectionUtils.isEmpty(onlyKeys)) {
            return map;
        }
        List<DiscountRateConfig> list = this.discountRateConfigRepository.lambdaQuery()
                .eq(DiscountRateConfig::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(DiscountRateConfig::getTenantCode, TenantUtils.getTenantCode())
                .in(DiscountRateConfig::getOnlyKey, onlyKeys)
                .list();
        if (CollectionUtils.isEmpty(list)) {
            return map;
        }
        return list.stream().collect(Collectors.toMap(DiscountRateConfig::getOnlyKey, Function.identity()));
    }
}
