package com.biz.crm.tpm.business.budget.discount.rate.local.service.internal;

import com.aliyun.tea.utils.Validate;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.budget.discount.rate.local.entity.DiscountRate;
import com.biz.crm.tpm.business.budget.discount.rate.local.repository.DiscountRateRepository;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.DiscountRateBudgetDto;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.DiscountRateDto;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.DiscountRatePlanCalDto;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.enums.DiscountRateDimensionEnum;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.enums.DiscountRateVersionEnum;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.service.DiscountRateSdkService;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.vo.DiscountRateBudgetVo;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.vo.DiscountRateVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author: chenlong
 * @date: 2023/1/7 11:16
 * @description: 折扣率管理(DiscountRate)表sdk服务实现类
 */
@Service("DiscountRateSdkService")
public class DiscountRateSdkServiceImpl implements DiscountRateSdkService {

    @Autowired(required = false)
    private DiscountRateRepository discountRateRepository;
    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 分页查询所有数据
     *
     * @param pageable 分页对象
     * @param rateDto  查询dto
     * @return 所有数据
     */
    @Override
    public Page<DiscountRateVo> findByConditions(Pageable pageable, DiscountRateDto rateDto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(rateDto)) {
            rateDto = new DiscountRateDto();
        }
        rateDto.setTenantCode(TenantUtils.getTenantCode());
        return this.discountRateRepository.findByConditions(pageable, rateDto);
    }

    /**
     * 根据预算获取折扣率
     *
     * @param budgetDtoList 预算数据
     * @return Map<String, DiscountRateBudgetVo>
     */
    @Override
    public Map<String, DiscountRateBudgetVo> getDiscountRateByDimensionCode(List<DiscountRateBudgetDto> budgetDtoList) {
        //默认版本为计划折扣率
        String version = DiscountRateVersionEnum.PLAN.getCode();
        String dimension = DiscountRateDimensionEnum.RETAILER_REGION.getCode();
        Map<String, DiscountRateBudgetVo> map = new HashMap<>();
        if (CollectionUtils.isEmpty(budgetDtoList)) {
            return map;
        }
        List<String> onlykeys = new ArrayList<>();
        for (DiscountRateBudgetDto dto : budgetDtoList) {
            Validate.isTrue(StringUtils.isNotBlank(dto.getMonthBudgetCode()), "预算编码不能为空");
            Validate.isTrue(StringUtils.isNotBlank(dto.getBusinessFormatCode()), "业态不能为空");
            Validate.isTrue(StringUtils.isNotBlank(dto.getBusinessUnitCode()), "业务单元不能为空");
            Validate.notNull(dto.getYearAndMonth(), "年月不能为空");
            String dateStr = DateUtil.dateToStr(DateUtil.date_yyyy_MM, dto.getYearAndMonth());
            Validate.isTrue(BusinessUnitEnum.VERTICAL.getCode().equals(dto.getBusinessUnitCode()), "业务单元必须为垂直");
            Validate.isTrue(StringUtils.isNotBlank(dto.getCustomerRetailerCode()), "零售商编码不能为空");
            Validate.isTrue(StringUtils.isNotBlank(dto.getBrandCode()), "品牌编码不能为空");
            Validate.isTrue(StringUtils.isNotBlank(dto.getRegionCode()), "区域编码不能为空");
            String onlykey = dto.getBusinessFormatCode() + dto.getBusinessUnitCode() + version + dimension + dateStr +
                    dto.getCustomerRetailerCode() + dto.getRegionCode();
            onlykeys.add(onlykey);
        }
        List<DiscountRate> rateList = this.discountRateRepository.lambdaQuery()
                .eq(DiscountRate::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(DiscountRate::getTenantCode, TenantUtils.getTenantCode())
                .in(DiscountRate::getOnlyKey, onlykeys)
                .list();
        if (CollectionUtils.isEmpty(rateList)) {
            return map;
        }
        Map<String, DiscountRate> rateMap = rateList.stream().collect(Collectors.toMap(DiscountRate::getOnlyKey, Function.identity()));
        for (DiscountRateBudgetDto budgetDto : budgetDtoList) {
            String dateStr = DateUtil.dateToStr(DateUtil.date_yyyy_MM, budgetDto.getYearAndMonth());
            String onlykey = budgetDto.getBusinessFormatCode() + budgetDto.getBusinessUnitCode() + version + dimension + dateStr +
                    budgetDto.getCustomerRetailerCode() + budgetDto.getRegionCode();
            if (rateMap.containsKey(onlykey)) {
                DiscountRate rate = rateMap.get(onlykey);
                DiscountRateBudgetVo budgetVo = this.nebulaToolkitService.copyObjectByBlankList(
                        rate, DiscountRateBudgetVo.class, null, null);
                map.put(budgetDto.getMonthBudgetCode(), budgetVo);
            }
        }
        return map;
    }

    /**
     * 根据业态+业务单元+客户+产品组装的key获取折扣率
     *
     * @param keys 分页对象
     * @return List<DiscountRateDto>
     */
    @Override
    public List<DiscountRateDto> getDiscountRateByKeys(List<List<String>> keys) {
        if (CollectionUtils.isEmpty(keys)) {
            return Lists.newArrayList();
        }
        return this.discountRateRepository.getDiscountRateByKeys(keys, TenantUtils.getTenantCode());
    }

    /**
     * 根据业态+业务单元+客户组装的key获取折扣率
     *
     * @param keys 分页对象
     * @return List<DiscountRateDto>
     */
    @Override
    public List<DiscountRateDto> getDiscountRateByKeysNoPro(List<List<String>> keys) {
        if (CollectionUtils.isEmpty(keys)) {
            return Lists.newArrayList();
        }
        return this.discountRateRepository.getDiscountRateByKeysNoPro(keys, TenantUtils.getTenantCode());
    }

    /**
     * 批量查询
     *
     * @param onlyKeyList
     * @return java.util.Map<java.lang.String, com.biz.crm.tpm.business.budget.discount.rate.sdk.vo.DiscountRateBudgetVo>
     * @author huojia
     * @date 2023/1/20 16:03
     **/
    @Override
    public Map<String, DiscountRateBudgetVo> getDiscountRateByOnlyKey(List<String> onlyKeyList) {
        Map<String, DiscountRateBudgetVo> map = new HashMap<>();
        if (CollectionUtils.isEmpty(onlyKeyList)) {
            return map;
        }
        List<DiscountRate> rateList = this.discountRateRepository.lambdaQuery()
                .eq(DiscountRate::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(DiscountRate::getTenantCode, TenantUtils.getTenantCode())
                .in(DiscountRate::getOnlyKey, onlyKeyList)
                .list();
        if (CollectionUtils.isEmpty(rateList)) {
            return map;
        }
        List<DiscountRateBudgetVo> discountRateBudgetVos = (List<DiscountRateBudgetVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
                rateList, DiscountRate.class, DiscountRateBudgetVo.class, LinkedHashSet.class, ArrayList.class
        );
        return discountRateBudgetVos.stream()
                .collect(Collectors.toMap(DiscountRateBudgetVo::getOnlyKey, Function.identity(), (oldVo, newVo) -> newVo));
    }

    /**
     * 销售计划获取折扣率
     *
     * @param calDto 筛选数据
     * @return Result<List < DiscountRateDto>>
     */
    @Override
    public List<DiscountRateDto> findRateByPlanConditions(DiscountRatePlanCalDto calDto) {
        return this.discountRateRepository.findRateByPlanConditions(calDto);
    }
}
