package com.biz.crm.tpm.business.budget.discount.rate.local.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.customer.retailer.sdk.service.CustomerRetailerVoService;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.product.brand.sdk.service.ProductBrandService;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.enums.YesOrNoEnum;
import com.biz.crm.tpm.business.budget.discount.rate.local.entity.DiscountRate;
import com.biz.crm.tpm.business.budget.discount.rate.local.entity.DiscountRateConfig;
import com.biz.crm.tpm.business.budget.discount.rate.local.repository.DiscountRateConfigRepository;
import com.biz.crm.tpm.business.budget.discount.rate.local.repository.DiscountRateRepository;
import com.biz.crm.tpm.business.budget.discount.rate.local.service.DiscountRateAsyncService;
import com.biz.crm.tpm.business.budget.discount.rate.local.service.DiscountRateLogService;
import com.biz.crm.tpm.business.budget.discount.rate.local.service.DiscountRateService;
import com.biz.crm.tpm.business.budget.discount.rate.local.service.DiscountRateVariableService;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.constant.DiscountRateConstant;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.dto.*;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.enums.DiscountRateDimensionEnum;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.event.log.DiscountRateEventListener;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.service.DiscountRateConfigSdkService;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.vo.DiscountRateConfigVo;
import com.biz.crm.tpm.business.budget.discount.rate.sdk.vo.DiscountRateVo;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.dto.ProcessStatusDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author: chenlong
 * @date: 2023/1/7 11:00
 * @description: 折扣率管理(DiscountRate)表服务实现类
 */
@Slf4j
@Service("discountRateService")
public class DiscountRateServiceImpl implements DiscountRateService {

    @Autowired(required = false)
    private DiscountRateRepository discountRateRepository;
    @Autowired(required = false)
    private DiscountRateConfigRepository discountRateConfigRepository;
    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;
    @Autowired(required = false)
    private GenerateCodeService generateCodeService;
    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;
    @Autowired(required = false)
    private LoginUserService loginUserService;
    @Autowired(required = false)
    private ProcessBusinessService processBusinessService;
    @Autowired(required = false)
    private DiscountRateVariableService discountRateVariableService;
    @Autowired(required = false)
    private DiscountRateLogService discountRateLogService;
    @Autowired(required = false)
    private DiscountRateConfigSdkService discountRateConfigSdkService;
    @Autowired(required = false)
    private CustomerRetailerVoService customerRetailerVoService;
    @Autowired(required = false)
    private TerminalVoService terminalVoService;
    @Autowired(required = false)
    private ProductBrandService productBrandService;
    @Autowired(required = false)
    private DiscountRateAsyncService discountRateAsyncService;
    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public DiscountRateVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        DiscountRate rate = this.discountRateRepository.lambdaQuery()
                .eq(DiscountRate::getId, id)
                .eq(DiscountRate::getTenantCode, TenantUtils.getTenantCode())
                .one();
        if (null == rate) {
            return null;
        }
        return this.nebulaToolkitService.copyObjectByWhiteList(rate, DiscountRateVo.class, null, null);
    }

    /**
     * 修改数据
     *
     * @param rateDto dto对象
     */
    @Override
    public void update(DiscountRateDto rateDto) {

    }

    /**
     * 删除数据
     *
     * @param ids 主键结合
     */
    @Override
    public void delete(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "删除数据时，主键集合不能为空！");
        String tenantCode = TenantUtils.getTenantCode();
        List<DiscountRate> rateList = this.discountRateRepository.lambdaQuery()
                .eq(DiscountRate::getTenantCode, tenantCode)
                .in(DiscountRate::getId, ids)
                .eq(DiscountRate::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
        if (CollectionUtils.isEmpty(rateList)) {
            return;
        }
        ArrayList<DiscountRate> rates = new ArrayList<>();
        rateList.forEach(item -> {
            DiscountRate rate = new DiscountRate();
            rate.setId(item.getId());
            rate.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
            rates.add(rate);

            //删除业务日志
            DiscountRateLogEventDto logEventDto = new DiscountRateLogEventDto();
            DiscountRateVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(item, DiscountRateVo.class, null, null);
            logEventDto.setOriginal(oldVo);
            DiscountRateDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(item, DiscountRateDto.class, null, null);
            newDto.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
            logEventDto.setNewest(newDto);
            SerializableBiConsumer<DiscountRateEventListener, DiscountRateLogEventDto> onDelete =
                    DiscountRateEventListener::onDelete;
            this.nebulaNetEventClient.publish(logEventDto, DiscountRateEventListener.class, onDelete);
        });
        this.discountRateRepository.updateBatchByIdAndTenantCode(rates, tenantCode);
    }

    /**
     * 启用
     *
     * @param ids 主键列表
     */
    @Override
    public void enableBatch(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "删除数据时，主键集合不能为空！");
        String tenantCode = TenantUtils.getTenantCode();
        List<DiscountRate> rateList = this.discountRateRepository.lambdaQuery()
                .eq(DiscountRate::getTenantCode, tenantCode)
                .in(DiscountRate::getId, ids)
                .eq(DiscountRate::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
        if (CollectionUtils.isEmpty(rateList)) {
            return;
        }
        ArrayList<DiscountRate> rates = new ArrayList<>();
        rateList.forEach(item -> {
            DiscountRate rate = new DiscountRate();
            rate.setId(item.getId());
            rate.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            rates.add(rate);

            //删除业务日志
            DiscountRateLogEventDto logEventDto = new DiscountRateLogEventDto();
            DiscountRateVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(item, DiscountRateVo.class, null, null);
            logEventDto.setOriginal(oldVo);
            DiscountRateDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(item, DiscountRateDto.class, null, null);
            newDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            logEventDto.setNewest(newDto);
            SerializableBiConsumer<DiscountRateEventListener, DiscountRateLogEventDto> onEnable =
                    DiscountRateEventListener::onEnable;
            this.nebulaNetEventClient.publish(logEventDto, DiscountRateEventListener.class, onEnable);
        });
        this.discountRateRepository.updateBatchByIdAndTenantCode(rates, tenantCode);
    }

    /**
     * 禁用
     *
     * @param ids 主键列表
     */
    @Override
    public void disableBatch(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "删除数据时，主键集合不能为空！");
        String tenantCode = TenantUtils.getTenantCode();
        List<DiscountRate> rateList = this.discountRateRepository.lambdaQuery()
                .eq(DiscountRate::getTenantCode, tenantCode)
                .in(DiscountRate::getId, ids)
                .eq(DiscountRate::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
        if (CollectionUtils.isEmpty(rateList)) {
            return;
        }
        ArrayList<DiscountRate> rates = new ArrayList<>();
        rateList.forEach(item -> {
            DiscountRate rate = new DiscountRate();
            rate.setId(item.getId());
            rate.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
            rates.add(rate);

            //删除业务日志
            DiscountRateLogEventDto logEventDto = new DiscountRateLogEventDto();
            DiscountRateVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(item, DiscountRateVo.class, null, null);
            logEventDto.setOriginal(oldVo);
            DiscountRateDto newDto = this.nebulaToolkitService.copyObjectByWhiteList(item, DiscountRateDto.class, null, null);
            newDto.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
            logEventDto.setNewest(newDto);
            SerializableBiConsumer<DiscountRateEventListener, DiscountRateLogEventDto> onEnable =
                    DiscountRateEventListener::onEnable;
            this.nebulaNetEventClient.publish(logEventDto, DiscountRateEventListener.class, onEnable);
        });
        this.discountRateRepository.updateBatchByIdAndTenantCode(rates, tenantCode);
    }

    /**
     * 大批量保存
     *
     * @param importList 导入数据列表
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void bulkImportSave(List<DiscountRateDto> importList) {
        try {
            if (CollectionUtils.isEmpty(importList)) {
                return;
            }
            //设置基础信息
            AbstractCrmUserIdentity loginDetails = this.loginUserService.getAbstractLoginUser();
            Date date = new Date();
            //循环设置信息并区分编辑和新增数据
            List<DiscountRateDto> addList = new ArrayList<>();
            List<DiscountRateDto> editList = new ArrayList<>();
            for (DiscountRateDto dto : importList) {
                if (StringUtils.isNotBlank(dto.getId())) {
                    dto.setModifyAccount(loginDetails.getUsername());
                    dto.setModifyName(loginDetails.getRealName());
                    dto.setModifyTime(date);
                    editList.add(dto);
                } else {
                    if (StringUtils.isBlank(dto.getApproveTag())) {
                        dto.setApproveTag(YesOrNoEnum.NO.getCode());
                    }
                    dto.setCreateAccount(loginDetails.getUsername());
                    dto.setCreateName(loginDetails.getRealName());
                    dto.setCreateTime(date);
                    dto.setTenantCode(loginDetails.getTenantCode());
                    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                    dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                    addList.add(dto);
                }
            }
            //新增数据
            if (!CollectionUtils.isEmpty(addList)) {
                log.info("开始执行大批量保存-----------------新增数据条数：{}", editList.size());
//                String ruleCode = StringUtils.join(DiscountRateConstant.DISCOUNT_RATE_PREFIX, DateFormatUtils.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY_NO_CH));
                List<String> codes = this.generateCodeService.generateCode(DiscountRateConstant.DISCOUNT_RATE_PREFIX, addList.size(), 5, 2, TimeUnit.DAYS);
                for (int i = 0; i < addList.size(); i++) {
                    DiscountRateDto rateDto = addList.get(i);
                    rateDto.setDiscountRateCode(codes.get(i));
                }
                Collection<DiscountRate> adds = this.nebulaToolkitService.copyCollectionByWhiteList(
                        addList, DiscountRateDto.class, DiscountRate.class, LinkedHashSet.class, ArrayList.class);
                List<DiscountRate> rateList = (List<DiscountRate>) adds;
                //分批保存
                List<List<DiscountRate>> rates = Lists.partition(rateList, 500);
                for (List<DiscountRate> list : rates) {

                    this.discountRateRepository.bulkSave(list);
                }
            }
            //更新数据
            List<DiscountRateLogEventDto> editLogs = new ArrayList<>();
            if (!CollectionUtils.isEmpty(editList)) {
                log.info("开始执行大批量保存-----------------编辑数据条数：{}", editList.size());
                //分批保存
                List<List<DiscountRateDto>> rates = Lists.partition(editList, 500);
                for (List<DiscountRateDto> list : rates) {
                    List<String> ids = list.stream().map(DiscountRateDto::getId).collect(Collectors.toList());
                    List<DiscountRate> oldList = this.discountRateRepository.lambdaQuery()
                            .in(DiscountRate::getId, ids)
                            .eq(DiscountRate::getTenantCode, TenantUtils.getTenantCode())
                            .list();
                    Collection<DiscountRateVo> oldDtoList = this.nebulaToolkitService.copyCollectionByWhiteList(
                            oldList, DiscountRate.class, DiscountRateVo.class, LinkedHashSet.class, ArrayList.class);
                    Map<String, DiscountRateVo> oldMap = oldDtoList.stream().collect(Collectors.toMap(DiscountRateVo::getId, Function.identity()));
                    for (DiscountRateDto dto : list) {
                        DiscountRateLogEventDto eventDto = new DiscountRateLogEventDto();
                        eventDto.setNewest(dto);
                        eventDto.setOriginal(oldMap.getOrDefault(dto.getId(), new DiscountRateVo()));
                        editLogs.add(eventDto);
                    }
                    Collection<DiscountRate> res = this.nebulaToolkitService.copyCollectionByWhiteList(
                            list, DiscountRateDto.class, DiscountRate.class, LinkedHashSet.class, ArrayList.class);
                    this.discountRateRepository.updateBatchByIdAndTenantCode(res, TenantUtils.getTenantCode());
                }
            }

            //新增日志
            if (!CollectionUtils.isEmpty(addList)) {
                log.info("开始执行大批量保存-----------------新增日志");
                discountRateLogService.addLogAsync(addList);
            }
            //编辑日志
            if (!CollectionUtils.isEmpty(editLogs)) {
                log.info("开始执行大批量保存-----------------编辑日志");
                discountRateLogService.updateLogAsync(editLogs);
            }
        } catch (Exception e) {
            log.error("", e);
            throw new NullPointerException(e.getMessage());
        }
    }

    /**
     * 提交工作流
     *
     * @param processDto 参数
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void submitApproval(DiscountRateProcessDto processDto) {
        Validate.isTrue(Objects.nonNull(processDto.getProcessBusiness()), "流程参数不能为空");
        DiscountRate rate = this.discountRateRepository.lambdaQuery()
                .eq(DiscountRate::getId, processDto.getId())
                .eq(DiscountRate::getTenantCode, TenantUtils.getTenantCode())
                .eq(DiscountRate::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .one();
        Validate.notNull(rate, "未找到对应的折扣率");
        DiscountRateConfig config = this.discountRateConfigRepository.lambdaQuery()
                .eq(DiscountRateConfig::getConfigCode, rate.getConfigCode())
                .eq(DiscountRateConfig::getTenantCode, TenantUtils.getTenantCode())
                .eq(DiscountRateConfig::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .one();
        Validate.notNull(config, "未找到对应的折扣率配置");
        Validate.isTrue(BooleanEnum.TRUE.getCapital().equals(config.getApproveTag()), "此折扣率无需审批");
        DiscountRateDto rateDto = nebulaToolkitService.copyObjectByWhiteList(rate, DiscountRateDto.class, HashSet.class, ArrayList.class);
        submitApprovalHandle(rateDto, processDto.getProcessBusiness());
    }

    /**
     * 提交工作流处理
     *
     * @param dto
     */
    public void submitApprovalHandle(DiscountRateDto dto, ProcessBusinessDto processBusiness) {
        processBusiness.setBusinessNo(dto.getDiscountRateCode());
        JSONObject jsonObject = JsonUtils.toJSONObject(dto);
        processBusiness.setBusinessFormJson(jsonObject.toJSONString());
        processBusiness.setBusinessCode(DiscountRateConstant.DISCOUNT_RATE_PROCESS);
        ProcessBusinessVo processBusinessVo = this.processBusinessService.processStart(processBusiness);

        DiscountRate entity = new DiscountRate();
        entity.setProcessStatus(ProcessStatusEnum.COMMIT.getDictCode());
        entity.setProcessNo(processBusinessVo.getProcessNo());
        entity.setId(dto.getId());
        this.discountRateRepository.updateByIdAndTenantCode(entity, TenantUtils.getTenantCode());
    }

    /**
     * 流程审批通过
     *
     * @param dto 流程参数
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void processPass(ProcessStatusDto dto) {
        discountRateRepository.updateProcessStatus(dto.getProcessNo(), dto.getProcessStatus());
    }

    /**
     * 审批驳回|流程追回
     *
     * @param dto 流程参数
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void processRejectAndRecover(ProcessStatusDto dto) {
        discountRateRepository.updateProcessStatus(dto.getProcessNo(), dto.getProcessStatus());
    }

    /**
     * 更新计算
     *
     * @param recalDto 流程参数
     */
//    @Async
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void recal(DiscountRateRecalDto recalDto) {
        Validate.isTrue(StringUtils.isNotBlank(recalDto.getBusinessFormatCode()), "业态不能为空");
        Validate.isTrue(StringUtils.isNotBlank(recalDto.getBusinessUnitCode()), "业务单元不能为空");
        Validate.isTrue(StringUtils.isNotBlank(recalDto.getRateVersion()), "版本不能为空");
        Validate.notNull(recalDto.getYearAndMonth(), "年月不能为空");
        //获取需要更新计算的数据
        LambdaQueryWrapper<DiscountRate> wrapper = new LambdaQueryWrapper<DiscountRate>()
                .eq(DiscountRate::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(DiscountRate::getTenantCode, TenantUtils.getTenantCode())
                .eq(DiscountRate::getYearAndMonth, recalDto.getYearAndMonth())
                .eq(DiscountRate::getBusinessFormatCode, recalDto.getBusinessFormatCode())
                .eq(DiscountRate::getBusinessUnitCode, recalDto.getBusinessUnitCode())
                .eq(DiscountRate::getRateVersion, recalDto.getRateVersion());
        if (StringUtils.isNotBlank(recalDto.getCustomerCode())) {
            wrapper.eq(DiscountRate::getCustomerCode, recalDto.getCustomerCode());
        }
        if (StringUtils.isNotBlank(recalDto.getRegionCode())) {
            wrapper.eq(DiscountRate::getRegionCode, recalDto.getRegionCode());
        }
        if (StringUtils.isNotBlank(recalDto.getCustomerRetailerCode())) {
            wrapper.eq(DiscountRate::getCustomerRetailerCode, recalDto.getCustomerRetailerCode());
        }
        if (StringUtils.isNotBlank(recalDto.getBrandCode())) {
            wrapper.eq(DiscountRate::getBrandCode, recalDto.getBrandCode());
        }
        List<DiscountRate> list = this.discountRateRepository.list(wrapper);
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        Map<String, List<DiscountRate>> configKeyMap = list.stream().collect(Collectors.groupingBy(i -> {
                    String key = i.getBusinessFormatCode() + i.getBusinessUnitCode() + i.getRateVersion();
                    if (StringUtils.isNotBlank(i.getBusinessDepartment())) {
                        key = key + i.getBusinessDepartment();
                    }
                    key = key + i.getDimensionType();
                    return key;
                }
        ));
        //查询折扣率配置
        List<DiscountRateConfig> configList = this.discountRateConfigRepository.lambdaQuery()
                .eq(DiscountRateConfig::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(DiscountRateConfig::getTenantCode, TenantUtils.getTenantCode())
                .in(DiscountRateConfig::getOnlyKey, configKeyMap.keySet())
                .list();

        if (CollectionUtils.isEmpty(configList)) {
            return;
        }
        //循环折扣率配置
        for (DiscountRateConfig config : configList) {
            //获取需要更新的折扣率数据
            List<DiscountRate> rateList = configKeyMap.getOrDefault(config.getOnlyKey(), null);
            if (CollectionUtils.isEmpty(rateList)) {
                continue;
            }
            //判断维度
            List<String> cusCodes = new ArrayList<>();
            List<String> proCodes = new ArrayList<>();
            List<String> reCodes = new ArrayList<>();
//            List<String> brandCodes = new ArrayList<>();
            List<String> regionCodes = new ArrayList<>();


            for (DiscountRate rate : rateList) {
                if (StringUtils.isNotBlank(rate.getCustomerCode())) {
                    cusCodes.add(rate.getCustomerCode());
                }
                if (StringUtils.isNotBlank(rate.getProductCode())) {
                    proCodes.add(rate.getProductCode());
                }
                if (StringUtils.isNotBlank(rate.getCustomerRetailerCode())) {
                    reCodes.add(rate.getCustomerRetailerCode());
                }
//                if (StringUtils.isNotBlank(rate.getBrandCode())) {
//                    brandCodes.add(rate.getBrandCode());
//                }
                if (StringUtils.isNotBlank(rate.getRegionCode())) {
                    regionCodes.add(rate.getRegionCode());
                }
            }
            CalculateDto calculateDto = new CalculateDto();
            calculateDto.setYearAndMonth(recalDto.getYearAndMonth());
            calculateDto.setDimensionType(config.getDimensionType());
            Map<String, BigDecimal> sysMap = null;
            Map<String, BigDecimal> planMap = null;
            if (DiscountRateDimensionEnum.CUSTOMER.getCode().equals(config.getDimensionType())) {
                calculateDto.setCustomerCodeList(cusCodes);
                calculateDto.setFormula(config.getSystemRateFormula());
                sysMap = discountRateVariableService.singleCalculateExpressionBatch(calculateDto);
                calculateDto.setFormula(config.getSystemRateFormula());
                planMap = discountRateVariableService.singleCalculateExpressionBatch(calculateDto);
            } else if (DiscountRateDimensionEnum.CUSTOMER_PRODUCT.getCode().equals(config.getDimensionType())) {
                calculateDto.setCustomerCodeList(cusCodes);
                calculateDto.setProductCodeList(proCodes);
                calculateDto.setFormula(config.getSystemRateFormula());
                sysMap = discountRateVariableService.singleCalculateExpressionBatch(calculateDto);
                calculateDto.setFormula(config.getSystemRateFormula());
                planMap = discountRateVariableService.singleCalculateExpressionBatch(calculateDto);
            } else if (DiscountRateDimensionEnum.PRODUCT.getCode().equals(config.getDimensionType())) {
                calculateDto.setProductCodeList(proCodes);
                calculateDto.setFormula(config.getSystemRateFormula());
                sysMap = discountRateVariableService.singleCalculateExpressionBatch(calculateDto);
                calculateDto.setFormula(config.getSystemRateFormula());
                planMap = discountRateVariableService.singleCalculateExpressionBatch(calculateDto);
            } else if (DiscountRateDimensionEnum.RETAILER_REGION.getCode().equals(config.getDimensionType())) {
                calculateDto.setCustomerRetailerCodeList(reCodes);
//                calculateDto.setBrandCodeList(brandCodes);
                calculateDto.setRegionCodeList(regionCodes);
                calculateDto.setFormula(config.getSystemRateFormula());
                planMap = discountRateVariableService.singleCalculateExpressionBatch(calculateDto);
            } else {
                continue;
            }
            updateDate(config, rateList, sysMap, planMap);
        }
    }

    /**
     * 通过折扣率配置自动生成折扣率
     */
    @Override
    public void verticalDiscountRateCal(String yearMonth) {
        //设置基础信息
        loginUserService.refreshAuthentication(null);
        AbstractCrmUserIdentity loginDetails = this.loginUserService.getAbstractLoginUser();

        //先查询折扣率配置
        DiscountRateConfigDto configDto = new DiscountRateConfigDto();
        configDto.setBusinessUnitCode(BusinessUnitEnum.VERTICAL.getCode());
        //这里目前只生成垂直-零售商+区域的期初
//        configDto.setDimensionType(DiscountRateDimensionEnum.RETAILER_REGION_BRAND.getCode());
        configDto.setDimensionType(DiscountRateDimensionEnum.RETAILER_REGION.getCode());
        //查询折扣率配置条数
        Integer count = discountRateConfigRepository.lambdaQuery()
                .eq(DiscountRateConfig::getTenantCode, TenantUtils.getTenantCode())
                .eq(DiscountRateConfig::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(DiscountRateConfig::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
                .eq(DiscountRateConfig::getDimensionType, configDto.getDimensionType())
                .eq(DiscountRateConfig::getBusinessUnitCode, configDto.getBusinessUnitCode())
                .count();
        if (count == 0) {
            return;
        }
        //查询零售商，区域，品牌的名称
        Map<String, String> sysMap = customerRetailerVoService.getAllRetailerCode();
        Map<String, String> regionMap = this.dictToolkitService.findMapByDictTypeCode(DiscountRateConstant.MDM_CUSTOMIZE_ORG);
//        Map<String, String> brandMap = productBrandService.getAllBrandCode();
        int i = 1;
        while (true) {
            Page<DiscountRateConfigVo> page = discountRateConfigSdkService.findByConditions(PageRequest.of(i, 50), configDto);
            if (CollectionUtils.isEmpty(page.getRecords())) {
                return;
            }
            //循环折扣率配置，根据公式统计数据
            for (DiscountRateConfigVo record : page.getRecords()) {
                //批量获取公式值
                CalculateDto calculateDto = new CalculateDto();
                calculateDto.setFormula(record.getPlanRateFormula());
                calculateDto.setAllTag(true);
                calculateDto.setBusinessUnitCode(record.getBusinessUnitCode());
                calculateDto.setBusinessFormatCode(record.getBusinessFormatCode());
                calculateDto.setYearAndMonth(DateUtil.getDateByFormat(yearMonth, DateUtil.DEFAULT_YEAR_MONTH));
                Map<String, BigDecimal> map = discountRateVariableService.singleCalculateExpressionBatch(calculateDto);
                //生成折扣率数据
                if (map.isEmpty()) {
                    continue;
                }
                //组装数据
                List<DiscountRateDto> rateList = new ArrayList<>();
                for (String s : map.keySet()) {
                    String[] arr = s.split("-");
                    if (arr.length < 2) {
                        continue;
                    }
                    DiscountRateDto rate = new DiscountRateDto();
                    rate.setApproveTag(record.getApproveTag());
                    if (YesOrNoEnum.YES.getCode().equals(record.getApproveTag())) {
                        rate.setProcessStatus(ProcessStatusEnum.PREPARE.getDictCode());
                    }
                    rate.setBusinessFormatCode(record.getBusinessFormatCode());
                    rate.setBusinessUnitCode(record.getBusinessUnitCode());
                    rate.setBusinessDepartment(record.getBusinessDepartment());
                    rate.setDimensionType(record.getDimensionType());
                    rate.setRateVersion(record.getRateVersion());
                    rate.setYearAndMonth(calculateDto.getYearAndMonth());
                    rate.setYearAndMonthStr(DateUtil.dateToStr(calculateDto.getYearAndMonth(), DateUtil.date_yyyy_MM));
                    rate.setCustomerRetailerCode(arr[0]);
                    rate.setCustomerRetailerName(sysMap.getOrDefault(rate.getCustomerRetailerCode(), null));
                    rate.setRegionCode(arr[1]);
                    rate.setRegionName(regionMap.getOrDefault(rate.getRegionCode(), null));
//                    rate.setBrandCode(arr[2]);
//                    rate.setBrandName(brandMap.getOrDefault(rate.getBrandCode(), null));
                    rate.setPlanRate(map.getOrDefault(s, BigDecimal.ZERO).setScale(3, BigDecimal.ROUND_HALF_UP));
                    rate.setPlanRateStr(rate.getPlanRate().multiply(new BigDecimal(100)).toString() + '%');
                    rate.setConfigCode(configDto.getConfigCode());
                    rate.setTenantCode(TenantUtils.getTenantCode());
                    String onlyKey = rate.getBusinessFormatCode() + rate.getBusinessUnitCode() + rate.getRateVersion() + rate.getDimensionType()
                            + yearMonth + rate.getCustomerRetailerCode() + rate.getRegionCode();
                    rate.setOnlyKey(onlyKey);
                    rateList.add(rate);
                }
                this.discountRateAsyncService.batchAddEdit(rateList);
            }
            i++;
        }
    }

    /**
     * 根据onlykeys获取id
     *
     * @param keys onlyKey列表
     * @return Map<String, String>
     */
    @Override
    public Map<String, String> getIdByKeys(List<String> keys) {
        if (CollectionUtils.isEmpty(keys)) {
            return Maps.newHashMap();
        }
        List<List<String>> keyList = Lists.partition(keys, 500);
        List<DiscountRate> rates = this.discountRateRepository.getIdByKeys(keyList, TenantUtils.getTenantCode());
        if (CollectionUtils.isEmpty(rates)) {
            return Maps.newHashMap();
        }
        return rates.stream().collect(Collectors.toMap(DiscountRate::getOnlyKey, DiscountRate::getId, (oldValue, newValue) -> newValue));
    }

    /**
     * 更新数据
     *
     * @param config   折扣率配置
     * @param rateList 折扣率
     * @param sysMap   系统折扣率公式结果
     * @param planMap  计划折扣率公式结果
     */
    public void updateDate(DiscountRateConfig config, List<DiscountRate> rateList,
                           Map<String, BigDecimal> sysMap, Map<String, BigDecimal> planMap) {
        //编辑
        if (!CollectionUtils.isEmpty(rateList)) {
            for (DiscountRate rate : rateList) {
                String key = null;
                if (DiscountRateDimensionEnum.CUSTOMER.getCode().equals(config.getDimensionType())) {
                    key = rate.getCustomerCode();
                } else if (DiscountRateDimensionEnum.CUSTOMER_PRODUCT.getCode().equals(config.getDimensionType())) {
                    key = rate.getCustomerCode() + rate.getProductCode();
                } else if (DiscountRateDimensionEnum.PRODUCT.getCode().equals(config.getDimensionType())) {
                    key = rate.getProductCode();
                } else if (DiscountRateDimensionEnum.RETAILER_REGION_BRAND.getCode().equals(config.getDimensionType())) {
                    key = rate.getCustomerRetailerCode() + rate.getBrandCode() + rate.getRegionCode();
                }else if (DiscountRateDimensionEnum.RETAILER_REGION.getCode().equals(config.getDimensionType())) {
                    key = rate.getCustomerRetailerCode() + rate.getRegionCode();
                } else {
                    continue;
                }
                if (null != sysMap) {
                    BigDecimal b = sysMap.getOrDefault(key, BigDecimal.ZERO);
                    rate.setSystemRate(b.multiply(BigDecimal.valueOf(100)).setScale(2, BigDecimal.ROUND_HALF_UP));
                    rate.setSystemRateStr(rate.getSystemRate().toString() + '%');
                }
                if (null != planMap) {
                    BigDecimal c = planMap.getOrDefault(key, BigDecimal.ZERO);
                    rate.setPlanRate(c.multiply(BigDecimal.valueOf(100)).setScale(2, BigDecimal.ROUND_HALF_UP));
                    rate.setPlanRateStr(rate.getPlanRate().toString() + '%');
                }
            }
            this.discountRateRepository.updateBatchByIdAndTenantCode(rateList, TenantUtils.getTenantCode());
        }
    }
}
