package com.biz.crm.tpm.business.budget.forecast.feign.service.internal;

import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.budget.forecast.sdk.constants.SubComBudgetForecastConstant;
import com.biz.crm.tpm.business.budget.forecast.sdk.service.SubComBudgetForecastLockService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author huojia
 * @date 2022年11月01日 14:51
 */
@Slf4j
@Service
public class SubComBudgetForecastLockServiceImpl implements SubComBudgetForecastLockService {

    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 根据预算预测编码加锁
     *
     * @param budgetForecastCode
     * @param timeUnit
     * @param time
     * @return boolean
     * @author huojia
     * @date 2022/11/1 21:03
     **/
    @Override
    public boolean lock(String budgetForecastCode, TimeUnit timeUnit, int time) {
        if (StringUtils.isEmpty(budgetForecastCode)) {
            throw new RuntimeException("预算预测加锁失败，预算预测编码不能为空");
        }
        if (ObjectUtils.isEmpty(timeUnit)) {
            timeUnit = TimeUnit.MILLISECONDS;
        }
        if (time <= 0) {
            time = SubComBudgetForecastConstant.DEFAULT_LOCK_TIME;
        }
        return this.redisLockService.tryLock(SubComBudgetForecastConstant.SUB_COM_BUDGET_FORECAST_LOCK + budgetForecastCode, timeUnit, time);
    }

    /**
     * 根据预算预测编码批量加锁
     *
     * @param budgetForecastCodeList
     * @param timeUnit
     * @param time
     * @return boolean
     * @author huojia
     * @date 2022/11/1 21:04
     **/
    @Override
    public boolean lock(List<String> budgetForecastCodeList, TimeUnit timeUnit, int time) {
        return this.lock(budgetForecastCodeList, timeUnit, time, SubComBudgetForecastConstant.DEFAULT_WAITE_TIME_FIVE);
    }


    /**
     * 根据预算预测编码批量加锁
     *
     * @param budgetForecastCodeList
     * @param timeUnit
     * @param lockTime
     * @param waiteTime
     * @return boolean
     * @author huojia
     * @date 2022/11/1 21:04
     **/
    @Override
    public boolean lock(List<String> budgetForecastCodeList, TimeUnit timeUnit, int lockTime, int waiteTime) {
        if (CollectionUtils.isEmpty(budgetForecastCodeList)) {
            throw new RuntimeException("预算预测加锁失败，预算预测编码不能为空");
        }
        if (ObjectUtils.isEmpty(timeUnit)) {
            timeUnit = TimeUnit.MILLISECONDS;
        }
        if (lockTime <= 0) {
            lockTime = SubComBudgetForecastConstant.DEFAULT_LOCK_TIME;
        }
        if (waiteTime <= 0) {
            waiteTime = SubComBudgetForecastConstant.DEFAULT_WAITE_TIME_FIVE;
        }

        boolean isLock = true;
        List<String> successKeys = new ArrayList<>();
        try {
            // 循环加锁，并记录成功的key
            for (String budgetForecastCode : budgetForecastCodeList) {
                isLock = this.redisLockService.tryLock(SubComBudgetForecastConstant.SUB_COM_BUDGET_FORECAST_LOCK + budgetForecastCode, timeUnit, lockTime, waiteTime);
                if (!isLock) {
                    return false;
                }
                successKeys.add(budgetForecastCode);
            }
        } finally {
            // 存在加锁失败的情况，则先将成功的解锁
            if (!isLock && !CollectionUtils.isEmpty(successKeys)) {
                successKeys.forEach(key -> {
                    redisLockService.unlock(SubComBudgetForecastConstant.SUB_COM_BUDGET_FORECAST_LOCK + key);
                });
            }
        }
        return true;
    }

    /**
     * 根据预算预测编码解锁
     *
     * @param budgetForecastCode
     * @author huojia
     * @date 2022/11/1 21:04
     **/
    @Override
    public void unLock(String budgetForecastCode) {
        if (StringUtils.isEmpty(budgetForecastCode)) {
            throw new RuntimeException("预算预测解锁失败，预算预测编码不能为空");
        }
        redisLockService.unlock(SubComBudgetForecastConstant.SUB_COM_BUDGET_FORECAST_LOCK + budgetForecastCode);
    }

    /**
     * 根据预算预测编码批量解锁
     *
     * @param budgetForecastCodeList
     * @author huojia
     * @date 2022/11/1 21:04
     **/
    @Override
    public void unLock(List<String> budgetForecastCodeList) {
        if (CollectionUtils.isEmpty(budgetForecastCodeList)) {
            throw new RuntimeException("预算预测解锁失败，预算预测编码不能为空");
        }
        budgetForecastCodeList.forEach(budgetForecastCode -> {
            redisLockService.unlock(SubComBudgetForecastConstant.SUB_COM_BUDGET_FORECAST_LOCK + budgetForecastCode);
        });
    }

}
