package com.biz.crm.tpm.business.budget.item.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.dto.TreeDto;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.vo.LazyTreeVo;
import com.biz.crm.mn.common.base.util.DataTooLongGenerateRespUtil;
import com.biz.crm.tpm.business.budget.item.local.entity.BudgetItemEntity;
import com.biz.crm.tpm.business.budget.item.sdk.dto.BudgetItemDto;
import com.biz.crm.tpm.business.budget.item.sdk.service.BudgetItemMainItemService;
import com.biz.crm.tpm.business.budget.item.sdk.service.BudgetItemService;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemDataTypeVo;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemMainItemVo;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * @author huojia
 * @date 2022年10月24日 15:33
 */
@RestController
@RequestMapping("/v1/budget/item")
@Slf4j
@Api(tags = "预算项目")
public class BudgetItemController {

    @Autowired(required = false)
    private BudgetItemService budgetItemService;

    @Autowired(required = false)
    private BudgetItemMainItemService budgetItemMainItemService;

    /**
     * 由于创建或修改导致的重复提交的问题。在创建和修改前，需要使用该方法获得预授权</br>
     * 预授权成功后，才能通过预授权信息进行添加，
     *
     * @return
     */
    @ApiOperation(value = "由于创建授权标签来避免重复提交的问题。所以在创建前，需要使用该方法获得预授权")
    @PostMapping(value = "/preSave")
    public Result<?> preSave(Result<Object> ok) {
        try {
            Object prefix = this.budgetItemService.preSave();
            return Result.ok(prefix);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 分页查询所有数据
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findByConditions")
    public Result<Page<BudgetItemVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                       @ApiParam(name = "budgetItemDto", value = "预算项目信息") BudgetItemDto dto) {
        try {
            Page<BudgetItemVo> page = this.budgetItemService.findByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据主体预算项目，查询关联关系
     *
     * @param mainBudgetItemCodeList 主体预算项目编码
     * @return 所有数据
     */
    @ApiOperation(value = "根据主体预算项目，查询关联关系")
    @PostMapping("listByMainBudgetItemCodeList")
    public Result<List<BudgetItemMainItemVo>> listByMainBudgetItemCodeList(@RequestBody List<String> mainBudgetItemCodeList) {
        try {
            List<BudgetItemMainItemVo> result = this.budgetItemMainItemService.listByMainBudgetItemCodeList(mainBudgetItemCodeList);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 分页查询上级
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询上级数据")
    @GetMapping("listParent")
    public Result<Page<BudgetItemVo>> listParent(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                 @ApiParam(name = "budgetItemDto", value = "预算项目信息") BudgetItemDto dto) {
        try {
            Page<BudgetItemVo> page = this.budgetItemService.listParent(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 查询上级对应数据类型
     *
     * @param dto 查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "查询上级对应数据类型")
    @PostMapping("listParentDataType")
    public Result<List<BudgetItemDataTypeVo>> listParentDataType(@RequestBody BudgetItemDto dto) {
        try {
            List<BudgetItemDataTypeVo> dataTypeVos = this.budgetItemService.listParentDataType(dto);
            return Result.ok(dataTypeVos);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 分页查询主体预算项目
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询主体预算项目")
    @GetMapping("listMainItem")
    public Result<Page<BudgetItemVo>> listMainItem(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                   @ApiParam(name = "budgetItemDto", value = "预算项目信息") BudgetItemDto dto) {
        try {
            Page<BudgetItemVo> page = this.budgetItemService.listMainItem(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 重置预算项目树降维编码
     *
     * @return com.biz.crm.business.common.sdk.model.Result<?>
     * @author huojia
     * @date 2022/10/25 10:03
     **/
    @ApiOperation("重置预算项目树降维编码")
    @PostMapping("/resetRuleCode")
    public Result<?> resetRuleCode() {
        try {
            budgetItemService.updateRuleCode();
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 预算项目树
     *
     * @param treeDto
     * @return
     */
    @ApiOperation(value = "预算科目树", notes = "什么都不传返回第一层，传parentCode返回这个编码的下一级，传name模糊查询并且查询出所有上级再构建成树形，优先级parentCode > name")
    @GetMapping("/budgetItemLazyTree")
    public Result<List<LazyTreeVo>> budgetItemLazyTree(@ApiParam(name = "treeDto", value = "查询Dto") TreeDto treeDto) {
        return Result.ok(budgetItemService.budgetSubjectsLazyTree(treeDto));
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @ApiOperation(value = "通过主键查询单条数据")
    @GetMapping("detail")
    public Result<BudgetItemVo> detail(@RequestParam("id") @ApiParam(name = "id", value = "主键id") String id) {
        try {
            BudgetItemVo budgetItemVo = this.budgetItemService.findById(id);
            return Result.ok(budgetItemVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过预算项目编码查询单条数据
     *
     * @param code 编号
     * @return 单条数据
     */
    @ApiOperation(value = "通过预算项目编码查询单条数据")
    @GetMapping("findByCode")
    public Result<BudgetItemVo> findByCode(@RequestParam("code") @ApiParam(name = "code", value = "编号") String code) {
        try {
            BudgetItemVo budgetItemVo = this.budgetItemService.findByCode(code, null);
            return Result.ok(budgetItemVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 新增数据
     *
     * @param dto 实体对象
     * @return 新增结果
     */
    @ApiOperation(value = "新增数据")
    @PostMapping
    public Result<BudgetItemVo> create(@ApiParam(name = "budgetItemDto", value = "TPM-预算科目") @RequestBody BudgetItemDto dto) {
        try {
            BudgetItemVo result = this.budgetItemService.create(dto);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 新增数据
     *
     * @param dto 实体对象
     * @return 新增结果
     */
    @ApiOperation(value = "新增数据-能力中心")
    @PostMapping("create")
    public Result<BudgetItemVo> feignCreate(@RequestBody BudgetItemDto dto) {
        try {
            BudgetItemVo result = this.budgetItemService.create(dto);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(DataTooLongGenerateRespUtil.generateRespStr(e, BudgetItemEntity.class));
        }
    }

    /**
     * 修改数据
     *
     * @param dto 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "修改数据")
    @PatchMapping
    public Result<BudgetItemVo> update(@ApiParam(name = "budgetItemDto", value = "TPM-预算科目") @RequestBody BudgetItemDto dto) {
        try {
            BudgetItemVo result = this.budgetItemService.update(dto);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 删除数据
     *
     * @param ids 主键结合
     * @return 删除结果
     */
    @DeleteMapping
    @ApiOperation(value = "删除数据")
    public Result<?> delete(@RequestParam("ids") List<String> ids) {
        try {
            this.budgetItemService.delete(ids);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 批量根据id启用
     */
    @ApiOperation(value = "批量根据id启用")
    @PatchMapping(value = "enable")
    public Result<?> enable(@RequestBody List<String> ids) {
        try {
            this.budgetItemService.enable(ids);
            return Result.ok("启用成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 批量根据id禁用
     */
    @ApiOperation(value = "批量根据id禁用")
    @PatchMapping(value = "disable")
    public Result<?> disable(@RequestBody List<String> ids) {
        try {
            this.budgetItemService.disable(ids);
            return Result.ok("禁用成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 批量根据id禁用
     */
    @ApiOperation(value = "批量根据id禁用")
    @PostMapping(value = "disablePost")
    public Result<?> disablePost(@RequestBody List<String> ids) {
        try {
            this.budgetItemService.disable(ids);
            return Result.ok("禁用成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 分页查询分子公司预算项目
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询分子公司预算项目")
    @GetMapping("listSubsidiaryCompaniesBudgetItem")
    public Result<Page<BudgetItemVo>> listSubsidiaryCompaniesBudgetItem(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                   @ApiParam(name = "budgetItemDto", value = "预算项目信息") BudgetItemDto dto) {
        try {
            Page<BudgetItemVo> page = this.budgetItemService.listSubsidiaryCompaniesBudgetItem(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

}