package com.biz.crm.tpm.business.budget.item.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.local.entity.TenantFlagOpEntity;
import com.biz.crm.business.common.local.entity.UuidEntity;
import com.biz.crm.business.common.local.entity.UuidFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.tpm.business.budget.item.local.entity.BudgetItemEntity;
import com.biz.crm.tpm.business.budget.item.local.mapper.BudgetItemMapper;
import com.biz.crm.tpm.business.budget.item.sdk.dto.BudgetItemDto;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Optional;

/**
 * @author huojia
 * @date 2022年10月24日 16:35
 */
@Component
public class BudgetItemRepository extends ServiceImpl<BudgetItemMapper, BudgetItemEntity> {


    @Autowired(required = false)
    private BudgetItemMapper budgetItemMapper;

    /**
     * 根据id、删除标记查询预算项目
     *
     * @param id
     * @return com.biz.crm.tpm.business.budget.item.sdk.vo.resp.BudgetItemRespVo
     * @author huojia
     * @date 2022/10/24 16:40
     **/
    public BudgetItemEntity getById(String id, String delFlag) {
        if (StringUtils.isEmpty(id)) {
            return null;
        }
        return this.lambdaQuery().eq(BudgetItemEntity::getId, id)
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(StringUtils.isNotEmpty(delFlag), UuidFlagOpEntity::getDelFlag, delFlag)
                .one();
    }

    /**
     * 根据编码预算项目编码查询
     *
     * @param code
     * @return com.biz.crm.tpm.business.budget.item.sdk.vo.resp.BudgetItemRespVo
     * @author huojia
     * @date 2022/10/24 16:40
     **/
    public BudgetItemEntity getByBudgetItemCode(String code, String enableStatus) {
        if (StringUtils.isEmpty(code)) {
            return null;
        }
        return this.lambdaQuery().eq(BudgetItemEntity::getBudgetItemCode, code)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(StringUtils.isNotEmpty(enableStatus), UuidFlagOpEntity::getEnableStatus, enableStatus)
                .one();
    }

    /**
     * 根据编码预算项目编码查询
     *
     * @param codes
     * @return com.biz.crm.tpm.business.budget.item.sdk.vo.resp.BudgetItemRespVo
     * @author huojia
     * @date 2022/10/24 16:40
     **/
    public List<BudgetItemEntity> getByBudgetItemCodes(List<String> codes, String enableStatus) {
        if (CollectionUtils.isEmpty(codes)) {
            return null;
        }
        return this.lambdaQuery().in(BudgetItemEntity::getBudgetItemCode, codes)
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(StringUtils.isNotEmpty(enableStatus), UuidFlagOpEntity::getEnableStatus, enableStatus)
                .list();
    }

    /**
     * 查询下级预算项目
     *
     * @param parentCode
     * @return java.util.List<com.biz.crm.tpm.business.budget.item.local.entity.BudgetItemEntity>
     * @author huojia
     * @date 2022/10/25 10:14
     **/
    public List<BudgetItemEntity> findChildrenListByParentCode(String parentCode) {
        parentCode = Optional.ofNullable(parentCode).orElse(StringUtils.EMPTY);
        return this.lambdaQuery()
                .eq(StringUtils.isNotEmpty(parentCode), BudgetItemEntity::getParentBudgetItemCode, parentCode)
                .and(StringUtils.isEmpty(parentCode), qw->qw.isNull(BudgetItemEntity::getParentBudgetItemCode).or().eq(BudgetItemEntity::getParentBudgetItemCode, StringUtils.EMPTY))
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 查询所有无上级预算项目
     *
     * @return java.util.List<com.biz.crm.tpm.business.budget.item.local.entity.BudgetItemEntity>
     * @author huojia
     * @date 2022/10/25 10:28
     **/
    public List<BudgetItemEntity> findListWithoutParentCode() {
        return this.lambdaQuery()
                .and(qw->qw.isNull(BudgetItemEntity::getParentBudgetItemCode).or().eq(BudgetItemEntity::getParentBudgetItemCode, StringUtils.EMPTY))
//                .eq(BudgetItemEntity::getParentBudgetItemCode, StringUtils.EMPTY)
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 模糊查询
     *
     * @param enableStatus
     * @param name
     * @return java.util.List<com.biz.crm.tpm.business.budget.item.local.entity.BudgetItemEntity>
     * @author huojia
     * @date 2022/10/25 11:14
     **/
    public List<BudgetItemEntity> findItemCodeAndRuleCodeByEnableStatusOptAndItemNameLike(String enableStatus, String name) {
        List<BudgetItemEntity> likeList =
                this.lambdaQuery()
                        .eq(StringUtils.isNotEmpty(enableStatus), BudgetItemEntity::getEnableStatus, enableStatus)
                        .like(BudgetItemEntity::getBudgetItemName, name)
                        .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                        .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                        .select(BudgetItemEntity::getBudgetItemCode, BudgetItemEntity::getRuleCode)
                        .list();
        return likeList;
    }

    /**
     * 根据ids、删除标记查询预算项目
     *
     * @param ids
     * @param delFlag
     * @return java.util.List<com.biz.crm.tpm.business.budget.item.local.entity.BudgetItemEntity>
     * @author huojia
     * @date 2022/10/27 11:42
     **/
    public List<BudgetItemEntity> listByIds(List<String> ids, String delFlag) {
        if (CollectionUtils.isEmpty(ids)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .in(UuidEntity::getId, ids)
                .eq(StringUtils.isNotEmpty(delFlag), UuidFlagOpEntity::getDelFlag, delFlag)
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 根据预算项目编码批量查询
     *
     * @param mainBudgetItemCodeList
     * @param enableStatus
     * @return java.util.List<com.biz.crm.tpm.business.budget.item.local.entity.BudgetItemEntity>
     * @author huojia
     * @date 2022/11/2 17:01
     **/
    public List<BudgetItemEntity> listByBudgetItemCodeList(List<String> mainBudgetItemCodeList, String enableStatus) {
        if (CollectionUtils.isEmpty(mainBudgetItemCodeList)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .in(BudgetItemEntity::getBudgetItemCode, mainBudgetItemCodeList)
                .eq(StringUtils.isNotEmpty(enableStatus), UuidFlagOpEntity::getEnableStatus, enableStatus)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 根据父级批量查询
     *
     * @param parentCodeList
     * @return java.util.List<com.biz.crm.tpm.business.budget.item.local.entity.BudgetItemEntity>
     * @author huojia
     * @date 2022/12/3 16:54
     **/
    public List<BudgetItemEntity> listByParent(List<String> parentCodeList) {
        if (CollectionUtils.isEmpty(parentCodeList)) {
            return null;
        }
        return this.lambdaQuery()
                .in(BudgetItemEntity::getParentBudgetItemCode, parentCodeList)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 通过主体预算项目找到分子公司预算项目
     * @param dto
     * @return {@link List}<{@link BudgetItemVo}>
     */
    public List<BudgetItemVo> findByMainBudgetItem(BudgetItemDto dto) {
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        dto.setTenantCode(TenantUtils.getTenantCode());
        return this.baseMapper.findByMainBudgetItem(dto);
    }

    /**
     * 分页查询分子公司预算项目
     *
     * @param page
     * @param dto
     * @return
     */
    public Page<BudgetItemVo> listSubsidiaryCompaniesBudgetItem(Page<BudgetItemVo> page, BudgetItemDto dto) {
        return this.budgetItemMapper.listSubsidiaryCompaniesBudgetItem(page,dto);
    }

    public List<BudgetItemEntity> listByParentCodes(List<String> parentCodeList) {
        if (CollectionUtils.isEmpty(parentCodeList)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .eq(BudgetItemEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(BudgetItemEntity::getTenantCode, TenantUtils.getTenantCode())
                .in(BudgetItemEntity::getParentBudgetItemCode, parentCodeList)
                .list();
    }
}
