package com.biz.crm.tpm.business.budget.item.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.dto.TreeDto;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.TreeRuleCodeStrategy;
import com.biz.crm.business.common.sdk.service.TreeRuleCodeStrategyHolder;
import com.biz.crm.business.common.sdk.utils.TreeUtil;
import com.biz.crm.business.common.sdk.vo.LazyTreeVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.budget.item.local.entity.BudgetItemControlConditionEntity;
import com.biz.crm.tpm.business.budget.item.local.entity.BudgetItemEntity;
import com.biz.crm.tpm.business.budget.item.local.entity.BudgetItemMainItemEntity;
import com.biz.crm.tpm.business.budget.item.local.mapper.BudgetItemMapper;
import com.biz.crm.tpm.business.budget.item.local.repository.BudgetItemControlConditionRepository;
import com.biz.crm.tpm.business.budget.item.local.repository.BudgetItemMainItemRepository;
import com.biz.crm.tpm.business.budget.item.local.repository.BudgetItemRepository;
import com.biz.crm.tpm.business.budget.item.sdk.constant.DictTypeCodeConstant;
import com.biz.crm.tpm.business.budget.item.sdk.constant.RedisKeysConstant;
import com.biz.crm.tpm.business.budget.item.sdk.dto.BudgetItemControlConditionDto;
import com.biz.crm.tpm.business.budget.item.sdk.dto.BudgetItemDto;
import com.biz.crm.tpm.business.budget.item.sdk.dto.BudgetItemLogEventDto;
import com.biz.crm.tpm.business.budget.item.sdk.event.BudgetItemEventListener;
import com.biz.crm.tpm.business.budget.item.sdk.event.BudgetItemLogEventListener;
import com.biz.crm.tpm.business.budget.item.sdk.service.BudgetItemService;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemControlConditionVo;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemDataTypeVo;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemMainItemVo;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetService;
import com.biz.crm.tpm.business.month.budget.sdk.service.SubComMonthBudgetService;
import com.biz.crm.tpm.business.month.budget.sdk.vo.SubComMonthBudgetVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author huojia
 * @date 2022年10月24日 15:43
 */
@Slf4j
@Service
public class BudgetItemServiceImpl implements BudgetItemService {

    @Autowired(required = false)
    private BudgetItemMapper budgetItemMapper;

    @Autowired(required = false)
    private RedisLockService redisLockService;
    @Autowired(required = false)
    private RedisMutexService redisMutexService;

    @Autowired(required = false)
    private MonthBudgetService monthBudgetService;

    @Resource
    private DictToolkitService dictToolkitService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private BudgetItemRepository budgetItemRepository;

    @Autowired(required = false)
    private SubComMonthBudgetService subComMonthBudgetService;

    @Autowired(required = false)
    private BudgetItemMainItemRepository budgetItemMainItemRepository;

    @Autowired(required = false)
    private TreeRuleCodeStrategyHolder treeRuleCodeStrategyHolder;

    @Autowired(required = false)
    private List<BudgetItemEventListener> budgetItemEventListenerList;

    @Autowired(required = false)
    private BudgetItemControlConditionRepository controlConditionRepository;

    @Override
    public String preSave() {
        String prefix = UUID.randomUUID().toString();
        // 1天后过期
        this.redisMutexService.setMCode(RedisKeysConstant.GENERATE_PREFIX, prefix, prefix, TimeUnit.MILLISECONDS.convert(1, TimeUnit.DAYS));
        return prefix;
    }

    /**
     * 分页查询
     *
     * @param pageable
     * @param dto
     * @return com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo>
     * @author huojia
     * @date 2022/11/1 20:24
     **/
    @Override
    public Page<BudgetItemVo> findByConditions(Pageable pageable, BudgetItemDto dto) {
        ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(dto)) {
            dto = new BudgetItemDto();
        }
        List<String> selectedCodeList =
                Optional.ofNullable(dto.getSelectedCodeList()).orElse(new ArrayList<>());
        if (StringUtils.isNotEmpty(dto.getSelectedCode())) {
            selectedCodeList.add(dto.getSelectedCode());
        }
        if (!CollectionUtils.isEmpty(selectedCodeList)) {
            dto.setSelectedCodeList(selectedCodeList);
        }
        Page<BudgetItemVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.budgetItemMapper.findByConditions(page, dto);
    }

    /**
     * 查询上级
     *
     * @param pageable
     * @param dto
     * @return com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo>
     * @author huojia
     * @date 2022/11/1 20:24
     **/
    @Override
    public Page<BudgetItemVo> listParent(Pageable pageable, BudgetItemDto dto) {
        ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 25));
        if (Objects.isNull(dto)) {
            dto = new BudgetItemDto();
        }
        if (StringUtils.isEmpty(dto.getBudgetItemLevelCode())) {
            throw new RuntimeException("查询上级预算失败，当前预算层级不能为空！");
        }
        int itemLevel = Integer.parseInt(dto.getBudgetItemLevelCode());
        if (itemLevel <= 1) {
            return new Page<BudgetItemVo>();
        }
        dto.setBudgetItemLevelCode(String.valueOf(Integer.parseInt(dto.getBudgetItemLevelCode()) - 1));
        Page<BudgetItemVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        Page<BudgetItemVo> pageResult = this.budgetItemMapper.listParent(page, dto);
        List<BudgetItemVo> records = pageResult.getRecords();
        if (!CollectionUtils.isEmpty(records)) {
            Map<String, String> dataTypeMap = dictToolkitService.findMapByDictTypeCode(DictTypeCodeConstant.TPM_BUDGET_DATA_TYPE);
            records.forEach(record -> {
                if (StringUtils.isNotEmpty(record.getDataTypeCode())) {
                    List<String> dataTypeCodeList = Arrays.asList(record.getDataTypeCode().split(","));
                    List<BudgetItemDataTypeVo> dataTypeList = new ArrayList<>();
                    dataTypeCodeList.forEach(dataTypeCode -> {
                        BudgetItemDataTypeVo budgetItemDataTypeVo = new BudgetItemDataTypeVo();
                        budgetItemDataTypeVo.setDataTypeCode(dataTypeCode);
                        budgetItemDataTypeVo.setDataTypeName(dataTypeMap.get(dataTypeCode));
                        dataTypeList.add(budgetItemDataTypeVo);
                    });
                    record.setDataTypeList(dataTypeList);
                }
            });
            pageResult.setRecords(records);
        }
        return pageResult;
    }

    /**
     * 查询上级对应数据类型
     *
     * @param dto
     * @return java.util.List<com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemDataTypeVo>
     * @author huojia
     * @date 2022/12/6 17:40
     **/
    @Override
    public List<BudgetItemDataTypeVo> listParentDataType(BudgetItemDto dto) {
        Validate.notNull(dto, "请求参数不能为空！");
        List<BudgetItemDataTypeVo> result = new ArrayList<>();

        List<DictDataVo> dictDataVos = dictDataVoService.findByDictTypeCode(DictTypeCodeConstant.TPM_BUDGET_DATA_TYPE);
        if (!CollectionUtils.isEmpty(dictDataVos)) {
            dictDataVos.forEach(dictDataVo -> {
                BudgetItemDataTypeVo budgetItemDataTypeVo = new BudgetItemDataTypeVo();
                budgetItemDataTypeVo.setDataTypeCode(dictDataVo.getDictCode());
                budgetItemDataTypeVo.setDataTypeName(dictDataVo.getDictValue());
                result.add(budgetItemDataTypeVo);
            });
        }

        /*if (StringUtils.isEmpty(dto.getParentBudgetItemCode())) {
            List<DictDataVo> dictDataVos = dictDataVoService.findByDictTypeCode(DictTypeCodeConstant.TPM_BUDGET_DATA_TYPE);
            if (!CollectionUtils.isEmpty(dictDataVos)) {
                dictDataVos.forEach(dictDataVo -> {
                    BudgetItemDataTypeVo budgetItemDataTypeVo = new BudgetItemDataTypeVo();
                    budgetItemDataTypeVo.setDataTypeCode(dictDataVo.getDictCode());
                    budgetItemDataTypeVo.setDataTypeName(dictDataVo.getDictValue());
                    result.add(budgetItemDataTypeVo);
                });
            }
        } else {
            BudgetItemEntity budgetItemEntity = this.budgetItemRepository.getByBudgetItemCode(dto.getParentBudgetItemCode(), EnableStatusEnum.ENABLE.getCode());
            if (ObjectUtils.isEmpty(budgetItemEntity)) {
                return result;
            }
            Map<String, String> dataTypeMap = dictToolkitService.findMapByDictTypeCode(DictTypeCodeConstant.TPM_BUDGET_DATA_TYPE);
            if (StringUtils.isNotEmpty(budgetItemEntity.getDataTypeCode())) {
                List<String> dataTypeCodeList = Arrays.asList(budgetItemEntity.getDataTypeCode().split(","));
                dataTypeCodeList.forEach(dataTypeCode -> {
                    BudgetItemDataTypeVo budgetItemDataTypeVo = new BudgetItemDataTypeVo();
                    budgetItemDataTypeVo.setDataTypeCode(dataTypeCode);
                    budgetItemDataTypeVo.setDataTypeName(dataTypeMap.get(dataTypeCode));
                    result.add(budgetItemDataTypeVo);
                });
            }
        }*/
        return result;
    }

    /**
     * 查询预算加载树
     *
     * @param dto
     * @return java.util.List<com.biz.crm.business.common.sdk.vo.LazyTreeVo>
     * @author huojia
     * @date 2022/11/1 20:25
     **/
    @Override
    public List<LazyTreeVo> budgetSubjectsLazyTree(TreeDto dto) {
        List<LazyTreeVo> list = new ArrayList<>();
        String excludeRuleCode = "";
        if (StringUtils.isNotEmpty(dto.getExcludeCodeAndChildren())) {
            // 根据 budgetItemCode 查询 ruleCode
            BudgetItemEntity one = this.budgetItemRepository.getByBudgetItemCode(dto.getExcludeCodeAndChildren(), null);
            if (one != null) {
                excludeRuleCode = one.getRuleCode();
            }
        }
        if (!StringUtils.isEmpty(dto.getParentCode())) {
            // 查询parentCode下一级, 启用状态, 只查询该编码下一级, 排除这个降维编码的下级
            List<LazyTreeVo> lazyTreeVos = this.budgetItemMapper.findBudgetItemLazyTreeList(dto.getEnableStatus(), null, dto.getParentCode(), null, null, excludeRuleCode);
            list.addAll(lazyTreeVos);
        } else if (!StringUtils.isEmpty(dto.getName())) {
            // 模糊查询并构建成树
            List<BudgetItemEntity> likeList = this.budgetItemRepository.findItemCodeAndRuleCodeByEnableStatusOptAndItemNameLike(dto.getEnableStatus(), dto.getName());
            if (!CollectionUtils.isEmpty(likeList)) {
                TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
                Set<String> parentRuleCodes = treeRuleCodeStrategy.findParentRuleCodeByRuleCodes(3, likeList.stream().map(BudgetItemEntity::getRuleCode).collect(Collectors.toList()));
                // 启用状态, 只查询这些降维编码, 排除这个降维编码的下级
                list.addAll(this.budgetItemMapper.findBudgetItemLazyTreeList(dto.getEnableStatus(), null, null, null, new ArrayList<>(parentRuleCodes), excludeRuleCode));
            }
        } else {
            // 查询第一层  启用状态，传true只查第一层， 排除这个降维编码的下级
            list.addAll(this.budgetItemMapper.findBudgetItemLazyTreeList(dto.getEnableStatus(), true, null, null, null, excludeRuleCode));
        }
        if (!CollectionUtils.isEmpty(list)) {
            list.forEach(item -> item.setHasChild(BooleanEnum.TRUE.getNumStr().equalsIgnoreCase(String.valueOf(item.getHasChildFlag()))));
            return TreeUtil.generateLazyTreeByParentCode(list);
        }
        return new ArrayList<>();
    }

    /**
     * 重置降维编码
     *
     * @author huojia
     * @date 2022/11/1 20:26
     **/
    @Override
    public void updateRuleCode() {
        // 查找parentCode不为空但找不到对应上级的数据，将其parentCode设为空
        this.budgetItemMapper.updateOrphanParentCodeNull();
        // 查找所有parentCode为空的数据（相当于第一层数据）
        List<BudgetItemEntity> topList = this.budgetItemRepository.findListWithoutParentCode();
        // 递归设置其ruleCode和其所有子级的ruleCode
        TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
        log.info("updateRuleCode topList={}", topList.size());
        for (int i = 0; i < topList.size(); i++) {
            log.info("updateRuleCode top={}", topList.get(i).getBudgetItemCode());
            // 递归调用
            this.updateCurAndChildrenRuleCode(
                    topList.get(i).getBudgetItemCode(),
                    treeRuleCodeStrategy.generateByNum(3, i + 1),
                    1);
        }
    }

    /**
     * 查询业务单元为主体的预算项目
     *
     * @param pageable
     * @param dto
     * @return com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo>
     * @author huojia
     * @date 2022/11/1 20:26
     **/
    @Override
    public Page<BudgetItemVo> listMainItem(Pageable pageable, BudgetItemDto dto) {
        ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 25));
        if (Objects.isNull(dto)) {
            dto = new BudgetItemDto();
        }
        // 查询业务单元为主体的预算项目
        List<BudgetItemControlConditionEntity> controlConditionEntityList = this.controlConditionRepository.listByBusinessUnitCode(BusinessUnitEnum.HEADQUARTERS.getCode());
        if (CollectionUtils.isEmpty(controlConditionEntityList)) {
            return new Page<>();
        }
        Set<String> budgetItemCodeSet = controlConditionEntityList.stream().map(BudgetItemControlConditionEntity::getBudgetItemCode).collect(Collectors.toSet());
        if (CollectionUtils.isEmpty(budgetItemCodeSet)) {
            return new Page<>();
        }
        List<String> selectCodeList = new ArrayList<>(budgetItemCodeSet);
        // 排除预算项目本身
        selectCodeList.remove(dto.getExcludeCode());
        if (CollectionUtils.isEmpty(selectCodeList)) {
            return new Page<>();
        }
        dto.setSelectedCodeList(selectCodeList);
        Page<BudgetItemVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        Page<BudgetItemVo> result = this.budgetItemMapper.listByBudgetItemCodeList(page, dto);
        return result;
    }

    /**
     * 批量导入新增
     *
     * @param importList
     * @author huojia
     * @date 2022/11/1 20:27
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void importSave(List<BudgetItemDto> importList) {
        if (CollectionUtils.isEmpty(importList)) {
            return;
        }
        List<BudgetItemControlConditionDto> controlConditionDtoList = new ArrayList<>();
        importList.forEach(dto -> {
            dto.setTenantCode(TenantUtils.getTenantCode());
            dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            dto.setId(UUID.randomUUID().toString().replaceAll("-", ""));
            dto.getControlConditionDtoList().forEach(controlConditionDto -> {
                controlConditionDto.setBudgetItemCode(dto.getBudgetItemCode());
                controlConditionDtoList.add(controlConditionDto);
            });
        });

        // 保存预算项目
        List<BudgetItemEntity> budgetItemEntities = new ArrayList<>(this.nebulaToolkitService.copyCollectionByWhiteList(importList, BudgetItemDto.class, BudgetItemEntity.class, LinkedHashSet.class, ArrayList.class));
        this.budgetItemRepository.saveBatch(budgetItemEntities);
        if (!CollectionUtils.isEmpty(budgetItemEventListenerList)) {
            importList.forEach(dto -> {
                for (BudgetItemEventListener budgetItemEventListener : budgetItemEventListenerList) {
                    budgetItemEventListener.onCreated(dto);
                }
            });
        }
        // 保存管控条件
        Collection<BudgetItemControlConditionEntity> controlConditionEntities = this.nebulaToolkitService.copyCollectionByBlankList(controlConditionDtoList, BudgetItemControlConditionDto.class, BudgetItemControlConditionEntity.class, LinkedHashSet.class, ArrayList.class);
        controlConditionRepository.saveBatch(new ArrayList<>(controlConditionEntities));
        // 业务日志新增
        importList.forEach(dto -> {
            BudgetItemLogEventDto logEventDto = new BudgetItemLogEventDto();
            logEventDto.setOriginal(null);
            logEventDto.setNewest(dto);
            SerializableBiConsumer<BudgetItemLogEventListener, BudgetItemLogEventDto> onCreate =
                    BudgetItemLogEventListener::onCreate;
            this.nebulaNetEventClient.publish(logEventDto, BudgetItemLogEventListener.class, onCreate);
        });
        // 重置降维编码
        this.updateRuleCode();
    }

    /**
     * 根据预算项目编码批量查询预算项目
     *
     * @param budgetItemCodeList
     * @return java.util.List<com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo>
     * @author huojia
     * @date 2022/11/4 15:52
     **/
    @Override
    public List<BudgetItemVo> listByCodes(List<String> budgetItemCodeList) {
        if (CollectionUtils.isEmpty(budgetItemCodeList)) {
            return Lists.newArrayList();
        }
        List<BudgetItemEntity> budgetItemEntities = this.budgetItemRepository.listByBudgetItemCodeList(budgetItemCodeList, EnableStatusEnum.ENABLE.getCode());
        if (CollectionUtils.isEmpty(budgetItemEntities)) {
            return Lists.newArrayList();
        }
        List<BudgetItemVo> budgetItemVos = Lists.newArrayList(
                this.nebulaToolkitService.copyCollectionByWhiteList(budgetItemEntities, BudgetItemEntity.class, BudgetItemVo.class, LinkedHashSet.class, ArrayList.class)
        );
        List<BudgetItemControlConditionEntity> controlConditionList = this.controlConditionRepository.listByBudgetItemCodeList(budgetItemCodeList);
        if (!CollectionUtils.isEmpty(controlConditionList)) {
            List<BudgetItemControlConditionVo> budgetItemControlConditionVos = (List<BudgetItemControlConditionVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
                    controlConditionList, BudgetItemControlConditionEntity.class, BudgetItemControlConditionVo.class, LinkedHashSet.class, ArrayList.class
            );
            Map<String, List<BudgetItemControlConditionVo>> controlConditionMap = budgetItemControlConditionVos.stream()
                    .collect(Collectors.groupingBy(BudgetItemControlConditionVo::getBudgetItemCode));
            budgetItemVos.forEach(budgetItemVo -> {
                budgetItemVo.setControlConditionDtoList(controlConditionMap.get(budgetItemVo.getBudgetItemCode()));
            });
        }
        return budgetItemVos;
    }

    /**
     * 根据预算项目编码批量查询预算项目
     *
     * @param budgetItemCodeList
     * @return
     */
    @Override
    public List<BudgetItemVo> listByCodesSimple(List<String> budgetItemCodeList) {
        if (CollectionUtils.isEmpty(budgetItemCodeList)) {
            return Lists.newArrayList();
        }
        List<BudgetItemEntity> budgetItemEntities = this.budgetItemRepository.listByBudgetItemCodeList(budgetItemCodeList, EnableStatusEnum.ENABLE.getCode());
        if (CollectionUtils.isEmpty(budgetItemEntities)) {
            return Lists.newArrayList();
        }
        List<BudgetItemVo> budgetItemVos = Lists.newArrayList(
                this.nebulaToolkitService.copyCollectionByWhiteList(budgetItemEntities, BudgetItemEntity.class, BudgetItemVo.class, LinkedHashSet.class, ArrayList.class)
        );
        return budgetItemVos;
    }

    /**
     * 根据主体预算项目编码批量查询
     *
     * @param mainBudgetItemList
     * @return java.util.List<com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo>
     * @author huojia
     * @date 2022/11/22 15:11
     **/
    @Override
    public List<BudgetItemVo> listByMainBudgetItem(List<String> mainBudgetItemList) {
        if (CollectionUtils.isEmpty(mainBudgetItemList)) {
            return Lists.newArrayList();
        }
        List<BudgetItemMainItemEntity> budgetItemMainItemEntityList = budgetItemMainItemRepository.listByMainBudgetItemCodeList(mainBudgetItemList);
        if (CollectionUtils.isEmpty(budgetItemMainItemEntityList)) {
            return Lists.newArrayList();
        }
        Map<String, List<BudgetItemMainItemEntity>> mainBudgetItemMap = budgetItemMainItemEntityList.stream().collect(Collectors.groupingBy(BudgetItemMainItemEntity::getBudgetItemCode));
        List<String> budgetItemCodeList = budgetItemMainItemEntityList.stream().map(BudgetItemMainItemEntity::getBudgetItemCode).collect(Collectors.toList());
        List<BudgetItemEntity> budgetItemEntities = this.budgetItemRepository.listByBudgetItemCodeList(budgetItemCodeList, null);
        if (CollectionUtils.isEmpty(budgetItemEntities)) {
            return Lists.newArrayList();
        }
        List<BudgetItemVo> budgetItemVos = (List<BudgetItemVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
                budgetItemEntities, BudgetItemEntity.class, BudgetItemVo.class, LinkedHashSet.class, ArrayList.class
        );
        budgetItemVos.forEach(budgetItemVo -> {
            budgetItemVo.setBudgetItemMainItemList(
                    (List<BudgetItemMainItemVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
                            mainBudgetItemMap.get(budgetItemVo.getBudgetItemCode()), BudgetItemMainItemEntity.class, BudgetItemMainItemVo.class, LinkedHashSet.class, ArrayList.class
                    )
            );
        });
        return budgetItemVos;
    }

    /**
     * 获取所有对应子级
     *
     * @param parentCodeList
     * @return java.util.List<com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo>
     * @author huojia
     * @date 2022/12/3 16:50
     **/
    @Override
    public List<BudgetItemVo> listByParent(List<String> parentCodeList) {
        if (CollectionUtils.isEmpty(parentCodeList)) {
            return null;
        }
        List<BudgetItemEntity> budgetItemEntities = this.budgetItemRepository.listByParent(parentCodeList);
        if (CollectionUtils.isEmpty(budgetItemEntities)) {
            return null;
        }
        return (List<BudgetItemVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
                budgetItemEntities, BudgetItemEntity.class, BudgetItemVo.class, LinkedHashSet.class, ArrayList.class
        );
    }

    /**
     * 更新自己及子集的降维编码
     */
    private void updateCurAndChildrenRuleCode(String budgetItemCode, String ruleCode, int levelNum) {
        log.info("updateRuleCode budgetItemCode：{}，ruleCode：{}，levelNum：{}", budgetItemCode, ruleCode, levelNum);
        // 更新当前
        BudgetItemEntity budgetItemEntity = budgetItemRepository.getByBudgetItemCode(budgetItemCode, null);
        if (budgetItemEntity == null) {
            return;
        }
        budgetItemEntity.setRuleCode(ruleCode);
        budgetItemEntity.setLevelNum(levelNum);
        budgetItemRepository.updateById(budgetItemEntity);
        // 查询下一层
        List<BudgetItemEntity> childrenList =
                budgetItemRepository.findChildrenListByParentCode(budgetItemCode);
        if (org.apache.commons.collections.CollectionUtils.isEmpty(childrenList)) {
            return;
        }
        // 遍历下级
        TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
        for (int i = 0; i < childrenList.size(); i++) {
            // 递归调用
            this.updateCurAndChildrenRuleCode(
                    childrenList.get(i).getBudgetItemCode(),
                    ruleCode + treeRuleCodeStrategy.generateByNum(3, i + 1),
                    levelNum + 1);
        }
    }

    /**
     * 根据主键查询预算项目
     *
     * @return com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo
     * @author huojia
     * @date 2022/11/1 20:27
     **/
    @Override
    public BudgetItemVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        BudgetItemEntity budgetItemEntity = budgetItemRepository.getById(id, DelFlagStatusEnum.NORMAL.getCode());
        Validate.notNull(budgetItemEntity, "数据不存在，请刷新后重试！");
        return buildVo(budgetItemEntity);
    }

    /**
     * 根据编码查询预算项目
     *
     * @param code
     * @param enableStatus
     * @return com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo
     * @author huojia
     * @date 2022/11/1 20:27
     **/
    @Override
    public BudgetItemVo findByCode(String code, String enableStatus) {
        if (StringUtils.isBlank(code)) {
            return null;
        }
        BudgetItemEntity budgetItemEntity = budgetItemRepository.getByBudgetItemCode(code, enableStatus);
        Validate.notNull(budgetItemEntity, "数据不存在，请刷新后重试！");
        return buildVo(budgetItemEntity);
    }

    @Override
    public List<BudgetItemVo> findByCodes(List<String> codes) {
        if (CollectionUtils.isEmpty(codes)) {
            return Lists.newArrayList();
        }
        return findByCodes(codes, EnableStatusEnum.ENABLE.getCode());
    }

    @Override
    public List<BudgetItemVo> findByCodes(List<String> codes, String enableStatus) {
        if (CollectionUtils.isEmpty(codes)) {
            return Lists.newArrayList();
        }
        List<BudgetItemEntity> list = budgetItemRepository.getByBudgetItemCodes(codes, enableStatus);
        return this.buildVos(list);
    }

    /**
     * 实体转换为vo并设置管控条件
     *
     * @param budgetItemEntity
     * @return com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo
     * @author huojia
     * @date 2022/10/24 17:49
     **/
    private BudgetItemVo buildVo(BudgetItemEntity budgetItemEntity) {
        if (ObjectUtils.isEmpty(budgetItemEntity)) {
            return null;
        }
        BudgetItemVo vo = this.nebulaToolkitService.copyObjectByWhiteList(budgetItemEntity, BudgetItemVo.class, LinkedHashSet.class, ArrayList.class);
        List<BudgetItemMainItemEntity> mainItemEntityList = budgetItemMainItemRepository.findByBudgetItemCode(vo.getBudgetItemCode());
        if (!CollectionUtils.isEmpty(mainItemEntityList)) {
            Collection<BudgetItemMainItemVo> budgetItemMainItemVos = this.nebulaToolkitService.copyCollectionByWhiteList(
                    mainItemEntityList, BudgetItemMainItemEntity.class, BudgetItemMainItemVo.class, LinkedHashSet.class, ArrayList.class
            );
            vo.setBudgetItemMainItemList(new ArrayList<>(budgetItemMainItemVos));
            vo.setMainBudgetItemCode(budgetItemMainItemVos.stream().map(BudgetItemMainItemVo::getMainBudgetItemCode).collect(Collectors.joining(",")));
        }
        List<BudgetItemControlConditionEntity> controlConditionEntityList = controlConditionRepository.findByBudgetItemCode(vo.getBudgetItemCode());
        if (!CollectionUtils.isEmpty(controlConditionEntityList)) {
            Collection<BudgetItemControlConditionVo> controlConditionDtoList = this.nebulaToolkitService.copyCollectionByBlankList(
                    controlConditionEntityList, BudgetItemControlConditionEntity.class, BudgetItemControlConditionVo.class, LinkedHashSet.class, ArrayList.class
            );
            vo.setControlConditionDtoList(new ArrayList<>(controlConditionDtoList));
        }
        return vo;
    }

    /**
     * 实体转换为vo并设置管控条件
     *
     * @param list
     * @return com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo
     * @author huojia
     * @date 2022/10/24 17:49
     **/
    private List<BudgetItemVo> buildVos(List<BudgetItemEntity> list) {
        if (CollectionUtils.isEmpty(list)) {
            return Lists.newArrayList();
        }
        List<BudgetItemVo> budgetItemVos = (List<BudgetItemVo>) nebulaToolkitService.copyCollectionByWhiteList(list, BudgetItemEntity.class, BudgetItemVo.class, HashSet.class, ArrayList.class);

        List<String> budgetItemCodes = budgetItemVos.stream().map(BudgetItemVo::getBudgetItemCode).distinct().collect(Collectors.toList());

        List<BudgetItemMainItemEntity> mainItemEntityList = budgetItemMainItemRepository.findByBudgetItemCodes(budgetItemCodes);

        Map<String, List<BudgetItemMainItemVo>> budgetMainItemMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(mainItemEntityList)) {
            List<BudgetItemMainItemVo> budgetItemMainItemVos = (List<BudgetItemMainItemVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
                    mainItemEntityList, BudgetItemMainItemEntity.class, BudgetItemMainItemVo.class, LinkedHashSet.class, ArrayList.class
            );
            budgetMainItemMap = budgetItemMainItemVos.stream().collect(Collectors.groupingBy(BudgetItemMainItemVo::getBudgetItemCode));
        }

        Map<String, List<BudgetItemControlConditionVo>> controlMap = new HashMap<>();
        List<BudgetItemControlConditionEntity> controlConditionEntityList = controlConditionRepository.findByBudgetItemCodes(budgetItemCodes);
        if (!CollectionUtils.isEmpty(controlConditionEntityList)) {
            List<BudgetItemControlConditionVo> controlConditionDtoList = (List<BudgetItemControlConditionVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                    controlConditionEntityList, BudgetItemControlConditionEntity.class, BudgetItemControlConditionVo.class, LinkedHashSet.class, ArrayList.class
            );
            controlMap = controlConditionDtoList.stream().collect(Collectors.groupingBy(BudgetItemControlConditionVo::getBudgetItemCode));
        }

        for (BudgetItemVo vo : budgetItemVos) {
            if (budgetMainItemMap.containsKey(vo.getBudgetItemCode())) {
                vo.setBudgetItemMainItemList(budgetMainItemMap.get(vo.getBudgetItemCode()));
                vo.setMainBudgetItemCode(vo.getBudgetItemMainItemList().stream().map(BudgetItemMainItemVo::getMainBudgetItemCode).collect(Collectors.joining(",")));
            }
            if (controlMap.containsKey(vo.getBudgetItemCode())) {
                vo.setControlConditionDtoList(controlMap.get(vo.getBudgetItemCode()));
            }
        }
        return budgetItemVos;
    }

    /**
     * 新增
     *
     * @param dto
     * @return com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo
     * @author huojia
     * @date 2022/11/1 20:27
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public BudgetItemVo create(BudgetItemDto dto) {
        this.createValidate(dto);
        // 保存预算项目
        BudgetItemEntity budgetItemEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, BudgetItemEntity.class, LinkedHashSet.class, ArrayList.class);
        // 生成降维编码
        generateRuleCodeByParentCode(budgetItemEntity);
        String mainBudgetItemCode = budgetItemEntity.getMainBudgetItemCode();
        budgetItemEntity.setMainBudgetItemCode(null);
        budgetItemEntity.setMainBudgetItemName(null);
        budgetItemEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        budgetItemEntity.setTenantCode(TenantUtils.getTenantCode());
        this.budgetItemRepository.saveOrUpdate(budgetItemEntity);
        dto.setId(budgetItemEntity.getId());
        if (!CollectionUtils.isEmpty(budgetItemEventListenerList)) {
            for (BudgetItemEventListener budgetItemEventListener : budgetItemEventListenerList) {
                budgetItemEventListener.onCreated(dto);
            }
        }
        // 保存关联主体预算项目
        if (!StringUtils.isEmpty(mainBudgetItemCode)) {
            List<String> mainBudgetList = Arrays.asList(mainBudgetItemCode.split(","));
            List<BudgetItemMainItemEntity> budgetItemMainItemEntities = new ArrayList<>();
            List<BudgetItemVo> budgetItemVos = listByCodes(mainBudgetList);
            Map<String, BudgetItemVo> map = budgetItemVos.stream().collect(Collectors.toMap(BudgetItemVo::getBudgetItemCode, Function.identity()));
            mainBudgetList.forEach(mainBudgetCode -> {
                BudgetItemMainItemEntity budgetItemMainItemEntity = new BudgetItemMainItemEntity();
                budgetItemMainItemEntity.setBudgetItemCode(budgetItemEntity.getBudgetItemCode());
                budgetItemMainItemEntity.setMainBudgetItemCode(mainBudgetCode);
                if (!MapUtils.isEmpty(map)) {
                    budgetItemMainItemEntity.setMainBudgetItemName(map.get(mainBudgetCode).getBudgetItemName());
                }
                budgetItemMainItemEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                budgetItemMainItemEntity.setTenantCode(TenantUtils.getTenantCode());
                budgetItemMainItemEntities.add(budgetItemMainItemEntity);
            });
            budgetItemMainItemRepository.saveBatch(budgetItemMainItemEntities);
        }
        // 保存管控条件
        Collection<BudgetItemControlConditionEntity> controlConditionEntities = this.nebulaToolkitService.copyCollectionByBlankList(
                dto.getControlConditionDtoList(), BudgetItemControlConditionDto.class, BudgetItemControlConditionEntity.class, LinkedHashSet.class, ArrayList.class
        );
        controlConditionRepository.saveBatch(new ArrayList<>(controlConditionEntities));
        // 业务日志新增
        BudgetItemLogEventDto logEventDto = new BudgetItemLogEventDto();
        logEventDto.setOriginal(null);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<BudgetItemLogEventListener, BudgetItemLogEventDto> onCreate =
                BudgetItemLogEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, BudgetItemLogEventListener.class, onCreate);
        BudgetItemVo budgetItemVo = this.nebulaToolkitService.copyObjectByWhiteList(dto, BudgetItemVo.class, LinkedHashSet.class, ArrayList.class);
        List<BudgetItemControlConditionVo> conditionVo = new ArrayList<>(this.nebulaToolkitService.copyCollectionByWhiteList(controlConditionEntities, BudgetItemControlConditionEntity.class, BudgetItemControlConditionVo.class, LinkedHashSet.class, ArrayList.class));
        budgetItemVo.setControlConditionDtoList(conditionVo);
        return budgetItemVo;
    }

    /**
     * 设置层级、降维编码
     *
     * @param budgetItemEntity
     * @return java.lang.String
     * @author huojia
     * @date 2022/10/25 10:38
     **/
    private void generateRuleCodeByParentCode(BudgetItemEntity budgetItemEntity) {
        int levelNum = 1;
        String parentRuleCode = null;
        // 先校验上级
        if (StringUtils.isNotEmpty(budgetItemEntity.getParentBudgetItemCode())) {
            BudgetItemEntity parent =
                    this.budgetItemRepository.getByBudgetItemCode(budgetItemEntity.getParentBudgetItemCode(), null);
            Validate.notNull(parent, "父节点不存在!");
            levelNum = parent.getLevelNum() + 1;
            parentRuleCode = parent.getRuleCode();
        }
        budgetItemEntity.setParentBudgetItemCode(Optional.ofNullable(budgetItemEntity.getParentBudgetItemCode()).orElse(StringUtils.EMPTY));
        // 获取父级预算项目对应所有下级
        List<BudgetItemEntity> childrenListByParentCode = this.budgetItemRepository.findChildrenListByParentCode(budgetItemEntity.getParentBudgetItemCode());
        List<TreeDto> childrenDto =
                Lists.newArrayList(
                        nebulaToolkitService.copyCollectionByWhiteList(
                                childrenListByParentCode,
                                BudgetItemEntity.class,
                                TreeDto.class,
                                HashSet.class,
                                ArrayList.class));
        TreeRuleCodeStrategy treeRuleCodeStrategy = treeRuleCodeStrategyHolder.getStrategy(null);
        // 生成降维编码
        String ruleCode = treeRuleCodeStrategy.generate(3, parentRuleCode, childrenDto);
        budgetItemEntity.setRuleCode(ruleCode);
        budgetItemEntity.setLevelNum(levelNum);
    }

    /**
     * 校验必要参数
     *
     * @param dto
     * @author huojia
     * @date 2022/10/24 16:56
     **/
    private void createValidate(BudgetItemDto dto) {
        Validate.notNull(dto, "新增时，对象信息不能为空！");
        dto.setId(null);
        // 验证重复操作
        // this.validationPrefix(dto);
        // 校验必填字段
        Validate.notBlank(dto.getBudgetItemCode(), "新增数据时，预算项目编码不能为空！");
        Validate.notBlank(dto.getBudgetItemName(), "新增数据时，预算项目名称不能为空！");
        Validate.notBlank(dto.getBudgetItemLevelCode(), "新增数据时，预算项目层级不能为空！");
        Validate.notBlank(dto.getFeeAttrCode(), "新增数据时，预算费用属性不能为空！");
        Validate.notEmpty(dto.getControlConditionDtoList(), "新增数据时，管控条件不能为空");
        // 编码验重
        BudgetItemEntity budgetItemEntity = this.budgetItemRepository.getByBudgetItemCode(dto.getBudgetItemCode(), null);
        Validate.isTrue(budgetItemEntity == null, "新增数据时，预算项目编号重复，请检查！");
        AtomicInteger sort = new AtomicInteger(1);
        AtomicReference<Boolean> tag = new AtomicReference<>(true);
        dto.getControlConditionDtoList().forEach(controlConditionVo -> {
            controlConditionVo.setBudgetItemCode(dto.getBudgetItemCode());
            controlConditionVo.setTenantCode(TenantUtils.getTenantCode());
            controlConditionVo.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            controlConditionVo.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            Validate.notBlank(controlConditionVo.getBusinessFormatCode(), "新增数据时，业态不能为空！");
            Validate.notBlank(controlConditionVo.getBusinessUnitCode(), "新增数据时，业务单元不能为空！");
            Validate.notBlank(controlConditionVo.getControlTypeCode(), "新增数据时，管控类型不能为空！");
            //判断第一条管控条件为分子公司
            if (sort.get() == 1 && BusinessUnitEnum.SON_COMPANY.getCode().equals(controlConditionVo.getBusinessUnitCode())) {
                tag.set(false);
            }
            sort.addAndGet(1);
        });
        //若管控条件里有分子公司，则可以不填写上级预算项目
        // 设置上级预算项目
        if (Integer.parseInt(dto.getBudgetItemLevelCode()) != 1 && tag.get()) {
            Validate.notBlank(dto.getParentBudgetItemCode(), "新增数据时，预算项目层级非一级预算项目，需要选择上级预算项目[上级预算项目编码不能为空]！");
            BudgetItemEntity parentBudgetItem = budgetItemRepository.getByBudgetItemCode(dto.getParentBudgetItemCode(), null);
            if (ObjectUtils.isEmpty(parentBudgetItem)) {
                throw new RuntimeException("上级预算项目" + dto.getParentBudgetItemCode() + "不存在，请检查！");
            }
            dto.setParentBudgetItemName(parentBudgetItem.getBudgetItemName());
        }
        // 设置主体关联预算
        if (StringUtils.isNotEmpty(dto.getMainBudgetItemCode())) {
            /*List<String> mainBudgetItemCodeList = Arrays.asList(dto.getMainBudgetItemCode().split(","));
            List<BudgetItemEntity> budgetItemEntities = budgetItemRepository.listByBudgetItemCodeList(mainBudgetItemCodeList, null);
            if (!CollectionUtils.isEmpty(budgetItemEntities)) {
                String mainBudgetItemName = budgetItemEntities.stream().map(BudgetItemEntity::getBudgetItemName).collect(Collectors.joining(","));
                dto.setMainBudgetItemName(mainBudgetItemName);
            }*/
        }
        // 设置历史预算项目
        if (StringUtils.isNotEmpty(dto.getHistoryBudgetItemCode())) {
            BudgetItemEntity historyBudgetItem = budgetItemRepository.getByBudgetItemCode(dto.getHistoryBudgetItemCode(), null);
            if (historyBudgetItem != null) {
                dto.setHistoryBudgetItemName(historyBudgetItem.getBudgetItemName());
            }
        }
        dto.setTenantCode(TenantUtils.getTenantCode());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    }

    /**
     * 创建数据加锁
     *
     * @param dto
     * @author huojia
     * @date 2022/10/24 16:56
     **/
    private void validationPrefix(BudgetItemDto dto) {
        // 验证重复提交标识
        String prefix = dto.getPrefix();
        Assert.hasLength(prefix, "错误的预操作标记，请检查!!");
        Assert.isTrue(StringUtils.isNotBlank(this.redisMutexService.getMCode(RedisKeysConstant.GENERATE_PREFIX, prefix)), "没有发现预操作标记，可能是因为重复操作的原因!");
        boolean lockSuccess = false;
        try {
            lockSuccess = this.redisLockService.tryLock(prefix, TimeUnit.MILLISECONDS, 1);
            Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            this.redisMutexService.setMCode(RedisKeysConstant.GENERATE_PREFIX, prefix, prefix, 1L);
        } finally {
            if (lockSuccess) {
                this.redisLockService.unlock(prefix);
            }
        }
    }

    /**
     * 编辑
     *
     * @param dto
     * @return com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo
     * @author huojia
     * @date 2022/11/1 20:28
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public BudgetItemVo update(BudgetItemDto dto) {
        this.updateValidate(dto);
        BudgetItemVo budgetItemVo = this.findById(dto.getId());
        if (ObjectUtils.isEmpty(budgetItemVo)) {
            throw new RuntimeException("修改数据失败，原数据不存在！");
        }
        // 校验是否能修改（只有未关联下级预算项目的可以编辑层级、预算项目编码不能修改）
        if (!StringUtils.equals(dto.getBudgetItemCode(), budgetItemVo.getBudgetItemCode())) {
            throw new RuntimeException("修改数据失败，预算项目编码不能修改！");
        }
        if (!StringUtils.equals(dto.getBudgetItemLevelCode(), budgetItemVo.getBudgetItemLevelCode())) {
            List<BudgetItemEntity> childrenListByParentCode = this.budgetItemRepository.findChildrenListByParentCode(budgetItemVo.getBudgetItemCode());
            if (!CollectionUtils.isEmpty(childrenListByParentCode)) {
                throw new RuntimeException("修改数据失败，当前预算项目已关联下级预算项目，不能修改层级！");
            }
        }
        // 保存预算项目
        BudgetItemDto oldVo = this.nebulaToolkitService.copyObjectByWhiteList(budgetItemVo, BudgetItemDto.class, LinkedHashSet.class, ArrayList.class);
        List<BudgetItemControlConditionEntity> controlConditionEntityList = this.controlConditionRepository.findByBudgetItemCode(dto.getBudgetItemCode());
        oldVo.setControlConditionDtoList(new ArrayList<>(this.nebulaToolkitService.copyCollectionByBlankList(controlConditionEntityList, BudgetItemControlConditionEntity.class, BudgetItemControlConditionDto.class, LinkedHashSet.class, ArrayList.class)));
        BudgetItemEntity budgetItemEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, BudgetItemEntity.class, LinkedHashSet.class, ArrayList.class);
        // 判断是否重置降维编码
        if (!StringUtils.equals(oldVo.getParentBudgetItemCode(), dto.getParentBudgetItemCode())) {
            generateRuleCodeByParentCode(budgetItemEntity);
        }
        String mainBudgetItemCode = budgetItemEntity.getMainBudgetItemCode();
        budgetItemEntity.setMainBudgetItemCode(null);
        budgetItemEntity.setMainBudgetItemName(null);
        this.budgetItemRepository.saveOrUpdate(budgetItemEntity);
        // 发送事件
        if (!CollectionUtils.isEmpty(budgetItemEventListenerList)) {
            for (BudgetItemEventListener budgetItemEventListener : budgetItemEventListenerList) {
                budgetItemEventListener.onUpdate(oldVo, dto);
            }
        }
        // 保存关联主体预算项目
        budgetItemMainItemRepository.delByBudgetItemCode(dto.getBudgetItemCode());
        if (!StringUtils.isEmpty(mainBudgetItemCode)) {
            List<String> mainBudgetList = Arrays.asList(mainBudgetItemCode.split(","));
            List<BudgetItemVo> budgetItemVos = listByCodes(mainBudgetList);
            Map<String, BudgetItemVo> map = budgetItemVos.stream().collect(Collectors.toMap(BudgetItemVo::getBudgetItemCode, Function.identity()));
            List<BudgetItemMainItemEntity> budgetItemMainItemEntities = new ArrayList<>();
            mainBudgetList.forEach(mainBudgetCode -> {
                BudgetItemMainItemEntity budgetItemMainItemEntity = new BudgetItemMainItemEntity();
                budgetItemMainItemEntity.setBudgetItemCode(budgetItemEntity.getBudgetItemCode());
                budgetItemMainItemEntity.setMainBudgetItemCode(mainBudgetCode);
                if (map.containsKey(mainBudgetCode)) {
                    budgetItemMainItemEntity.setMainBudgetItemName(map.get(mainBudgetCode).getBudgetItemName());
                }
                budgetItemMainItemEntity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                budgetItemMainItemEntity.setTenantCode(TenantUtils.getTenantCode());
                budgetItemMainItemEntities.add(budgetItemMainItemEntity);
            });
            budgetItemMainItemRepository.saveBatch(budgetItemMainItemEntities);
        }
        // 保存管控条件
        Collection<BudgetItemControlConditionEntity> controlConditionEntities = this.nebulaToolkitService.copyCollectionByBlankList(dto.getControlConditionDtoList(), BudgetItemControlConditionDto.class, BudgetItemControlConditionEntity.class, LinkedHashSet.class, ArrayList.class);
        controlConditionRepository.delByBudgetItemCode(dto.getBudgetItemCode());
        controlConditionRepository.saveOrUpdateBatch(new ArrayList<>(controlConditionEntities));
        // 业务日志编辑
        BudgetItemLogEventDto logEventDto = new BudgetItemLogEventDto();
        logEventDto.setOriginal(oldVo);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<BudgetItemLogEventListener, BudgetItemLogEventDto> onUpdate =
                BudgetItemLogEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, BudgetItemLogEventListener.class, onUpdate);
        return this.nebulaToolkitService.copyObjectByWhiteList(dto, BudgetItemVo.class, LinkedHashSet.class, ArrayList.class);
    }

    /**
     * 校验必要参数
     *
     * @param dto
     * @author huojia
     * @date 2022/10/24 19:27
     **/
    private void updateValidate(BudgetItemDto dto) {
        Validate.notNull(dto, "修改时，对象信息不能为空！");
        // 验证重复操作
        // this.validationPrefix(dto);
        Validate.notBlank(dto.getId(), "修改数据时，主键不能为空！");
        Validate.notBlank(dto.getBudgetItemCode(), "修改数据时，预算项目编码不能为空！");
        Validate.notBlank(dto.getBudgetItemName(), "修改数据时，预算项目名称不能为空！");
        Validate.notBlank(dto.getBudgetItemLevelCode(), "修改数据时，预算项目层级不能为空！");
        Validate.notBlank(dto.getFeeAttrCode(), "修改数据时，预算费用属性不能为空！");
        Validate.notEmpty(dto.getControlConditionDtoList(), "修改数据时，管控条件不能为空！");
        // 设置基础字段
        if (StringUtils.isEmpty(dto.getTenantCode())) {
            dto.setTenantCode(TenantUtils.getTenantCode());
        }
        AtomicReference<Boolean> tag = new AtomicReference<>(true);
        AtomicInteger sort = new AtomicInteger(1);
        dto.getControlConditionDtoList().forEach(controlConditionVo -> {
            controlConditionVo.setBudgetItemCode(dto.getBudgetItemCode());
            controlConditionVo.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            controlConditionVo.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            Validate.notBlank(controlConditionVo.getBusinessFormatCode(), "编辑数据时，业态不能为空！");
            Validate.notBlank(controlConditionVo.getBusinessUnitCode(), "编辑数据时，业务单元不能为空！");
            Validate.notBlank(controlConditionVo.getControlTypeCode(), "编辑数据时，管控类型不能为空！");
            //判断第一条管控条件为分子公司
            if (sort.get() == 1 && BusinessUnitEnum.SON_COMPANY.getCode().equals(controlConditionVo.getBusinessUnitCode())) {
                tag.set(false);
            }
            sort.addAndGet(1);
            if (StringUtils.isEmpty(controlConditionVo.getTenantCode())) {
                controlConditionVo.setTenantCode(TenantUtils.getTenantCode());
            }
        });
        //若管控条件里有分子公司，则可以不填写上级预算项目
        if (Integer.parseInt(dto.getBudgetItemLevelCode()) != 1 && tag.get()) {
            Validate.notBlank(dto.getParentBudgetItemCode(), "修改数据时，预算项目层级非一级预算项目，需要选择上级预算项目[上级预算项目编码不能为空]！");
            Validate.notBlank(dto.getParentBudgetItemName(), "修改数据时，预算项目层级非一级预算项目，需要选择上级预算项目[上级预算项目名称不能为空]！");
        }
        if (StringUtils.isNotEmpty(dto.getMainBudgetItemCode())) {
            /*List<String> mainBudgetItemCodeList = Arrays.asList(dto.getMainBudgetItemCode().split(","));
            List<BudgetItemEntity> budgetItemEntities = budgetItemRepository.listByBudgetItemCodeList(mainBudgetItemCodeList, null);
            if (!CollectionUtils.isEmpty(budgetItemEntities)) {
                String mainBudgetItemName = budgetItemEntities.stream().map(BudgetItemEntity::getBudgetItemName).collect(Collectors.joining(","));
                dto.setMainBudgetItemName(mainBudgetItemName);
            }*/
        }
        if (StringUtils.isNotEmpty(dto.getHistoryBudgetItemCode())) {
            BudgetItemEntity historyBudgetItem = budgetItemRepository.getByBudgetItemCode(dto.getHistoryBudgetItemCode(), null);
            if (historyBudgetItem != null) {
                dto.setHistoryBudgetItemName(historyBudgetItem.getBudgetItemName());
            }
        }
    }

    /**
     * 删除
     *
     * @param ids
     * @author huojia
     * @date 2022/11/1 20:28
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new RuntimeException("请选择要删除的数据！");
        }
        List<BudgetItemEntity> budgetItemEntities = this.budgetItemRepository.listByIds(ids);
        if (CollectionUtils.isEmpty(budgetItemEntities)) {
            return;
        }
        // 关联月度预算的不能删
        List<String> budgetItemCodeList = budgetItemEntities.stream().map(BudgetItemEntity::getBudgetItemCode).collect(Collectors.toList());
        /*List<MonthBudgetVo> monthBudgetVoList = monthBudgetService.listByBudgetItemCodeList(budgetItemCodeList);
        if (!CollectionUtils.isEmpty(monthBudgetVoList)) {
            throw new RuntimeException("预算项目" + monthBudgetVoList.get(0).getBudgetItemCode() + "关联月度预算" + monthBudgetVoList.get(0).getMonthBudgetCode() + "无法删除！");
        }*/
        // 关联分子公司月度预算的不能删
        List<SubComMonthBudgetVo> subComMonthBudgetVoList = subComMonthBudgetService.listByBudgetItemCodeList(budgetItemCodeList);
        if (!CollectionUtils.isEmpty(subComMonthBudgetVoList)) {
            throw new RuntimeException("预算项目" + subComMonthBudgetVoList.get(0).getBudgetItemCode() + "关联分子公司月度预算" + subComMonthBudgetVoList.get(0).getMonthBudgetCode() + "无法删除！");
        }
        // 有下级的不能删
        budgetItemEntities.forEach(budgetItemEntity -> {
            List<BudgetItemEntity> childrenListByParentCode = this.budgetItemRepository.findChildrenListByParentCode(budgetItemEntity.getBudgetItemCode());
            if (!CollectionUtils.isEmpty(childrenListByParentCode)) {
                throw new RuntimeException("预算项目:" + budgetItemEntity.getBudgetItemCode() + "已维护下级预算项目，不能删除");
            }
            budgetItemEntity.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
        });
        this.budgetItemRepository.updateBatchById(budgetItemEntities);
        List<String> budgetItemCodes = budgetItemEntities.stream().map(BudgetItemEntity::getBudgetItemCode).collect(Collectors.toList());
        this.budgetItemMainItemRepository.delByBudgetItemCodes(budgetItemCodes);
        this.controlConditionRepository.delByBudgetItemCodes(budgetItemCodes);
        // 业务日志删除
        budgetItemEntities.forEach(budgetItemEntity -> {
            BudgetItemLogEventDto logEventDto = new BudgetItemLogEventDto();
            logEventDto.setOriginal(this.nebulaToolkitService.copyObjectByWhiteList(
                    budgetItemEntity, BudgetItemDto.class, LinkedHashSet.class, ArrayList.class
            ));
            logEventDto.setNewest(null);
            SerializableBiConsumer<BudgetItemLogEventListener, BudgetItemLogEventDto> onDelete =
                    BudgetItemLogEventListener::onDelete;
            this.nebulaNetEventClient.publish(logEventDto, BudgetItemLogEventListener.class, onDelete);
        });
    }

    /**
     * 启用
     *
     * @param ids
     * @author huojia
     * @date 2022/11/1 20:28
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enable(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new RuntimeException("请选择要启用的数据！");
        }
        List<BudgetItemEntity> budgetItemEntities = this.budgetItemRepository.listByIds(ids, DelFlagStatusEnum.NORMAL.getCode());
        if (CollectionUtils.isEmpty(budgetItemEntities)) {
            return;
        }
        budgetItemEntities.forEach(budgetItemEntity -> {
            if (EnableStatusEnum.ENABLE.getCode().equals(budgetItemEntity.getEnableStatus())) {
                throw new RuntimeException("启用失败，只能选择禁用状态预算项目");
            }
            budgetItemEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        });
        this.budgetItemRepository.updateBatchById(budgetItemEntities);
        // 业务日志启用
        budgetItemEntities.forEach(budgetItemEntity -> {
            BudgetItemLogEventDto logEventDto = new BudgetItemLogEventDto();
            logEventDto.setOriginal(this.nebulaToolkitService.copyObjectByWhiteList(
                    budgetItemEntity, BudgetItemDto.class, LinkedHashSet.class, ArrayList.class
            ));
            logEventDto.setNewest(null);
            SerializableBiConsumer<BudgetItemLogEventListener, BudgetItemLogEventDto> onEnable =
                    BudgetItemLogEventListener::onEnable;
            this.nebulaNetEventClient.publish(logEventDto, BudgetItemLogEventListener.class, onEnable);
        });
    }

    /**
     * 禁用
     *
     * @param ids
     * @author huojia
     * @date 2022/11/1 20:28
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disable(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new RuntimeException("请选择要禁用的数据");
        }
        List<BudgetItemEntity> budgetItemEntities = this.budgetItemRepository.listByIds(ids, DelFlagStatusEnum.NORMAL.getCode());
        if (CollectionUtils.isEmpty(budgetItemEntities)) {
            return;
        }
        List<String> budgetItemCodeList = budgetItemEntities.stream().map(BudgetItemEntity::getBudgetItemCode).collect(Collectors.toList());
        List<BudgetItemEntity> list = budgetItemRepository.listByParentCodes(budgetItemCodeList);
        List<String> relateCodeList = list.stream().map(BudgetItemEntity::getParentBudgetItemCode).distinct().collect(Collectors.toList());

        budgetItemEntities.forEach(budgetItemEntity -> {
            if (EnableStatusEnum.DISABLE.getCode().equals(budgetItemEntity.getEnableStatus())) {
                throw new RuntimeException("禁用失败，只能选择启用状态预算项目");
            }
            Validate.isTrue(!relateCodeList.contains(budgetItemEntity.getBudgetItemCode()), "预算项目[%s]已被关联，无法禁用", budgetItemEntity.getBudgetItemCode());
            budgetItemEntity.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
        });
        this.budgetItemRepository.updateBatchById(budgetItemEntities);
        // 业务日志禁用
        budgetItemEntities.forEach(budgetItemEntity -> {
            BudgetItemLogEventDto logEventDto = new BudgetItemLogEventDto();
            logEventDto.setOriginal(this.nebulaToolkitService.copyObjectByWhiteList(
                    budgetItemEntity, BudgetItemDto.class, LinkedHashSet.class, ArrayList.class
            ));
            logEventDto.setNewest(null);
            SerializableBiConsumer<BudgetItemLogEventListener, BudgetItemLogEventDto> onDisable =
                    BudgetItemLogEventListener::onDisable;
            this.nebulaNetEventClient.publish(logEventDto, BudgetItemLogEventListener.class, onDisable);
        });
    }

    /**
     * 通过主体预算项目找到分子公司预算项目
     *
     * @param mainBudgetItemCodes
     * @param businessFormatCode
     * @return {@link List}<{@link BudgetItemVo}>
     */
    @Override
    public List<BudgetItemVo> findByMainBudgetItem(List<String> mainBudgetItemCodes, String businessFormatCode) {
        BudgetItemDto dto = new BudgetItemDto();
        dto.setBusinessUnitCode(BusinessUnitEnum.SON_COMPANY.getCode());
        dto.setMainBudgetItemCodes(mainBudgetItemCodes);
        dto.setBusinessFormatCode(businessFormatCode);
        return this.budgetItemRepository.findByMainBudgetItem(dto);
    }

    /**
     * 分页查询分子公司预算项目
     * 过滤：业态为分子公司、未删除、启用
     *
     * @param pageable
     * @param dto
     * @return
     */
    @Override
    public Page<BudgetItemVo> listSubsidiaryCompaniesBudgetItem(Pageable pageable, BudgetItemDto dto) {
        ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (ObjectUtils.isEmpty(dto)) {
            dto = new BudgetItemDto();
        }
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setTenantCode(TenantUtils.getTenantCode());
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        dto.setBusinessUnitCode(BusinessUnitEnum.SON_COMPANY.getCode());
        Page<BudgetItemVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        Page<BudgetItemVo> result = budgetItemRepository.listSubsidiaryCompaniesBudgetItem(page, dto);
        return result;
    }
}