package com.biz.crm.tpm.business.budget.item.local.service.process;

import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.tpm.business.budget.item.sdk.constant.DictTypeCodeConstant;
import com.biz.crm.tpm.business.budget.item.sdk.dto.BudgetItemControlConditionDto;
import com.biz.crm.tpm.business.budget.item.sdk.dto.BudgetItemDto;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemImportsVo;
import com.biz.crm.tpm.business.budget.item.sdk.service.BudgetItemService;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 预算项目导入
 *
 * @author huojia
 * @date 2022年10月25日 15:28
 */
@Slf4j
@Component
public class BudgetItemImportsProcess implements ImportProcess<BudgetItemImportsVo> {

    @Resource
    private BudgetItemService budgetItemService;

    @Resource
    private DictToolkitService dictToolkitService;

    @Resource
    private NebulaToolkitService nebulaToolkitService;

    private Map<String, String> tpmBudgetFeeAttrMap = new HashMap<>();
    private Map<String, String> tpmBudgetItemLevelMap = new HashMap<>();
    private Map<String, String> tpmBudgetDataTypeMap = new HashMap<>();
    private Map<String, String> mdmBusinessFormatMap = new HashMap<>();
    private Map<String, String> mdmBusinessUnitMap = new HashMap<>();
    private Map<String, String> tpmBudgetControlTypeMap = new HashMap<>();
    private Map<String, String> tpmFeeBelongMap = new HashMap<>();
    private Map<String, String> tpmDepartmentBelongMap = new HashMap<>();
    private Map<String, BudgetItemVo> budgetItemVoMap = new HashMap<>();

    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, BudgetItemImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Validate.notEmpty(data, "导入数据不能为空！");

        List<BudgetItemDto> importList = this.validate(data);

        budgetItemService.importSave(importList);
        return null;
    }

    /**
     * 校验并转换数据
     *
     * @param data
     * @return java.util.List<com.biz.crm.tpm.business.year.budget.sdk.dto.YearBudgetDto>
     * @author huojia
     * @date 2022/10/28 15:13
     **/
    private List<BudgetItemDto> validate(LinkedHashMap<Integer, BudgetItemImportsVo> data) {
        List<BudgetItemImportsVo> budgetItemImportsVos = new ArrayList<>(data.values());

        // 分组前先校验预算项目编码
        budgetItemImportsVos.forEach(budgetItemImportsVo -> {
            if (StringUtils.isEmpty(budgetItemImportsVo.getBudgetItemCode())) {
                throw new RuntimeException("预算项目编码不能为空！");
            }
        });

        List<BudgetItemDto> budgetItemDtoList = new ArrayList<>();
        // 构建基础数据map
        this.buildBaseMap(budgetItemImportsVos);

        // 校验参数
        Map<String, List<BudgetItemImportsVo>> itemMap = budgetItemImportsVos.stream().collect(Collectors.groupingBy(BudgetItemImportsVo::getBudgetItemCode));
        itemMap.forEach((key, value) -> {
            if (CollectionUtils.isEmpty(value)) {
                return;
            }
            // 以第一条数据作为主表数据
            BudgetItemImportsVo budgetItemImportsVo = value.get(0);

            // 校验基础数据
            this.buildBaseData(budgetItemImportsVo);

            // 校验明细数据
            this.buildDetailData(value);

            List<BudgetItemControlConditionDto> controlConditionDtoList = new ArrayList<>(this.nebulaToolkitService.copyCollectionByWhiteList(value, BudgetItemImportsVo.class, BudgetItemControlConditionDto.class, LinkedHashSet.class, ArrayList.class));
            BudgetItemDto budgetItemDto = this.nebulaToolkitService.copyObjectByWhiteList(value.get(0), BudgetItemDto.class, LinkedHashSet.class, ArrayList.class);
            budgetItemDto.setControlConditionDtoList(controlConditionDtoList);
            budgetItemDtoList.add(budgetItemDto);
        });
        return budgetItemDtoList;
    }

    /**
     * 校验明细数据
     *
     * @param value
     * @author huojia
     * @date 2022/11/11 15:31
     **/
    private void buildDetailData(List<BudgetItemImportsVo> value) {
        // 数据字典名称转编码
        value.forEach(importsVo -> {
            Validate.notBlank(importsVo.getBusinessFormatCode(), "业态不能为空！");
            Validate.notBlank(importsVo.getBusinessUnitCode(), "业务单元不能为空！");
            Validate.notBlank(importsVo.getControlTypeCode(), "管控类型不能为空！");
            Validate.notBlank(importsVo.getFeeBelongCode(), "费用归口不能为空！");
            Validate.notBlank(importsVo.getDepartmentBelongCode(), "部门归口不能为空！");

            if (mdmBusinessFormatMap.containsKey(importsVo.getBusinessFormatCode())) {
                importsVo.setBusinessFormatCode(mdmBusinessFormatMap.get(importsVo.getBusinessFormatCode()));
            } else {
                throw new RuntimeException("业态" + importsVo.getBusinessFormatCode() + "错误，请检查！");
            }
            if (mdmBusinessUnitMap.containsKey(importsVo.getBusinessUnitCode())) {
                importsVo.setBusinessUnitCode(mdmBusinessUnitMap.get(importsVo.getBusinessUnitCode()));
            } else {
                throw new RuntimeException("业务单元" + importsVo.getBusinessUnitCode() + "错误，请检查！");
            }
            if (tpmBudgetControlTypeMap.containsKey(importsVo.getControlTypeCode())) {
                importsVo.setControlTypeCode(tpmBudgetControlTypeMap.get(importsVo.getControlTypeCode()));
            } else {
                throw new RuntimeException("管控类型" + importsVo.getControlTypeCode() + "错误，请检查！");
            }
            if (tpmFeeBelongMap.containsKey(importsVo.getFeeBelongCode())) {
                importsVo.setFeeBelongCode(tpmFeeBelongMap.get(importsVo.getFeeBelongCode()));
            } else {
                throw new RuntimeException("费用归口" + importsVo.getFeeBelongCode() + "错误，请检查！");
            }
            if (tpmDepartmentBelongMap.containsKey(importsVo.getDepartmentBelongCode())) {
                importsVo.setDepartmentBelongCode(tpmDepartmentBelongMap.get(importsVo.getDepartmentBelongCode()));
            } else {
                throw new RuntimeException("部门归口" + importsVo.getDepartmentBelongCode() + "错误，请检查！");
            }
        });
    }

    /**
     * 校验基础数据
     *
     * @param budgetItemImportsVo
     * @author huojia
     * @date 2022/11/11 15:27
     **/
    private void buildBaseData(BudgetItemImportsVo budgetItemImportsVo) {
        // 校验必填
        Validate.notEmpty(budgetItemImportsVo.getBudgetItemCode(), "预算项目编码不能为空！");
        Validate.notEmpty(budgetItemImportsVo.getBudgetItemName(), "预算项目名称不能为空！");
        Validate.notEmpty(budgetItemImportsVo.getBudgetItemLevelCode(), "预算项目层级不能为空！");
        Validate.notBlank(budgetItemImportsVo.getFeeAttrCode(), "预算费用属性不能为空！");

        // 校验预算项目是否重复
        if (budgetItemVoMap.containsKey(budgetItemImportsVo.getBudgetItemCode())) {
            throw new RuntimeException("预算项目编码" + budgetItemImportsVo.getBudgetItemCode() + "已存在，请检查！");
        }

        // 预算项目层级
        if (tpmBudgetItemLevelMap.containsKey(budgetItemImportsVo.getBudgetItemLevelCode())) {
            budgetItemImportsVo.setBudgetItemLevelCode(tpmBudgetItemLevelMap.get(budgetItemImportsVo.getBudgetItemLevelCode()));
            if (Integer.parseInt(budgetItemImportsVo.getBudgetItemLevelCode()) != 1) {
                Validate.notBlank(budgetItemImportsVo.getParentBudgetItemCode(), "预算项目层级非一级预算项目，上级预算项目编码不能为空！");
            }
        } else {
            throw new RuntimeException("预算项目层级" + budgetItemImportsVo.getBudgetItemLevelCode() + "错误，请检查！");
        }

        // 费用属性
        if (tpmBudgetFeeAttrMap.containsKey(budgetItemImportsVo.getFeeAttrCode())) {
            budgetItemImportsVo.setFeeAttrCode(tpmBudgetFeeAttrMap.get(budgetItemImportsVo.getFeeAttrCode()));
        } else {
            throw new RuntimeException("费用属性" + budgetItemImportsVo.getFeeAttrCode() + "错误，请检查！");
        }

        // 校验数据类型
        if (!StringUtils.isEmpty(budgetItemImportsVo.getDataTypeCode())) {
            if (tpmBudgetDataTypeMap.containsKey(budgetItemImportsVo.getDataTypeCode())) {
                budgetItemImportsVo.setDataTypeCode(tpmBudgetDataTypeMap.get(budgetItemImportsVo.getDataTypeCode()));
            } else {
                throw new RuntimeException("数据类型" + budgetItemImportsVo.getDataTypeCode() + "错误，请检查！");
            }
        }

        // 上级预算项目
        if (!StringUtils.isEmpty(budgetItemImportsVo.getParentBudgetItemCode())) {
            if (budgetItemVoMap.containsKey(budgetItemImportsVo.getParentBudgetItemCode())) {
                budgetItemImportsVo.setParentBudgetItemName(budgetItemVoMap.get(budgetItemImportsVo.getParentBudgetItemCode()).getBudgetItemName());
            } else {
                throw new RuntimeException("上级预算编码" + budgetItemImportsVo.getParentBudgetItemCode() + "错误，请检查！");
            }
        }

        // 校验主体预算项目
        if (!StringUtils.isEmpty(budgetItemImportsVo.getMainBudgetItemCode())) {
            List<String> mainBudgetItemCodeList = Arrays.asList(budgetItemImportsVo.getMainBudgetItemCode().split(","));
            List<String> mainBudgetItemNameList = new ArrayList<>();
            mainBudgetItemCodeList.forEach(mainBudgetItemCode -> {
                if (!budgetItemVoMap.containsKey(mainBudgetItemCode)) {
                    throw new RuntimeException("主体预算项目编码" + budgetItemImportsVo.getMainBudgetItemCode() + "错误，请检查！");
                } else {
                    mainBudgetItemNameList.add(budgetItemVoMap.get(mainBudgetItemCode).getBudgetItemName());
                }
            });
            budgetItemImportsVo.setMainBudgetItemName(String.join(",", mainBudgetItemNameList));
        }

        // 校验历史预算项目
        if (!StringUtils.isEmpty(budgetItemImportsVo.getHistoryBudgetItemCode())) {
            if (budgetItemVoMap.containsKey(budgetItemImportsVo.getHistoryBudgetItemCode())) {
                budgetItemImportsVo.setHistoryBudgetItemName(budgetItemVoMap.get(budgetItemImportsVo.getHistoryBudgetItemCode()).getBudgetItemName());
            } else {
                throw new RuntimeException("历史预算项目编码" + budgetItemImportsVo.getHistoryBudgetItemCode() + "错误，请检查！");
            }
        }
    }

    /**
     * 构建基础数据map
     *
     * @param budgetItemImportsVos
     * @author huojia
     * @date 2022/11/11 15:23
     **/
    private void buildBaseMap(List<BudgetItemImportsVo> budgetItemImportsVos) {
        // 查询对应数字典
        tpmBudgetFeeAttrMap = dictToolkitService.findConvertMapByDictTypeCode(DictTypeCodeConstant.TPM_BUDGET_FEE_ATTR);
        tpmBudgetItemLevelMap = dictToolkitService.findConvertMapByDictTypeCode(DictTypeCodeConstant.TPM_BUDGET_ITEM_LEVEL);
        tpmBudgetDataTypeMap = dictToolkitService.findConvertMapByDictTypeCode(DictTypeCodeConstant.TPM_BUDGET_DATA_TYPE);
        mdmBusinessFormatMap = dictToolkitService.findConvertMapByDictTypeCode(DictTypeCodeConstant.MDM_BUSINESS_FORMAT);
        mdmBusinessUnitMap = dictToolkitService.findConvertMapByDictTypeCode(DictTypeCodeConstant.MDM_BUSINESS_UNIT);
        tpmBudgetControlTypeMap = dictToolkitService.findConvertMapByDictTypeCode(DictTypeCodeConstant.TPM_BUDGET_CONTROL_TYPE);
        tpmFeeBelongMap = dictToolkitService.findConvertMapByDictTypeCode(DictTypeCodeConstant.TPM_FEE_BELONG);
        tpmDepartmentBelongMap = dictToolkitService.findConvertMapByDictTypeCode(DictTypeCodeConstant.TPM_DEPARTMENT_BELONG);

        // 查询预算项目
        List<String> budgetItemCodeList = budgetItemImportsVos.stream().map(BudgetItemImportsVo::getBudgetItemCode).filter(Objects::nonNull).collect(Collectors.toList());
        List<String> historyBudgetItemCodeList = budgetItemImportsVos.stream().map(BudgetItemImportsVo::getHistoryBudgetItemCode).filter(Objects::nonNull).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(historyBudgetItemCodeList)) {
            budgetItemCodeList.addAll(historyBudgetItemCodeList);
        }
        List<String> mainBudgetItemCodes = budgetItemImportsVos.stream().map(BudgetItemImportsVo::getMainBudgetItemCode).filter(Objects::nonNull).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(mainBudgetItemCodes)) {
            mainBudgetItemCodes.forEach(mainBudgetItemCode -> {
                budgetItemCodeList.addAll(Arrays.asList(mainBudgetItemCode.split(",")));
            });
        }
        List<BudgetItemVo> budgetItemVos = budgetItemService.listByCodes(budgetItemCodeList);
        budgetItemVoMap = budgetItemVos.stream().collect(Collectors.toMap(BudgetItemVo::getBudgetItemCode, Function.identity(), (oldVo, newVo) -> newVo));

    }

    @Override
    public Class<BudgetItemImportsVo> findCrmExcelVoClass() {
        return BudgetItemImportsVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "TPM_BUDGET_ITEM_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "TPM预算项目导入";
    }
}
