package com.biz.crm.tpm.business.channel.price.monitor.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.price.sdk.service.PriceVoService;
import com.biz.crm.mdm.business.price.sdk.vo.PriceFeeVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.channel.price.monitor.local.entity.ChannelPriceEntity;
import com.biz.crm.tpm.business.channel.price.monitor.local.repository.ChannelPriceRepository;
import com.biz.crm.tpm.business.channel.price.monitor.local.service.ChannelPriceService;
import com.biz.crm.tpm.business.channel.price.monitor.sdk.constant.ChannelPriceConstant;
import com.biz.crm.tpm.business.channel.price.monitor.sdk.dto.ChannelPriceDto;
import com.biz.crm.tpm.business.day.sales.sdk.dto.TpmDaySalesSpliceDto;
import com.biz.crm.tpm.business.day.sales.sdk.service.TpmDaySalesSpliceService;
import com.biz.crm.tpm.business.day.sales.sdk.vo.TpmDaySalesSpliceVo;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.CurrentMonthSaleDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.PromotionPlanService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Maps;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 渠道价格接口实现
 *
 * @author zhouyang
 * @date 2023-07-26 18:15:45
 */
@Slf4j
@Service
public class ChannelPriceServiceImpl implements ChannelPriceService {

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private ChannelPriceRepository channelPriceRepository;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private TpmDaySalesSpliceService tpmDaySalesSpliceService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private ProductVoService productVoService;

    @Autowired(required = false)
    private PriceVoService priceVoService;

//    @Autowired(required = false)
//    private ActivityDailyEstimatedPriceVoService activityDailyEstimatedPriceVoService;

    @Autowired(required = false)
    private PromotionPlanService promotionPlanService;


    @Override
    public Page<ChannelPriceDto> findByConditions(Pageable pageable, ChannelPriceDto dto) {
        return channelPriceRepository.findByConditions(pageable != null ? pageable : PageRequest.of(0, 50), dto);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.channelPriceRepository.removeByIds(ids);
    }

    @Override
    @Async
    public void getSalesSpliceDataAsync(String yearAndMonth, AbstractCrmUserIdentity abstractLoginUser) {
        boolean lockSuccess = false;
        try {
            lockSuccess = redisLockService.tryLock(ChannelPriceConstant.CHANNEL_PRICE_LOCK, TimeUnit.HOURS, 2);
            Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            loginUserService.refreshAuthentication(abstractLoginUser);
            this.getSalesSpliceData(yearAndMonth);
        } catch (Exception e) {
            log.error("=====>   渠道价格数据 异常   <=====");
            log.error("", e);
        } finally {
            if (lockSuccess) {
                redisLockService.unlock(ChannelPriceConstant.CHANNEL_PRICE_LOCK);
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class, propagation = Propagation.NOT_SUPPORTED)
    public void getSalesSpliceData(String yearMonthLy) {
        Validate.notBlank(yearMonthLy, "请填入年月");
        boolean isLock = redisLockService.isLock(ChannelPriceConstant.CHANNEL_PRICE_LOCK);
        Assert.isTrue(isLock, "未加锁,本次不执行!");
        this.removeByYearMonth(yearMonthLy);
        Pageable pageable = PageRequest.of(1, 800);
        Page<TpmDaySalesSpliceVo> daySalesPage = null;

        TpmDaySalesSpliceDto dto = new TpmDaySalesSpliceDto();
        dto.setYearMonthLy(yearMonthLy);
        do {
            daySalesPage = this.tpmDaySalesSpliceService.findByYearAndMonth(pageable, dto);
            log.info("=====>    查询渠道价格数据 年月[{}]  页[{}/{}][{}]条    <=====", yearMonthLy, daySalesPage.getCurrent(), daySalesPage.getPages(), daySalesPage.getSize());
            if (CollectionUtil.isEmpty(daySalesPage.getRecords())) {
                return;
            }
            List<ChannelPriceEntity> channelPriceList = this.buildData(daySalesPage.getRecords(), yearMonthLy);
            this.saveDataList(channelPriceList);
        } while (daySalesPage.hasNext());

    }

    private List<ChannelPriceEntity> buildData(List<TpmDaySalesSpliceVo> records, String yearMonthLy) {
        if (CollectionUtil.isEmpty(records)) {
            return Collections.emptyList();
        }
        String tenantCode = TenantUtils.getTenantCode();
        List<ChannelPriceEntity> channelPriceList = new ArrayList<>();
        records.forEach(item -> {
            List<TpmDaySalesSpliceVo> daySalesSpliceVos = tpmDaySalesSpliceService.getByCustomerAndProduct(yearMonthLy, item.getCustomer(), item.getGoodsCode());
            if (CollectionUtil.isEmpty(daySalesSpliceVos)) {
                return;
            }
            // 根据业态+渠道+销售机构+客户+产品+日期+电商渠道+活动形式的维度进行去重
            List<TpmDaySalesSpliceVo> resultList = daySalesSpliceVos.stream().collect(
                    Collectors.collectingAndThen(
                            Collectors.toCollection(() -> new TreeSet<>(Comparator.comparing(TpmDaySalesSpliceVo::getBusinessFormatCode)
                                    .thenComparing(TpmDaySalesSpliceVo::getChannelCode)
                                    .thenComparing(TpmDaySalesSpliceVo::getSalesInstitutionCode)
                                    .thenComparing(TpmDaySalesSpliceVo::getCustomer)
                                    .thenComparing(TpmDaySalesSpliceVo::getGoodsCode)
                                    .thenComparing(TpmDaySalesSpliceVo::getTradeDate)
                                    .thenComparing(TpmDaySalesSpliceVo::getChannel)
                                    .thenComparing(TpmDaySalesSpliceVo::getActivityForm))), ArrayList::new)
            );
            // 客户信息
            List<String> customerCodes = resultList.stream().map(a -> a.getCustomer() +
                    a.getSalesInstitutionCode() + a.getChannelCode() + a.getBusinessFormatCode()).distinct().collect(Collectors.toList());
            List<CustomerVo> customerVoList = customerVoService.findByCustomerCodes(customerCodes);
            Map<String, CustomerVo> customerVoMap = CollectionUtils.isEmpty(customerVoList)
                    ? Maps.newHashMap() : customerVoList.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, v -> v, (n, o) -> n));
            // 产品信息
            List<String> productCode = resultList.stream().map(TpmDaySalesSpliceVo::getGoodsCode).collect(Collectors.toList());
            List<ProductVo> productVoList = productVoService.findByCodes(productCode);
            Map<String, ProductVo> productVoMap = CollectionUtils.isEmpty(productVoList)
                    ? Maps.newHashMap() : productVoList.stream().collect(Collectors.toMap(ProductVo::getProductCode, v -> v, (n, o) -> n));
            resultList.forEach(e -> {
                ChannelPriceEntity entity = new ChannelPriceEntity();
                entity.setBusinessFormatCode(e.getBusinessFormatCode());
                entity.setBusinessFormatName(e.getBusinessFormatName());
                entity.setChannelCode(e.getChannelCode());
                entity.setChannelName(e.getChannelName());
                entity.setSalesInstitutionCode(e.getSalesInstitutionCode());
                entity.setSalesInstitutionName(e.getSalesInstitutionName());
                entity.setCustomerCode(e.getCustomer());
                entity.setCustomerName(e.getCustomerName());
                entity.setGoodsCode(e.getGoodsCode());
                entity.setGoodsName(e.getGoodsName());
                entity.setTradeDate(e.getTradeDate());
                entity.setEstoreChannel(e.getChannel());
                entity.setActivityFormCode(e.getActivityForm());
                entity.setActivityFormName(e.getActivityFormName());
                if (null != e.getAmount() && e.getNum().compareTo(BigDecimal.ZERO) != 0) {
                    entity.setSalePrice(Optional.ofNullable(e.getAmount()).orElse(BigDecimal.ZERO).divide(e.getNum(), 2, BigDecimal.ROUND_DOWN));
                }
                // 客户信息
                StringBuilder customerCode = new StringBuilder();
                customerCode.append(e.getCustomer());
                customerCode.append(e.getSalesInstitutionCode());
                customerCode.append(e.getChannelCode());
                customerCode.append(e.getBusinessFormatCode());
                if (customerVoMap.containsKey(customerCode.toString())) {
                    entity.setBusinessModelCode(customerVoMap.get(customerCode.toString()).getBusinessModelCode());
                    entity.setPlatformCode(customerVoMap.get(customerCode.toString()).getEstorePlatform());
                }
                // 产品信息
                if (productVoMap.containsKey(e.getGoodsCode())) {
                    entity.setProductBrandCode(productVoMap.get(e.getGoodsCode()).getProductBrandCode());
                    entity.setProductBrandName(productVoMap.get(e.getGoodsCode()).getProductBrandName());
                }
                // 标准零售价（元）	根据产品+日期，匹配价格管理中维护的标准零售价
                // 红线价（元）	根据产品+日期+业务模式，匹配价格管理中维护的红线价
                // 促销选品价（元）	根据客户+产品，匹配价格管理中维护的促销选品价中的最低价格
                try {
                    PriceFeeVo priceFeeVo = priceVoService.findByGoodsCode(entity.getGoodsCode(), entity.getCustomerCode(), entity.getBusinessModelCode(), yearMonthLy + "-01");
                    if (null != priceFeeVo) {
                        entity.setSuggestedRetailPrice(priceFeeVo.getStandardRetailPrice());
                        entity.setRedLinePrice(priceFeeVo.getRedLinePrice());
                        entity.setPromotionPrice(priceFeeVo.getPromotionalSelectionPrice());
                    }
                } catch (Exception error) {
                    log.error("日价格监控价格查询失败：" + error.getMessage(), e);
                }
                // 预估最低价（元）	根据业态+渠道+销售机构+客户+产品+日期，匹配日预估价格管理中的的预估最低价数据，若匹配到多条，则取多条中的最小数据;
//                ActivityDailyEstimatedPriceDto activityDailyEstimatedPriceDto = new ActivityDailyEstimatedPriceDto();
//                activityDailyEstimatedPriceDto.setEstoreChannel(entity.getEstoreChannel());
//                activityDailyEstimatedPriceDto.setCustomerCode(entity.getCustomerCode());
//                activityDailyEstimatedPriceDto.setProductCode(entity.getGoodsCode());
//                activityDailyEstimatedPriceDto.setMonitorDateStr(yearMonthLy);
//                BigDecimal estimateLowerPrice = activityDailyEstimatedPriceVoService.getMinPriceByCondition(activityDailyEstimatedPriceDto);
//                entity.setEstimateLowerPrice(estimateLowerPrice);
                // 促销规划价格（元）	根据客户+产品+日期匹配 促销规划 当月销售 的 活动底价 数据，若匹配到多条，则取多条中的最小数据
                CurrentMonthSaleDto currentMonthSaleDto = new CurrentMonthSaleDto();
                currentMonthSaleDto.setErpCode(entity.getCustomerCode());
                currentMonthSaleDto.setProductCode(entity.getGoodsCode());
                currentMonthSaleDto.setMonitorMonthStr(yearMonthLy);
                BigDecimal activityBasePrice = promotionPlanService.getMinActivityBasePrice(currentMonthSaleDto);
                entity.setActivityLowerPrice(activityBasePrice);
                // 实际销售价（元）	根据业态+渠道+销售机构+客户+产品+日期，查询平台销售数据管理表中的“总量/总金额“；未查到不显示
                // 实际销售价VS红线价	实际销售价-红线价；实际销售价没有值不计算
                // 实际销售价VS促销选品价	实际销售价-促销选品价；实际销售价没有值不计算
                // 实际销售价VS预估最低价	实际销售价-预估最低价；实际销售价没有值不计算
                // 实际销售价VS促销规划价格	实际销售价-促销规划价格；实际销售价没有值不计算
                if (null != entity.getSalePrice()) {
                    entity.setSaleVsRedLinePrice(entity.getSalePrice().subtract(Optional.ofNullable(entity.getRedLinePrice()).orElse(BigDecimal.ZERO)));
                    entity.setSaleVsPromotionPrice(entity.getSalePrice().subtract(Optional.ofNullable(entity.getPromotionPrice()).orElse(BigDecimal.ZERO)));
                    entity.setSaleVsEstimateLowerPrice(entity.getSalePrice().subtract(Optional.ofNullable(entity.getEstimateLowerPrice()).orElse(BigDecimal.ZERO)));
                    entity.setSaleVsActivityLowerPrice(entity.getSalePrice().subtract(Optional.ofNullable(entity.getActivityLowerPrice()).orElse(BigDecimal.ZERO)));
                }
                entity.setTenantCode(tenantCode);
                entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                entity.setPriceYearMonth(yearMonthLy);
                channelPriceList.add(entity);
            });
        });
        return channelPriceList;
    }

    @Transactional(rollbackFor = Exception.class, propagation = Propagation.NOT_SUPPORTED)
    public void saveDataList(List<ChannelPriceEntity> channelPriceList) {
        if (CollectionUtil.isEmpty(channelPriceList)) {
            return;
        }
        this.channelPriceRepository.saveOrUpdateBatch(channelPriceList);
    }

    @Transactional(rollbackFor = Exception.class, propagation = Propagation.NOT_SUPPORTED)
    public void removeByYearMonth(String yearAndMonth) {
        if (StringUtil.isEmpty(yearAndMonth)) {
            return;
        }
        // 删除旧的
        this.channelPriceRepository.removeByYearMonth(yearAndMonth);
    }
}
