package com.biz.crm.tpm.business.day.price.monitor.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.price.sdk.service.PriceVoService;
import com.biz.crm.mdm.business.price.sdk.vo.PriceFeeVo;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.day.price.monitor.local.entity.DayPriceMonitor;
import com.biz.crm.tpm.business.day.price.monitor.local.repository.DayPriceMonitorRepository;
import com.biz.crm.tpm.business.day.price.monitor.sdk.dto.DayPriceMonitorDto;
import com.biz.crm.tpm.business.day.price.monitor.sdk.service.DayPriceMonitorService;
import com.biz.crm.tpm.business.day.price.monitor.sdk.vo.DayPriceMonitorVo;
import com.biz.crm.tpm.business.day.sales.sdk.dto.TpmDaySalesSearchDto;
import com.biz.crm.tpm.business.day.sales.sdk.service.TpmDaySalesService;
import com.biz.crm.tpm.business.day.sales.sdk.vo.TpmDaySalesVo;
import com.biz.crm.tpm.business.promotion.plan.sdk.dto.CurrentMonthSaleDto;
import com.biz.crm.tpm.business.promotion.plan.sdk.service.PromotionPlanService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 日价格监控表(tpm_day_price_monitor)服务实现类
 *
 * @author : qiancheng
 * @date : 2022-11-11
 */
@Slf4j
@Service("dayPriceMonitorService")
public class DayPriceMonitorServiceImpl implements DayPriceMonitorService {

    @Autowired(required = false)
    private DayPriceMonitorRepository dayPriceMonitorRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private TpmDaySalesService tpmDaySalesService;
    @Autowired(required = false)
    private CustomerVoService customerVoService;
    @Autowired(required = false)
    private PriceVoService priceVoService;

//    @Autowired(required = false)
//    private ActivityDailyEstimatedPriceVoService activityDailyEstimatedPriceVoService;


    @Autowired(required = false)
    private PromotionPlanService promotionPlanService;


    /**
     * 分页查询数据
     * @param pageable        分页对象
     * @param dayPriceMonitor 实体对象
     * @return 分页对象
     */
    @Override
    public Page<DayPriceMonitorVo> findByConditions(Pageable pageable, DayPriceMonitorDto dayPriceMonitor) {
        if(pageable == null){
            pageable = PageRequest.of(1,20);
        }
        if(dayPriceMonitor == null){
            dayPriceMonitor = new DayPriceMonitorDto();
        }
        if(StringUtils.isBlank(dayPriceMonitor.getDelFlag())){
            dayPriceMonitor.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        }
        if(StringUtils.isBlank(dayPriceMonitor.getTenantCode())){
            dayPriceMonitor.setTenantCode(TenantUtils.getTenantCode());
        }

        return this.dayPriceMonitorRepository.findByConditions(pageable,dayPriceMonitor);
    }

    /**
     * 通过主键查询单条数据
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public DayPriceMonitorVo findById(String id) {
        Validate.notBlank(id,"主键不能够为空");
        DayPriceMonitor dayPriceMonitorEntity = dayPriceMonitorRepository.getById(id);
        return nebulaToolkitService.copyObjectByWhiteList(dayPriceMonitorEntity, DayPriceMonitorVo.class, HashSet.class, ArrayList.class);
    }


    /**
     * 从促销规划中获取数据
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void getDataFromPromotionPlan(DayPriceMonitorDto dto) {
        Validate.notBlank(dto.getMonitorDateStr(),"日期不能为空");
        Date startDate = DateUtil.getDateByFormat(dto.getMonitorDateStr(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        //先删掉历史数据
        dayPriceMonitorRepository.deleteByDate(dto.getMonitorDateStr());

        TpmDaySalesSearchDto tpmDaySalesSearchDto = new TpmDaySalesSearchDto();
        tpmDaySalesSearchDto.setStartDate(startDate);
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(startDate);
        calendar.add(Calendar.DATE,1);
        tpmDaySalesSearchDto.setEndDate(calendar.getTime());
        List<TpmDaySalesVo> dataFromPromotionPlan = tpmDaySalesService.getDataFromPromotionPlan(tpmDaySalesSearchDto);
        if (CollectionUtils.isEmpty(dataFromPromotionPlan)){
            return;
        }
        List<String> customerCodeList = dataFromPromotionPlan.stream()
                .filter(item -> StringUtils.isNotEmpty(item.getSalesInstitutionCode()))
                .filter(item -> StringUtils.isNotEmpty(item.getCustomer()))
                .map(item -> {
                    //客户编码的拼接规则：ERP编码+机构+渠道+业态
                    return item.getCustomer() + item.getSalesInstitutionCode() + item.getChannelCode() + item.getBusinessFormatCode();
                }).distinct().collect(Collectors.toList());
        Map<String,CustomerVo> customerVoMap = Maps.newHashMap();
        if (!CollectionUtils.isEmpty(customerCodeList)){
            List<CustomerVo> customerList = customerVoService.findBaseByCustomerCodes(customerCodeList);
            if (!CollectionUtils.isEmpty(customerList)){
                customerVoMap = customerList.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, Function.identity(),(o,n)->n));
            }
        }

        List<DayPriceMonitor> dayPriceMonitorList = new ArrayList<>();

        for (TpmDaySalesVo tpmDaySalesVo : dataFromPromotionPlan) {
            DayPriceMonitor dayPriceMonitor = new DayPriceMonitor();
            dayPriceMonitor.setBusinessFormatCode(tpmDaySalesVo.getBusinessFormatCode());
            dayPriceMonitor.setChannelCode(tpmDaySalesVo.getChannelCode());
            dayPriceMonitor.setChannelName(tpmDaySalesVo.getChannelName());
            dayPriceMonitor.setSalesOrgCode(tpmDaySalesVo.getSalesInstitutionCode());
            dayPriceMonitor.setSalesOrgName(tpmDaySalesVo.getSalesInstitutionName());
            dayPriceMonitor.setCustomerErpCode(tpmDaySalesVo.getCustomer());
            //客户编码的拼接规则：ERP编码+机构+渠道+业态
            String customerCode = dayPriceMonitor.getCustomerErpCode() + dayPriceMonitor.getSalesOrgCode() + dayPriceMonitor.getChannelCode() + dayPriceMonitor.getBusinessFormatCode();
            dayPriceMonitor.setCustomerCode(customerCode);
            dayPriceMonitor.setCustomerName(tpmDaySalesVo.getCustomerName());
            dayPriceMonitor.setProductCode(tpmDaySalesVo.getGoodsCode());
            dayPriceMonitor.setProductName(tpmDaySalesVo.getGoodsName());
            dayPriceMonitor.setQuantity(tpmDaySalesVo.getNum());
            dayPriceMonitor.setAmount(tpmDaySalesVo.getAmount());
            if (null != dayPriceMonitor.getQuantity()
                    && dayPriceMonitor.getQuantity().compareTo(BigDecimal.ZERO) != 0) {
                dayPriceMonitor.setSalePrice(Optional.ofNullable(dayPriceMonitor.getAmount()).orElse(BigDecimal.ZERO).divide(dayPriceMonitor.getQuantity(), 6, BigDecimal.ROUND_DOWN));
            }
            dayPriceMonitor.setMonitorDate(startDate);

            if (StringUtils.isNotEmpty(dayPriceMonitor.getCustomerCode())){
                CustomerVo customerVo = customerVoMap.get(dayPriceMonitor.getCustomerCode());
                if (null != customerVo){
//                业务模式	根据客户+渠道+业态，匹配客户管理中的业务模式
//                平台	根据客户+渠道+业态，匹配客户管理中的电商平台
                    dayPriceMonitor.setBusinessModel(customerVo.getBusinessModelCode());
                    dayPriceMonitor.setPlatformCode(customerVo.getEstorePlatform());
                }
            }

//            标准零售价（元）	根据产品+日期，匹配价格管理中维护的标准零售价
//            红线价（元）	根据产品+日期+业务模式，匹配价格管理中维护的红线价
//            促销选品价（元）	根据客户+产品，匹配价格管理中维护的促销选品价中的最低价格
            try {
                PriceFeeVo priceFeeVo = priceVoService.findByGoodsCode(dayPriceMonitor.getProductCode(), dayPriceMonitor.getCustomerErpCode(), dayPriceMonitor.getBusinessModel(), dto.getMonitorDateStr());
                if (null != priceFeeVo){
                    dayPriceMonitor.setSuggestedRetailPrice(priceFeeVo.getStandardRetailPrice());
                    dayPriceMonitor.setRedLinePrice(priceFeeVo.getRedLinePrice());
                    dayPriceMonitor.setPromotionPrice(priceFeeVo.getPromotionalSelectionPrice());
                }
            }catch (Exception e){
                log.error("日价格监控价格查询失败："+e.getMessage(),e);
            }


//            预估最低价（元）	根据业态+渠道+销售机构+客户+产品+日期，匹配日预估价格管理中的的预估最低价数据，若匹配到多条，则取多条中的最小数据;
//            ActivityDailyEstimatedPriceDto activityDailyEstimatedPriceDto = new ActivityDailyEstimatedPriceDto();
//            activityDailyEstimatedPriceDto.setEstoreChannel(dayPriceMonitor.getChannelCode());
//            activityDailyEstimatedPriceDto.setCustomerCode(dayPriceMonitor.getCustomerCode());
//            activityDailyEstimatedPriceDto.setProductCode(dayPriceMonitor.getProductCode());
//            activityDailyEstimatedPriceDto.setMonitorDateStr(dto.getMonitorDateStr());
//            BigDecimal estimateLowerPrice = activityDailyEstimatedPriceVoService.getMinPriceByCondition(activityDailyEstimatedPriceDto);
//            dayPriceMonitor.setEstimateLowerPrice(estimateLowerPrice);
//            促销规划价格（元）	根据客户+产品+日期匹配 促销规划 当月销售 的 活动底价 数据，若匹配到多条，则取多条中的最小数据
            CurrentMonthSaleDto currentMonthSaleDto = new CurrentMonthSaleDto();
            currentMonthSaleDto.setCustomerCode(dayPriceMonitor.getCustomerCode());
            currentMonthSaleDto.setProductCode(dayPriceMonitor.getProductCode());
            currentMonthSaleDto.setMonitorDateStr(dto.getMonitorDateStr());
            BigDecimal activityBasePrice = promotionPlanService.getMinActivityBasePrice(currentMonthSaleDto);
            dayPriceMonitor.setActivityLowerPrice(activityBasePrice);


//            实际销售价（元）	根据业态+渠道+销售机构+客户+产品+日期，查询平台销售数据管理表中的“总量/总金额“；未查到不显示
//            实际销售价VS红线价	实际销售价-红线价；实际销售价没有值不计算
//            实际销售价VS促销选品价	实际销售价-促销选品价；实际销售价没有值不计算
//            实际销售价VS预估最低价	实际销售价-预估最低价；实际销售价没有值不计算
//            实际销售价VS促销规划价格	实际销售价-促销规划价格；实际销售价没有值不计算
            if (null != dayPriceMonitor.getSalePrice()){
                dayPriceMonitor.setSaleVsRedPrice(dayPriceMonitor.getSalePrice().subtract(Optional.ofNullable(dayPriceMonitor.getRedLinePrice()).orElse(BigDecimal.ZERO)));
                dayPriceMonitor.setSaleVsPromotionPrice(dayPriceMonitor.getSalePrice().subtract(Optional.ofNullable(dayPriceMonitor.getPromotionPrice()).orElse(BigDecimal.ZERO)));
                dayPriceMonitor.setSaleVsEslowerPrice(dayPriceMonitor.getSalePrice().subtract(Optional.ofNullable(dayPriceMonitor.getEstimateLowerPrice()).orElse(BigDecimal.ZERO)));
                dayPriceMonitor.setSaleVsLowerPrice(dayPriceMonitor.getSalePrice().subtract(Optional.ofNullable(dayPriceMonitor.getActivityLowerPrice()).orElse(BigDecimal.ZERO)));
            }
            dayPriceMonitor.setTenantCode(TenantUtils.getTenantCode());
            dayPriceMonitorList.add(dayPriceMonitor);
        }
        if(dayPriceMonitorList.size() > 0){
            dayPriceMonitorRepository.saveOrUpdateBatch(dayPriceMonitorList);
        }
    }

    @Override
    public void delete(List<String> ids) {
        dayPriceMonitorRepository.deleteByIds(ids);
    }
}
