package com.biz.crm.tpm.business.day.sales.local.controller;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.day.sales.sdk.constants.DaySalesConstants;
import com.biz.crm.tpm.business.day.sales.sdk.dto.TpmDaySalesSearchDto;
import com.biz.crm.tpm.business.day.sales.sdk.dto.TpmDaySalesSpliceDto;
import com.biz.crm.tpm.business.day.sales.sdk.service.TpmDaySalesService;
import com.biz.crm.tpm.business.day.sales.sdk.vo.TpmDaySalesVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

import java.util.Date;
import java.util.List;

/**
 * 日销售数据管理
 *
 * @author yaoyongming
 * @since 2022-10-27 16:50:54
 */
@RestController
@RequestMapping("/v1/tpmDaySales")
@Slf4j
@Api(tags = "日销售数据管理")
public class TpmDaySalesController {

    /**
     * 服务对象
     */
    @Autowired(required = false)
    private TpmDaySalesService tpmDaySalesService;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 分页查询所有数据
     *
     * @param pageable    分页对象
     * @param tpmDaySales 查询DTO
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findByConditions")
    public Result<Page<TpmDaySalesVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                        @ApiParam(name = "tpmDaySales", value = "日销售数据DTO") TpmDaySalesSearchDto tpmDaySales) {
        try {
            Page<TpmDaySalesVo> page = this.tpmDaySalesService.findByConditions(pageable, tpmDaySales);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 查询所有数据
     *
     * @param tpmDaySales 查询所有数据
     */
    @ApiOperation(value = "查询所有数据")
    @PostMapping("findByConditionsNoPage")
    public Result<List<TpmDaySalesVo>> findByConditionsNoPage(@ApiParam(name = "tpmDaySales", value = "日销售数据DTO") @RequestBody TpmDaySalesSearchDto tpmDaySales) {
        try {
            return Result.ok(this.tpmDaySalesService.findByConditionsNoPage(tpmDaySales));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过主键列表查询多条数据
     *
     * @param idList 主键列表
     * @return 多条数据
     */
    @ApiOperation(value = "通过主键列表查询多条数据")
    @PatchMapping("findByIdList")
    public Result<?> findByIdList(@ApiParam(name = "idList", value = "主键列表") @RequestBody List<String> idList) {
        try {
            List<TpmDaySalesVo> tpmDaySales = this.tpmDaySalesService.findByIdList(idList);
            return Result.ok(tpmDaySales);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 拆分
     *
     * @param dto 日销售拆分DTO列表
     * @return
     */
    @ApiOperation(value = "拆分")
    @PostMapping("splice")
    public Result<?> splice(@ApiParam(name = "dto", value = "日销售拆分DTO") @RequestBody TpmDaySalesSpliceDto dto) {
        try {
            this.tpmDaySalesService.splice(dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 批量拆分
     *
     * @param dtoList 日销售拆分DTO列表
     * @return
     */
    @ApiOperation(value = "批量拆分")
    @PostMapping("spliceBatch")
    public Result<?> spliceBatch(@ApiParam(name = "dtoList", value = "日销售拆分DTO列表") @RequestBody List<TpmDaySalesSpliceDto> dtoList) {
        try {
            this.tpmDaySalesService.spliceBatch(dtoList);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 新增数据
     *
     * @param tpmDaySalesVo VO对象
     * @return 新增结果
     */
    @ApiOperation(value = "新增数据")
    @PostMapping
    public Result<?> create(@ApiParam(name = "tpmDaySalesVo", value = "日销售数据") @RequestBody TpmDaySalesVo tpmDaySalesVo) {
        try {
            this.tpmDaySalesService.create(tpmDaySalesVo);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 同步指定月份销售数据
     *
     * @param yearMonth 年月
     * @return 新增结果
     */
    @ApiOperation(value = "同步指定月份销售数据")
    @GetMapping("/getSalesData")
    public Result<?> getSalesData(@RequestParam("yearMonth") String yearMonth) {
        try {
            Assert.hasLength(yearMonth, "年月参数不能为空!");
            try {
                Date date = DateUtil.date_yyyy_MM.parse(yearMonth);
                Assert.isTrue(yearMonth.equals(DateUtil.date_yyyy_MM.format(date)), "年月格式错误【yyyy-MM】");
            } catch (Exception e) {
                throw new IllegalArgumentException("年月格式错误【yyyy-MM】");
            }
            boolean isLock = redisLockService.isLock(DaySalesConstants.TPM_SALES_DATA_LOCK);
            Assert.isTrue(!isLock, "其他人正在操作数据,加锁失败,请稍后重试!");
            this.tpmDaySalesService.getSalesData(yearMonth, loginUserService.getAbstractLoginUser());
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 删除数据
     *
     * @param ids 主键集合
     * @return 删除结果
     */
    @ApiOperation(value = "删除数据")
    @DeleteMapping
    public Result<?> delete(@ApiParam(name = "ids", value = "主键集合") @RequestParam("ids") List<String> ids) {
        try {
            this.tpmDaySalesService.delete(ids);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
}
