package com.biz.crm.tpm.business.day.sales.local.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.channel.sdk.service.CustomerChannelVoService;
import com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.tpm.business.day.sales.sdk.vo.DaySalesImportVo;
import com.biz.crm.tpm.business.day.sales.sdk.service.TpmDaySalesService;
import com.biz.crm.tpm.business.day.sales.sdk.vo.TpmDaySalesVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * TPM日销售数据导入
 *
 * @author: yaoyongming
 * @date: 2022/10/28 17:12
 */
@Component
@Slf4j
public class DaySalesImportsProcess implements ImportProcess<DaySalesImportVo> {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private TpmDaySalesService tpmDaySalesService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private ProductVoService productVoService;

    @Autowired(required = false)
    private CustomerChannelVoService customerChannelVoService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    private static final String MDM_BUSINESS_FORMAT = "mdm_business_format";

    /**
     * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
     * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
     *
     * @return
     */
    @Override
    public boolean importBeforeValidationFlag() {
        return Boolean.TRUE;
    }

    /**
     * 数据校验
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> tryVerify(LinkedHashMap<Integer, DaySalesImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        for (Map.Entry<Integer, DaySalesImportVo> row : data.entrySet()) {
            int rowNum = row.getKey();
            DaySalesImportVo vo = row.getValue();
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getCustomer()), "客户编码不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getGoodsCode()), "产品编码不能为空！");
            this.validateIsTrue(vo.getTradeDate() != null, "交易日期不能为空！");
            this.validateIsTrue(vo.getAmount() != null, "金额不能为空！");
            this.validateIsTrue(vo.getNum() != null, "数量不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getSalesInstitutionCode()), "销售组织编码不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getChannelCode()), "渠道编码不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getBusinessFormatCode()), "业态不能为空！");
            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(rowNum, errInfo);
            }
        }
        return errMap;
    }

    /**
     * 数据处理
     *
     * @param data
     * @param paramsVo
     * @param params
     * @return
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, DaySalesImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        try {
            Validate.notEmpty(data, "导入数据不能为空！");
            Map<Integer, String> errMap = new HashMap<>();
            List<TpmDaySalesVo> importList = this.validate(data, errMap);

            if (errMap.isEmpty()) {
                tpmDaySalesService.createBatch(importList);
            }
            return errMap;
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            e.printStackTrace();
            throw e;
        }
    }

    private List<TpmDaySalesVo> validate(LinkedHashMap<Integer, DaySalesImportVo> data, Map<Integer, String> errMap) {
        //关联数据查询
        Map<String, List<CustomerVo>> customerVoMap = new HashMap<>();
        Map<String, ProductVo> productVoMap = new HashMap<>();
        Map<String, List<SalesOrgVo>> orgVoMap = new HashMap<>();
        Map<String, CustomerChannelVo> channelVoMap = new HashMap<>();
        Set<String> customerCodeSet = new HashSet<>();
        Set<String> productCodeSet = new HashSet<>();
        Set<String> channelCodeSet = new HashSet<>();
        Set<String> orgCodeSet = new HashSet<>();
        for (DaySalesImportVo dto : data.values()) {
            customerCodeSet.add(dto.getCustomer());
            productCodeSet.add(dto.getGoodsCode());
            channelCodeSet.add(dto.getChannelCode());
            orgCodeSet.add(dto.getSalesInstitutionCode());
        }
        // 查询数据字典
        Map<String, List<DictDataVo>> dictDataMap = dictDataVoService.findByDictTypeCodeList(Arrays.asList(MDM_BUSINESS_FORMAT));
        //查询渠道
        List<CustomerChannelVo> channelList = customerChannelVoService.findByCodes(new ArrayList<>(channelCodeSet));
        if (!CollectionUtils.isEmpty(channelList)) {
            channelVoMap = channelList.stream().collect(Collectors.toMap(e -> e.getCustomerChannelCode(), Function.identity()));
        }
        // 客户
        List<CustomerVo> customerVos = customerVoService.findByCustomerMdgCodes(new ArrayList<>(customerCodeSet));
        if (!CollectionUtils.isEmpty(customerVos)) {
            customerVoMap = customerVos.stream().collect(Collectors.groupingBy(e -> e.getErpCode()));
        }
        // 产品
        List<ProductVo> productVoList = productVoService.findDetailsByIdsOrProductCodes(null, new ArrayList<>(productCodeSet));
        if (!CollectionUtils.isEmpty(productVoList)) {
            productVoMap = productVoList.stream().collect(Collectors.toMap(ProductVo::getProductCode, Function.identity()));
        }
        //销售机构
        List<SalesOrgVo> orgVos = salesOrgVoService.findByErpCodeList(new ArrayList<>(orgCodeSet));
        if (org.apache.commons.collections.CollectionUtils.isNotEmpty(orgVos)) {
            orgVoMap = orgVos.stream().collect(Collectors.groupingBy(SalesOrgVo::getErpCode));
        }
        //关联数据校验
        for (Map.Entry<Integer, DaySalesImportVo> row : data.entrySet()) {
            if (productVoMap.containsKey(row.getValue().getGoodsCode())) {
                row.getValue().setGoodsName(productVoMap.get(row.getValue().getGoodsCode()).getProductName());
            } else {
                this.validateIsTrue(false, "产品编码【" + row.getValue().getGoodsCode() + "】错误，请检查！");
            }
            //业态
            DictDataVo promotionType = dictDataMap.get(MDM_BUSINESS_FORMAT).stream().filter(e -> e.getDictValue().equals(row.getValue().getBusinessFormatCode())).findFirst().orElse(null);
            if (promotionType != null) {
                row.getValue().setBusinessFormatCode(promotionType.getDictCode());
            } else {
                this.validateIsTrue(false, "业态【" + row.getValue().getBusinessFormatCode() + "】错误，请检查！");
            }
            if (channelVoMap.containsKey(row.getValue().getChannelCode())) {
                row.getValue().setChannelName(channelVoMap.get(row.getValue().getChannelCode()).getCustomerChannelName());
            } else {
                this.validateIsTrue(false, "渠道编码【" + row.getValue().getChannelCode() + "】错误，请检查！");
            }
            if (orgVoMap.containsKey(row.getValue().getSalesInstitutionCode())) {
                List<SalesOrgVo> list = orgVoMap.get(row.getValue().getSalesInstitutionCode());
                SalesOrgVo salesOrgVo = list.stream().filter(e -> row.getValue().getChannelCode().equals(e.getChannelCode()) && row.getValue().getBusinessFormatCode().equals(e.getBusinessFormatCode()))
                        .findFirst().orElse(null);
                if (salesOrgVo != null) {
                    row.getValue().setSalesInstitutionName(salesOrgVo.getSalesOrgName());
                    row.getValue().setSalesInstitutionCode(salesOrgVo.getSalesOrgCode());
                } else {
                    this.validateIsTrue(false, "销售机构编码【" + row.getValue().getSalesInstitutionCode() + "】错误，请检查！");
                }
            } else {
                this.validateIsTrue(false, "销售机构编码【" + row.getValue().getSalesInstitutionCode() + "】错误，请检查！");
            }
            if (customerVoMap.containsKey(row.getValue().getCustomer())) {
                List<CustomerVo> list = customerVoMap.get(row.getValue().getCustomer());
                CustomerVo customerVo = list.stream().filter(e -> row.getValue().getChannelCode().equals(e.getCustomerChannelCode()) && row.getValue().getSalesInstitutionCode().equals(e.getSalesInstitutionErpCode()) && row.getValue().getBusinessFormatCode().equals(e.getBusinessFormatCode()))
                        .findFirst().orElse(null);
                if (customerVo == null) {
                    this.validateIsTrue(false, "客户编码【" + row.getValue().getCustomer() + "】未找到对应客户，请检查！");
                } else {
                    row.getValue().setCustomerName(customerVo.getCustomerName());
                }
            } else {
                this.validateIsTrue(false, "客户编码【" + row.getValue().getCustomer() + "】错误，请检查！");
            }

            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(row.getKey(), errInfo);
            }
        }

        if (!errMap.isEmpty()) {
            return new ArrayList<>();
        }

        List<TpmDaySalesVo> dtoList = new ArrayList<>(nebulaToolkitService.copyCollectionByWhiteList(data.values(), DaySalesImportVo.class, TpmDaySalesVo.class, LinkedHashSet.class, ArrayList.class));
        return dtoList;
    }

    /**
     * 获取数据实体
     *
     * @return
     */
    @Override
    public Class<DaySalesImportVo> findCrmExcelVoClass() {
        return DaySalesImportVo.class;
    }

    /**
     * 获取业务对应的模板编码，全局唯一
     *
     * @return
     */
    @Override
    public String getTemplateCode() {
        return "TMP_DAY_SALES_IMPORTS";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return
     */
    @Override
    public String getTemplateName() {
        return "TPM日销售数据导入";
    }

    /**
     * 单次缓存的数量
     *
     * @return
     */
    @Override
    public Integer getBatchCount() {
        return 10000;
    }
}
