package com.biz.crm.tpm.business.day.sales.local.service.internal;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.tpm.business.day.sales.sdk.vo.DaySalesSpliceImportVo;
import com.biz.crm.tpm.business.day.sales.sdk.service.TpmDaySalesSpliceService;
import com.biz.crm.tpm.business.day.sales.sdk.vo.TpmDaySalesSpliceVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;

/**
 * TPM日销售数据拆分明细导入
 *
 * @author: yaoyongming
 * @date: 2022/10/28 17:13
 */
@Component
@Slf4j
public class DaySalesSpliceImportsProcess implements ImportProcess<DaySalesSpliceImportVo> {

    @Autowired
    private NebulaToolkitService nebulaToolkitService;

    @Autowired
    private TpmDaySalesSpliceService tpmDaySalesSpliceService;

    /**
     * 单次缓存的数量
     *
     * @return
     */
    @Override
    public Integer getBatchCount() {
        return Integer.MAX_VALUE;
    }

    /**
     * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
     * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
     *
     * @return
     */
    @Override
    public boolean importBeforeValidationFlag() {
        return Boolean.TRUE;
    }

    /**
     * 数据校验
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> tryVerify(LinkedHashMap<Integer, DaySalesSpliceImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Validate.notEmpty(data, "导入数据不能为空！");
        Map<Integer, String> errMap = new HashMap<>();
        for (Map.Entry<Integer, DaySalesSpliceImportVo> row : data.entrySet()) {
            int rowNum = row.getKey();
            DaySalesSpliceImportVo vo = row.getValue();
            this.validateIsTrue(vo.getAmount() != null, "金额不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getCustomer()), "客户编码不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getCustomerName()), "客户名称不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getGoodsCode()), "货品编号不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getGoodsName()), "货品名称不能为空！");
            this.validateIsTrue(vo.getNum() != null, "数量不能为空！");
            this.validateIsTrue(vo.getTradeDate() != null, "交易日期不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getActivityFormName()), "活动形式名称不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getChannel()), "渠道不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getBusinessFormatName()), "业态不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getChannelCode()), "渠道不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getSalesInstitutionCode()), "销售机构编码不能为空！");
            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(rowNum, errInfo);
            }
        }
        return errMap;
    }

    /**
     * 数据处理
     *
     * @param data
     * @param paramsVo
     * @param params
     * @return
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, DaySalesSpliceImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        List<DaySalesSpliceImportVo> importList = new ArrayList<>(data.values());

        Collection<TpmDaySalesSpliceVo> tpmDaySalesVos = nebulaToolkitService.copyCollectionByBlankList(importList, DaySalesSpliceImportVo.class, TpmDaySalesSpliceVo.class, HashSet.class, ArrayList.class);

        tpmDaySalesSpliceService.createBatch(tpmDaySalesVos);
        return null;
    }

    /**
     * 获取数据实体
     *
     * @return
     */
    @Override
    public Class<DaySalesSpliceImportVo> findCrmExcelVoClass() {
        return DaySalesSpliceImportVo.class;
    }

    /**
     * 获取业务对应的模板编码，全局唯一
     *
     * @return
     */
    @Override
    public String getTemplateCode() {
        return "TMP_DAY_SALES_SPLICE_IMPORTS";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return
     */
    @Override
    public String getTemplateName() {
        return "TPM日销售数据拆分明细导入";
    }

}
