package com.biz.crm.tpm.business.day.sales.local.service.internal;


import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.customer.channel.sdk.service.CustomerChannelVoService;
import com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerDto;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.day.sales.local.entity.TpmDaySales;
import com.biz.crm.tpm.business.day.sales.local.entity.TpmDaySalesSplice;
import com.biz.crm.tpm.business.day.sales.local.repository.TpmDaySalesRepository;
import com.biz.crm.tpm.business.day.sales.local.repository.TpmDaySalesSpliceRepository;
import com.biz.crm.tpm.business.day.sales.sdk.constants.DaySalesConstants;
import com.biz.crm.tpm.business.day.sales.sdk.dto.TpmDaySalesSearchDto;
import com.biz.crm.tpm.business.day.sales.sdk.dto.TpmDaySalesSpliceDto;
import com.biz.crm.tpm.business.day.sales.sdk.dto.log.DaySalesLogEventDto;
import com.biz.crm.tpm.business.day.sales.sdk.event.log.DaySalesLogEventListener;
import com.biz.crm.tpm.business.day.sales.sdk.service.TpmDaySalesService;
import com.biz.crm.tpm.business.day.sales.sdk.vo.TpmDaySalesVo;
import com.biz.crm.tpm.business.third.system.sdk.dto.Ce1MnjtFullDto;
import com.biz.crm.tpm.business.third.system.sdk.service.Ce1MnjtSdkService;
import com.biz.crm.tpm.business.third.system.sdk.vo.Ce1MnjtFullVo2;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import com.google.common.collect.Lists;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 日销售数据实体类(TpmDaySales)表服务实现类
 *
 * @author yaoyongming
 * @since 2022-10-27 16:50:56
 */
@Slf4j
@Service("tpmDaySalesService")
public class TpmDaySalesServiceImpl implements TpmDaySalesService {

    @Autowired(required = false)
    private TpmDaySalesRepository tpmDaySalesRepository;

    @Autowired(required = false)
    private TpmDaySalesSpliceRepository tpmDaySalesSpliceRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private Ce1MnjtSdkService ce1MnjtSdkService;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    @Autowired(required = false)
    private CustomerChannelVoService customerChannelVoService;

    @Autowired(required = false)
    private ProductVoService productVoService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 分页查询数据
     *
     * @param pageable    分页对象
     * @param tpmDaySales 实体对象
     * @return
     */
    @Override
    public Page<TpmDaySalesVo> findByConditions(Pageable pageable, TpmDaySalesSearchDto tpmDaySales) {
        ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(tpmDaySales)) {
            tpmDaySales = new TpmDaySalesSearchDto();
        }
        return this.tpmDaySalesRepository.findByConditions(pageable, tpmDaySales);
    }

    /**
     * 查询数据
     *
     * @param tpmDaySales
     * @return
     */
    @Override
    public List<TpmDaySalesVo> findByConditionsNoPage(TpmDaySalesSearchDto tpmDaySales) {
        if (Objects.isNull(tpmDaySales)) {
            tpmDaySales = new TpmDaySalesSearchDto();
        }
        return this.tpmDaySalesRepository.findByConditionsNoPage(tpmDaySales);
    }

    /**
     * 查询数据
     *
     * @param tpmDaySales
     * @return
     */
    @Override
    public List<TpmDaySalesVo> findByConditionsNoPage2(TpmDaySalesSearchDto tpmDaySales) {
        if (Objects.isNull(tpmDaySales)) {
            tpmDaySales = new TpmDaySalesSearchDto();
        }
        return this.tpmDaySalesRepository.findByConditionsNoPage2(tpmDaySales);
    }

    @Override
    public BigDecimal getDaySalesTotalAmount(TpmDaySalesSearchDto daySalesSearch) {
        return this.tpmDaySalesRepository.getDaySalesTotalAmount(daySalesSearch);
    }

    /**
     * 通过主键列表查询多条数据
     *
     * @param idList 主键列表
     * @return 多条数据
     */
    @Override
    public List<TpmDaySalesVo> findByIdList(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "查询数据时，主键集合不能为空！");
        return Lists.newArrayList(
                nebulaToolkitService.copyCollectionByWhiteList(
                        this.tpmDaySalesRepository.findByIdList(idList), TpmDaySales.class, TpmDaySalesVo.class, HashSet.class, ArrayList.class));
    }

    /**
     * 拆分
     *
     * @param dto
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void splice(TpmDaySalesSpliceDto dto) {
        spliceExecute(Arrays.asList(dto));
    }

    /**
     * 批量拆分
     *
     * @param dtoList
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void spliceBatch(List<TpmDaySalesSpliceDto> dtoList) {
        spliceExecute(dtoList);
    }

    /**
     * 进行拆分
     *
     * @param dtoList
     */
    private void spliceExecute(List<TpmDaySalesSpliceDto> dtoList) {
        Set<String> ids = spliceValidate(dtoList);
        //查询日销售数据
        List<TpmDaySales> daySalesList = tpmDaySalesRepository.listByIds(ids);
        Validate.isTrue(daySalesList.size() == ids.size(), "未找到所要拆分的日销售数据！");

        //按日销售数据分组
        Map<String, List<TpmDaySalesSpliceDto>> map = dtoList.stream().collect(Collectors.groupingBy(TpmDaySalesSpliceDto::getParentId));
        List<TpmDaySalesSplice> spliceList = new ArrayList<>();
        map.forEach((k, v) -> {
            TpmDaySales daySales = daySalesList.stream().filter(e -> e.getId().equals(k)).findFirst().orElse(null);
            Integer num = 0;
            BigDecimal amount = BigDecimal.ZERO;
            for (TpmDaySalesSpliceDto dto : v) {
                num += dto.getNumSplice();
                amount = amount.add(dto.getAmountSplice());
                Validate.isTrue(num <= daySales.getNumEnable(), "拆分数量大于可拆分数量");
                Validate.isTrue(amount.compareTo(daySales.getAmountEnable()) <= 0, "拆分金额大于可拆分金额");

                TpmDaySalesSplice daySalesSplice = nebulaToolkitService.copyObjectByWhiteList(daySales, TpmDaySalesSplice.class, HashSet.class, ArrayList.class);
                daySalesSplice.setId(null);
                daySalesSplice.setNum(BigDecimal.valueOf(dto.getNumSplice()));
                daySalesSplice.setAmount(dto.getAmountSplice());
                daySalesSplice.setChannel(dto.getChannel());
                daySalesSplice.setActivityForm(dto.getActivityForm());
                daySalesSplice.setActivityFormName(dto.getActivityFormName());
                spliceList.add(daySalesSplice);
            }
            //更新日销售数据
            daySales.setNumEnable(daySales.getNumEnable() - num);
            daySales.setAmountEnable(daySales.getAmountEnable().subtract(amount));
            tpmDaySalesRepository.saveOrUpdate(daySales);
        });
        try {
            tpmDaySalesSpliceRepository.saveBatch(spliceList);
        } catch (DuplicateKeyException e) {
            throw new RuntimeException("存在重复的拆分明细，不允许拆分");
        }
    }

    /**
     * 批量拆分校验
     *
     * @param dtoList
     */
    Set<String> spliceValidate(List<TpmDaySalesSpliceDto> dtoList) {
        Set<String> idSet = new HashSet<>();
        Set<String> uniqueSet = new HashSet<>();
        dtoList.forEach(e -> {
            Validate.notNull(e, "对象信息不能为空！");
            Validate.notBlank(e.getParentId(), "日销售主键id不能为空！");
            Validate.notBlank(e.getChannel(), "渠道不能为空！");
            Validate.notBlank(e.getActivityForm(), "活动形式不能为空！");
            Validate.notNull(e.getAmountSplice(), "拆分金额不能为空！");
            Validate.notNull(e.getNumSplice(), "拆分数量不能为空！");
            Validate.notNull(e.getParentId(), "日销售主键id不能为空！");
            Validate.isTrue(e.getNumSplice() > 0 || e.getAmountSplice().compareTo(BigDecimal.ZERO) > 0, "拆分金额和拆分数量至少有一个大于0才允许拆分");

            idSet.add(e.getParentId());
            uniqueSet.add(e.getParentId() + "_" + e.getChannel() + "_" + e.getActivityForm());
        });
        Validate.isTrue(uniqueSet.size() == dtoList.size(), "相同日销售数据不可重复拆分！");
        return idSet;
    }

    /**
     * 新建
     *
     * @param daySalesVo
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void create(TpmDaySalesVo daySalesVo) {
        //重复校验有重复，做覆盖
        String id = createValidate(daySalesVo);

        store(daySalesVo, id);
    }

    /**
     * 批量新建
     *
     * @param daySalesVos
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void createBatch(Collection<TpmDaySalesVo> daySalesVos) {
        daySalesVos.forEach(e -> createImport(e));
    }

    @Override
    @Transactional(propagation = Propagation.NOT_SUPPORTED, rollbackFor = Exception.class)
    public void batchSave(List<TpmDaySalesVo> daySalesVos) {
        if (CollectionUtils.isEmpty(daySalesVos)) {
            return;
        }
        Collection<TpmDaySales> tpmDaySales = this.nebulaToolkitService.copyCollectionByBlankList(daySalesVos, TpmDaySalesVo.class, TpmDaySales.class, LinkedHashSet.class, ArrayList.class);
        tpmDaySales.forEach(item -> item.setTenantCode(TenantUtils.getTenantCode()));
        this.tpmDaySalesRepository.saveBatch(tpmDaySales);
    }

    @Override
    @Transactional(propagation = Propagation.NOT_SUPPORTED, rollbackFor = Exception.class)
    public void deleteByVerifyCodes(List<String> verifyCodes) {
        if (CollectionUtils.isEmpty(verifyCodes)) {
            return;
        }
        this.tpmDaySalesRepository.deleteByVerifyCode(verifyCodes);
    }

    @Override
    public List<TpmDaySalesVo> getDataFromPromotionPlan(TpmDaySalesSearchDto dto) {
        Validate.isTrue(null != dto.getStartDate() && null != dto.getEndDate(), "开始结束时间不能为空");
        return tpmDaySalesRepository.getDataFromPromotionPlan(dto);
    }

    /**
     * 同步指定月份销售数据
     *
     * @param yearMonth
     */
    @Override
    @Async
    public void getSalesData(String yearMonth, AbstractCrmUserIdentity userIdentity) {
        boolean lockSuccess = redisLockService.tryLock(DaySalesConstants.TPM_SALES_DATA_LOCK, TimeUnit.HOURS, 4);
        Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
        try {
            loginUserService.refreshAuthentication(userIdentity);
            this.findSalesData(yearMonth);
        } finally {
            redisLockService.unlock(DaySalesConstants.TPM_SALES_DATA_LOCK);
        }
    }

    @Override
    @DynamicTaskService(cornExpression = "0 0 1 * * ?", taskDesc = "凌晨1点平台销售数据管理同步日销售数据")
    @Transactional(propagation = Propagation.NOT_SUPPORTED, rollbackFor = Exception.class)
    public void synchronizationSalesData() {
        loginUserService.refreshAuthentication(null);
        boolean lockSuccess = redisLockService.tryLock(DaySalesConstants.TPM_SALES_DATA_LOCK, TimeUnit.HOURS, 4);
        Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
        try {
            this.findSalesData(DateUtil.date_yyyy_MM.format(new Date()));
        } finally {
            redisLockService.unlock(DaySalesConstants.TPM_SALES_DATA_LOCK);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选择要删除的数据！");
        tpmDaySalesRepository.removeByIds(ids);
    }

    private void findSalesData(String yearMonth) {
        Assert.hasLength(yearMonth, "年月参数不能为空!");
        try {
            Date date = DateUtil.date_yyyy_MM.parse(yearMonth);
            Assert.isTrue(yearMonth.equals(DateUtil.date_yyyy_MM.format(date)), "年月格式错误【yyyy-MM】");
        } catch (Exception e) {
            throw new IllegalArgumentException("年月格式错误【yyyy-MM】");
        }
        List<String> businessModelCodeList = new ArrayList<>();
        CustomerDto customerDto = new CustomerDto();
        businessModelCodeList.add(DaySalesConstants.SELF_OPERATED_MODE);
        businessModelCodeList.add(DaySalesConstants.DISTRIBUTION_MODE);
        customerDto.setBusinessModelCodeList(businessModelCodeList);
        List<CustomerVo> customerVos = customerVoService.getCustomerByConditions(customerDto);
        Validate.isTrue(!CollectionUtils.isEmpty(customerVos), "未查询到主数据的自营客户信息");
        List<String> codeList = customerVos.stream().map(CustomerVo::getErpCode).filter(StringUtils::isNotBlank).collect(Collectors.toList());
        Validate.isTrue(!CollectionUtils.isEmpty(codeList), "未查询到主数据的自营客户MDG编码");
        Map<String, String> customerMap = customerVos.stream().filter(vo -> !StringUtils.isEmpty(vo.getErpCode())).collect(Collectors.toMap(CustomerVo::getErpCode, CustomerVo::getCustomerName, (a, b) -> a));
        Ce1MnjtFullDto ce1MnjtFullDto = new Ce1MnjtFullDto();
        ce1MnjtFullDto.setYearMonth(yearMonth.replace("-", ""));
        ce1MnjtFullDto.setCodeList(codeList);
        Pageable pageable = PageRequest.of(1, 800);
        Page<Ce1MnjtFullVo2> pageResult = null;
        String tenantCode = TenantUtils.getTenantCode();
        String code = DelFlagStatusEnum.NORMAL.getCode();
        this.removeByYearMonth(yearMonth);
        do {
            pageResult = ce1MnjtSdkService.findSalesByConditions(pageable, ce1MnjtFullDto);
            pageable = pageable.next();
            log.info("=====>    查询到日销售数据 年月[{}]  页[{}/{}][{}]条    <=====", yearMonth, pageResult.getCurrent(), pageResult.getPages(), pageResult.getSize());
            buildAndSaveData(pageResult.getRecords(), tenantCode, code, customerMap, yearMonth);
        } while (pageResult.hasNext());

    }

    @Transactional(rollbackFor = Exception.class, propagation = Propagation.NOT_SUPPORTED)
    public void removeByYearMonth(String yearMonth) {
        if (StringUtil.isEmpty(yearMonth)) {
            return;
        }
        tpmDaySalesRepository.removeByYearMonth(yearMonth);
    }

    /**
     * 构建参数
     *
     * @param records
     */
    private void buildAndSaveData(List<Ce1MnjtFullVo2> records, String tenantCode, String
            code, Map<String, String> customerMap, String yearMonth) {
        List<String> channelCodeList = records.stream().map(Ce1MnjtFullVo2::getDistributionChannelCode).filter(StringUtils::isNotBlank).collect(Collectors.toList());
        List<String> companyCodeList = records.stream().map(Ce1MnjtFullVo2::getCompanyCode).filter(StringUtils::isNotBlank).collect(Collectors.toList());
        List<String> productCodeList = records.stream().map(Ce1MnjtFullVo2::getProductCode).filter(StringUtils::isNotBlank).collect(Collectors.toList());
        Map<String, SalesOrgVo> salesInstitutionMap = new HashMap<>();
        Map<String, List<CustomerChannelVo>> customerChannelMap = new HashMap<>();
        Map<String, List<ProductVo>> productMap = new HashMap<>();
        List<SalesOrgVo> salesInstitutions = salesOrgVoService.findBySalesOrgCodesPost(companyCodeList);
        if (CollectionUtil.isNotEmpty(salesInstitutions)) {
            salesInstitutionMap.putAll(salesInstitutions.stream().collect(Collectors.toMap(SalesOrgVo::getSalesOrgCode, v -> v, (n, o) -> n)));
        }
        List<CustomerChannelVo> byCodes = customerChannelVoService.findByCodes(channelCodeList);
        if (CollectionUtil.isNotEmpty(byCodes)) {
            customerChannelMap.putAll(byCodes.stream().collect(Collectors.groupingBy(CustomerChannelVo::getCustomerChannelCode)));
        }
        List<ProductVo> byProductCodes = productVoService.findByCodes(productCodeList);
        if (CollectionUtil.isNotEmpty(byProductCodes)) {
            productMap.putAll(byProductCodes.stream().collect(Collectors.groupingBy(ProductVo::getProductCode)));
        }
        List<TpmDaySales> list = new ArrayList<>();
        for (Ce1MnjtFullVo2 record : records) {
            TpmDaySales tpmDaySales = new TpmDaySales();
            tpmDaySales.setId(record.getCId());
            tpmDaySales.setBusinessFormatCode(record.getProductTeamCode());
            String distributionChannelCode = record.getDistributionChannelCode();
            tpmDaySales.setChannelCode(distributionChannelCode);
            List<CustomerChannelVo> customerChannelVos = customerChannelMap.get(distributionChannelCode);
            if (!CollectionUtils.isEmpty(customerChannelVos)) {
                tpmDaySales.setChannelName(customerChannelVos.get(0).getCustomerChannelName());
            }
            String companyCode = record.getCompanyCode();
            tpmDaySales.setSalesInstitutionCode(companyCode);
            SalesOrgVo salesOrgVos = salesInstitutionMap.get(distributionChannelCode + tpmDaySales.getBusinessFormatCode() + companyCode);
            if (Objects.nonNull(salesOrgVos)) {
                tpmDaySales.setSalesInstitutionName(salesOrgVos.getSalesOrgName());
            }
//            String senderCode = record.getSenderCode();
            tpmDaySales.setCustomer(record.getCustomerCode());
            tpmDaySales.setCustomerName(customerMap.get(record.getCustomerCode()));
            String productCode = record.getProductCode();
            tpmDaySales.setGoodsCode(productCode);
            List<ProductVo> productVos = productMap.get(productCode);
            if (!CollectionUtils.isEmpty(productVos)) {
                tpmDaySales.setGoodsName(productVos.get(0).getProductName());
            }
            String salesIncludingTaxNet = record.getSalesIncludingTaxNet();
            if (StringUtils.isNotBlank(salesIncludingTaxNet)) {
                tpmDaySales.setAmount(new BigDecimal(salesIncludingTaxNet));
            }
            tpmDaySales.setNum(record.getSalesVolumes());
            try {
                tpmDaySales.setTradeDate(DateUtil.parse(record.getPostDate(), DateUtil.DEFAULT_YEAR_MONTH_DAY_NO_CH));
            } catch (Exception e) {
                throw new IllegalArgumentException("交易日期解析错误");
            }
            tpmDaySales.setTenantCode(tenantCode);
            tpmDaySales.setDelFlag(code);
            tpmDaySales.setAmountEnable(BigDecimal.ZERO);
            tpmDaySales.setNumEnable(0);
            tpmDaySales.setYearMonthLy(yearMonth);
            list.add(tpmDaySales);
        }
        tpmDaySalesRepository.saveOrUpdateBatch(list);
    }

    /**
     * 导入创建
     *
     * @param daySalesVo
     */
    private void createImport(TpmDaySalesVo daySalesVo) {
        String id = validateRepeatabilityOnCreate(daySalesVo);

        store(daySalesVo, id);
    }

    /**
     * 保存
     *
     * @param daySalesVo
     * @param id
     */
    private void store(TpmDaySalesVo daySalesVo, String id) {
        TpmDaySales daySales = nebulaToolkitService.copyObjectByWhiteList(daySalesVo, TpmDaySales.class, HashSet.class, ArrayList.class);
        daySales.setId(id);
        daySales.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        daySales.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        daySales.setTenantCode(TenantUtils.getTenantCode());
        daySales.setNumEnable(daySales.getNum().intValue());
        daySales.setAmountEnable(daySales.getAmount());

        tpmDaySalesRepository.saveOrUpdate(daySales);

        // 业务日志创建
        DaySalesLogEventDto logEventDto = new DaySalesLogEventDto();
        logEventDto.setOriginal(null);
        logEventDto.setNewest(daySalesVo);
        SerializableBiConsumer<DaySalesLogEventListener, DaySalesLogEventDto> onCreate =
                DaySalesLogEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, DaySalesLogEventListener.class, onCreate);
    }

    /**
     * 创建验证
     *
     * @param tpmDaySales
     */
    private String createValidate(TpmDaySalesVo tpmDaySales) {
        Validate.notNull(tpmDaySales, "对象信息不能为空！");
        Validate.notNull(tpmDaySales.getAmount(), "金额不能为空！");
        Validate.notBlank(tpmDaySales.getCustomer(), "客户编码不能为空！");
        Validate.notBlank(tpmDaySales.getCustomerName(), "客户名称不能为空！");
        Validate.notBlank(tpmDaySales.getGoodsCode(), "货品编号不能为空！");
        Validate.notBlank(tpmDaySales.getGoodsName(), "货品名称不能为空！");
        Validate.notNull(tpmDaySales.getNum(), "数量不能为空！");
        Validate.notNull(tpmDaySales.getTradeDate(), "交易日期不能为空！");
        return validateRepeatabilityOnCreate(tpmDaySales);
    }


    /**
     * 校验重复数据
     *
     * @param tpmDaySales
     * @return
     */
    private String validateRepeatabilityOnCreate(TpmDaySalesVo tpmDaySales) {
        List<TpmDaySales> list = tpmDaySalesRepository.findOnCreate(tpmDaySales);
        return CollectionUtils.isEmpty(list) ? null : list.get(0).getId();
    }

    /**
     * 修改验证
     *
     * @param tpmDaySales
     */
    private void updateValidate(TpmDaySales tpmDaySales) {
        Validate.notNull(tpmDaySales, "修改时，对象信息不能为空！");
        Validate.notBlank(tpmDaySales.getId(), "新增数据时，不能为空！");
        Validate.notBlank(tpmDaySales.getTenantCode(), "新增数据时，租户编号不能为空！");
        Validate.notNull(tpmDaySales.getAmount(), "新增数据时，金额不能为空！");
        Validate.notBlank(tpmDaySales.getCustomer(), "新增数据时，客户不能为空！");
        Validate.notBlank(tpmDaySales.getGoodsCode(), "新增数据时，货品编号不能为空！");
        Validate.notBlank(tpmDaySales.getGoodsName(), "新增数据时，货品名称不能为空！");
        Validate.notNull(tpmDaySales.getNum(), "新增数据时，数量不能为空！");
        Validate.notNull(tpmDaySales.getTradeDate(), "新增数据时，交易日期不能为空！");

    }
}

