package com.biz.crm.tpm.business.day.sales.local.service.internal;


import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.customer.channel.sdk.service.CustomerChannelVoService;
import com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.tpm.business.day.sales.sdk.constants.DaySalesConstants;
import com.biz.crm.tpm.business.day.sales.sdk.dto.TpmDaySalesSpliceDto;
import com.biz.crm.tpm.business.day.sales.sdk.dto.TpmDaySalesSpliceSearchDto;
import com.biz.crm.tpm.business.day.sales.local.repository.TpmDaySalesSpliceRepository;
import com.biz.crm.tpm.business.day.sales.local.entity.TpmDaySalesSplice;
import com.biz.crm.tpm.business.day.sales.sdk.service.TpmDaySalesSpliceService;
import com.biz.crm.tpm.business.day.sales.sdk.vo.TpmDaySalesSpliceVo;
import com.biz.crm.tpm.business.day.sales.sdk.vo.TpmDaySalesVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Maps;
import jodd.util.StringUtil;
import org.springframework.stereotype.Service;
import org.springframework.beans.factory.annotation.Autowired;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.data.domain.Pageable;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 日销售数据拆分明细实体类(TpmDaySalesSplice)表服务实现类
 *
 * @author yaoyongming
 * @since 2022-10-27 16:50:59
 */
@Service("tpmDaySalesSpliceService")
public class TpmDaySalesSpliceServiceImpl implements TpmDaySalesSpliceService {

    @Autowired(required = false)
    private TpmDaySalesSpliceRepository tpmDaySalesSpliceRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired
    private DictDataVoService dictDataVoService;
    @Autowired
    private SalesOrgVoService salesOrgVoService;
    @Autowired
    private CustomerChannelVoService customerChannelVoService;

    /**
     * 分页查询数据
     *
     * @param pageable          分页对象
     * @param tpmDaySalesSplice DTO对象
     * @return
     */
    @Override
    public Page<TpmDaySalesSpliceVo> findByConditions(Pageable pageable, TpmDaySalesSpliceSearchDto tpmDaySalesSplice) {
        ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(tpmDaySalesSplice)) {
            tpmDaySalesSplice = new TpmDaySalesSpliceSearchDto();
        }
        return this.tpmDaySalesSpliceRepository.findByConditions(pageable, tpmDaySalesSplice);
    }

    /**
     * 新建
     *
     * @param daySalesSpliceVo
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void create(TpmDaySalesSpliceVo daySalesSpliceVo) {
        //重复校验有重复，做覆盖
        String id = createValidate(daySalesSpliceVo);

        TpmDaySalesSplice daySalesSplice = nebulaToolkitService.copyObjectByWhiteList(daySalesSpliceVo, TpmDaySalesSplice.class, HashSet.class, ArrayList.class);

        daySalesSplice.setId(id);
        daySalesSplice.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        daySalesSplice.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        daySalesSplice.setTenantCode(TenantUtils.getTenantCode());

        tpmDaySalesSpliceRepository.saveOrUpdate(daySalesSplice);
    }

    /**
     * 批量新建
     *
     * @param daySalesSpliceVos
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void createBatch(Collection<TpmDaySalesSpliceVo> daySalesSpliceVos) {
        // 业态数据字典:mdm_business_format
        Map<String, String> businessFormatMap = Maps.newHashMap();
        List<DictDataVo> businessFormatList = dictDataVoService.findByDictTypeCode(DaySalesConstants.BUSINESS_FORMAT);
        if (!CollectionUtils.isEmpty(businessFormatList)) {
            businessFormatMap.putAll(businessFormatList.stream().collect(Collectors.toMap(DictDataVo::getDictValue, DictDataVo::getDictCode, (o, n) -> o)));
        }
        // 销售机构信息
        List<String> salesOrgCodes = daySalesSpliceVos.stream().map(a -> {
            StringBuffer salesOrgCode = new StringBuffer();
            salesOrgCode.append(a.getChannelCode());
            salesOrgCode.append(businessFormatMap.get(a.getBusinessFormatName()));
            salesOrgCode.append(a.getSalesInstitutionCode());
            return salesOrgCode.toString();
        }).collect(Collectors.toList());
        List<SalesOrgVo> salesOrgVos = salesOrgVoService.findBySalesOrgCodes(salesOrgCodes);
        Map<String, SalesOrgVo> salesOrgVoMap = CollectionUtils.isEmpty(salesOrgVos)
                ? Maps.newHashMap()
                : salesOrgVos.stream().collect(Collectors.toMap(SalesOrgVo::getSalesOrgCode, Function.identity()));
        // 渠道
        List<String> channelCodes = daySalesSpliceVos.stream().map(TpmDaySalesVo::getChannelCode).collect(Collectors.toList());
        List<CustomerChannelVo> channelVos = customerChannelVoService.findByCodes(channelCodes);
        Map<String, CustomerChannelVo> customerChannelVoMap = CollectionUtils.isEmpty(channelVos)
                ? Maps.newHashMap()
                : channelVos.stream().collect(Collectors.toMap(CustomerChannelVo::getCustomerChannelCode, Function.identity()));
        daySalesSpliceVos.forEach(e -> {
            // 业态编码
            if (businessFormatMap.containsKey(e.getBusinessFormatName())) {
                e.setBusinessFormatCode(businessFormatMap.get(e.getBusinessFormatName()));
            }
            // 销售机构编码
            StringBuffer salesOrgCode = new StringBuffer();
            salesOrgCode.append(e.getChannelCode());
            salesOrgCode.append(e.getBusinessFormatName());
            salesOrgCode.append(e.getSalesInstitutionCode());
            if (salesOrgVoMap.containsKey(salesOrgCode.toString())) {
                e.setBusinessFormatName(salesOrgVoMap.get(salesOrgCode.toString()).getSalesOrgName());
            }
            // 渠道
            if (customerChannelVoMap.containsKey(e.getChannelCode())) {
                e.setChannelName(customerChannelVoMap.get(e.getChannelCode()).getCustomerChannelName());
            }
            e.setActivityForm(e.getActivityFormName());
            create(e);
        });
    }

    /**
     * 获取GMV
     *
     * @param daySalesSpliceVo
     * @return
     */
    @Override
    public BigDecimal getGmv(TpmDaySalesSpliceVo daySalesSpliceVo) {
        Validate.notNull(daySalesSpliceVo, "对象信息不能为空！");
        Validate.notBlank(daySalesSpliceVo.getCustomer(), "客户编码不能为空！");
//        Validate.notBlank(daySalesSpliceVo.getGoodsCode(), "产品编码不能为空！");
        Validate.notBlank(daySalesSpliceVo.getChannel(), "渠道不能为空！");
        Validate.notBlank(daySalesSpliceVo.getActivityForm(), "活动形式不能为空！");

        return tpmDaySalesSpliceRepository.getGmv(daySalesSpliceVo);
    }

    @Override
    public Page<TpmDaySalesSpliceVo> findByYearAndMonth(Pageable pageable, TpmDaySalesSpliceDto dto) {
        return this.tpmDaySalesSpliceRepository.findByYearAndMonth(pageable, dto);
    }

    /**
     * 创建验证
     *
     * @param tpmDaySalesSplice
     */
    private String createValidate(TpmDaySalesSpliceVo tpmDaySalesSplice) {
        Validate.notNull(tpmDaySalesSplice, "对象信息不能为空！");
        Validate.notNull(tpmDaySalesSplice.getAmount(), "金额不能为空！");
        Validate.notBlank(tpmDaySalesSplice.getCustomer(), "客户编码不能为空！");
        Validate.notBlank(tpmDaySalesSplice.getCustomerName(), "客户名称不能为空！");
        Validate.notBlank(tpmDaySalesSplice.getGoodsCode(), "货品编号不能为空！");
        Validate.notBlank(tpmDaySalesSplice.getGoodsName(), "货品名称不能为空！");
        Validate.notNull(tpmDaySalesSplice.getNum(), "数量不能为空！");
        Validate.notNull(tpmDaySalesSplice.getTradeDate(), "交易日期不能为空！");
        Validate.notBlank(tpmDaySalesSplice.getActivityFormName(), "活动形式名称不能为空！");
        Validate.notBlank(tpmDaySalesSplice.getChannel(), "渠道不能为空！");
        Validate.notBlank(tpmDaySalesSplice.getBusinessFormatName(), "业态不能为空！");
        Validate.notBlank(tpmDaySalesSplice.getChannelCode(), "渠道不能为空！");
        Validate.notBlank(tpmDaySalesSplice.getSalesInstitutionCode(), "销售机构编码不能为空！");
        return validateRepeatabilityOnCreate(tpmDaySalesSplice);
    }

    /**
     * 校验重复数据
     *
     * @param tpmDaySalesSplice
     * @return
     */
    private String validateRepeatabilityOnCreate(TpmDaySalesSpliceVo tpmDaySalesSplice) {
        List<TpmDaySalesSplice> list = tpmDaySalesSpliceRepository.findOnCreate(tpmDaySalesSplice);
        return CollectionUtils.isEmpty(list) ? null : list.get(0).getId();
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> ids) {
        Validate.isTrue(!CollectionUtils.isEmpty(ids), "请选择要删除的数据！");
        tpmDaySalesSpliceRepository.removeByIds(ids);
    }

    /**
     * 获取明细数据
     *
     * @param yearMonthLy 年月
     * @param customer     客户编码
     * @param goodsCode    商品编码
     * @return java.util.List<com.biz.crm.tpm.business.day.sales.sdk.vo.TpmDaySalesSpliceVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-12-06 16:04
     */
    @Override
    public List<TpmDaySalesSpliceVo> getByCustomerAndProduct(String yearMonthLy, String customer, String goodsCode) {
        if (StringUtil.isEmpty(yearMonthLy)
                || StringUtil.isEmpty(customer)
                || StringUtil.isEmpty(goodsCode)) {
            return Collections.emptyList();
        }
        return tpmDaySalesSpliceRepository.getByCustomerAndProduct(yearMonthLy, customer, goodsCode);
    }
}

