package com.biz.crm.tpm.business.deduction.detail.mapping.local.service.internal;

import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSON;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.retailer.sdk.service.CustomerRetailerVoService;
import com.biz.crm.mdm.business.customer.retailer.sdk.vo.CustomerRetailerVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.mn.common.base.eunm.BusinessFormatEnum;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.tpm.business.activity.form.sdk.dto.ActivityFormDto;
import com.biz.crm.tpm.business.activity.form.sdk.service.ActivityFormService;
import com.biz.crm.tpm.business.activity.form.sdk.vo.ActivityFormVo;
import com.biz.crm.tpm.business.activity.form.sdk.vo.SubRelatedMainActivityFormVo;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.constant.TpmDeductionDetailMappingConstant;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.dto.TpmDeductionDetailMappingCustomerDto;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.dto.TpmDeductionDetailMappingDto;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.dto.TpmDeductionDetailMappingRelationActivityConfigDto;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.dto.TpmDeductionDetailMappingTextDto;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.service.TpmDeductionDetailMappingService;
import com.biz.crm.tpm.business.deduction.detail.mapping.sdk.vo.TpmDeductionDetailMappingImportsVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author duyiran
 * @create 2022-10-29 17:25
 */
@Slf4j
@Component
public class TpmDeductionDetailMappingImportsProcess implements ImportProcess<TpmDeductionDetailMappingImportsVo> {

    @Autowired(required = false)
    TpmDeductionDetailMappingService tpmDeductionDetailMappingService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Resource
    private ActivityFormService activityFormService;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private CustomerRetailerVoService customerRetailerVoService;

    //默认渠道编码
    private static final String DEFAULT_CHANNEL_CODE = "A1";


    /**
     * 数据处理
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    @Transactional
    public Map<Integer, String> execute(LinkedHashMap<Integer, TpmDeductionDetailMappingImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        log.info("商超扣费映射导入开始执行，参数【{}】", JSON.toJSONString(data));
        try {
            Validate.notEmpty(data, "导入数据不能为空！");
            this.baseValidate(data);
            this.validateRepeatAndSave(data);
        } catch (Exception e) {
            log.error("商超扣费映射导入异常,{}",e);
            if (StrUtil.isEmpty(e.getMessage())){
                throw new IllegalArgumentException("空指针异常",e);
            }else {
                throw new IllegalArgumentException(e.getMessage(),e);
            }
        }
        return null;
    }

    /**
     * 校验数据
     *
     * @param data data
     * @author duyiran
     * @date 2022-10-29 15:04
     **/
    private void baseValidate(LinkedHashMap<Integer, TpmDeductionDetailMappingImportsVo> data) {
        log.info("商超扣费映射导入,校验数据");
        data.forEach((index, importsVo) -> {
            Validate.notBlank(importsVo.getName(),"映射名称不能为空");
            Validate.isTrue(importsVo.getName().length() <= 255,"扣费映射名称超长，最长支持255");
            Validate.notBlank(importsVo.getBusinessUnitCode(),"业务单元不能为空");
            Validate.notBlank(importsVo.getBusinessFormatCode(),"业态不能为空");
            Validate.notBlank(importsVo.getResaleCommercialCode(),"零售商编码不能为空");
            Validate.notBlank(importsVo.getDeductionDetailItems(),"扣费明细项不能为空");
            Validate.notBlank(importsVo.getRelationActivityFormCodes(),"关联活动形式编码不能为空");
        });
    }


    private void validateRepeatAndSave(LinkedHashMap<Integer, TpmDeductionDetailMappingImportsVo> data) {
        log.info("商超扣费映射导入开始转换数据");
        List<TpmDeductionDetailMappingImportsVo> importsVos = new ArrayList<>(data.values());
        if (CollectionUtils.isEmpty(importsVos)) {
            return ;
        }

        Set<String> salesInstitutionCodes = new HashSet<>();
        Set<String> customerCodes = new HashSet<>();
        Set<String> activityFormCodes = new HashSet<>();
        Set<String> resaleCommercialCodes = new HashSet<>();
        importsVos.forEach(i -> {
            BusinessFormatEnum businessFormatEnum = BusinessFormatEnum.descToEnum(i.getBusinessFormatCode().trim());
            Validate.notNull(businessFormatEnum,"未知的业态！【"+i.getBusinessFormatCode()+"】");
            i.setBusinessFormatCode(businessFormatEnum.getCode());
            BusinessUnitEnum businessUnitEnum = BusinessUnitEnum.descToEnum(i.getBusinessUnitCode().trim());
            Validate.notNull(businessUnitEnum,"未知的业务单元！【"+i.getBusinessFormatCode()+"】");
            i.setBusinessUnitCode(businessUnitEnum.getCode());

            String firstChannelCode = DEFAULT_CHANNEL_CODE;
            if (BusinessUnitEnum.SON_COMPANY.getCode().equals(i.getBusinessUnitCode()) && StringUtils.isNotBlank(i.getChannelCode())) {
                firstChannelCode = i.getChannelCode();
            }
            if (BusinessUnitEnum.VERTICAL.getCode().equals(i.getBusinessUnitCode()) && StringUtils.isNotBlank(i.getFirstChannelCode())) {
                firstChannelCode = i.getFirstChannelCode();
            }

            if (StringUtils.isNotBlank(i.getSalesInstitutionErpCode())) {
                String salesInstitutionCode = firstChannelCode + i.getBusinessFormatCode() + i.getSalesInstitutionErpCode();
                salesInstitutionCodes.add(salesInstitutionCode);
            }
            if (StringUtils.isNotBlank(i.getErpCode())) {
                Validate.notBlank(i.getSalesInstitutionErpCode(),"填写了客户信息，必须填写销售机构信息！");
                String[] split = i.getErpCode().split(",");
                for (int a = 0 ; a < split.length ; a++ ) {
                    String customerCode = split[a] + i.getSalesInstitutionErpCode() + firstChannelCode + i.getBusinessFormatCode();
                    customerCodes.add(customerCode);

                }
            }

            if (StringUtils.isNotBlank(i.getRelationActivityFormCodes())) {
                String[] split = i.getRelationActivityFormCodes().split(",");
                for (int a = 0 ; a < split.length ; a++) {
                    activityFormCodes.add(split[a]);
                }
            }

            if (StringUtils.isNotBlank(i.getResaleCommercialCode())) {
                resaleCommercialCodes.add(i.getResaleCommercialCode());
            }
        });

        Map<String, SalesOrgVo> salesOrgVoMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(salesInstitutionCodes)) {
            List<SalesOrgVo> salesOrgVoList = salesOrgVoService.findBySalesOrgCodesPost(new ArrayList<>(salesInstitutionCodes));
            Validate.notEmpty(salesOrgVoList,"根据填写的所有销售机构编码，未查询到任何销售机构！");
            if (!CollectionUtils.isEmpty(salesOrgVoList)) {
                salesOrgVoMap = salesOrgVoList.stream().collect(Collectors.toMap(SalesOrgVo::getSalesOrgCode, Function.identity(), (v1, v2) -> v2));
            }
        }

        Map<String, CustomerVo> customerVoMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(customerCodes)) {
            List<CustomerVo> customerVoList = customerVoService.findBaseByCustomerCodes(new ArrayList<>(customerCodes));
            Validate.notEmpty(customerVoList,"根据填写的所有客户编码，未查询到任何客户信息！");
            if (!CollectionUtils.isEmpty(customerVoList)) {
                customerVoMap = customerVoList.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, Function.identity(), (v1, v2) -> v2));
            }
        }

        Map<String, ActivityFormVo> activityFormVoMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(activityFormCodes)) {
            List<ActivityFormVo> activityFormVoList = activityFormService.findFormAndRelatedTypeCodes(new ArrayList<>(activityFormCodes), TenantUtils.getTenantCode());
            Validate.notEmpty(activityFormVoList,"根据填写的所有活动形式编码，未查询到任何活动形式！");
            if (!CollectionUtils.isEmpty(activityFormVoList)) {
                activityFormVoMap = activityFormVoList.stream().collect(Collectors.toMap(ActivityFormVo::getActivityFormCode, Function.identity(), (v1, v2) -> v2));
            }
        }

        Map<String, CustomerRetailerVo> retailerVoMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(resaleCommercialCodes)) {
            List<CustomerRetailerVo> retailerVoList = customerRetailerVoService.findByCodes(new ArrayList<>(resaleCommercialCodes));
            Validate.notEmpty(retailerVoList,"根据填写的所有零售商编码，未查询到零售商信息！");
            if (!CollectionUtils.isEmpty(retailerVoList)) {
                retailerVoMap = retailerVoList.stream().collect(Collectors.toMap(CustomerRetailerVo::getCustomerRetailerCode, Function.identity(), (v1, v2) -> v2));
            }
        }

        List<TpmDeductionDetailMappingDto> mappingDtoList = new ArrayList<>(importsVos.size());
        for (TpmDeductionDetailMappingImportsVo importsVo : importsVos) {
            TpmDeductionDetailMappingDto mappingDto = new TpmDeductionDetailMappingDto();
            List<TpmDeductionDetailMappingCustomerDto> customerDtoList = new ArrayList<>();
            List<TpmDeductionDetailMappingTextDto> textDtoList = new ArrayList<>();
            List<TpmDeductionDetailMappingRelationActivityConfigDto> activityConfigDtoList = new ArrayList<>();

            mappingDto.setName(importsVo.getName());
            mappingDto.setChannelCode(importsVo.getChannelCode());
            mappingDto.setFirstChannelCode(importsVo.getFirstChannelCode());
            mappingDto.setSecondChannelCode(importsVo.getSecondChannelCode());
            mappingDto.setResaleCommercialCode(importsVo.getResaleCommercialCode());
            mappingDto.setResaleCommercialName(importsVo.getResaleCommercialName());
            mappingDto.setBusinessFormatCode(importsVo.getBusinessFormatCode());
            mappingDto.setBusinessUnitCode(importsVo.getBusinessUnitCode());

            String firstChannelCode = DEFAULT_CHANNEL_CODE;
            if (BusinessUnitEnum.SON_COMPANY.getCode().equals(importsVo.getBusinessUnitCode()) && StringUtils.isNotBlank(importsVo.getChannelCode())) {
                firstChannelCode = importsVo.getChannelCode();
            }
            if (BusinessUnitEnum.VERTICAL.getCode().equals(importsVo.getBusinessUnitCode()) && StringUtils.isNotBlank(importsVo.getFirstChannelCode())) {
                firstChannelCode = importsVo.getFirstChannelCode();
            }
            if (!CollectionUtils.isEmpty(salesOrgVoMap) && StringUtils.isNotBlank(importsVo.getSalesInstitutionErpCode())) {
                String salesInstitutionCode = firstChannelCode + importsVo.getBusinessFormatCode() + importsVo.getSalesInstitutionErpCode();
                SalesOrgVo salesOrgVo = salesOrgVoMap.get(salesInstitutionCode);
                Validate.notNull(salesOrgVo,"未查询到销售机构！渠道【"+firstChannelCode+"】,业态【"+importsVo.getBusinessFormatCode()+"】,销售机构编码【"+importsVo.getSalesInstitutionErpCode()+"】");
                mappingDto.setSalesInstitutionCode(salesOrgVo.getSalesOrgCode());
                mappingDto.setSalesInstitutionErpCode(salesOrgVo.getErpCode());
                mappingDto.setSalesInstitutionName(salesOrgVo.getSalesOrgName());
            }

            if (!CollectionUtils.isEmpty(customerVoMap) && StringUtils.isNotBlank(importsVo.getErpCode())) {
                String[] split = importsVo.getErpCode().split(",");
                for (int a = 0 ; a < split.length ; a++ ) {
                    String customerCode = split[a] + importsVo.getSalesInstitutionErpCode() + firstChannelCode + importsVo.getBusinessFormatCode() ;
                    CustomerVo customerVo = customerVoMap.get(customerCode);
                    Validate.notNull(customerVo,"未查询到客户！渠道【"+firstChannelCode+"】,业态【"+importsVo.getBusinessFormatCode()+"】,客户编码【"+split[a]+"】");
                    TpmDeductionDetailMappingCustomerDto customerDto = new TpmDeductionDetailMappingCustomerDto();
                    customerDto.setCustomerCode(customerVo.getCustomerCode());
                    customerDto.setErpCode(customerVo.getErpCode());
                    customerDto.setCustomerName(customerVo.getCustomerName());
                    customerDtoList.add(customerDto);
                }
            }

            if (StringUtils.isNotBlank(importsVo.getDeductionDetailItems())) {
                String[] split = importsVo.getDeductionDetailItems().split(",");
                for (int a = 0 ; a < split.length ; a++) {
                    Validate.isTrue(split[a].length() <= 255,"扣费明细项名称【"+split[a]+"】超长，最长支持255");
                    TpmDeductionDetailMappingTextDto textDto = new TpmDeductionDetailMappingTextDto();
                    textDto.setText(split[a]);
                    textDtoList.add(textDto);
                }
            }

            if (!CollectionUtils.isEmpty(activityFormVoMap) && StringUtils.isNotBlank(importsVo.getRelationActivityFormCodes())) {
                String[] split = importsVo.getRelationActivityFormCodes().split(",");
                for (int a = 0 ; a < split.length ; a++) {
                    ActivityFormVo activityFormVo = activityFormVoMap.get(split[a]);
                    Validate.notNull(activityFormVo,"未查询到活动形式！"+split[a]);
                    TpmDeductionDetailMappingRelationActivityConfigDto configDto = new TpmDeductionDetailMappingRelationActivityConfigDto();
                    configDto.setActivityFormCode(activityFormVo.getActivityFormCode());
                    configDto.setActivityFormName(activityFormVo.getActivityFormName());
                    configDto.setActivityTypeCode(activityFormVo.getActivityTypeCode());
                    configDto.setActivityTypeName(activityFormVo.getActivityTypeName());
                    activityConfigDtoList.add(configDto);
                }
            }

            if (!CollectionUtils.isEmpty(retailerVoMap) && StringUtils.isNotBlank(importsVo.getResaleCommercialCode())) {
                CustomerRetailerVo customerRetailerVo = retailerVoMap.get(importsVo.getResaleCommercialCode());
                Validate.notNull(customerRetailerVo,"未知的零售商！【"+importsVo.getResaleCommercialCode()+"】");
                mappingDto.setResaleCommercialCode(customerRetailerVo.getCustomerRetailerCode());
                mappingDto.setResaleCommercialName(customerRetailerVo.getCustomerRetailerName());
            }

            mappingDto.setCustomerList(customerDtoList);
            mappingDto.setDeductionDetailMappingTextList(textDtoList);
            mappingDto.setDeductionDetailMappingRelationActivityConfigList(activityConfigDtoList);
            mappingDtoList.add(mappingDto);
        }

        log.info("开始保存");
        tpmDeductionDetailMappingService.importSave(mappingDtoList);
    }

    /**
     * 获取数据实体
     *
     * @return
     */
    @Override
    public Class<TpmDeductionDetailMappingImportsVo> findCrmExcelVoClass() {
        return TpmDeductionDetailMappingImportsVo.class;
    }

    /**
     * 获取数据实体
     *
     * @return
     */
    @Override
    public String getTemplateCode() {
        return "TPM_DEDUCTION_DETAIL_MAPPING_IMPORT";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return
     */
    @Override
    public String getTemplateName() {
        return "TPM-商超扣费明细映射";
    }
}
