package com.biz.crm.tpm.business.distrib.close.data.policy.process;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.channel.sdk.service.CustomerChannelVoService;
import com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.tpm.business.distrib.close.data.policy.service.PlatformLogisticsService;
import com.biz.crm.tpm.business.distrib.close.policy.sdk.vo.PlatformLogisticsImportVo;
import com.biz.crm.tpm.business.distrib.close.policy.sdk.vo.PlatformLogisticsVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Author: zhouyang
 * @Date: 2023/7/20
 */
@Slf4j
@Component
public class PlatformLogisticsImportProcess implements ImportProcess<PlatformLogisticsImportVo> {

    private static final Integer startRow = 2;
    @Autowired
    private NebulaToolkitService nebulaToolkitService;

    @Autowired
    private PlatformLogisticsService platformLogisticsService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    @Autowired(required = false)
    private CustomerChannelVoService customerChannelVoService;

    @Override
    public Integer getBatchCount() {
        return Integer.MAX_VALUE;
    }

    /**
     * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
     * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
     *
     * @return
     */
    @Override
    public boolean importBeforeValidationFlag() {
        return Boolean.TRUE;
    }

    /**
     * 数据校验
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> tryVerify(LinkedHashMap<Integer, PlatformLogisticsImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Validate.notNull(data, "导入数据不能为空！");
        Validate.isTrue(data.values().size() <= getBatchCount(), "单次导入数据不能超过%s条", getBatchCount());
        Map<Integer, String> errMap = new HashMap<>();
        for (Map.Entry<Integer, PlatformLogisticsImportVo> row : data.entrySet()) {
            int rowNum = row.getKey();
            PlatformLogisticsImportVo vo = row.getValue();
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getSapProductCode()), "[sap代码]不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getOrderCode()), "[订单号]不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getOrderStatus()), "[订单状态]不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getAfterSalesStatus()), "[售后状态]不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getDeliveryCode()), "[快递单号]不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getBusinessFormatCode()), "[业态]不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getChannelCode()), "[渠道编码]不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getSalesOrgCode()), "[销售组织]不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getCustomerCode()), "[客户编码]不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getYearAndMonth()), "[月份]不能为空！");
            this.validateIsTrue(vo.getProductNumber() != null, "[商品数量]不能为空！");
            this.validateIsTrue(vo.getNumber() != null, "[提数]不能为空！");
            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(rowNum, errInfo);
            }
        }
        return errMap;
    }

    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, PlatformLogisticsImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        List<PlatformLogisticsVo> list = this.validate(data, errMap);
        if (errMap.isEmpty()) {
            this.platformLogisticsService.createBatch(list);
        }
        return errMap;
    }

    private List<PlatformLogisticsVo> validate(LinkedHashMap<Integer, PlatformLogisticsImportVo> data, Map<Integer, String> errMap) {
        List<PlatformLogisticsVo> list = Lists.newArrayList();
        Calendar calendar =Calendar.getInstance();
        calendar.setTime(new Date());
        int year = calendar.get(Calendar.YEAR);
        int month = calendar.get(Calendar.MONTH)+1;
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");
        SimpleDateFormat sdf2 = new SimpleDateFormat("yyyy-MM-dd HH:mm");

        Map<String, List<SalesOrgVo>> saleOrgVoMap = new HashMap<>();
        Map<String, List<CustomerVo>> customerVoMap = new HashMap<>();
        Map<String, String> channelVoMap = new HashMap<>();

        Set<String> salesOrgCodeSet = data.values().stream().map(PlatformLogisticsImportVo::getSalesOrgCode).filter(StringUtils::isNotEmpty).collect(Collectors.toSet());
        Set<String> customerCodeSet = data.values().stream().map(PlatformLogisticsImportVo::getCustomerCode).filter(StringUtils::isNotEmpty).collect(Collectors.toSet());

        Map<String, List<DictDataVo>> mapDict = this.dictDataVoService.findByDictTypeCodeList(Lists.newArrayList("mdm_business_format"));
        Map<String, String> formatMap = mapDict.get("mdm_business_format").stream().collect(Collectors.toMap(DictDataVo::getDictValue, DictDataVo::getDictCode, (e1, e2) -> e1));

        //查询渠道
        List<String> channelCodeSet = data.values().stream().map(PlatformLogisticsImportVo::getChannelCode).filter(StringUtils::isNotBlank).collect(Collectors.toList());
        List<CustomerChannelVo> channelList = customerChannelVoService.findByCodes(new ArrayList<>(channelCodeSet));
        if (CollectionUtils.isNotEmpty(channelList)) {
            channelVoMap = channelList.stream().collect(Collectors.toMap(CustomerChannelVo::getCustomerChannelCode, CustomerChannelVo::getCustomerChannelName));
        }

        List<SalesOrgVo> saleOrgVoList = salesOrgVoService.findByErpCodeList(Lists.newArrayList(salesOrgCodeSet));
        List<CustomerVo> customerVoList = customerVoService.findByCustomerMdgCodes(Lists.newArrayList(customerCodeSet));

        if (CollectionUtils.isNotEmpty(saleOrgVoList)) {
            saleOrgVoMap = saleOrgVoList.stream().collect(Collectors.groupingBy(e -> e.getErpCode()));
        }
        if (CollectionUtils.isNotEmpty(customerVoList)) {
            customerVoMap = customerVoList.stream().collect(Collectors.groupingBy(e -> e.getErpCode()));
        }

        Map<String, String> finalChannelVoMap = channelVoMap;
        Map<String, List<SalesOrgVo>> finalSaleOrgVoMap = saleOrgVoMap;
        Map<String, List<CustomerVo>> finalCustomerVoMap = customerVoMap;
        data.forEach((k, value) -> {
            this.validateIsTrue(value.getProductNumber() > 0 && value.getProductNumber() <= 10,"[商品数量]不能小于1或大于10");
            this.validateIsTrue(value.getNumber() > 0,"[提数]不能小于1");
            try {
                // 年月校验
                Date yearAndMonth = sdf.parse(value.getYearAndMonth());
                Calendar calendar2 = Calendar.getInstance();
                calendar2.setTime(yearAndMonth);
                int importYear = calendar2.get(Calendar.YEAR);
                int importMonth = calendar2.get(Calendar.MONTH)+1;
                if ((year == importYear && month <= importMonth) ) {
                    this.validateIsTrue(false, String.format("只能导入当前月之前数据"));
                } else if (year < importYear){
                    this.validateIsTrue(false, String.format("只能导入当前月之前数据"));
                }
            } catch (ParseException e) {
                // 时间格式错误
                this.validateIsTrue(false, String.format("月份格式错误。正确格式为：2023-01"));
            }
            PlatformLogisticsVo vo = this.nebulaToolkitService.copyObjectByWhiteList(value,PlatformLogisticsVo.class,HashSet.class,ArrayList.class);
            try {
                vo.setPayTime(Objects.nonNull(value.getImportPayTime()) ? sdf2.parse(value.getImportPayTime()) : null);
                vo.setPromiseDeliveryTime(Objects.nonNull(value.getImportPromiseDeliveryTime()) ? sdf2.parse(value.getImportPromiseDeliveryTime()) : null);
                vo.setDeliveryTime(Objects.nonNull(value.getImportDeliveryTime()) ? sdf2.parse(value.getImportDeliveryTime()) : null);
            } catch (ParseException e) {
                this.validateIsTrue(false, String.format("时间格式错误。正确格式为：2023-01-01 01:01"));
            }

            this.validateIsTrue(finalChannelVoMap.containsKey(value.getChannelCode()), "未找到【"+ value.getChannelCode() +"】对应的渠道");
            if (formatMap.containsKey(value.getBusinessFormatCode())) {
                vo.setBusinessFormatCode(formatMap.get(value.getBusinessFormatCode()));
            } else {
                this.validateIsTrue(false, "请填写正确的业态名称");
            }

            if (finalSaleOrgVoMap.containsKey(value.getSalesOrgCode())) {
                List<SalesOrgVo> list1 = finalSaleOrgVoMap.get(value.getSalesOrgCode());
                SalesOrgVo salesOrgVo = list1.stream().filter(e -> value.getChannelCode().equals(e.getChannelCode()) && vo.getBusinessFormatCode().equals(e.getBusinessFormatCode()))
                        .findFirst().orElse(null);
                if (salesOrgVo != null) {
                    vo.setSalesOrgName(salesOrgVo.getSalesOrgName());
                } else {
                    this.validateIsTrue(false, "销售组织编码【" + value.getSalesOrgCode() + "】错误，请检查！");
                }
            } else {
                this.validateIsTrue(false, "未找到【"+ value.getSalesOrgCode() +"】对应的组织");
            }
            if (finalCustomerVoMap.containsKey(value.getCustomerCode())) {
                List<CustomerVo> list2 = finalCustomerVoMap.get(value.getCustomerCode());
                CustomerVo customerVo = list2.stream().filter(e -> value.getChannelCode().equals(e.getCustomerChannelCode()) && value.getSalesOrgCode().equals(e.getSalesInstitutionErpCode()) && vo.getBusinessFormatCode().equals(e.getBusinessFormatCode()))
                        .findFirst().orElse(null);
                if (customerVo != null) {
                    vo.setCustomerName(customerVo.getCustomerName());
                } else {
                    this.validateIsTrue(false, "客户编码【" + value.getCustomerCode() + "】错误，请检查！");
                }
            } else {
                this.validateIsTrue(false, "未找到【"+ value.getCustomerCode() +"】对应的客户");
            }

            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(k, errInfo);
            }

            list.add(vo);
        });
        return list;
    }

    @Override
    public Class<PlatformLogisticsImportVo> findCrmExcelVoClass() {
        return PlatformLogisticsImportVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "TPM_PLATFORM_LOGISTICS_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "TPM平台物流底表导入";
    }
}
