package com.biz.crm.tpm.business.distribution.wages.local.process;

import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.channel.sdk.service.CustomerChannelVoService;
import com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.distribution.wages.sdk.dto.DistributionWagesDto;
import com.biz.crm.tpm.business.distribution.wages.sdk.service.DistributionWagesService;
import com.biz.crm.tpm.business.distribution.wages.sdk.vo.DistributionWagesImportsVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 导入
 *
 * @author: yaoyongming
 * @date: 2022/12/29 10:52
 */
@Slf4j
@Component
public class DistributionWagesImportsProcess implements ImportProcess<DistributionWagesImportsVo> {

    @Resource
    private DistributionWagesService distributionWagesService;

    @Resource
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private CustomerChannelVoService customerChannelVoService;

    @Autowired(required = false)
    private DictDataVoService dictDataVoService;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    private static final String MDM_BUSINESS_FORMAT = "mdm_business_format";
    private static final String REGEX = "\\d{4}-\\d{2}";

    /**
     * 单次缓存的数量
     *
     * @return
     */
    @Override
    public Integer getBatchCount() {
        return 10000;
    }

    /**
     * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
     * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
     *
     * @return
     */
    @Override
    public boolean importBeforeValidationFlag() {
        return Boolean.TRUE;
    }

    /**
     * 数据校验
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> tryVerify(LinkedHashMap<Integer, DistributionWagesImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        for (Map.Entry<Integer, DistributionWagesImportsVo> row : data.entrySet()) {
            int rowNum = row.getKey();
            DistributionWagesImportsVo vo = row.getValue();
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getOperators()),"运营人员不能为空！");
            this.validateIsTrue(validateSalaryAmount(vo.getSalaryAmount()),"工资金额不能为空，或者不能为负！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getCustomerCode()),"客户编码不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getChannelCode()),"渠道编码不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getSalesInstitutionCode()),"销售机构编码不能为空！");
            this.validateIsTrue(StringUtils.isNotEmpty(vo.getYears()),"年月不能为空！");
            this.validateIsTrue(yearBooleanMethod(vo.getYears()),"年月格式有误,例 2023-10");
            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(rowNum, errInfo);
            }
        }
        return errMap;
    }

    private Boolean validateSalaryAmount(BigDecimal salaryAmount) {
        if (salaryAmount == null || salaryAmount.compareTo(BigDecimal.ZERO) < 0){
            return false;
        }
        return true;
    }

    private Boolean yearBooleanMethod(String years) {
        if (years.length() != 7 || !years.matches(REGEX)){
            return false;
        }
        return true;
    }

    /**
     * 数据处理
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, DistributionWagesImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Validate.notEmpty(data, "导入数据不能为空！");

        Map<Integer, String> errMap = new HashMap<>();
        Map<String, List<CustomerVo>> customerVoMap = new HashMap<>();
        Map<String, List<SalesOrgVo>> orgVoMap = new HashMap<>();
        Map<String, CustomerChannelVo> channelVoMap = new HashMap<>();
        Set<String> customerCodeSet = new HashSet<>();
        Set<String> channelCodeSet = new HashSet<>();
        Set<String> orgCodeSet = new HashSet<>();
        data.values().forEach(dto -> {
            customerCodeSet.add(dto.getCustomerCode());
            channelCodeSet.add(dto.getChannelCode());
            orgCodeSet.add(dto.getSalesInstitutionCode());
        });
        // 查询数据字典
        Map<String, List<DictDataVo>> dictDataMap = dictDataVoService.findByDictTypeCodeList(Arrays.asList(MDM_BUSINESS_FORMAT));
        //查询渠道
        List<CustomerChannelVo> channelList = customerChannelVoService.findByCodes(new ArrayList<>(channelCodeSet));
        if (!CollectionUtils.isEmpty(channelList)) {
            channelVoMap = channelList.stream().collect(Collectors.toMap(e -> e.getCustomerChannelCode(), Function.identity()));
        }
        // 客户
        List<CustomerVo> customerVos = customerVoService.findByCustomerMdgCodes(new ArrayList<>(customerCodeSet));
        if (!CollectionUtils.isEmpty(customerVos)) {
            customerVoMap = customerVos.stream().collect(Collectors.groupingBy(e -> e.getErpCode()));
        }
        //销售机构
        List<SalesOrgVo> orgVos = salesOrgVoService.findByErpCodeList(new ArrayList<>(orgCodeSet));
        if (org.apache.commons.collections.CollectionUtils.isNotEmpty(orgVos)) {
            orgVoMap = orgVos.stream().collect(Collectors.groupingBy(e -> e.getErpCode()));
        }
        Map<String, CustomerChannelVo> finalChannelVoMap = channelVoMap;
        Map<String, List<SalesOrgVo>> finalOrgVoMap = orgVoMap;
        Map<String, List<CustomerVo>> finalCustomerVoMap = customerVoMap;
        data.forEach((k, dto) -> {
            //业态
            DictDataVo promotionType = dictDataMap.get(MDM_BUSINESS_FORMAT).stream().filter(e -> e.getDictValue().equals(dto.getBusinessFormatCode())).findFirst().orElse(null);
            if (promotionType != null) {
                dto.setBusinessFormatCode(promotionType.getDictCode());
            } else {
                this.validateIsTrue(false, "业态【" + dto.getBusinessFormatCode() + "】错误，请检查！");
            }
            if (finalChannelVoMap.containsKey(dto.getChannelCode())) {
                dto.setChannelName(finalChannelVoMap.get(dto.getChannelCode()).getCustomerChannelName());
            } else {
                this.validateIsTrue(false, "渠道编码【" + dto.getChannelCode() + "】错误，请检查！");
            }
            SalesOrgVo salesOrgVo = null;
            if (finalOrgVoMap.containsKey(dto.getSalesInstitutionCode())) {
                List<SalesOrgVo> list = finalOrgVoMap.get(dto.getSalesInstitutionCode());
                salesOrgVo = list.stream().filter(e -> dto.getChannelCode().equals(e.getChannelCode()) && dto.getBusinessFormatCode().equals(e.getBusinessFormatCode()))
                        .findFirst().orElse(null);
                if (salesOrgVo == null) {
                    this.validateIsTrue(false, "销售机构编码【" + dto.getSalesInstitutionCode() + "】未找到对应销售机构，请检查！");
                } else {
                    dto.setSalesInstitutionName(salesOrgVo.getSalesOrgName());
                }
            } else {
                this.validateIsTrue(false, "销售机构编码【" + dto.getSalesInstitutionCode() + "】错误，请检查！");
            }
            if (finalCustomerVoMap.containsKey(dto.getCustomerCode())) {
                List<CustomerVo> list = finalCustomerVoMap.get(dto.getCustomerCode());
                SalesOrgVo finalSalesOrgVo = salesOrgVo;
                CustomerVo customerVo = list.stream().filter(e -> dto.getChannelCode().equals(e.getCustomerChannelCode()) && finalSalesOrgVo.getSalesInstitutionCode().equals(e.getSalesInstitutionCode()) && dto.getBusinessFormatCode().equals(e.getBusinessFormatCode()))
                        .findFirst().orElse(null);
                if (customerVo == null) {
                    this.validateIsTrue(false, "客户编码【" + dto.getCustomerCode() + "】未找到对应客户，请检查！");
                } else {
                    dto.setCustomerName(customerVo.getCustomerName());
                }
            } else {
                this.validateIsTrue(false, "客户编码【" + dto.getCustomerCode() + "】错误，请检查！");
            }

            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(k, errInfo);
            }
        });

        if (errMap.isEmpty()) {
            List<DistributionWagesImportsVo> importList = new ArrayList<>(data.values());
            Collection<DistributionWagesDto> dtoList = nebulaToolkitService.copyCollectionByWhiteList(importList, DistributionWagesImportsVo.class, DistributionWagesDto.class, LinkedHashSet.class, ArrayList.class);

            distributionWagesService.importSave(new ArrayList<>(dtoList));
        }
        return errMap;
    }

    /**
     * 获取数据实体
     *
     * @return
     */
    @Override
    public Class<DistributionWagesImportsVo> findCrmExcelVoClass() {
        return DistributionWagesImportsVo.class;
    }

    /**
     * 获取业务对应的模板编码，全局唯一
     *
     * @return
     */
    @Override
    public String getTemplateCode() {
        return "TPM_DISTRIBUTION_WAGES_IMPORT";
    }

    /**
     * 获取业务对应的模板描述
     *
     * @return
     */
    @Override
    public String getTemplateName() {
        return "TPM分销客户人员工资表导入";
    }
}
