package com.biz.crm.tpm.business.key.indicators.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.tpm.business.key.indicators.local.entity.KeyIndicators;
import com.biz.crm.tpm.business.key.indicators.local.repository.KeyIndicatorsRepository;
import com.biz.crm.tpm.business.key.indicators.sdk.dto.KeyIndicatorsDto;
import com.biz.crm.tpm.business.key.indicators.sdk.dto.KeyIndicatorsLogEventDto;
import com.biz.crm.tpm.business.key.indicators.sdk.enums.YesOrNoEnum;
import com.biz.crm.tpm.business.key.indicators.sdk.event.log.KeyIndicatorsLogEventListener;
import com.biz.crm.tpm.business.key.indicators.sdk.service.KeyIndicatorsService;
import com.biz.crm.tpm.business.key.indicators.sdk.vo.KeyIndicatorsVO;
import com.biz.crm.tpm.business.variable.sdk.enums.VariableFunctionEnum;
import com.biz.crm.tpm.business.variable.sdk.register.FormulaCustomVariableRegister;
import com.biz.crm.tpm.business.variable.sdk.register.FormulaVariableRegister;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 关键指标表(tpm_key_indicators)服务实现类
 *
 * @author : qiancheng
 * @date : 2022-11-2
 */
@Slf4j
@Service("keyIndicatorsService")
public class KeyIndicatorsServiceImpl implements KeyIndicatorsService {


    @Autowired
    private KeyIndicatorsRepository keyIndicatorsRepository;

    @Autowired
    private NebulaToolkitService nebulaToolkitService;

    @Autowired
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired
    private GenerateCodeService generateCodeService;

    @Autowired(required = false)
    private List<FormulaVariableRegister> formulaVariableRegisterList;

    @Autowired(required = false)
    private List<FormulaCustomVariableRegister> formulaCustomVariableRegisterList;

    /**
     * 分页查询数据
     *
     * @param pageable         分页对象
     * @param keyIndicatorsDto 实体对象
     */
    @Override
    public Page<KeyIndicatorsVO> findByConditions(Pageable pageable, KeyIndicatorsDto keyIndicatorsDto) {
        if (pageable == null) {
            pageable = PageRequest.of(1, 20);
        }
        if (keyIndicatorsDto == null) {
            keyIndicatorsDto = new KeyIndicatorsDto();
        }
        if (StringUtils.isBlank(keyIndicatorsDto.getDelFlag())) {
            keyIndicatorsDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        }
        if (StringUtils.isBlank(keyIndicatorsDto.getTenantCode())) {
            keyIndicatorsDto.setTenantCode(TenantUtils.getTenantCode());
        }

        return this.keyIndicatorsRepository.findByConditions(pageable, keyIndicatorsDto);
    }

    /**
     * 模糊查询bean名称
     *
     * @param beanName
     * @return
     */
    @Override
    public List<String> findBeanName(String beanName) {
        List<String> beanList = new ArrayList<>();
        beanList.add(beanName);
        return beanList;
    }

    @Override
    public KeyIndicatorsVO findBeanByCondition(String beanCode, String beanTypes) {
        Validate.notBlank(beanCode, "关键指标编码不能为空！");
        Validate.notBlank(beanTypes, "指标类型不能为空！");
        List<String> beanTypeList = Arrays.asList(beanTypes.split(","));
        String code = null;
        String name = null;
        Integer sort = null;
        List<FormulaVariableRegister> filterList = formulaVariableRegisterList.stream()
                .filter(e -> {
                    if (CollectionUtils.isEmpty(e.getFunctionStringList())) {
                        return false;
                    }
                    for (String beanType : beanTypeList) {
                        if (!e.getFunctionStringList().contains(beanType)) {
                            return false;
                        }
                    }
                    return StringUtils.equals(beanCode, e.getVariableCode());
                }).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(filterList)) {
            List<FormulaCustomVariableRegister> filterCustomList = formulaCustomVariableRegisterList.stream()
                    .filter(e -> {
                        if (CollectionUtils.isEmpty(e.getFunctionStringList())) {
                            return false;
                        }
                        for (String beanType : beanTypeList) {
                            if (!e.getFunctionStringList().contains(beanType)) {
                                return false;
                            }
                        }
                        return StringUtils.equals(beanCode, e.getVariableCode());
                    }).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(filterCustomList)) {
                FormulaCustomVariableRegister formulaVariableRegister = filterCustomList.get(0);
                code = formulaVariableRegister.getVariableCode();
                name = formulaVariableRegister.getVariableName();
                sort = formulaVariableRegister.getSort();
            }
        } else {
            FormulaVariableRegister formulaVariableRegister = filterList.get(0);
            code = formulaVariableRegister.getVariableCode();
            name = formulaVariableRegister.getVariableName();
            sort = formulaVariableRegister.getSort();
        }
        Validate.notBlank(code, "关键指标编码[%s]未找到对应注册器！", beanCode);
        KeyIndicatorsVO keyIndicatorsVO = new KeyIndicatorsVO();
        keyIndicatorsVO.setKeyIndicatorsCode(code);
        keyIndicatorsVO.setKeyIndicatorsName(name);
        keyIndicatorsVO.setSortIndex(sort);
        return keyIndicatorsVO;
    }

    /**
     * 通过beanName 查询bean路径
     *
     * @param beanName bean名称
     * @return
     */
    @Override
    public KeyIndicatorsVO findBeanPath(String beanName) {
        KeyIndicatorsVO vo = new KeyIndicatorsVO();
        vo.setBeanPath("/" + beanName);
        return vo;
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public KeyIndicatorsVO findById(String id) {
        if (!StringUtils.isBlank(id)) {
            KeyIndicators keyIndicators = keyIndicatorsRepository.findById(id);
            if (keyIndicators != null) {
                return nebulaToolkitService.copyObjectByWhiteList(keyIndicators, KeyIndicatorsVO.class, HashSet.class, ArrayList.class);
            }
        }
        return null;
    }

    /**
     * 通过主键集合查询数据
     *
     * @param idList 主键
     * @return 数据集合
     */
    @Override
    public List<KeyIndicatorsVO> findByIds(List<String> idList) {
        if (idList.size() > 0) {
            List<KeyIndicators> keyIndicatorsList = keyIndicatorsRepository.listByIds(idList);
            if (keyIndicatorsList.size() > 0) {
                return Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(keyIndicatorsList, KeyIndicators.class, KeyIndicatorsVO.class, HashSet.class, ArrayList.class));
            }
        }
        return null;
    }

    /**
     * 新增数据
     *
     * @param keyIndicators 实体对象
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public KeyIndicatorsDto create(KeyIndicatorsDto keyIndicators) {
        this.createValidate(keyIndicators);
        this.checkType(keyIndicators);
        this.checkConfigurable(keyIndicators);
        List<KeyIndicators> existList = this.keyIndicatorsRepository.findByCode(keyIndicators.getKeyIndicatorsCode(), null, null);
//        Validate.isTrue(existList.size() == 0, "关键指标已存在！");
        if (existList.size() == 0){
            if (StringUtils.isEmpty(keyIndicators.getConfigurable())) {
                keyIndicators.setConfigurable(YesOrNoEnum.NO.getCode());
            }
            keyIndicators.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            keyIndicators.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            keyIndicators.setTenantCode(TenantUtils.getTenantCode());
//        String ruleCode = KeyIndicatorsConstant.KEY_INDICATORS_CODE_PREFIX;
//        keyIndicators.setKeyIndicatorsCode(generateCodeService.generateCode(ruleCode, 1).get(0));
            KeyIndicators entity = nebulaToolkitService.copyObjectByWhiteList(keyIndicators, KeyIndicators.class, HashSet.class, ArrayList.class);
            keyIndicatorsRepository.save(entity);
            keyIndicators.setId(entity.getId());

            //新增业务日志
            KeyIndicatorsLogEventDto logEventDto = new KeyIndicatorsLogEventDto();
            logEventDto.setOriginal(null);
            logEventDto.setNewest(keyIndicators);
            SerializableBiConsumer<KeyIndicatorsLogEventListener, KeyIndicatorsLogEventDto> onCreate = KeyIndicatorsLogEventListener::onCreate;
            this.nebulaNetEventClient.publish(logEventDto, KeyIndicatorsLogEventListener.class, onCreate);
        } else {
            for (KeyIndicators keyIndicator : existList) {
                Validate.isTrue(DelFlagStatusEnum.DELETE.getCode().equals(keyIndicator.getDelFlag()), "关键指标已存在！");
                keyIndicators.setId(keyIndicator.getId());
                keyIndicators.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                keyIndicators.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                KeyIndicators newKeyIndicators = nebulaToolkitService.copyObjectByWhiteList(keyIndicators, KeyIndicators.class, HashSet.class, ArrayList.class);
                keyIndicatorsRepository.updateById(newKeyIndicators);
                //编辑业务日志
                KeyIndicatorsLogEventDto logEventDto = new KeyIndicatorsLogEventDto();
                KeyIndicatorsDto oldDto = nebulaToolkitService.copyObjectByWhiteList(keyIndicator, KeyIndicatorsDto.class, HashSet.class, ArrayList.class);
                KeyIndicatorsDto newDto = nebulaToolkitService.copyObjectByWhiteList(newKeyIndicators, KeyIndicatorsDto.class, HashSet.class, ArrayList.class);
                logEventDto.setOriginal(oldDto);
                logEventDto.setNewest(newDto);
                SerializableBiConsumer<KeyIndicatorsLogEventListener, KeyIndicatorsLogEventDto> onUpdate = KeyIndicatorsLogEventListener::onUpdate;
                this.nebulaNetEventClient.publish(logEventDto, KeyIndicatorsLogEventListener.class, onUpdate);
            }
        }

        return keyIndicators;
    }

    /**
     * 编辑数据
     *
     * @param keyIndicators 实体对象
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public KeyIndicatorsDto update(KeyIndicatorsDto keyIndicators) {
        this.updateValidate(keyIndicators);
        this.checkType(keyIndicators);
        this.checkConfigurable(keyIndicators);
        List<KeyIndicators> existList = this.keyIndicatorsRepository.findByCode(keyIndicators.getKeyIndicatorsCode(), DelFlagStatusEnum.NORMAL.getCode(), null);
        existList = existList.stream().filter(e -> !keyIndicators.getId().equals(e.getId())).collect(Collectors.toList());
        Validate.isTrue(existList.size() == 0, "关键指标已存在！");

        KeyIndicators dbKeyIndicators = keyIndicatorsRepository.getById(keyIndicators.getId());
        Validate.notNull(dbKeyIndicators, "根据指定的id键值，未能获取到相应信息");
        Validate.isTrue(dbKeyIndicators.getKeyIndicatorsCode().equals(keyIndicators.getKeyIndicatorsCode()), "关键指标编码不可修改！");
        KeyIndicators newKeyIndicators = nebulaToolkitService.copyObjectByWhiteList(keyIndicators, KeyIndicators.class, HashSet.class, ArrayList.class);
        keyIndicatorsRepository.updateById(newKeyIndicators);

        //编辑业务日志
        KeyIndicatorsLogEventDto logEventDto = new KeyIndicatorsLogEventDto();
        KeyIndicatorsDto oldDto = nebulaToolkitService.copyObjectByWhiteList(dbKeyIndicators, KeyIndicatorsDto.class, HashSet.class, ArrayList.class);
        KeyIndicatorsDto newDto = nebulaToolkitService.copyObjectByWhiteList(newKeyIndicators, KeyIndicatorsDto.class, HashSet.class, ArrayList.class);
        logEventDto.setOriginal(oldDto);
        logEventDto.setNewest(newDto);
        SerializableBiConsumer<KeyIndicatorsLogEventListener, KeyIndicatorsLogEventDto> onUpdate = KeyIndicatorsLogEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, KeyIndicatorsLogEventListener.class, onUpdate);
        return keyIndicators;
    }


    /**
     * 逻辑删除数据
     *
     * @param idList 主键集合
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delete(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        List<KeyIndicators> keyIndicatorsList = keyIndicatorsRepository.findByIds(idList);
        for (KeyIndicators keyIndicators : keyIndicatorsList) {
            keyIndicators.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
        }
        this.keyIndicatorsRepository.updateBatchById(keyIndicatorsList);

        //删除业务日志
        List<KeyIndicatorsDto> kiDtoList = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(keyIndicatorsList, KeyIndicators.class, KeyIndicatorsDto.class, HashSet.class, ArrayList.class));
        SerializableBiConsumer<KeyIndicatorsLogEventListener, KeyIndicatorsLogEventDto> onDelete =
                KeyIndicatorsLogEventListener::onDelete;
        kiDtoList.forEach(kiDto -> {
            KeyIndicatorsLogEventDto logEventDto = new KeyIndicatorsLogEventDto();
            logEventDto.setOriginal(kiDto);
            logEventDto.setNewest(null);
            nebulaNetEventClient.publish(logEventDto, KeyIndicatorsLogEventListener.class, onDelete);
        });

    }


    /**
     * 启用、禁用
     *
     * @param idList       主键集合
     * @param enableStatus 启用/禁用状态
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateEnableStatus(List<String> idList, String enableStatus) {
        this.updateEnableStatusValidate(idList, enableStatus);
        List<KeyIndicators> entityList = keyIndicatorsRepository.findByIds(idList);
        Assert.notEmpty(entityList, "数据不存在!");
        String desc = EnableStatusEnum.getDesc(enableStatus);
        Assert.hasLength(desc, "启禁用状态不合法!");
        entityList.forEach(item -> {
            if (enableStatus.equals(item.getEnableStatus())) {
                throw new IllegalArgumentException("数据已[" + desc + "],不可重复[" + desc + "]");
            }
        });
        ArrayList<KeyIndicators> keyIndicatorsList = new ArrayList<>();
        idList.forEach(id -> {
            KeyIndicators keyIndicators = new KeyIndicators();
            keyIndicators.setId(id);
            keyIndicators.setEnableStatus(enableStatus);
            keyIndicatorsList.add(keyIndicators);
        });
        keyIndicatorsRepository.updateBatchById(keyIndicatorsList);

        //启用禁用业务日志
        List<KeyIndicatorsDto> kiDtoList = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(keyIndicatorsList, KeyIndicators.class, KeyIndicatorsDto.class, HashSet.class, ArrayList.class));
        if (EnableStatusEnum.ENABLE.getCode().equals(enableStatus)) {
            SerializableBiConsumer<KeyIndicatorsLogEventListener, KeyIndicatorsLogEventDto> onEnable =
                    KeyIndicatorsLogEventListener::onEnable;
            kiDtoList.forEach(kiDto -> {
                KeyIndicatorsLogEventDto logEventDto = new KeyIndicatorsLogEventDto();
                logEventDto.setOriginal(kiDto);
                logEventDto.setNewest(null);
                nebulaNetEventClient.publish(logEventDto, KeyIndicatorsLogEventListener.class, onEnable);
            });
        } else {
            SerializableBiConsumer<KeyIndicatorsLogEventListener, KeyIndicatorsLogEventDto> onDisable =
                    KeyIndicatorsLogEventListener::onDisable;
            kiDtoList.forEach(kiDto -> {
                KeyIndicatorsLogEventDto logEventDto = new KeyIndicatorsLogEventDto();
                logEventDto.setOriginal(kiDto);
                logEventDto.setNewest(null);
                nebulaNetEventClient.publish(logEventDto, KeyIndicatorsLogEventListener.class, onDisable);
            });
        }

    }

    /**
     * 更新状态验证
     *
     * @param idList       主键集合
     * @param enableStatus 启用、禁用状态
     */
    private void updateEnableStatusValidate(List<String> idList, String enableStatus) {
        Validate.notEmpty(idList, "主键集合不能为空");
        Validate.notBlank(enableStatus, "启禁用状态不能为空");
        Validate.isTrue(EnableStatusEnum.contains(enableStatus), "未知的启禁用状态");
    }

    /**
     * 创建验证
     *
     * @param keyIndicators 实体对象
     */
    private void createValidate(KeyIndicatorsDto keyIndicators) {
        Validate.notNull(keyIndicators, "对象信息不能为空！");
        Validate.notBlank(keyIndicators.getBusinessFormatCode(), "业态不能为空！");
        Validate.notBlank(keyIndicators.getBusinessUnitCode(), "业务单元不能为空！");
        Validate.notBlank(keyIndicators.getKeyIndicatorsCode(), "关键指标编码不能为空！");
        Validate.notBlank(keyIndicators.getKeyIndicatorsName(), "关键指标名称不能为空！");
        Validate.notBlank(keyIndicators.getIndicatorsType(), "指标类型不能为空！");
        String[] split = keyIndicators.getIndicatorsType().split(",");
        List<String> strings = Arrays.asList(split);
        strings.forEach(type -> {
            Validate.notNull(VariableFunctionEnum.findByCode(type), "未知的指标类型");
        });
    }

    /**
     * 修改验证
     *
     * @param keyIndicators 实体对象
     */
    private void updateValidate(KeyIndicatorsDto keyIndicators) {
        Validate.notNull(keyIndicators, "对象信息不能为空！");
        Validate.notNull(keyIndicators.getId(), "ID不能为空！");
        Validate.notBlank(keyIndicators.getBusinessFormatCode(), "业态不能为空！");
        Validate.notBlank(keyIndicators.getBusinessUnitCode(), "业务单元不能为空！");
        Validate.notBlank(keyIndicators.getKeyIndicatorsName(), "关键指标名称不能为空！");
        Validate.notBlank(keyIndicators.getIndicatorsType(), "指标类型不能为空！");
        String[] split = keyIndicators.getIndicatorsType().split(",");
        List<String> strings = Arrays.asList(split);
        strings.forEach(type -> {
            Validate.notNull(VariableFunctionEnum.findByCode(type), "未知的指标类型");
        });
    }

    private void checkType(KeyIndicatorsDto keyIndicators) {
        List<FormulaVariableRegister> filterList = formulaVariableRegisterList.stream().filter(e -> StringUtils.equals(e.getVariableCode(), keyIndicators.getKeyIndicatorsCode())).collect(Collectors.toList());
        Validate.isTrue(filterList.size() > 0, "关键指标编码[%s]未找到对应注册器！", keyIndicators.getKeyIndicatorsCode());
        List<String> beanTypeList = Arrays.asList(keyIndicators.getIndicatorsType().split(","));
        Set<String> nameSet = new HashSet<>();
        long count = formulaVariableRegisterList.stream()
                .filter(e -> {
                    if (!StringUtils.equals(keyIndicators.getKeyIndicatorsCode(), e.getVariableCode())) {
                        return false;
                    }
                    if (CollectionUtils.isEmpty(e.getFunctionStringList())) {
                        return false;
                    }
                    for (String beanType : beanTypeList) {
                        if (!e.getFunctionStringList().contains(beanType)) {
                            VariableFunctionEnum functionEnum = VariableFunctionEnum.findByCode(beanType);
                            if (Objects.nonNull(functionEnum)) {
                                nameSet.add(functionEnum.getDesc());
                            } else {
                                nameSet.add(beanType);
                            }
                            return false;
                        }
                    }
                    return true;
                }).count();
        Validate.isTrue(count > 0, "关键指标编码[%s]未预设指标类型[%s]！", keyIndicators.getKeyIndicatorsCode(), String.join(",", nameSet));
    }

    private void checkConfigurable(KeyIndicatorsDto keyIndicators) {
        if (StringUtils.isEmpty(keyIndicators.getConfigurable())) {
            keyIndicators.setConfigurable(YesOrNoEnum.NO.getCode());
        }
        List<FormulaVariableRegister> filterList = formulaVariableRegisterList.stream()
                .filter(e -> StringUtils.equals(e.getVariableCode(), keyIndicators.getKeyIndicatorsCode()))
                .collect(Collectors.toList());
        FormulaVariableRegister formulaVariableRegister = filterList.get(0);
        if (YesOrNoEnum.YES.getCode().equals(keyIndicators.getConfigurable())) {
            Validate.isTrue(formulaVariableRegister.isConfigurable(), "关键指标编码[%s]是自定义变量！", keyIndicators.getKeyIndicatorsCode());
        } else {
            Validate.isTrue(!formulaVariableRegister.isConfigurable(), "关键指标编码[%s]非自定义变量！", keyIndicators.getKeyIndicatorsCode());
        }
    }
}
