package com.biz.crm.tpm.business.large.date.config.local.service.internal;


import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.tpm.business.large.date.config.local.entity.TpmLargeDateConfig;
import com.biz.crm.tpm.business.large.date.config.local.repository.TpmLargeDateConfigRepository;
import com.biz.crm.tpm.business.large.date.config.sdk.constant.TpmLargeDateConfigConstant;
import com.biz.crm.tpm.business.large.date.config.sdk.dto.TpmLargeDateConfigDto;
import com.biz.crm.tpm.business.large.date.config.sdk.dto.TpmProductRangeDto;
import com.biz.crm.tpm.business.large.date.config.sdk.dto.TpmSalesOrgDto;
import com.biz.crm.tpm.business.large.date.config.sdk.dto.log.TpmLargeDateConfigLogEventDto;
import com.biz.crm.tpm.business.large.date.config.sdk.event.log.TpmLargeDateConfigEventListener;
import com.biz.crm.tpm.business.large.date.config.sdk.service.TpmLargeDateConfigService;
import com.biz.crm.tpm.business.large.date.config.sdk.service.TpmProductRangeService;
import com.biz.crm.tpm.business.large.date.config.sdk.service.TpmSalesOrgService;
import com.biz.crm.tpm.business.large.date.config.sdk.vo.TpmLargeDateConfigVo;
import com.biz.crm.tpm.business.large.date.config.sdk.vo.TpmProductRangeVo;
import com.biz.crm.tpm.business.large.date.config.sdk.vo.TpmSalesOrgVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 大日期配置(TpmLargeDateConfig)服务实现类
 *
 * @author heding
 * @create 2022-12-05 15:50:12
 */
@Service("tpmLargeDateConfigService")
public class TpmLargeDateConfigServiceImpl implements TpmLargeDateConfigService {
    @Autowired(required = false)
    private TpmLargeDateConfigRepository tpmLargeDateConfigRepository;
    @Autowired(required = false)
    private TpmProductRangeService tpmProductRangeService;
    @Autowired(required = false)
    private TpmSalesOrgService tpmSalesOrgService;
    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;
    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;
    @Autowired(required = false)
    private GenerateCodeService generateCodeService;


    /**
     * 通过主键查询单条数据回显
     *
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public TpmLargeDateConfigVo findById(String id) {
        // 判断id是否为空
        if (StringUtils.isBlank(id)) {
            return null;
        }
        // 根据id查询销量渠道管理主表信息
        TpmLargeDateConfig tpmLargeDateConfig = tpmLargeDateConfigRepository.getById(id);
        if (ObjectUtils.isEmpty(tpmLargeDateConfig)) {
            return null;
        }
        TpmLargeDateConfigVo tpmLargeDateConfigVo = nebulaToolkitService.copyObjectByWhiteList(tpmLargeDateConfig, TpmLargeDateConfigVo.class, LinkedHashSet.class, ArrayList.class);
        List<TpmProductRangeVo> productRangeVoList = tpmProductRangeService.findByListLargeDateCode(tpmLargeDateConfigVo.getLargeDateCode());
        List<TpmSalesOrgVo> tpmSalesOrgVoList = tpmSalesOrgService.findByListLargeDateCode(tpmLargeDateConfigVo.getLargeDateCode());
        tpmLargeDateConfigVo.setProductRangeVoList(productRangeVoList);
        tpmLargeDateConfigVo.setSalesOrgVoList(tpmSalesOrgVoList);
        return tpmLargeDateConfigVo;

    }

    /**
     * 删除数据
     *
     * @param idList 主键集合
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
        List<TpmLargeDateConfig> businessPolicyList1 = tpmLargeDateConfigRepository.listByIds(idList);
        Validate.isTrue(!CollectionUtils.isEmpty(businessPolicyList1), "要删除的数据不能为空");
        List<TpmLargeDateConfig> tpmLargeDateConfigList = new ArrayList<>();
        businessPolicyList1.forEach(item -> {
            TpmLargeDateConfig tpmLargeDateConfig = new TpmLargeDateConfig();
            tpmLargeDateConfig.setId(item.getId());
            tpmLargeDateConfig.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
            tpmLargeDateConfigList.add(tpmLargeDateConfig);
            String largeDateCode = item.getLargeDateCode();
            if (TpmLargeDateConfigConstant.BUSINESS_UNIT_CODE.contains(item.getBusinessUnitCode())) {
                tpmSalesOrgService.deleteByLargeDateCode(largeDateCode);
            }
            tpmProductRangeService.deleteByLargeDateCode(largeDateCode);
            //删除业务日志
            TpmLargeDateConfigLogEventDto logEventDto = new TpmLargeDateConfigLogEventDto();
            TpmLargeDateConfigVo oldVo = nebulaToolkitService.copyObjectByWhiteList(item, TpmLargeDateConfigVo.class, LinkedHashSet.class, ArrayList.class);
            logEventDto.setOriginal(oldVo);
            TpmLargeDateConfigDto tpmLargeDateConfigDto = nebulaToolkitService.copyObjectByWhiteList(item, TpmLargeDateConfigDto.class, LinkedHashSet.class, ArrayList.class);
            tpmLargeDateConfigDto.setDelFlag(tpmLargeDateConfig.getDelFlag());
            logEventDto.setNewest(tpmLargeDateConfigDto);
            SerializableBiConsumer<TpmLargeDateConfigEventListener, TpmLargeDateConfigLogEventDto> onDelete =
                    TpmLargeDateConfigEventListener::onDelete;
            nebulaNetEventClient.publish(logEventDto, TpmLargeDateConfigEventListener.class, onDelete);
        });
        tpmLargeDateConfigRepository.updateBatchById(tpmLargeDateConfigList);
    }

    /**
     * 新增数据
     *
     * @param dto 实体对象
     * @return 新增结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void create(TpmLargeDateConfigDto dto) {
        createValidate(dto);

        // 配置编码
        // 重复校验
        List<TpmProductRangeDto> productRangeDtoList = dto.getProductRangeDtoList();
        List<TpmSalesOrgDto> salesOrgDtoList = dto.getSalesOrgDtoList();
        boolean salesOrgResult = CollectionUtils.isNotEmpty(salesOrgDtoList);
        boolean result = CollectionUtils.isNotEmpty(productRangeDtoList);
        String tenantCode = TenantUtils.getTenantCode();
        String oldCode = repeatValidate(productRangeDtoList, salesOrgDtoList, salesOrgResult, dto);
        if (result && StringUtils.isNotBlank(oldCode)) {
            dto.setTenantCode(tenantCode);
            dto.setLargeDateCode(oldCode);
            tpmLargeDateConfigRepository.updateByConditions(dto);
            return;
        }
        //大日期配置主表
        String delCode = DelFlagStatusEnum.NORMAL.getCode();
        String enableCode = EnableStatusEnum.ENABLE.getCode();
        dto.setDelFlag(delCode);
        dto.setTenantCode(tenantCode);
        dto.setEnableStatus(enableCode);
        String code = generateCodeService.generateCode(TpmLargeDateConfigConstant.LARGE_DATE_CONFIG_PREFIX, 1, 5, 2, TimeUnit.DAYS).get(0);
        dto.setLargeDateCode(code);
        // 保存关联产品范围
        if (result) {
            String codeSet = saveProductData(code, productRangeDtoList, tenantCode, delCode, enableCode);
            dto.setCodeSet(codeSet);
        }
        // 保存关联销售机构
        if (salesOrgResult) {
            String salesOrgCodeSet = saveSalesData(code, salesOrgDtoList, tenantCode, delCode, enableCode);
            dto.setSalesOrgCodeSet(salesOrgCodeSet);
        }
        TpmLargeDateConfig tpmLargeDateConfig = nebulaToolkitService.copyObjectByWhiteList(dto,
                TpmLargeDateConfig.class, LinkedHashSet.class, ArrayList.class);
        //新增业务日志
        TpmLargeDateConfigLogEventDto logEventDto = new TpmLargeDateConfigLogEventDto();
        logEventDto.setOriginal(null);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<TpmLargeDateConfigEventListener, TpmLargeDateConfigLogEventDto> onCreate =
                TpmLargeDateConfigEventListener::onCreate;
        nebulaNetEventClient.publish(logEventDto, TpmLargeDateConfigEventListener.class, onCreate);
        tpmLargeDateConfigRepository.save(tpmLargeDateConfig);

    }

    /**
     * 保存销售机构信息
     *
     * @param code            配置编码
     * @param salesOrgDtoList 销售机构信息
     * @param tenantCode      租户编码
     * @param delCode         删除状态
     * @param enableCode      启用状态
     * @return 销售机构编码集
     */
    private String saveSalesData(String code, List<TpmSalesOrgDto> salesOrgDtoList, String tenantCode, String delCode, String enableCode) {
        String salesOrgCodeSet = "";
        for (TpmSalesOrgDto tpmSalesOrgDto : salesOrgDtoList) {
            salesOrgCodeSet = StringUtils.join(salesOrgCodeSet, " ", tpmSalesOrgDto.getSalesOrgCode());
            tpmSalesOrgDto.setLargeDateCode(code);
            tpmSalesOrgDto.setDelFlag(delCode);
            tpmSalesOrgDto.setEnableStatus(enableCode);
            tpmSalesOrgDto.setTenantCode(tenantCode);
            tpmSalesOrgDto.setId(null);
            tpmSalesOrgService.create(tpmSalesOrgDto);
        }
        return salesOrgCodeSet;
    }

    /**
     * 保存产品范围信息
     *
     * @param code                配置编码
     * @param productRangeDtoList 产品范围信息
     * @param tenantCode          租户编码
     * @param delCode             删除状态
     * @param enableCode          启用状态
     * @return 产品编码集
     */
    private String saveProductData(String code, List<TpmProductRangeDto> productRangeDtoList, String tenantCode, String delCode, String enableCode) {
        String codeSet = "";
        for (TpmProductRangeDto tpmProductRangeDto : productRangeDtoList) {
            Validate.notBlank(tpmProductRangeDto.getProductCode(), "产品编码不能为空！");
            codeSet = StringUtils.join(codeSet, " ", tpmProductRangeDto.getProductCode());
            tpmProductRangeDto.setLargeDateCode(code);
            tpmProductRangeDto.setDelFlag(delCode);
            tpmProductRangeDto.setEnableStatus(enableCode);
            tpmProductRangeDto.setTenantCode(tenantCode);
            tpmProductRangeDto.setId(null);
            tpmProductRangeService.create(tpmProductRangeDto);
        }
        return codeSet;
    }


    /**
     * 分页查询数据
     *
     * @param pageable 分页对象
     * @param dto      查询条件实体
     * @return 所有数据
     */
    @Override
    public Page<TpmLargeDateConfigVo> findByConditions(Pageable pageable, TpmLargeDateConfigDto dto) {
        ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(dto)) {
            dto = new TpmLargeDateConfigDto();
        }
        return tpmLargeDateConfigRepository.findByConditions(pageable, dto);
    }

    /**
     * 编辑数据
     *
     * @param dto 实体对象
     * @return 编辑结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void update(TpmLargeDateConfigDto dto) {
        updateValidate(dto);
        // 销量渠道管理主表
        TpmLargeDateConfigVo oldVo = findById(dto.getId());
        String tenantCode = dto.getTenantCode();
        String delCode = DelFlagStatusEnum.NORMAL.getCode();
        String enableCode = EnableStatusEnum.ENABLE.getCode();
        String code = dto.getLargeDateCode();
        List<TpmProductRangeDto> productRangeDtoList = dto.getProductRangeDtoList();
        List<TpmSalesOrgDto> salesOrgDtoList = dto.getSalesOrgDtoList();
        boolean salesOrgResult = CollectionUtils.isNotEmpty(salesOrgDtoList);
        boolean result = CollectionUtils.isNotEmpty(productRangeDtoList);
        // 保存关联产品范围
        if (result) {
            tpmProductRangeService.deleteByLargeDateCode(code);
            String codeSet = saveProductData(code, productRangeDtoList, tenantCode, delCode, enableCode);
            dto.setCodeSet(codeSet);
        }
        // 保存关联销售机构
        if (salesOrgResult) {
            tpmSalesOrgService.deleteByLargeDateCode(code);
            String salesOrgCodeSet = saveSalesData(code, salesOrgDtoList, tenantCode, delCode, enableCode);
            dto.setSalesOrgCodeSet(salesOrgCodeSet);
        }
        TpmLargeDateConfig tpmLargeDateConfig = nebulaToolkitService.copyObjectByWhiteList(dto,
                TpmLargeDateConfig.class, LinkedHashSet.class, ArrayList.class);
        tpmLargeDateConfigRepository.saveOrUpdate(tpmLargeDateConfig);
        //编辑业务日志
        TpmLargeDateConfigLogEventDto logEventDto = new TpmLargeDateConfigLogEventDto();
        logEventDto.setOriginal(oldVo);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<TpmLargeDateConfigEventListener, TpmLargeDateConfigLogEventDto> onUpdate =
                TpmLargeDateConfigEventListener::onUpdate;
        nebulaNetEventClient.publish(logEventDto, TpmLargeDateConfigEventListener.class, onUpdate);

    }


    /**
     * 启用
     *
     * @param idList 主键集合
     * @return 启用结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void enable(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "启用数据时，主键集合不能为空！");
        List<TpmLargeDateConfig> tpmLargeDateConfigList = tpmLargeDateConfigRepository.listByIds(idList);
        Validate.isTrue(!CollectionUtils.isEmpty(tpmLargeDateConfigList), "启用的数据不能为空");
        ArrayList<TpmLargeDateConfig> activityContractConfigList = new ArrayList<>();
        tpmLargeDateConfigList.forEach(item -> {
            TpmLargeDateConfig tpmLargeDateConfig = new TpmLargeDateConfig();
            tpmLargeDateConfig.setId(item.getId());
            tpmLargeDateConfig.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            activityContractConfigList.add(tpmLargeDateConfig);

            //启用业务日志
            TpmLargeDateConfigLogEventDto logEventDto = new TpmLargeDateConfigLogEventDto();
            TpmLargeDateConfigVo oldVo = nebulaToolkitService.copyObjectByWhiteList(item, TpmLargeDateConfigVo.class, LinkedHashSet.class, ArrayList.class);
            logEventDto.setOriginal(oldVo);
            TpmLargeDateConfigDto tpmLargeDateConfigDto = nebulaToolkitService.copyObjectByWhiteList(item, TpmLargeDateConfigDto.class, LinkedHashSet.class, ArrayList.class);
            tpmLargeDateConfigDto.setEnableStatus(tpmLargeDateConfig.getEnableStatus());
            logEventDto.setNewest(tpmLargeDateConfigDto);
            SerializableBiConsumer<TpmLargeDateConfigEventListener, TpmLargeDateConfigLogEventDto> onEnable =
                    TpmLargeDateConfigEventListener::onEnable;
            nebulaNetEventClient.publish(logEventDto, TpmLargeDateConfigEventListener.class, onEnable);
        });
        tpmLargeDateConfigRepository.updateBatchById(activityContractConfigList);
    }

    /**
     * 禁用
     *
     * @param idList 主键集合
     * @return 禁用结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void disable(List<String> idList) {
        Validate.isTrue(!CollectionUtils.isEmpty(idList), "禁用数据时，主键集合不能为空！");
        List<TpmLargeDateConfig> tpmLargeDateConfigList = tpmLargeDateConfigRepository.listByIds(idList);
        Validate.isTrue(!CollectionUtils.isEmpty(tpmLargeDateConfigList), "禁用的数据不能为空");
        ArrayList<TpmLargeDateConfig> activityContractConfigList = new ArrayList<>();
        tpmLargeDateConfigList.forEach(item -> {
            TpmLargeDateConfig tpmLargeDateConfig = new TpmLargeDateConfig();
            tpmLargeDateConfig.setId(item.getId());
            tpmLargeDateConfig.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
            activityContractConfigList.add(tpmLargeDateConfig);

            //禁用业务日志
            TpmLargeDateConfigLogEventDto logEventDto = new TpmLargeDateConfigLogEventDto();
            TpmLargeDateConfigVo oldVo = nebulaToolkitService.copyObjectByWhiteList(item, TpmLargeDateConfigVo.class, LinkedHashSet.class, ArrayList.class);
            logEventDto.setOriginal(oldVo);
            TpmLargeDateConfigDto tpmLargeDateConfigDto = nebulaToolkitService.copyObjectByWhiteList(item, TpmLargeDateConfigDto.class, LinkedHashSet.class, ArrayList.class);
            tpmLargeDateConfigDto.setEnableStatus(tpmLargeDateConfig.getEnableStatus());
            logEventDto.setNewest(tpmLargeDateConfigDto);
            SerializableBiConsumer<TpmLargeDateConfigEventListener, TpmLargeDateConfigLogEventDto> onDisable =
                    TpmLargeDateConfigEventListener::onDisable;
            nebulaNetEventClient.publish(logEventDto, TpmLargeDateConfigEventListener.class, onDisable);
        });
        tpmLargeDateConfigRepository.updateBatchById(activityContractConfigList);
    }

    @Override
    public Map<String, List<TpmLargeDateConfigVo>> findAllProductConfig() {
        List<TpmLargeDateConfigVo> list = tpmLargeDateConfigRepository.findAllProductConfig(TpmLargeDateConfigConstant.FORMULANAME,
                EnableStatusEnum.ENABLE.getCode());
        if (CollectionUtils.isNotEmpty(list)) {
            List<TpmLargeDateConfigVo> largeDateConfigVos = new ArrayList<>();
            String lessThan = TpmLargeDateConfigConstant.LESS_THAN;
            list.forEach(l->{
                String formulaName = l.getFormulaName();
                if (StringUtils.isNotBlank(formulaName) && formulaName.contains(lessThan)) {
                    largeDateConfigVos.add(l);
                }
            });
            if (CollectionUtils.isNotEmpty(largeDateConfigVos) && largeDateConfigVos.size() > 0) {
                return largeDateConfigVos.stream().collect(Collectors.groupingBy(l -> l.getBusinessFormatCode() + l.getBusinessUnitCode()));
            }
        }
        return null;
    }


    /**
     * 创建验证
     *
     * @param dto
     */
    private void createValidate(TpmLargeDateConfigDto dto) {
        Validate.notNull(dto, "新增时，对象信息不能为空！");
        dto.setId(null);
        Validate.notBlank(dto.getLargeDateName(), "新增数据时，配置名称不能为空！");
        Validate.notBlank(dto.getBusinessFormatCode(), "新增数据时，业态不能为空！");
        Validate.notBlank(dto.getBusinessUnitCode(), "新增数据时，业务单元不能为空！");
        if (TpmLargeDateConfigConstant.BUSINESS_UNIT_CODE.contains(dto.getBusinessUnitCode())){
            Validate.notNull(dto.getSalesOrgDtoList(),"新增数据时，销售机构信息不能为空！");
        }

    }

    /**
     * 修改验证
     *
     * @param dto
     */
    private void updateValidate(TpmLargeDateConfigDto dto) {
        Validate.notNull(dto, "修改时，对象信息不能为空！");
        Validate.notBlank(dto.getId(), "修改数据时，主键id不能为空！");
        Validate.notBlank(dto.getTenantCode(), "修改数据时，租户编号不能为空！");
        Validate.notBlank(dto.getLargeDateCode(), "修改数据时，配置编码不能为空！");
        Validate.notBlank(dto.getLargeDateName(), "修改数据时，配置名称不能为空！");
        Validate.notBlank(dto.getBusinessFormatCode(), "修改数据时，业态不能为空！");
        Validate.notBlank(dto.getBusinessUnitCode(), "修改数据时，业务单元不能为空！");
        if (TpmLargeDateConfigConstant.BUSINESS_UNIT_CODE.contains(dto.getBusinessUnitCode())){
            Validate.notNull(dto.getSalesOrgDtoList(),"修改数据时，销售机构信息不能为空！");
        }
    }

    /**
     * 数据库验重 业态+业务单元+（如果业务单元是分子公司，那么维度还有个销售机构）+产品
     *
     * @param dto 实体对象
     */
    private String repeatValidate(List<TpmProductRangeDto> productRangeDtoList, List<TpmSalesOrgDto> salesOrgDtoList, boolean salesOrgResult, TpmLargeDateConfigDto dto) {
        List<TpmLargeDateConfigVo> tpmLargeDateConfigVos = tpmLargeDateConfigRepository.repeatValidate(dto);
        if (CollectionUtils.isNotEmpty(tpmLargeDateConfigVos)) {
            String codeSet = "";
            for (TpmProductRangeDto tpmProductRangeDto : productRangeDtoList) {
                codeSet = StringUtils.join(codeSet, " ", tpmProductRangeDto.getProductCode());
            }
            for (TpmLargeDateConfigVo tpmLargeDateConfigVo : tpmLargeDateConfigVos) {
                if (tpmLargeDateConfigVo.getCodeSet().equals(codeSet)) {
                    String largeDateCode = tpmLargeDateConfigVo.getLargeDateCode();
                    if (salesOrgResult) {
                        String salesOrgCodeSet = "";
                        for (TpmSalesOrgDto tpmSalesOrgDto : salesOrgDtoList) {
                            salesOrgCodeSet = StringUtils.join(salesOrgCodeSet, " ", tpmSalesOrgDto.getSalesOrgCode());
                        }
                        String sales = tpmLargeDateConfigVo.getSalesOrgCodeSet();
                        return salesOrgCodeSet.equals(sales) ? largeDateCode : null;
                    }
                    return largeDateCode;
                }
            }
        }
        return null;
    }
}
