package com.biz.crm.tpm.business.material.purchasing.order.local.controller;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.tpm.business.material.purchasing.order.local.consumer.ActivityDetailPlanPassMaterialPurchasingOrderConsumer;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.constant.TpmMaterialPurchasingOrderLockConstant;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.dto.TpmMaterialPurchasingOrderDetailDto;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.dto.TpmMaterialPurchasingOrderDto;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.dto.TpmMaterialPurchasingOrderfindCustomerDto;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.service.TpmMaterialPurchasingOrderService;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.vo.*;
import com.bizunited.nebula.common.util.JsonUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * TPM-物料采购单(TpmMaterialPurchasingOrder)表相关的http接口
 * @author duyiran
 * @since 2022-11-12 17:55:48
 */
@RestController
@RequestMapping("/v1/tpmMaterialPurchasingOrder")
@Slf4j
@Api(tags = "TPM-物料采购单")
public class TpmMaterialPurchasingOrderController {
    /**
     * 服务对象
     */
    @Autowired(required = false)
    private TpmMaterialPurchasingOrderService tpmMaterialPurchasingOrderService;

    @Autowired(required = false)
    private ActivityDetailPlanPassMaterialPurchasingOrderConsumer activityDetailPlanPassMaterialPurchasingOrderConsumer;

    @Autowired(required = false)
    private RedisLockService redisLockService;


    /**
     * 分页查询所有数据
     * @param pageable 分页对象
     * @param dto      查询dto
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findByConditions")
    public Result<Page<TpmMaterialPurchasingOrderVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                       @ApiParam(name = "dto", value = "TPM-物料采购单") TpmMaterialPurchasingOrderDto dto) {
        try {
            Page<TpmMaterialPurchasingOrderVo> page = this.tpmMaterialPurchasingOrderService.findByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 添加详情
     * @param
     * @return result
     */
    @ApiOperation(value = "添加详情")
    @PostMapping("createDetailList")
    public Result<?> addDetailList(@RequestBody TpmMaterialPurchasingOrderDto dto) {
        try {
            this.tpmMaterialPurchasingOrderService.addDetailList(dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据详情id查询
     * @param
     * @return result
     */
    @ApiOperation(value = "根据详情id查询")
    @GetMapping("findDetailById")
    public Result<TpmMaterialPurchasingOrderDetailVo> findDetailById(@RequestParam("id") String id) {
        try {
            TpmMaterialPurchasingOrderDetailVo detailVo = this.tpmMaterialPurchasingOrderService.findDetailById(id);
            return Result.ok(detailVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 物料组编辑
     * @param
     * @return result
     */
    @ApiOperation(value = "物料组编辑")
    @PostMapping("materialGroupUpdate")
    public Result<?> materialGroupUpdate(@RequestBody TpmMaterialPurchasingOrderDto dto) {
        boolean lockSuccess = false;
        try {
            Validate.notEmpty(dto.getId(), "主键参数不能为空");
            //加个锁
            lockSuccess = this.redisLockService.tryLock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK + dto.getId(), TimeUnit.SECONDS, 30);
            Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            this.tpmMaterialPurchasingOrderService.materialGroupUpdate(dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }finally {
            if (lockSuccess) {
                this.redisLockService.unlock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK + dto.getId());
            }
        }
    }

    /**
     * 供应商编辑
     * @param
     */
    @ApiOperation(value = "供应商编辑")
    @PostMapping("supplierUpdate")
    public Result<?> supplierUpdate(@RequestBody TpmMaterialPurchasingOrderDto dto) {
        try {
            Validate.notEmpty(dto.getId(), "主键参数不能为空");
            //加个锁
            boolean lock = this.redisLockService.tryLock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK + dto.getId(), TimeUnit.SECONDS, 30);
            Validate.isTrue(lock, "其他人正在操作该条数据,请稍后再提交");
            this.tpmMaterialPurchasingOrderService.supplierUpdate(dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }finally {
            this.redisLockService.unlock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK + dto.getId());
        }
    }

    /**
     * 出库单确认
     * @param
     */
    @ApiOperation(value = "出库单确认")
    @PostMapping("outConfirm")
    public Result<?> outConfirm(@RequestBody TpmMaterialPurchasingOrderDto dto) {
        try {
            Validate.notEmpty(dto.getIds(), "ids不能为空");
            //加个锁
            boolean lock = this.redisLockService.batchLock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK, dto.getIds(), TimeUnit.SECONDS, 60);
            Validate.isTrue(lock, "防重复提交,请稍后再提交");
            this.tpmMaterialPurchasingOrderService.outConfirm(dto);
            return Result.ok();
        } catch (Exception e) {
            log.info("确认报错,{}", ExceptionUtils.getStackTrace(e));
            return Result.error(e.getMessage());
        } finally {
            if (CollectionUtils.isNotEmpty(dto.getIds())) {
                this.redisLockService.batchUnLock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK, dto.getIds());
            }
        }
    }

    /**
     * 出库单撤回
     * @param
     */
    @ApiOperation(value = "出库单撤回")
    @PostMapping("outCancel")
    public Result<?> outCancel(@RequestBody TpmMaterialPurchasingOrderDto dto) {
        try {
            Validate.notEmpty(dto.getIds(), "ids不能为空");
            //加个锁
            boolean lock = this.redisLockService.batchLock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK, dto.getIds(), TimeUnit.SECONDS, 60);
            Validate.isTrue(lock, "防重复提交,请稍后再提交");
            this.tpmMaterialPurchasingOrderService.outCancel(dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (CollectionUtils.isNotEmpty(dto.getIds())) {
                this.redisLockService.batchUnLock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK, dto.getIds());
            }
        }
    }

    /**
     * 出库单催签
     * @param
     */
    @ApiOperation(value = "出库单催签")
    @PostMapping("outExpediting")
    public Result<?> outExpediting(@RequestBody TpmMaterialPurchasingOrderDto dto) {
        try {
            Validate.notEmpty(dto.getIds(), "ids不能为空");
            //加个锁
            boolean lock = this.redisLockService.batchLock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK, dto.getIds(), TimeUnit.SECONDS, 60);
            Validate.isTrue(lock, "防重复提交,请稍后再提交");
            this.tpmMaterialPurchasingOrderService.outExpediting(dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (CollectionUtils.isNotEmpty(dto.getIds())) {
                this.redisLockService.batchUnLock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK, dto.getIds());
            }
        }
    }

    /**
     * 批量下载订单确认表
     * @param ids
     */
    @ApiOperation(value = "批量下载订单确认表")
    @GetMapping("downloadOrderConfirmBatch")
    public Result<List<TpmMaterialPurchasingOrderFileVo>> downloadOrderConfirmBatch(@RequestParam("ids") List<String> ids) {
        try {
            List<TpmMaterialPurchasingOrderFileVo> result = this.tpmMaterialPurchasingOrderService.downloadOrderConfirmBatch(ids);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 批量下载签收单
     * @param ids
     */
    @ApiOperation(value = "批量下载签收单")
    @GetMapping("downloadOutConfirmBatch")
    public Result<List<TpmMaterialPurchasingOrderFileVo>> downloadOutConfirmBatch(@RequestParam("ids") List<String> ids) {
        try {
            List<TpmMaterialPurchasingOrderFileVo> result = this.tpmMaterialPurchasingOrderService.downloadOutConfirmBatch(ids);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 上传签收单
     * @param dto
     */
    @ApiOperation(value = "上传签收单")
    @PostMapping("uploadOutConfirm")
    public Result<?> uploadOutConfirm(@RequestBody TpmMaterialPurchasingOrderDetailDto dto) {
        try {
            this.tpmMaterialPurchasingOrderService.uploadOutConfirm(dto);
            return Result.ok();
        } catch (Exception e) {
            log.info("上传签收单报错了:{}", ExceptionUtils.getStackTrace(e));
            return Result.error(e.getMessage());
        }
    }

    /**
     * 批量提交审批
     * @param dto
     */
    @ApiOperation(value = "批量提交审批")
    @PostMapping("submitProcess")
    public Result<?> submitProcess(@RequestBody TpmMaterialPurchasingOrderDto dto) {
        try {
            Validate.notEmpty(dto.getDetailList(), "明细不能为空");
            //加个锁
            boolean lock = this.redisLockService.batchLock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK, dto.getDetailList().stream().map(TpmMaterialPurchasingOrderDetailDto::getId).collect(Collectors.toList()), TimeUnit.SECONDS, 60);
            Validate.isTrue(lock, "防重复提交,请稍后再提交");
            this.tpmMaterialPurchasingOrderService.submitProcess(dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (CollectionUtils.isNotEmpty(dto.getDetailList())) {
                this.redisLockService.batchUnLock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK, dto.getDetailList().stream().map(TpmMaterialPurchasingOrderDetailDto::getId).collect(Collectors.toList()));
            }
        }
    }


    /**
     * 通过主键查询单条数据
     * @param id 主键
     * @return 单条数据
     */
    @ApiOperation(value = "通过主键查询单条数据")
    @GetMapping("{id}")
    public Result<TpmMaterialPurchasingOrderVo> findById(@PathVariable @ApiParam(name = "id", value = "主键id") String id) {
        try {
            TpmMaterialPurchasingOrderVo tpmMaterialPurchasingOrder = this.tpmMaterialPurchasingOrderService.findById(id);
            return Result.ok(tpmMaterialPurchasingOrder);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 上传核销资料
     * @param dto dto
     */
    @ApiOperation(value = "上传核销资料")
    @PatchMapping("updateAuditFile")
    public Result<?> updateAuditFile(@ApiParam(name = "dto", value = "dto") @RequestBody TpmMaterialPurchasingOrderDto dto) {
        try {
            this.tpmMaterialPurchasingOrderService.updateAuditFile(dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("物料费结案查询核销资料")
    @GetMapping("auditFindMaterialAuditInfo")
    public Result<Page<TpmMaterialPurchasingOrderAuditFileVo>> auditFindMaterialAuditInfo(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                                          @ApiParam(name = "dto", value = "TPM-物料采购单") TpmMaterialPurchasingOrderDto dto){
        try {

            log.info("物料费结案查询核销资料,materialPurchasingOrders：{}", JsonUtils.obj2JsonString(dto.getMaterialPurchasingOrders()));
            return Result.ok(this.tpmMaterialPurchasingOrderService.auditFindMaterialAuditInfo(pageable,dto.getMaterialPurchasingOrders()));
        } catch (Exception e){
            log.error(e.getMessage(),e);
            return Result.error(e.getMessage());
        }
    }
    
    @ApiOperation("批量删除物料采购单")
    @PostMapping("batchDeleteMaterialPurchaseOrder")
    public Result<?> batchDeleteMaterialPurchaseOrder(@RequestBody List<String> ids){
        try {
            tpmMaterialPurchasingOrderService.batchDeleteMaterialPurchaseOrder(ids);
            return Result.ok();
        }catch (Exception e){
            log.error(e.getMessage(),e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 查询核销资料
     * @param id
     */
    @ApiOperation(value = "查询核销资料")
    @GetMapping("getAuditFile")
    public Result<TpmMaterialPurchasingOrderVo> getAuditFile(@ApiParam(name = "id", value = "id") @RequestParam("id") String id) {
        try {
            TpmMaterialPurchasingOrderVo orderVo = this.tpmMaterialPurchasingOrderService.getAuditFile(id);
            return Result.ok(orderVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 采购单确认
     * @param dto dto
     * @return 确认结果
     */
    @ApiOperation(value = "采购单确认")
    @PatchMapping("confirm")
    public Result<?> confirm(@ApiParam(name = "idList", value = "主键集合") @RequestBody TpmMaterialPurchasingOrderDto dto) {
        boolean lockSuccess = false;
        try {
            Validate.notEmpty(dto.getIds(), "ids不能为空");
            //加个锁
            lockSuccess = this.redisLockService.batchLock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK, dto.getIds(), TimeUnit.SECONDS, 60);
            Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            this.tpmMaterialPurchasingOrderService.confirm(dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess
                    && CollectionUtils.isNotEmpty(dto.getIds())) {
                this.redisLockService.batchUnLock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK, dto.getIds());
            }
        }
    }


    /**
     * 分页查询所有明细（核销用）
     * @param pageable 分页对象
     * @param dto      查询dto
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据（核销用）")
    @GetMapping("findDetailListByCodeForAuditPage")
    public Result<Page<TpmMaterialPurchasingOrderDetailVo>> findDetailListByCodeForAuditPage(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                                             @ApiParam(name = "dto", value = "TPM-物料采购单") TpmMaterialPurchasingOrderDto dto) {
        try {
            Page<TpmMaterialPurchasingOrderDetailVo> page = this.tpmMaterialPurchasingOrderService.findDetailListByCodeForAuditPage(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 分页查询当前采购单所有客户
     * @param pageable 分页对象
     * @param dto      查询dto
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询当前采购单所有客户")
    @GetMapping("findCustomerById")
    public Result<Page<TpmMaterialPurchasingOrderFindCustomerVo>> findCustomerById(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                                   @ApiParam(name = "dto", value = "TPM-物料采购单-查询客户") TpmMaterialPurchasingOrderfindCustomerDto dto) {
        try {
            Page<TpmMaterialPurchasingOrderFindCustomerVo> page = this.tpmMaterialPurchasingOrderService.findCustomerByCode(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 分页查询当前采购单所有可发货明细
     * @param pageable 分页对象
     * @param dto      查询dto
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询当前采购单所有可发货明细")
    @GetMapping("findOutStorageAbleDetailById")
    public Result<Page<TpmMaterialPurchasingOrderDetailVo>> findOutStorageAbleDetailById(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                                         @ApiParam(name = "dto", value = "TPM-物料采购单-查询客户") TpmMaterialPurchasingOrderDto dto) {
        try {
            Page<TpmMaterialPurchasingOrderDetailVo> page = this.tpmMaterialPurchasingOrderService.findOutStorageAbleDetailById(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 修改客户地址
     * @param dto dto
     * @return 修改结果
     */
    @ApiOperation(value = "修改客户地址")
    @PatchMapping("editCustomerById")
    public Result<?> editCustomerById(@ApiParam(name = "dto", value = "TPM-物料采购单-查询客户") @RequestBody TpmMaterialPurchasingOrderfindCustomerDto dto) {
        boolean lockSuccess = false;
        try {
            //加个锁
            lockSuccess = this.redisLockService.tryLock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK + dto.getId(), TimeUnit.SECONDS, 30);
            Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            this.tpmMaterialPurchasingOrderService.editCustomerAddressById(dto);
            return Result.ok("修改客户地址成功！");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess) {
                this.redisLockService.unlock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK + dto.getId());
            }
        }
    }

    /**
     * 完成出库
     * @return 完成出库结果
     */
    @ApiOperation(value = "完成出库")
    @PatchMapping("finishOutStorage")
    public Result<?> finishOutStorage(@ApiParam(name = "dto", value = "TPM-物料采购单") @RequestBody TpmMaterialPurchasingOrderDto dto) {
        boolean lockSuccess = false;
        try {
            //加个锁
            lockSuccess = this.redisLockService.tryLock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK + dto.getId(), TimeUnit.SECONDS, 30);
            Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
            tpmMaterialPurchasingOrderService.finishOutStorage(dto);
            return Result.ok("完成出库成功！");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            if (lockSuccess) {
                this.redisLockService.unlock(TpmMaterialPurchasingOrderLockConstant.TEMPLATE_CHECK_REPEAT_LOCK + dto.getId());
            }
        }
    }


    /**
     * 结案核销修改 已结案金额、是否完全结案状态
     */
    @ApiOperation(value = "结案核销修改 已结案金额、是否完全结案状态")
    @PatchMapping("audit")
    public Result<?> audit(@ApiParam(name = "dto", value = "物料采购单") @RequestBody List<TpmMaterialPurchasingOrderDto> dto) {
        try {
            this.tpmMaterialPurchasingOrderService.audit(dto);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
}
