package com.biz.crm.tpm.business.material.purchasing.order.local.imports;

import cn.hutool.core.collection.CollectionUtil;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.tpm.business.material.purchasing.order.local.service.TpmMaterialPurchasingOrderDetailService;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.dto.TpmMaterialPurchasingOrderDetailDto;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.enums.TpmMaterialPurchasingOrderEnums;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.vo.TpmMaterialPurchasingOrderSupplierImportVo;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 物料采购单供应商编辑导入
 * @author zhouyang
 * @date 2023/09/22
 */
@Slf4j
@Component
public class TpmMaterialPurchasingOrderSupplierImportsProcess implements ImportProcess<TpmMaterialPurchasingOrderSupplierImportVo> {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired
    private TpmMaterialPurchasingOrderDetailService detailService;


    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, TpmMaterialPurchasingOrderSupplierImportVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Object cacheKey = params.get("cacheKey");
        Validate.notNull(cacheKey, "缓存cacheKey不能为空");

        SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd");
        if (CollectionUtil.isNotEmpty(data)){
            Collection<TpmMaterialPurchasingOrderSupplierImportVo> importVos = data.values();
            if (CollectionUtil.isNotEmpty(importVos)){
                List<TpmMaterialPurchasingOrderSupplierImportVo> importVoList = Lists.newArrayList(importVos);
                for (int i = 0; i < importVoList.size(); i++) {
                    TpmMaterialPurchasingOrderSupplierImportVo importVo = importVoList.get(i);
                    Integer index = i+1;
                    Validate.notEmpty(importVo.getId(),"第"+index+"行,主键不能为空");
                    if(StringUtils.isNotEmpty(importVo.getDeliveryDateStr())) {
                        try {
                            df.parse(importVo.getDeliveryDateStr());
                        } catch (ParseException e) {
                            Validate.notEmpty(importVo.getDeliveryDateStr(), "第" + index + "行,发货时间格式不正确");
                        }
                    }
                    if(StringUtils.isNotEmpty(importVo.getRecipientDateStr())) {
                        try {
                            df.parse(importVo.getRecipientDateStr());
                        } catch (ParseException e) {
                            Validate.notEmpty(importVo.getRecipientDateStr(), "第" + index + "行,收货时间格式不正确");
                        }
                    }
                }
                List<TpmMaterialPurchasingOrderDetailDto> detailVoList = this.detailService.findCacheList((String) cacheKey);
                Map<String,TpmMaterialPurchasingOrderSupplierImportVo> importVoMap = importVos.stream().collect(Collectors.toMap(TpmMaterialPurchasingOrderSupplierImportVo::getId, Function.identity()));
                detailVoList.forEach(detailVo -> {
                    if (importVoMap.containsKey(detailVo.getId())) {
                        TpmMaterialPurchasingOrderSupplierImportVo importVo = importVoMap.get(detailVo.getId());
                        detailVo.setDeliveryDate(importVo.getDeliveryDateStr());
                        detailVo.setRecipientDate(importVo.getRecipientDateStr());
                        detailVo.setMaterialCategory(importVo.getMaterialCategory());
                        detailVo.setPackingBoxNumber(importVo.getPackingBoxNumber());
                        detailVo.setUnit(importVo.getUnit());
                        detailVo.setMaterialBaseStandard(importVo.getMaterialBaseStandard());
                        detailVo.setWeixinNo(importVo.getWeixinNo());
                        detailVo.setDeliveryContacts(importVo.getDeliveryContacts());
                        detailVo.setDeliveryPhone(importVo.getDeliveryPhone());
                        if(ProcessStatusEnum.COMMIT.getDictCode().equals(detailVo.getDetailStatus())||
                                ProcessStatusEnum.PASS.getDictCode().equals(detailVo.getDetailStatus())
                        ){} else {
                            detailVo.setDeliveryQuantity(importVo.getDeliveryQuantity());
                            Validate.isTrue(importVo.getDeliveryQuantity().compareTo(Optional.ofNullable(detailVo.getPurchasingQuantity()).orElse(BigDecimal.ZERO)) <= 0, "发货数量不能大于采购数量");
                            if(TpmMaterialPurchasingOrderEnums.ReceiveStatus.PORTIAL_DELIVERIED.getValue().equals(detailVo.getReceivingStatus())) {
                                Validate.isTrue(importVo.getDeliveryQuantity().compareTo(Optional.ofNullable(detailVo.getReceivingQuantity()).orElse(BigDecimal.ZERO)) >= 0, "发货数量应大于等于签收数量");
                            }

                        }
                    }
                });
                this.detailService.addItemCache((String) cacheKey, detailVoList);
            }
        }
        return null;
    }

    @Override
    public Class<TpmMaterialPurchasingOrderSupplierImportVo> findCrmExcelVoClass() {
        return TpmMaterialPurchasingOrderSupplierImportVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "TPM_MATERIAL_PURCHASING_ORDER_SUPPLIER_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "TPM-物料采购单明细导入-供应商";
    }

    @Override
    public String getBusinessCode() {
        return "TPM_MATERIAL_PURCHASING_ORDER_SUPPLIER_IMPORT";
    }

    @Override
    public String getBusinessName() {
        return "TPM-物料采购单明细供应商导入";
    }

}
