package com.biz.crm.tpm.business.material.purchasing.order.local.repository;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.tpm.business.material.purchasing.order.local.entity.TpmMaterialPurchasingOrderDetail;
import com.biz.crm.tpm.business.material.purchasing.order.local.mapper.TpmMaterialPurchasingOrderDetailMapper;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.dto.TpmMaterialPurchasingOrderDetailDto;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.dto.TpmMaterialPurchasingOrderDto;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.dto.TpmMaterialPurchasingOrderfindCustomerDto;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.vo.TpmMaterialPurchasingOrderDetailVo;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.vo.TpmMaterialPurchasingOrderFindCustomerVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.*;


/**
 * TPM-物料采购单-明细表(TpmMaterialPurchasingOrderDetail)表数据库访问层
 * @author duyiran
 * @since 2022-11-12 17:56:44
 */
@Component
@Slf4j
public class TpmMaterialPurchasingOrderDetailRepository extends ServiceImpl<TpmMaterialPurchasingOrderDetailMapper, TpmMaterialPurchasingOrderDetail> {

    @Autowired(required = false)
    private TpmMaterialPurchasingOrderDetailMapper tpmMaterialPurchasingOrderDetailMapper;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 根据编码查询明细
     * @param code 编码
     * @return
     */
    public List<TpmMaterialPurchasingOrderDetailVo> findByCode(String code) {
        if (StringUtils.isEmpty(code)) {
            return Lists.newArrayList();
        }
        String tenantCode = TenantUtils.getTenantCode();
        List<TpmMaterialPurchasingOrderDetail> details = this.lambdaQuery()
                .eq(TpmMaterialPurchasingOrderDetail::getCode, code)
                .eq(TpmMaterialPurchasingOrderDetail::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TpmMaterialPurchasingOrderDetail::getTenantCode, tenantCode)
                .list();
        Collection<TpmMaterialPurchasingOrderDetailVo> conditions = nebulaToolkitService.copyCollectionByWhiteList(details, TpmMaterialPurchasingOrderDetail.class,
                                                                                                                   TpmMaterialPurchasingOrderDetailVo.class, HashSet.class, ArrayList.class);
        return Lists.newArrayList(conditions);
    }

    /**
     * 根据编码查询明细
     * @param codList 编码
     * @return
     */
    public List<TpmMaterialPurchasingOrderDetail> findByCodeList(List<String> codList) {
        if (CollectionUtils.isEmpty(codList)) {
            return Lists.newArrayList();
        }
        String tenantCode = TenantUtils.getTenantCode();
        return this.lambdaQuery()
                .in(TpmMaterialPurchasingOrderDetail::getCode, codList)
                .eq(TpmMaterialPurchasingOrderDetail::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TpmMaterialPurchasingOrderDetail::getTenantCode, tenantCode)
                .orderByDesc(TpmMaterialPurchasingOrderDetail::getSubCode)
                .list();
    }

    /**
     * 根据编码查询明细
     * @param dto dto
     * @return
     */
    public List<TpmMaterialPurchasingOrderDetail> findByCodeAndCustomerCodeList(TpmMaterialPurchasingOrderfindCustomerDto dto) {
        String code = dto.getCode();
        String tenantCode = TenantUtils.getTenantCode();
        return this.lambdaQuery()
                .eq(TpmMaterialPurchasingOrderDetail::getCode, code)
                .eq(TpmMaterialPurchasingOrderDetail::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TpmMaterialPurchasingOrderDetail::getTenantCode, tenantCode)
                .in(TpmMaterialPurchasingOrderDetail::getCustomerCode, dto.getCustomerCodeList())
                .list();
    }

    /**
     * 根据细案明细编码查询明细
     * @param detailPlanItemCodeList detailPlanItemCodeList
     * @return 明细
     */
    public List<TpmMaterialPurchasingOrderDetail> findByDetailPlanItemCodeList(List<String> detailPlanItemCodeList) {
        if (CollectionUtils.isEmpty(detailPlanItemCodeList)) {
            return Lists.newArrayList();
        }
        String tenantCode = TenantUtils.getTenantCode();
        return this.lambdaQuery()
                .eq(TpmMaterialPurchasingOrderDetail::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TpmMaterialPurchasingOrderDetail::getTenantCode, tenantCode)
                .in(TpmMaterialPurchasingOrderDetail::getActivityDetailItemCode, detailPlanItemCodeList)
                .list();
    }

    /**
     * 根据code删除明细
     * @param code
     */
    public void removeByCode(String code) {
        if (StringUtils.isEmpty(code)) {
            return;
        }
        String tenantCode = TenantUtils.getTenantCode();
        this.lambdaUpdate()
                .eq(TpmMaterialPurchasingOrderDetail::getCode, code)
                .eq(TpmMaterialPurchasingOrderDetail::getTenantCode, tenantCode)
                .remove();
    }

    /**
     * 根据编码查询明细
     * @param pageable 分页对象
     * @param dto      dto
     * @return
     */
    public Page<TpmMaterialPurchasingOrderFindCustomerVo> findCustomerByCode(Pageable pageable, TpmMaterialPurchasingOrderfindCustomerDto dto) {
        Page<TpmMaterialPurchasingOrderFindCustomerVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.tpmMaterialPurchasingOrderDetailMapper.findCustomerByCode(page, dto);
    }

    /**
     * 分页查询当前采购单所有可发货明细
     * @param pageable 分页对象
     * @param dto      查询dto
     * @return 所有数据
     */
    public Page<TpmMaterialPurchasingOrderDetailVo> findOutStorageAbleDetailById(Pageable pageable, TpmMaterialPurchasingOrderDto dto) {
        Page<TpmMaterialPurchasingOrderDetailVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.tpmMaterialPurchasingOrderDetailMapper.findOutStorageAbleDetailById(page, dto);
    }

    /**
     * 根据明细编码查找发货明细实体
     * @param subCode subCode
     * @return 实体
     */
    public TpmMaterialPurchasingOrderDetail getBySubCode(String subCode) {
        return this.lambdaQuery()
                .eq(TpmMaterialPurchasingOrderDetail::getSubCode, subCode)
                .eq(TpmMaterialPurchasingOrderDetail::getTenantCode, TenantUtils.getTenantCode())
                .one();
    }

    public List<TpmMaterialPurchasingOrderDetail> getByCode(String code) {
        if (StringUtils.isEmpty(code)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .eq(TpmMaterialPurchasingOrderDetail::getCode, code)
                .eq(TpmMaterialPurchasingOrderDetail::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TpmMaterialPurchasingOrderDetail::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    public void deleteByCode(String code) {
        this.lambdaUpdate()
                .eq(TpmMaterialPurchasingOrderDetail::getCode, code)
                .remove();
    }

    public List<TpmMaterialPurchasingOrderDetail> findByActivityDetailItemCodes(List<String> activityDetailPlanItemCodes) {
        if (CollectionUtils.isEmpty(activityDetailPlanItemCodes)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .in(TpmMaterialPurchasingOrderDetail::getActivityDetailItemCode, activityDetailPlanItemCodes)
                .list();
    }

    public TpmMaterialPurchasingOrderDetail findByDetailCode(String detailCode) {
        if (StringUtils.isEmpty(detailCode)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(TpmMaterialPurchasingOrderDetail::getDetailCode, detailCode)
                .one();
    }

    public TpmMaterialPurchasingOrderDetail findByContractId(String contractId) {
        if (StringUtils.isEmpty(contractId)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(TpmMaterialPurchasingOrderDetail::getContractId, contractId)
                .one();
    }


    /**
     * 分页查询所有明细（核销用）
     * @param pageable 分页对象
     * @param dto      查询dto
     * @return 所有数据
     */
    public Page<TpmMaterialPurchasingOrderDetailVo> findDetailListByCodeForAuditPage(Pageable pageable, TpmMaterialPurchasingOrderDto dto) {
        Page<TpmMaterialPurchasingOrderDetailVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.tpmMaterialPurchasingOrderDetailMapper.findDetailListByCodeForAuditPage(page, dto);
    }

    /**
     * 根据编码查询总数
     * @param code
     * @return
     */
    public Integer getToTalByCode(String code) {
        if (StringUtils.isEmpty(code)) {
            return 0;
        }
        return this.lambdaQuery()
                .eq(TpmMaterialPurchasingOrderDetail::getCode, code)
                .count();
    }

    public List<TpmMaterialPurchasingOrderDetail> findDetailListByCode(String code) {
        if (StringUtils.isEmpty(code)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .eq(TpmMaterialPurchasingOrderDetail::getCode, code)
                .list();
    }

    public void updateReceivingStatusByCodes(List<String> codes, String status) {
        this.lambdaUpdate()
                .eq(TpmMaterialPurchasingOrderDetail::getTenantCode, TenantUtils.getTenantCode())
                .in(TpmMaterialPurchasingOrderDetail::getCode, codes)
                .set(TpmMaterialPurchasingOrderDetail::getReceivingStatus, status)
                .update();
    }

    /**
     * 分页查询可以提交审批的明细数据
     * @param pageable
     * @param dto
     * @return
     */
    public Page<TpmMaterialPurchasingOrderDetailVo> findSupplierApprovalList(Pageable pageable, TpmMaterialPurchasingOrderDto dto) {
        Page<TpmMaterialPurchasingOrderDetailVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.tpmMaterialPurchasingOrderDetailMapper.findSupplierApprovalList(page, dto);
    }

    public void updateDeliveryDate(String date, String code) {
        this.lambdaUpdate()
                .eq(TpmMaterialPurchasingOrderDetail::getTenantCode, TenantUtils.getTenantCode())
                .eq(TpmMaterialPurchasingOrderDetail::getCode, code)
                .set(TpmMaterialPurchasingOrderDetail::getDeliveryDate, date)
                .update();
    }

    public void updateRecipientDate(String date, String code) {
        this.lambdaUpdate()
                .eq(TpmMaterialPurchasingOrderDetail::getTenantCode, TenantUtils.getTenantCode())
                .eq(TpmMaterialPurchasingOrderDetail::getCode, code)
                .set(TpmMaterialPurchasingOrderDetail::getRecipientDate, date)
                .update();
    }

    public void updateReceivingFieldNull(TpmMaterialPurchasingOrderDetail detail) {
        this.lambdaUpdate()
                .eq(TpmMaterialPurchasingOrderDetail::getId, detail.getId())
                .set(TpmMaterialPurchasingOrderDetail::getReceivingStatus, detail.getReceivingStatus())
                .set(TpmMaterialPurchasingOrderDetail::getReceivingAmount, null)
                .set(TpmMaterialPurchasingOrderDetail::getReceivingQuantity, null)
                .update();
    }

    public Page<TpmMaterialPurchasingOrderDetailVo> auditFindMaterialPurchasingOrderDetailByCondition(Pageable pageable, TpmMaterialPurchasingOrderDetailDto dto) {
        Page<TpmMaterialPurchasingOrderDetailVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());

        return this.getBaseMapper().auditFindMaterialPurchasingOrderDetailByCondition(page,dto);
    }

    public void supplierProcessBack(List<TpmMaterialPurchasingOrderDetailVo> detailVos, String processStatus) {
        if(CollectionUtils.isEmpty(detailVos)){
            return;
        }
        List<TpmMaterialPurchasingOrderDetail> detailList = new ArrayList<>();
        for (TpmMaterialPurchasingOrderDetailVo detailVo : detailVos) {
            TpmMaterialPurchasingOrderDetail orderDetail = new TpmMaterialPurchasingOrderDetail();
            orderDetail.setId(detailVo.getId());
            orderDetail.setDetailStatus(processStatus);
            detailList.add(orderDetail);
        }
        log.info("物料采购单提交审批审批通过,detailList:{}", JsonUtils.obj2JsonString(detailList));
        this.updateBatchById(detailList);
    }

    public List<TpmMaterialPurchasingOrderDetailVo> findByDetailCodes(List<String> detailCodeList) {
        if(CollectionUtils.isEmpty(detailCodeList)){
            return Lists.newArrayList();
        }
        List<TpmMaterialPurchasingOrderDetail> list = this.lambdaQuery().in(TpmMaterialPurchasingOrderDetail::getDetailCode, detailCodeList)
                .list();
        if(CollectionUtils.isEmpty(list)){
            return Lists.newArrayList();
        }
        Collection<TpmMaterialPurchasingOrderDetailVo> tpmMaterialPurchasingOrderDetailVos = this.nebulaToolkitService.copyCollectionByWhiteList(list, TpmMaterialPurchasingOrderDetail.class, TpmMaterialPurchasingOrderDetailVo.class, LinkedHashSet.class, ArrayList.class);
        return (List<TpmMaterialPurchasingOrderDetailVo>) tpmMaterialPurchasingOrderDetailVos;
    }

    public void deleteByCodes(List<String> codes) {
        if(CollectionUtils.isNotEmpty(codes)){
            this.lambdaUpdate()
                    .set(TpmMaterialPurchasingOrderDetail::getDelFlag,DelFlagStatusEnum.DELETE.getCode())
                    .in(TpmMaterialPurchasingOrderDetail::getCode,codes)
                    .update();
        }
    }

    public Page<TpmMaterialPurchasingOrderDetailVo> findSupplierSubmitApprovalDetailInfo(Pageable pageable, TpmMaterialPurchasingOrderDetailDto dto) {
        Page<TpmMaterialPurchasingOrderDetailVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.tpmMaterialPurchasingOrderDetailMapper.findSupplierSubmitApprovalDetailInfo(page, dto);
    }
}

