package com.biz.crm.tpm.business.material.purchasing.order.local.service.internal;


import cn.hutool.core.lang.Snowflake;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.news.notice.sdk.service.NewsNoticeVoService;
import com.biz.crm.mdm.business.promotion.material.sdk.dto.OrderQualityLevelDto;
import com.biz.crm.mdm.business.promotion.material.sdk.service.PromotionMaterialService;
import com.biz.crm.mdm.business.promotion.material.sdk.vo.OrderQualityLevelVO;
import com.biz.crm.mdm.business.promotion.material.sdk.vo.PromotionMaterialVO;
import com.biz.crm.mdm.business.supplier.sdk.service.SupplierVoService;
import com.biz.crm.mdm.business.supplier.sdk.vo.SupplierVo;
import com.biz.crm.mn.common.base.eunm.YesOrNoEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.third.system.electronic.signature.center.sdk.vo.CreateContractVo;
import com.biz.crm.mn.third.system.electronic.signature.center.sdk.vo.ElectronicSignatureVo;
import com.biz.crm.tpm.business.activity.detail.plan.sdk.service.ActivityDetailPlanItemSdkService;
import com.biz.crm.tpm.business.material.purchasing.order.local.entity.*;
import com.biz.crm.tpm.business.material.purchasing.order.local.repository.*;
import com.biz.crm.tpm.business.material.purchasing.order.local.service.TpmMaterialPurchasingOrderAuditInvoiceService;
import com.biz.crm.tpm.business.material.purchasing.order.local.service.TpmMaterialPurchasingOrderDetailService;
import com.biz.crm.tpm.business.material.purchasing.order.local.service.TpmMaterialPurchasingOrderPdfService;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.constant.TpmMaterialPurchasingOrderConstant;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.dto.*;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.dto.log.TpmMaterialPurchasingOrderLogEventDto;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.enums.TpmMaterialPurchasingOrderEnums;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.event.log.TpmMaterialPurchasingOrderLogEventListener;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.service.TpmMaterialPurchasingOrderService;
import com.biz.crm.tpm.business.material.purchasing.order.sdk.vo.*;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBatchBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * TPM-物料采购单(TpmMaterialPurchasingOrder)表服务实现类
 * @author duyiran
 * @since 2022-11-12 17:55:49
 */
@Slf4j
@Service("tpmMaterialPurchasingOrderService")
public class TpmMaterialPurchasingOrderServiceImpl implements TpmMaterialPurchasingOrderService {

    @Autowired(required = false)
    private TpmMaterialPurchasingOrderRepository tpmMaterialPurchasingOrderRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private TpmMaterialPurchasingOrderDetailRepository tpmMaterialPurchasingOrderDetailRepository;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    @Autowired(required = false)
    private TpmMaterialPurchasingOrderDetailService tpmMaterialPurchasingOrderDetailService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private SupplierVoService supplierVoService;

    @Autowired
    private PromotionMaterialService promotionMaterialService;

    @Autowired(required = false)
    private ProcessBatchBusinessService processBatchBusinessService;

    @Autowired
    private TpmMaterialPurchasingOrderPdfService pdfService;

    @Autowired
    private TpmMaterialPurchasingOrderFileRepository fileRepository;

    @Autowired
    private TpmMaterialPurchasingBaseElectronicSignature baseElectronicSignature;

    @Autowired
    private TpmMaterialPurchasingOrderAuditInvoiceRepository auditInvoiceRepository;

    @Autowired
    private TpmMaterialPurchasingOrderAuditFileRepository auditFileRepository;

    @Autowired
    private TpmMaterialPurchasingOrderAuditInvoiceService tpmMaterialPurchasingOrderAuditInvoiceService;

    @Autowired(required = false)
    private NewsNoticeVoService newsNoticeVoService;

    @Autowired(required = false)
    private ActivityDetailPlanItemSdkService activityDetailPlanItemSdkService;

    /**
     * 分页查询数据
     * @param pageable 分页对象
     * @param dto      dto
     * @return
     */
    @Override
    public Page<TpmMaterialPurchasingOrderVo> findByConditions(Pageable pageable, TpmMaterialPurchasingOrderDto dto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(dto)) {
            dto = new TpmMaterialPurchasingOrderDto();
        }
        dto.setTenantCode(TenantUtils.getTenantCode());
        return this.tpmMaterialPurchasingOrderRepository.findByConditions(pageable, dto);
    }


    /**
     * 保存活动细案的物料采购单(不包括明细)
     * （明细信息单独接口添加）
     * @param tpmMaterialPurchasingOrderDtoList tpmMaterialPurchasingOrderDtoList
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveBatch(List<TpmMaterialPurchasingOrderDto> tpmMaterialPurchasingOrderDtoList) {
        if (CollectionUtils.isEmpty(tpmMaterialPurchasingOrderDtoList)) {
            return;
        }
        String tenantCode = TenantUtils.getTenantCode();
        List<TpmMaterialPurchasingOrder> entityList = (List<TpmMaterialPurchasingOrder>) nebulaToolkitService.copyCollectionByWhiteList(tpmMaterialPurchasingOrderDtoList, TpmMaterialPurchasingOrderDto.class, TpmMaterialPurchasingOrder.class, HashSet.class, ArrayList.class);
        List<String> codeList = generateCodeService.generateCode(TpmMaterialPurchasingOrderConstant.TPM_MATERIAL_PURCHASING_ORDER_FORMULA_PREFIX, entityList.size(), 5, 2, TimeUnit.DAYS);
        List<String> supplierCodes = tpmMaterialPurchasingOrderDtoList.stream().map(TpmMaterialPurchasingOrderDto::getSupplierCode).collect(Collectors.toList());
        List<SupplierVo> supplierVoList = this.supplierVoService.findBySupplierCodes(supplierCodes);
        Map<String, SupplierVo> supplierVoMap = supplierVoList.stream().collect(Collectors.toMap(SupplierVo::getSupplierCode, Function.identity()));
        AtomicInteger index = new AtomicInteger(0);
        entityList.forEach(entity -> {
            String code = codeList.get(index.getAndIncrement());
            entity.setCode(code);
            entity.setTenantCode(tenantCode);
            entity.setStatus(TpmMaterialPurchasingOrderEnums.OrderStatus.TO_SUBMIT_10.getValue());
            //entity.setProcessStatus(TpmMaterialPurchasingOrderEnums.OrderStatus.TO_SUBMIT_10.getValue());
            if (Objects.nonNull(supplierVoMap.get(entity.getSupplierCode()))) {
                entity.setSupplierOrg(supplierVoMap.get(entity.getSupplierCode()).getHrOrgCode());
                entity.setSupplierOrgName(supplierVoMap.get(entity.getSupplierCode()).getHrOrgName());
            }
            log.info("新生成物料采购单生成确认表");
            TpmMaterialPurchasingOrderVo orderVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, TpmMaterialPurchasingOrderVo.class, HashSet.class, ArrayList.class);
            TpmMaterialPurchasingOrderFile file = this.pdfService.orderConfirmPdf(orderVo);
            if (Objects.nonNull(file)) {
                this.fileRepository.save(file);
            }
        });
        this.tpmMaterialPurchasingOrderRepository.saveBatch(entityList);
    }

    /**
     * 分页查询数据（核销用）
     * @param pageable 分页对象
     * @param dto      dto
     * @return
     */
    @Override
    public Page<TpmMaterialPurchasingOrderVo> findByConditionsForAuditPage(Pageable pageable, TpmMaterialPurchasingOrderDto dto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(dto)) {
            dto = new TpmMaterialPurchasingOrderDto();
        }
        //核销只取已完成的物料采购单
        dto.setStatus(TpmMaterialPurchasingOrderEnums.OrderStatus.BACK_40.getValue());
        dto.setTenantCode(TenantUtils.getTenantCode());
        return this.tpmMaterialPurchasingOrderRepository.findByConditionsForAuditPage(pageable, dto);
    }

    /**
     * 服务对象
     */
    @Autowired(required = false)
    private TpmMaterialPurchasingOrderService tpmMaterialPurchasingOrderService;

    /**
     * 分页查询所有明细（核销用）
     * @param pageable 分页对象
     * @param dto      查询dto
     * @return 所有数据
     */
    @Override
    public Page<TpmMaterialPurchasingOrderDetailVo> findDetailListByCodeForAuditPage(Pageable pageable, TpmMaterialPurchasingOrderDto dto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(dto)) {
            dto = new TpmMaterialPurchasingOrderDto();
        }
        Validate.notBlank(dto.getBusinessFormatCode(), "请填写业态！");
        Validate.notBlank(dto.getBusinessUnitCode(), "请填写业务单元！");
        dto.setTenantCode(TenantUtils.getTenantCode());
        dto.setStatus(TpmMaterialPurchasingOrderEnums.OrderStatus.CONFIRM_30.getValue());
        List<String> detailReceiveStatusList = Lists.newArrayList();
        detailReceiveStatusList.add(TpmMaterialPurchasingOrderEnums.ReceiveStatus.DELIVERIED.getValue());
        detailReceiveStatusList.add(TpmMaterialPurchasingOrderEnums.ReceiveStatus.PORTIAL_DELIVERIED.getValue());
        dto.setDetailReceiveStatusList(detailReceiveStatusList);
//        if (StringUtils.isBlank(dto.getWhetherSpeciallyApproved()) || BooleanEnum.FALSE.getCapital().equals(dto.getWhetherSpeciallyApproved())) {
//            dto.setWholeAudit(BooleanEnum.FALSE.getCapital());
//        }
        return this.tpmMaterialPurchasingOrderDetailRepository.findDetailListByCodeForAuditPage(pageable, dto);

    }

    /**
     * 添加详情
     * @param dto
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void addDetailList(TpmMaterialPurchasingOrderDto dto) {
        this.addDetailValidate(dto);
        TpmMaterialPurchasingOrder entity = this.tpmMaterialPurchasingOrderRepository.getByCode(dto.getCode());
        Validate.notNull(entity, "未找到物料采购单");
        List<TpmMaterialPurchasingOrderDetailDto> dtos = this.tpmMaterialPurchasingOrderDetailService.findCacheList(dto.getCacheKey());
        Validate.notEmpty(dtos, "明细为空");
        if(StringUtils.equals(TpmMaterialPurchasingOrderEnums.OrderStatus.CONFIRM_30.getValue(), entity.getStatus())){
            //确认状态只修改 收货地址、收货联系人、收货人电话，微信号、发货联系人、联系电话、发货数量,发货时间，收货时间
            updateDetailForConfirmStatus(dtos,entity);

            return;
        }
        Validate.isTrue(!StringUtils.equals(TpmMaterialPurchasingOrderEnums.OrderStatus.CONFIRM_30.getValue(), entity.getStatus()), "已确认的采购单不能添加明细");

        // 判断明细是否已存在
        List<String> activityDetailPlanItemCodes = dtos.stream().map(TpmMaterialPurchasingOrderDetailDto::getActivityDetailItemCode).collect(Collectors.toList());
        List<TpmMaterialPurchasingOrderDetail> oldDetailList = this.tpmMaterialPurchasingOrderDetailRepository.findByDetailPlanItemCodeList(activityDetailPlanItemCodes);
        if (CollectionUtils.isNotEmpty(oldDetailList)) {
            List<String> codes = oldDetailList.stream().map(TpmMaterialPurchasingOrderDetail::getCode).distinct().collect(Collectors.toList());
            for (String code : codes) {
                Validate.isTrue(code.equals(entity.getCode()), "明细行在其他物料采购单中已存在");
            }
        }
        // 先删除旧的
        List<TpmMaterialPurchasingOrderDetailVo> oldDetailList2 = this.tpmMaterialPurchasingOrderDetailRepository.findByCode(dto.getCode());
        this.tpmMaterialPurchasingOrderDetailRepository.deleteByCode(dto.getCode());
        List<TpmMaterialPurchasingOrderDetail> detailList = (List<TpmMaterialPurchasingOrderDetail>)
                this.nebulaToolkitService.copyCollectionByWhiteList(dtos,
                                                                    TpmMaterialPurchasingOrderDetailDto.class,
                                                                    TpmMaterialPurchasingOrderDetail.class,
                                                                    HashSet.class, ArrayList.class);
        log.info("物料明细信息：{}", JSONObject.toJSONString(detailList));
        //采购数量/发货数量/签收数量
        AtomicReference<BigDecimal> totalPurchasingQuantity = new AtomicReference<>(BigDecimal.ZERO);
        AtomicReference<BigDecimal> totalDeliveryQuantity = new AtomicReference<>(BigDecimal.ZERO);
        AtomicReference<BigDecimal> totalReceivingQuantity = new AtomicReference<>(BigDecimal.ZERO);
        detailList.forEach(detail -> {
            if (Objects.nonNull(detail.getPurchasingQuantity())) {
                totalPurchasingQuantity.set(totalPurchasingQuantity.get().add(detail.getPurchasingQuantity()));
            }
            if (Objects.nonNull(detail.getDeliveryQuantity())) {
                totalDeliveryQuantity.set(totalDeliveryQuantity.get().add(detail.getDeliveryQuantity()));
            }
            if (Objects.nonNull(detail.getReceivingQuantity())) {
                totalReceivingQuantity.set(totalReceivingQuantity.get().add(detail.getReceivingQuantity()));
            }
        });
        entity.setQuantity(totalPurchasingQuantity.get());
        entity.setDeliveryQuantity(totalDeliveryQuantity.get());
        entity.setReceivingQuantity(totalReceivingQuantity.get());

        // 查询合同单价
        OrderQualityLevelDto qualityLevelDto = new OrderQualityLevelDto();
        qualityLevelDto.setMaterialCode(dto.getMaterialCode());
        qualityLevelDto.setMaterialQuantityStr(String.valueOf(totalPurchasingQuantity.get().intValue()));
        OrderQualityLevelVO qualityLevelVO = promotionMaterialService.findPriceByMaterialCode(qualityLevelDto);
        log.info("物料砍价：{}", JSONObject.toJSONString(qualityLevelVO));
        Validate.notNull(qualityLevelVO, "未找到物料坎价");
        List<String> codeList = Lists.newArrayList();
        if (detailList.stream().anyMatch(e -> StringUtils.isBlank(e.getDetailCode()))) {
            codeList.addAll(generateCodeService.generateCode(TpmMaterialPurchasingOrderConstant.TPM_MATERIAL_PURCHASING_ORDER_DETAIL_FORMULA_PREFIX, (int) detailList.stream().filter(e -> StringUtils.isBlank(e.getDetailCode())).count(), 5, 2, TimeUnit.DAYS));
        }

        //获取物料信息
        List<String> materialCodes = detailList.stream().map(TpmMaterialPurchasingOrderDetail::getMaterialCode).distinct().collect(Collectors.toList());
        Map<String, PromotionMaterialVO> materialVOMap = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(materialCodes)) {
            List<PromotionMaterialVO> materialVOS = promotionMaterialService.findByCodeList(materialCodes);
            materialVOMap.putAll(materialVOS.stream().collect(Collectors.toMap(PromotionMaterialVO::getMaterialCode, Function.identity(), (v1, v2) -> v1)));
        }

        //查询客户
        List<String> customerCodes = detailList.stream().map(TpmMaterialPurchasingOrderDetail::getCustomerCode).distinct().collect(Collectors.toList());
        Map<String, CustomerVo> customerVoMap = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(customerCodes)) {
            List<CustomerVo> customerVoList = this.customerVoService.findByCustomerCodes(customerCodes);
            if (CollectionUtils.isNotEmpty(customerVoList)) {
                customerVoMap.putAll(customerVoList.stream().collect(Collectors.toMap(CustomerVo::getCustomerCode, Function.identity(), (v1, v2) -> v1)));
            }
        }

        // 计算
        String tenantCode = TenantUtils.getTenantCode();
        Validate.notNull(qualityLevelVO.getMoney(), "未找到物料坎价");
        BigDecimal money = qualityLevelVO.getMoney();
        entity.setAmount(totalPurchasingQuantity.get().multiply(money));
        entity.setDeliveryAmount(totalDeliveryQuantity.get().multiply(money));
        entity.setReceivingAmount(totalReceivingQuantity.get().multiply(money));
        entity.setContractPrice(money);
        AtomicInteger index = new AtomicInteger(0);


        detailList.forEach(detail -> {
            if (StringUtils.isBlank(detail.getDetailCode())) {
                String detailCode = codeList.get(index.getAndIncrement());
                detail.setDetailCode(detailCode);
            }
            detail.setTenantCode(tenantCode);
            detail.setCode(entity.getCode());
            detail.setAmount(money.multiply(detail.getPurchasingQuantity()));
            detail.setContractPrice(money);
            detail.setSupplierCode(entity.getSupplierCode());
            detail.setSupplierName(entity.getSupplierName());
            detail.setReceivingStatus(TpmMaterialPurchasingOrderEnums.ReceiveStatus.WAIT_RECEIVE.getValue());
            detail.setDetailStatus(ProcessStatusEnum.PREPARE.getDictCode());
            if (Objects.nonNull(detail.getDeliveryQuantity())) {
                detail.setDeliveryAmount(detail.getDeliveryQuantity().multiply(money));
            }
            if (Objects.nonNull(detail.getReceivingQuantity())) {
                detail.setReceivingAmount(detail.getReceivingQuantity().multiply(money));
            }
            if (materialVOMap.containsKey(detail.getMaterialCode())) {
                detail.setUnit(materialVOMap.get(detail.getMaterialCode()).getUnit());
                detail.setPackingBoxNumber(Objects.nonNull(materialVOMap.get(detail.getMaterialCode()).getBoxSpecification())
                                                   ? new BigDecimal(materialVOMap.get(detail.getMaterialCode()).getBoxSpecification()) : null);
                detail.setMaterialBaseStandard(materialVOMap.get(detail.getMaterialCode()).getQualityStandard());
            }
            if (customerVoMap.containsKey(detail.getCustomerCode())) {
                detail.setDeliveryDealer(customerVoMap.get(detail.getCustomerCode()).getCityName());
            }
        });
        this.tpmMaterialPurchasingOrderRepository.updateById(entity);
        this.tpmMaterialPurchasingOrderDetailRepository.saveOrUpdateBatch(detailList);
        this.handleChangeStatus(Lists.newArrayList(entity.getCode()));

        //更新活动细案
        updateActivityDetail(entity.getCode(),activityDetailPlanItemCodes,oldDetailList2);

    }

    private void updateActivityDetail(String code, List<String> activityDetailPlanItemCodes, List<TpmMaterialPurchasingOrderDetailVo> oldDetailList2) {
        //1.新增加的活动明细编码
        List<String> newActivityDetailCodes = activityDetailPlanItemCodes;
        //2.老的活动明细编码
        List<String> oldActivityDetailCodes = oldDetailList2.stream().map(TpmMaterialPurchasingOrderDetailVo::getActivityDetailItemCode).collect(Collectors.toList());
        //3.本次新增的活动明细编码
        List<String> thisNewActivityDetailCodes = new ArrayList<>();
        //4.本次删除的活动明细编码
        List<String> thisDelActivityDetailCodes = new ArrayList<>();
        if(CollectionUtils.isEmpty(oldActivityDetailCodes)){
            thisNewActivityDetailCodes = newActivityDetailCodes;
        }else {
            thisNewActivityDetailCodes = newActivityDetailCodes.stream().filter(n -> oldActivityDetailCodes.stream().noneMatch(o -> o.equals(n))).collect(Collectors.toList());
        }
        if(CollectionUtils.isNotEmpty(oldActivityDetailCodes)){
            thisDelActivityDetailCodes = oldActivityDetailCodes.stream().filter(o -> newActivityDetailCodes.stream().noneMatch(n -> n.equals(o))).collect(Collectors.toList());
        }

        //活动细案明细更新数据
        activityDetailPlanItemSdkService.updateActivityDetailPurchasingOrder(code,thisNewActivityDetailCodes,thisDelActivityDetailCodes);
    }

    private void updateDetailForConfirmStatus(List<TpmMaterialPurchasingOrderDetailDto> dtos, TpmMaterialPurchasingOrder entity) {

        //总采购数量
        BigDecimal totalPurchasingQuantity = BigDecimal.ZERO;

        TpmMaterialPurchasingOrder tpmMaterialPurchasingOrder = new TpmMaterialPurchasingOrder();
        tpmMaterialPurchasingOrder.setId(entity.getId());
        tpmMaterialPurchasingOrder.setDeliveryQuantity(BigDecimal.ZERO);

        List<TpmMaterialPurchasingOrderDetail> detailList = new ArrayList<>();
        for (TpmMaterialPurchasingOrderDetailDto dto : dtos) {
            TpmMaterialPurchasingOrderDetail orderDetail = new TpmMaterialPurchasingOrderDetail();
            orderDetail.setId(dto.getId());
            orderDetail.setRecipientAddress(dto.getRecipientAddress());
            orderDetail.setRecipientContacts(dto.getRecipientContacts());
            orderDetail.setRecipientPhone(dto.getRecipientPhone());
            orderDetail.setWeixinNo(dto.getWeixinNo());
            orderDetail.setDeliveryContacts(dto.getDeliveryContacts());
            orderDetail.setDeliveryPhone(dto.getDeliveryPhone());
            orderDetail.setDeliveryQuantity(dto.getDeliveryQuantity());
            orderDetail.setDeliveryDate(dto.getDeliveryDate());
            orderDetail.setRecipientDate(dto.getRecipientDate());

            detailList.add(orderDetail);

            tpmMaterialPurchasingOrder.setDeliveryQuantity(tpmMaterialPurchasingOrder.getDeliveryQuantity().add(Optional.ofNullable(dto.getDeliveryQuantity()).orElse(BigDecimal.ZERO)));

            totalPurchasingQuantity = totalPurchasingQuantity.add(Optional.ofNullable(dto.getPurchasingQuantity()).orElse(BigDecimal.ZERO));
        }

        // 查询合同单价
        OrderQualityLevelDto qualityLevelDto = new OrderQualityLevelDto();
        qualityLevelDto.setMaterialCode(entity.getMaterialCode());
        qualityLevelDto.setMaterialQuantityStr(String.valueOf(totalPurchasingQuantity.intValue()));
        OrderQualityLevelVO qualityLevelVO = promotionMaterialService.findPriceByMaterialCode(qualityLevelDto);
        log.info("物料砍价：{}", JSONObject.toJSONString(qualityLevelVO));
        Validate.notNull(qualityLevelVO, "未找到物料坎价");
        Validate.notNull(qualityLevelVO.getMoney(), "未找到物料坎价");
        BigDecimal money = qualityLevelVO.getMoney();
        tpmMaterialPurchasingOrder.setDeliveryAmount(Optional.ofNullable(tpmMaterialPurchasingOrder.getDeliveryAmount()).orElse(BigDecimal.ZERO).multiply(money));

        for (TpmMaterialPurchasingOrderDetail orderDetail : detailList) {
            if (Objects.nonNull(orderDetail.getDeliveryQuantity())) {
                orderDetail.setDeliveryAmount(orderDetail.getDeliveryQuantity().multiply(money));
            }
        }

        this.tpmMaterialPurchasingOrderRepository.updateById(tpmMaterialPurchasingOrder);

        this.tpmMaterialPurchasingOrderDetailRepository.updateBatchById(detailList);
    }

    /**
     * 根据编码修改状态
     * @param codes
     * @param status
     */
    @Override
    @Transactional
    public void processBack(List<String> codes, String status) {
        if (CollectionUtils.isEmpty(codes) || StringUtils.isEmpty(status)) {
            return;
        }
        this.tpmMaterialPurchasingOrderRepository.updateStatusByCodes(codes, status);
    }

    @Override
    @Transactional
    public void materialGroupUpdate(TpmMaterialPurchasingOrderDto dto) {
        this.updateValidate(dto);
        TpmMaterialPurchasingOrder entity = this.tpmMaterialPurchasingOrderRepository.getById(dto.getId());
        Validate.notNull(entity, "未找到物料报损等级信息");
        entity.setPhone(dto.getPhone());
        entity.setRemark(dto.getRemark());
        entity.setDeliveryDateAsk(dto.getDeliveryDateAsk());
        entity.setNodeEndCaseDate(dto.getNodeEndCaseDate());
        entity.setMaterialBaseStandard(dto.getMaterialBaseStandard());
        entity.setMaterialGroupEdit(YesOrNoEnum.YES.getCode());
        this.tpmMaterialPurchasingOrderRepository.updateById(entity);
        this.tpmMaterialPurchasingOrderDetailRepository.updateRecipientDate(dto.getDeliveryDateAsk(), entity.getCode());
        // 生成订单确认表PDF
        TpmMaterialPurchasingOrderVo orderVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, TpmMaterialPurchasingOrderVo.class, HashSet.class, ArrayList.class);
        TpmMaterialPurchasingOrderFile file = this.pdfService.orderConfirmPdf(orderVo);
        // 删除旧的文件
        this.fileRepository.deleteByBusinessCode(entity.getCode());
        this.fileRepository.save(file);
    }


    /**
     * 出库确认
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void outConfirm(TpmMaterialPurchasingOrderDto dto) {
        Validate.notNull(dto, "参数错误");
        Validate.notBlank(dto.getCacheKey(), "cacheKey不能为空");
        List<TpmMaterialPurchasingOrderDetail> detailList = this.tpmMaterialPurchasingOrderDetailRepository.listByIds(dto.getIds());
        Validate.notEmpty(detailList, "未找到详情信息");
        List<String> codes = detailList.stream().map(TpmMaterialPurchasingOrderDetail::getCode).distinct().collect(Collectors.toList());
        List<TpmMaterialPurchasingOrder> orderList = this.tpmMaterialPurchasingOrderRepository.findByCodeList(codes);
        Validate.notEmpty(orderList, "未找到采购单信息");
        Map<String, TpmMaterialPurchasingOrder> orderMap = orderList.stream().collect(Collectors.toMap(TpmMaterialPurchasingOrder::getCode, Function.identity()));
        detailList.forEach(detail -> {
            TpmMaterialPurchasingOrder order = orderMap.get(detail.getCode());
            Validate.notNull(order, detail.getDetailCode() + ":未找到采购单信息");
            if (!(TpmMaterialPurchasingOrderEnums.OrderStatus.CONFIRM_30.getValue().equals(order.getStatus())
                    && StringUtils.equalsAny(detail.getReceivingStatus(), TpmMaterialPurchasingOrderEnums.ReceiveStatus.WAIT_RECEIVE.getValue())
                    && !TpmMaterialPurchasingOrderEnums.OrderChangeStatus.CHANGEING.getValue().equals(detail.getActivityDetailChangeStatus()))) {
                throw new IllegalArgumentException("只能对采购状态为【已确认】且签收状态为【待出库】且活动细案调整状态不为【调整中】的物料采购单出库确认");
            }
        });
        
        //更新提交时间
        List<TpmMaterialPurchasingOrder> orderListUpdate = new ArrayList<>();
        for (TpmMaterialPurchasingOrder order : orderList) {
            TpmMaterialPurchasingOrder orderUpdate = new TpmMaterialPurchasingOrder();
            orderUpdate.setId(order.getId());
            orderUpdate.setSubmitTime(DateUtil.dateStrNowAll());
            orderListUpdate.add(orderUpdate);
        }

        this.tpmMaterialPurchasingOrderRepository.updateBatchById(orderListUpdate);
        
        // 生成PDF文件并保存
        List<TpmMaterialPurchasingOrderFile> fileList = Lists.newArrayList();
        detailList.forEach(detail -> {
            TpmMaterialPurchasingOrderFile file = pdfService.outConfirmPdf(detail);
            fileList.add(file);
            try {
                Thread.sleep(2000);
            } catch (InterruptedException e) {
                log.info("睡不着了,{}", ExceptionUtils.getStackTrace(e));
            }
            // 推送第三方签章
            ElectronicSignatureVo<CreateContractVo> signatureVo = this.baseElectronicSignature.pushElectronicSignature(detail, file);
            log.info("第三方签章返回：{}", JSONObject.toJSONString(signatureVo));
            Optional<ElectronicSignatureVo<CreateContractVo>> electronicSignatureVo =
                    Optional.ofNullable(signatureVo);
            if (electronicSignatureVo.isPresent()) {
                ElectronicSignatureVo<CreateContractVo> voElectronicSignatureVo = electronicSignatureVo.get();
                if (Objects.isNull(voElectronicSignatureVo.getResult())) {
                    return;
                }
                detail.setContractId(String.valueOf(voElectronicSignatureVo.getResult().getContractId()));
                detail.setDocumentsId(String.valueOf(voElectronicSignatureVo.getResult().getDocumentId()));
//                detail.setReceivingQuantity(detail.getDeliveryQuantity());
//                detail.setReceivingAmount(detail.getReceivingQuantity().multiply(detail.getContractPrice()));
                detail.setAlreadyDeliveryQuantity(detail.getDeliveryQuantity());
            }
            if (Optional.ofNullable(detail.getDeliveryQuantity()).orElse(BigDecimal.ZERO).compareTo(detail.getPurchasingQuantity()) >= 0) {
                detail.setReceivingStatus(TpmMaterialPurchasingOrderEnums.ReceiveStatus.RECEIVED.getValue());
            }
            if (Optional.ofNullable(detail.getDeliveryQuantity()).orElse(BigDecimal.ZERO).compareTo(detail.getPurchasingQuantity()) < 0) {
                detail.setReceivingStatus(TpmMaterialPurchasingOrderEnums.ReceiveStatus.PORTIAL_RECEIVE.getValue());
            }
        });
        // 删除旧的文件
        List<String> detailCodes = detailList.stream().map(TpmMaterialPurchasingOrderDetail::getDetailCode).collect(Collectors.toList());
        this.fileRepository.deleteByBusinessCodes(detailCodes);
        this.fileRepository.saveBatch(fileList);
        this.tpmMaterialPurchasingOrderDetailRepository.updateBatchById(detailList);
        //计算签收数据
        this.handleReceiving(detailList);
        List<TpmMaterialPurchasingOrderDetailDto> transferDto =
                (List<TpmMaterialPurchasingOrderDetailDto>) this.nebulaToolkitService.copyCollectionByWhiteList(detailList,
                                                                                                                TpmMaterialPurchasingOrderDetail.class,
                                                                                                                TpmMaterialPurchasingOrderDetailDto.class,
                                                                                                                HashSet.class, ArrayList.class);
        this.tpmMaterialPurchasingOrderDetailService.addItemCache(dto.getCacheKey(), transferDto);
    }

    private void handleReceiving(List<TpmMaterialPurchasingOrderDetail> detailList) {
        if (CollectionUtils.isEmpty(detailList)) return;
        //找到头code
        List<String> codes = detailList.stream().map(TpmMaterialPurchasingOrderDetail::getCode).distinct().collect(Collectors.toList());
        List<TpmMaterialPurchasingOrder> orderList = this.tpmMaterialPurchasingOrderRepository.findByCodeList(codes);
        if (CollectionUtils.isEmpty(orderList)) return;

        List<TpmMaterialPurchasingOrderDetail> details = this.tpmMaterialPurchasingOrderDetailRepository.findByCodeList(codes);
        Map<String, List<TpmMaterialPurchasingOrderDetail>> groupBy = details.stream().collect(Collectors.groupingBy(TpmMaterialPurchasingOrderDetail::getCode));

        orderList.forEach(order -> {
            if (groupBy.containsKey(order.getCode())) {
                TpmMaterialPurchasingOrder update = new TpmMaterialPurchasingOrder();
                update.setId(order.getId());
                AtomicReference<BigDecimal> totalReceivingQuantity = new AtomicReference<>(BigDecimal.ZERO);
                AtomicReference<BigDecimal> totalReceivingAmount = new AtomicReference<>(BigDecimal.ZERO);
                groupBy.get(order.getCode()).forEach(detail -> {
                    if (Objects.nonNull(detail.getReceivingQuantity())) {
                        totalReceivingQuantity.set(totalReceivingQuantity.get().add(detail.getReceivingQuantity()));
                    }
                    if (Objects.nonNull(detail.getReceivingAmount())) {
                        totalReceivingAmount.set(totalReceivingAmount.get().add(detail.getReceivingAmount()));
                    }
                });
                update.setReceivingAmount(totalReceivingAmount.get());
                update.setReceivingQuantity(totalReceivingQuantity.get());
                this.tpmMaterialPurchasingOrderRepository.updateById(update);
            }
        });
    }


    /**
     * 批量下载订单确认表
     * @param ids
     * @return
     */
    @Override
    public List<TpmMaterialPurchasingOrderFileVo> downloadOrderConfirmBatch(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Lists.newArrayList();
        }
        List<TpmMaterialPurchasingOrder> entityList = this.tpmMaterialPurchasingOrderRepository.listByIds(ids);
        Validate.notEmpty(entityList, "未找到物料采购单信息");
        List<String> codes = entityList.stream().map(TpmMaterialPurchasingOrder::getCode).collect(Collectors.toList());
        List<TpmMaterialPurchasingOrderFile> fileList = this.fileRepository.findByBusinessCodes(codes);
        Validate.notEmpty(fileList, "未找到订单确认表");
        return (List<TpmMaterialPurchasingOrderFileVo>) this.nebulaToolkitService.copyCollectionByWhiteList(fileList, TpmMaterialPurchasingOrderFile.class, TpmMaterialPurchasingOrderFileVo.class, HashSet.class, ArrayList.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void outCancel(TpmMaterialPurchasingOrderDto dto) {
        Validate.notNull(dto, "参数错误");
        Validate.notBlank(dto.getCacheKey(), "cacheKey不能为空");
        List<TpmMaterialPurchasingOrderDetail> detailList = this.tpmMaterialPurchasingOrderDetailRepository.listByIds(dto.getIds());
        Validate.notEmpty(detailList, "未找到详情信息");
        List<String> codes = detailList.stream().map(TpmMaterialPurchasingOrderDetail::getCode).distinct().collect(Collectors.toList());
        List<TpmMaterialPurchasingOrder> orderList = this.tpmMaterialPurchasingOrderRepository.findByCodeList(codes);
        Validate.notEmpty(orderList, "未找到采购单信息");
        Map<String, TpmMaterialPurchasingOrder> orderMap = orderList.stream().collect(Collectors.toMap(TpmMaterialPurchasingOrder::getCode, Function.identity()));
        detailList.forEach(detail -> {
            TpmMaterialPurchasingOrder order = orderMap.get(detail.getCode());
            Validate.notNull(order, detail.getDetailCode() + ":未找到采购单信息");
            if (StringUtils.equals(detail.getReceivingStatus(), TpmMaterialPurchasingOrderEnums.ReceiveStatus.WAIT_RECEIVE.getValue())) {
                throw new IllegalArgumentException("签收状态为【待出库】的不能撤回");
            }
            if (StringUtils.equalsAny(detail.getReceivingStatus(), TpmMaterialPurchasingOrderEnums.ReceiveStatus.DELIVERIED.getValue(), TpmMaterialPurchasingOrderEnums.ReceiveStatus.PORTIAL_DELIVERIED.getValue())) {
                throw new IllegalArgumentException("签收状态为【已收货】、【部分收货】的不能撤回");
            }
        });
        StringBuffer errorMsg = new StringBuffer();
        detailList.forEach(detail -> {
            // 推送第三方签章
            try {
                ElectronicSignatureVo electronicSignatureVo = this.baseElectronicSignature.withdrawalElectronicSignature(detail);
                if (Objects.nonNull(electronicSignatureVo) && electronicSignatureVo.isSuccess()) {
                    //只要撤回成功 通通改成待出库！！！
                    //if (detail.getReceivingQuantity().compareTo(BigDecimal.ZERO) == 0) {
                    detail.setReceivingStatus(TpmMaterialPurchasingOrderEnums.ReceiveStatus.WAIT_RECEIVE.getValue());
                    //}
                    //if (detail.getDeliveryQuantity().compareTo(BigDecimal.ZERO) > 0) {
                    //detail.setReceivingStatus(TpmMaterialPurchasingOrderEnums.ReceiveStatus.PORTIAL_RECEIVE.getValue());
                    //}
                    detail.setReceivingQuantity(null);
                    detail.setReceivingAmount(null);
                    this.tpmMaterialPurchasingOrderDetailRepository.updateReceivingFieldNull(detail);
                } else {
                    errorMsg.append("[" + detail.getDetailCode() + "]撤销失败");
                }
            } catch (Exception e) {
                errorMsg.append("[" + detail.getDetailCode() + "]撤销失败");
            }
        });
        // 删除旧的文件
        List<String> detailCodes = detailList.stream().map(TpmMaterialPurchasingOrderDetail::getDetailCode).collect(Collectors.toList());
        this.fileRepository.deleteByBusinessCodes(detailCodes);
        //this.tpmMaterialPurchasingOrderDetailRepository.updateBatchById(detailList);
        List<TpmMaterialPurchasingOrderDetailDto> transferDto =
                (List<TpmMaterialPurchasingOrderDetailDto>) this.nebulaToolkitService.copyCollectionByWhiteList(detailList,
                                                                                                                TpmMaterialPurchasingOrderDetail.class,
                                                                                                                TpmMaterialPurchasingOrderDetailDto.class,
                                                                                                                HashSet.class, ArrayList.class);
        this.tpmMaterialPurchasingOrderDetailService.addItemCache(dto.getCacheKey(), transferDto);
    }

    /**
     * 出库单催签
     */
    @Override
    @Transactional
    public void outExpediting(TpmMaterialPurchasingOrderDto dto) {
        Validate.notNull(dto, "参数错误");
        Validate.notBlank(dto.getCacheKey(), "cacheKey不能为空");
        List<TpmMaterialPurchasingOrderDetail> detailList = this.tpmMaterialPurchasingOrderDetailRepository.listByIds(dto.getIds());
        Validate.notEmpty(detailList, "未找到详情信息");
        List<String> codes = detailList.stream().map(TpmMaterialPurchasingOrderDetail::getCode).distinct().collect(Collectors.toList());
        List<TpmMaterialPurchasingOrder> orderList = this.tpmMaterialPurchasingOrderRepository.findByCodeList(codes);
        Validate.notEmpty(orderList, "未找到采购单信息");
        Map<String, TpmMaterialPurchasingOrder> orderMap = orderList.stream().collect(Collectors.toMap(TpmMaterialPurchasingOrder::getCode, Function.identity()));
        detailList.forEach(detail -> {
            TpmMaterialPurchasingOrder order = orderMap.get(detail.getCode());
            Validate.notNull(order, detail.getDetailCode() + ":未找到采购单信息");
            if (!StringUtils.equalsAny(detail.getReceivingStatus(), TpmMaterialPurchasingOrderEnums.ReceiveStatus.RECEIVED.getValue(), TpmMaterialPurchasingOrderEnums.ReceiveStatus.PORTIAL_RECEIVE.getValue())) {
                throw new IllegalArgumentException("签收状态为【已出库】、【部分出库】的才能催签");
            }
        });
        detailList.forEach(detail -> {
            ElectronicSignatureVo electronicSignatureVo = this.baseElectronicSignature.expeditingElectronicSignature(detail);
        });
        List<TpmMaterialPurchasingOrderDetailDto> transferDto =
                (List<TpmMaterialPurchasingOrderDetailDto>) this.nebulaToolkitService.copyCollectionByWhiteList(detailList,
                                                                                                                TpmMaterialPurchasingOrderDetail.class,
                                                                                                                TpmMaterialPurchasingOrderDetailDto.class,
                                                                                                                HashSet.class, ArrayList.class);
        this.tpmMaterialPurchasingOrderDetailService.addItemCache(dto.getCacheKey(), transferDto);
    }

    @Override
    public List<TpmMaterialPurchasingOrderFileVo> downloadOutConfirmBatch(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Lists.newArrayList();
        }
        List<TpmMaterialPurchasingOrderDetail> detailList = this.tpmMaterialPurchasingOrderDetailRepository.listByIds(ids);
        Validate.notEmpty(detailList, "未找到物料采购单明细信息");
        List<String> codes = detailList.stream().map(TpmMaterialPurchasingOrderDetail::getDetailCode).collect(Collectors.toList());
        List<TpmMaterialPurchasingOrderFile> fileList = this.fileRepository.findByBusinessCodes(codes);
        Validate.notEmpty(fileList, "未找到订单签收表");
        return (List<TpmMaterialPurchasingOrderFileVo>) this.nebulaToolkitService.copyCollectionByWhiteList(fileList, TpmMaterialPurchasingOrderFile.class, TpmMaterialPurchasingOrderFileVo.class, HashSet.class, ArrayList.class);

    }

    @Override
    public TpmMaterialPurchasingOrderDetailVo findDetailById(String id) {
        Validate.notEmpty(id, "参数错误");
        TpmMaterialPurchasingOrderDetail detail = this.tpmMaterialPurchasingOrderDetailRepository.getById(id);
        Validate.notNull(detail, "未找到详情信息");
        TpmMaterialPurchasingOrderDetailVo detailVo = this.nebulaToolkitService.copyObjectByWhiteList(detail, TpmMaterialPurchasingOrderDetailVo.class, HashSet.class, ArrayList.class);
        // 查询签收单
        List<TpmMaterialPurchasingOrderFile> fileEntityList = this.fileRepository.findByBusinessCode(detailVo.getDetailCode());
        //Validate.notEmpty("未找到签收信息,请确认后再试");
        if (!CollectionUtils.isEmpty(fileEntityList)) {
            List<TpmMaterialPurchasingOrderFileVo> fileVoList = (List<TpmMaterialPurchasingOrderFileVo>) this.nebulaToolkitService.copyCollectionByWhiteList(fileEntityList, TpmMaterialPurchasingOrderFile.class, TpmMaterialPurchasingOrderFileVo.class, HashSet.class, ArrayList.class);
            detailVo.setFileList(fileVoList);
        }
        return detailVo;
    }

    @Override
    @Transactional
    public void uploadOutConfirm(TpmMaterialPurchasingOrderDetailDto dto) {
        Validate.notNull(dto, "上传时，对象不能为空");
        Validate.notBlank(dto.getCacheKey(), "上传时，cacheKey不能为空");
        Validate.notNull(dto.getReceivingQuantity(), "签收数量不能为空");
        Validate.notNull(dto.getId(), "详情id不能为空");
        Validate.isTrue(dto.getReceivingQuantity().compareTo(BigDecimal.ZERO) > 0, "签收数量必须大于0");
        Validate.isTrue(dto.getReceivingQuantity().toPlainString().length() <= 16, "签收数量填写数值过大");
        Validate.notEmpty(dto.getFileList(), "附件不能为空");
        TpmMaterialPurchasingOrderDetail detail = this.tpmMaterialPurchasingOrderDetailRepository.getById(dto.getId());
        TpmMaterialPurchasingOrderDetail update = new TpmMaterialPurchasingOrderDetail();
        update.setId(detail.getId());
        update.setReceivingQuantity(Optional.ofNullable(detail.getReceivingQuantity()).orElse(BigDecimal.ZERO).add(dto.getReceivingQuantity()));
        update.setReceivingAmount(update.getReceivingQuantity().multiply(detail.getContractPrice()));
        update.setAlreadyDeliveryQuantity(update.getReceivingQuantity());
        update.setReceivingDate(DateUtil.formatDate());
        Validate.notNull(detail.getDeliveryQuantity(), "未导入发货数量");
        Validate.isTrue(detail.getDeliveryQuantity().compareTo(update.getAlreadyDeliveryQuantity()) >= 0, "已发货数量不能大于发货数量");
        Validate.isTrue(!(TpmMaterialPurchasingOrderEnums.ReceiveStatus.RECEIVED.getValue().equals(detail.getReceivingStatus())||
                        TpmMaterialPurchasingOrderEnums.ReceiveStatus.PORTIAL_RECEIVE.getValue().equals(detail.getReceivingStatus())),
                "签收状态为已出库和部分出库不能被签收");
        //update.setRecipientDate(DateUtil.formatDate());
        if (update.getReceivingQuantity().compareTo(Optional.ofNullable(detail.getDeliveryQuantity()).orElse(BigDecimal.ZERO)) >= 0) {
            update.setReceivingStatus(TpmMaterialPurchasingOrderEnums.ReceiveStatus.DELIVERIED.getValue());
        }
        if (update.getReceivingQuantity().compareTo(Optional.ofNullable(detail.getDeliveryQuantity()).orElse(BigDecimal.ZERO)) < 0) {
            update.setReceivingStatus(TpmMaterialPurchasingOrderEnums.ReceiveStatus.PORTIAL_DELIVERIED.getValue());
        }
        List<TpmMaterialPurchasingOrderFile> fileEntityList = (List<TpmMaterialPurchasingOrderFile>) this.nebulaToolkitService.copyCollectionByWhiteList(dto.getFileList(), TpmMaterialPurchasingOrderFileDto.class, TpmMaterialPurchasingOrderFile.class, HashSet.class, ArrayList.class);
        String tenantCode = TenantUtils.getTenantCode();
        fileEntityList.forEach(file -> {
            file.setBusinessCode(detail.getDetailCode());
            file.setTenantCode(tenantCode);
            file.setFileType(TpmMaterialPurchasingOrderEnums.FileType.RECEIVE.getValue());
        });
        this.fileRepository.deleteByBusinessCode(detail.getDetailCode());
        this.fileRepository.saveBatch(fileEntityList);
        this.tpmMaterialPurchasingOrderDetailRepository.updateById(update);
        //计算签收数据
        this.handleReceiving(Lists.newArrayList(update));
        TpmMaterialPurchasingOrderDetail newDetail = this.tpmMaterialPurchasingOrderDetailRepository.getById(dto.getId());
        TpmMaterialPurchasingOrderDetailDto transferDto = this.nebulaToolkitService.copyObjectByWhiteList(newDetail,
                                                                                                          TpmMaterialPurchasingOrderDetailDto.class,
                                                                                                          HashSet.class, ArrayList.class);
        this.tpmMaterialPurchasingOrderDetailService.addItemCache(dto.getCacheKey(), Lists.newArrayList(transferDto));
    }

    @Override
    @Transactional
    public void submitProcess(TpmMaterialPurchasingOrderDto dto) {
        Validate.notNull(dto, "提交审批时参数不能为空");
        Validate.notEmpty(dto.getDetailList(), "数据不能为空");
        Validate.notNull(dto.getProcessBusiness(), "审批参数不能为空");
        //这里由于流程里面业务表单的设计为展示采购单列表。这里传给流程的是采购单编码 可能要改
        //处理明细数据的状态
        List<String> ids = dto.getDetailList().stream().map(TpmMaterialPurchasingOrderDetailDto::getId).collect(Collectors.toList());
        List<TpmMaterialPurchasingOrderDetail> detailList = this.tpmMaterialPurchasingOrderDetailRepository.listByIds(ids);
        this.approceProcess(detailList, dto);
        List<TpmMaterialPurchasingOrderDetail> updateList = Lists.newArrayList();
        detailList.forEach(e -> {
            TpmMaterialPurchasingOrderDetail update = new TpmMaterialPurchasingOrderDetail();
            update.setId(e.getId());
            update.setDetailStatus(ProcessStatusEnum.COMMIT.getDictCode());
            update.setProcessNo(e.getProcessNo());
            updateList.add(update);
        });
        this.tpmMaterialPurchasingOrderDetailRepository.updateBatchById(updateList);
    }

    /**
     * 上传核销资料
     * @param dto
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateAuditFile(TpmMaterialPurchasingOrderDto dto) {
        Validate.notNull(dto, "上传时，对象不能为空");
        Validate.notEmpty(dto.getId(), "主键参数不能为空");
        TpmMaterialPurchasingOrder order = this.tpmMaterialPurchasingOrderRepository.getById(dto.getId());
        Validate.notNull(order, "未找到物料采购单信息");
        Validate.notBlank(dto.getCacheKey(), "缓存key不能为空");
        String tenantCode = TenantUtils.getTenantCode();
        // 删除旧的发票
        this.auditInvoiceRepository.deleteByCode(order.getCode());
        List<TpmMaterialPurchasingOrderAuditInvoiceDto> invoiceDtos = this.tpmMaterialPurchasingOrderAuditInvoiceService.findCacheList(dto.getCacheKey());
        if (CollectionUtils.isNotEmpty(invoiceDtos)) {
            List<TpmMaterialPurchasingOrderAuditInvoice> auditInvoiceEntityList = (List<TpmMaterialPurchasingOrderAuditInvoice>) this.nebulaToolkitService.copyCollectionByWhiteList(invoiceDtos, TpmMaterialPurchasingOrderAuditInvoiceDto.class, TpmMaterialPurchasingOrderAuditInvoice.class, HashSet.class, ArrayList.class);
            auditInvoiceEntityList.forEach(auditInvoice -> {
                auditInvoice.setId(null);
                auditInvoice.setCode(order.getCode());
                auditInvoice.setTenantCode(tenantCode);
            });
            this.auditInvoiceRepository.saveBatch(auditInvoiceEntityList);
        }
        // 删除旧的附件信息
        this.auditFileRepository.deleteByBusinessCode(order.getCode());
        if (CollectionUtils.isNotEmpty(dto.getAuditFileList())) {
            List<TpmMaterialPurchasingOrderAuditFile> auditFileList = (List<TpmMaterialPurchasingOrderAuditFile>) this.nebulaToolkitService.copyCollectionByWhiteList(dto.getAuditFileList(), TpmMaterialPurchasingOrderAuditFileDto.class, TpmMaterialPurchasingOrderAuditFile.class, HashSet.class, ArrayList.class);
            auditFileList.forEach(auditFile -> {
                auditFile.setId(null);
                auditFile.setBusinessCode(order.getCode());
                auditFile.setTenantCode(tenantCode);
            });
            this.auditFileRepository.saveBatch(auditFileList);
        }

        this.tpmMaterialPurchasingOrderAuditInvoiceService.clearCache(dto.getCacheKey());
    }

    /**
     * 查询核销资料
     * @param id
     * @return
     */
    @Override
    public TpmMaterialPurchasingOrderVo getAuditFile(String id) {
        Validate.notEmpty(id, "主键参数不能为空");
        TpmMaterialPurchasingOrder order = this.tpmMaterialPurchasingOrderRepository.getById(id);
        Validate.notNull(order, "未找到物料采购单信息");
        TpmMaterialPurchasingOrderVo orderVo = this.nebulaToolkitService.copyObjectByWhiteList(order, TpmMaterialPurchasingOrderVo.class, HashSet.class, ArrayList.class);
        // 发票
        List<TpmMaterialPurchasingOrderAuditInvoice> auditInvoiceEntityList = this.auditInvoiceRepository.findByCode(order.getCode());
        if (CollectionUtils.isNotEmpty(auditInvoiceEntityList)) {
            List<TpmMaterialPurchasingOrderAuditInvoiceVo> auditInvoiceVoList = (List<TpmMaterialPurchasingOrderAuditInvoiceVo>) this.nebulaToolkitService.copyCollectionByWhiteList(auditInvoiceEntityList, TpmMaterialPurchasingOrderAuditInvoice.class, TpmMaterialPurchasingOrderAuditInvoiceVo.class, HashSet.class, ArrayList.class);
            orderVo.setAuditInvoiceList(auditInvoiceVoList);
        }
        // // 查询订单确认表
        List<TpmMaterialPurchasingOrderFile> confirmFileList = this.fileRepository.findByBusinessCode(order.getCode());
        // 签收单
        List<TpmMaterialPurchasingOrderFile> signFileList = this.fileRepository.findSignByCode(order.getCode());
        // 附件信息
        List<TpmMaterialPurchasingOrderAuditFile> auditFileList = this.auditFileRepository.findByBusinessCode(order.getCode());
        //if (CollectionUtils.isNotEmpty(auditFileList)) {
        List<TpmMaterialPurchasingOrderAuditFileVo> allList = Lists.newArrayList();
        List<TpmMaterialPurchasingOrderAuditFile> fileList = auditFileList.stream()
                .filter(a -> !StringUtils.equalsAny(a.getTypeCode(), TpmMaterialPurchasingOrderEnums.AuditFileTypeCode.CONFIRM.getValue(), TpmMaterialPurchasingOrderEnums.AuditFileTypeCode.SIGN.getValue()))
                .collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(fileList)) {
            List<TpmMaterialPurchasingOrderAuditFileVo> auditFileVoList = (List<TpmMaterialPurchasingOrderAuditFileVo>) this.nebulaToolkitService.copyCollectionByWhiteList(auditFileList, TpmMaterialPurchasingOrderAuditFile.class, TpmMaterialPurchasingOrderAuditFileVo.class, HashSet.class, ArrayList.class);
            allList.addAll(auditFileVoList);
        }
        // 订单确认表
        if (CollectionUtils.isNotEmpty(confirmFileList)) {
            List<TpmMaterialPurchasingOrderAuditFileVo> auditFileVoList = (List<TpmMaterialPurchasingOrderAuditFileVo>) this.nebulaToolkitService.copyCollectionByWhiteList(confirmFileList, TpmMaterialPurchasingOrderFile.class, TpmMaterialPurchasingOrderAuditFileVo.class, HashSet.class, ArrayList.class);
            auditFileVoList.forEach(file -> {
                file.setTypeCode(TpmMaterialPurchasingOrderEnums.AuditFileTypeCode.CONFIRM.getValue());
                file.setBusinessName(TpmMaterialPurchasingOrderEnums.AuditFileTypeCode.CONFIRM.getDesc());
            });
            allList.addAll(auditFileVoList);
        }
        // 签收单
        if (CollectionUtils.isNotEmpty(signFileList)) {
            List<TpmMaterialPurchasingOrderAuditFileVo> auditFileVoList = (List<TpmMaterialPurchasingOrderAuditFileVo>) this.nebulaToolkitService.copyCollectionByWhiteList(signFileList, TpmMaterialPurchasingOrderFile.class, TpmMaterialPurchasingOrderAuditFileVo.class, HashSet.class, ArrayList.class);
            auditFileVoList.forEach(file -> {
                file.setTypeCode(TpmMaterialPurchasingOrderEnums.AuditFileTypeCode.SIGN.getValue());
                file.setBusinessName(TpmMaterialPurchasingOrderEnums.AuditFileTypeCode.SIGN.getDesc());
            });
            allList.addAll(auditFileVoList);
        }
        orderVo.setAuditFileList(allList);
        //}
        return orderVo;
    }

    private void approceProcess(List<TpmMaterialPurchasingOrderDetail> detailList, TpmMaterialPurchasingOrderDto dto) {
        List<String> detailCodes = detailList.stream().map(TpmMaterialPurchasingOrderDetail::getDetailCode).collect(Collectors.toList());
        ProcessBusinessDto processBusiness = dto.getProcessBusiness();
        Validate.notNull(processBusiness, "提交工作流时，未传工作流对象信息!");
        processBusiness.setBusinessNoList(detailCodes);
        processBusiness.setBusinessNo(new Snowflake().nextIdStr());
        processBusiness.setBusinessFormJson(JsonUtils.obj2JsonString(dto));
        processBusiness.setBusinessCode(TpmMaterialPurchasingOrderConstant.TPM_MATERIAL_PURCHASING_ORDER_SUPPLIER_PROCESS);
        log.info("采购单流程提交参数(提交审批),{}", JSON.toJSONString(processBusiness));
        ProcessBusinessVo processBusinessVo = this.processBatchBusinessService.processStart(processBusiness);
        detailList.forEach(item->{
            item.setProcessNo(processBusinessVo.getProcessNo());
        });
        log.info("采购单流程提交审批：detailList：{}",JsonUtils.obj2JsonString(detailList));
    }

    /**
     * 物料组编辑参数验证
     * @param dto
     */
    private void updateValidate(TpmMaterialPurchasingOrderDto dto) {
        Validate.notNull(dto, "编辑时，对象信息不能为空！");
        Validate.notEmpty(dto.getId(), "主键参数不能为空");
        Validate.isTrue(StringUtils.isBlank(dto.getRemark()) || dto.getRemark().length() < 255, "备注已经超过255长度");
        Validate.isTrue(StringUtils.isBlank(dto.getMaterialBaseStandard()) || dto.getMaterialBaseStandard().length() < 255, "物料基本工艺标准已经超过255长度");
        Validate.isTrue(StringUtils.isBlank(dto.getPhone()) || dto.getPhone().length() <= 20, "电话已经超过20长度");
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void supplierUpdate(TpmMaterialPurchasingOrderDto dto) {
        this.updateValidate(dto);
        TpmMaterialPurchasingOrder entity = this.tpmMaterialPurchasingOrderRepository.getById(dto.getId());
        Validate.notNull(entity, "未找到物料报损等级信息");
        entity.setPhone(dto.getPhone());
        entity.setRemark(dto.getRemark());
        entity.setNodeMoldDate(dto.getNodeMoldDate());
        entity.setNodeSendedDate(dto.getNodeSendedDate());
        entity.setNodeReceiveDate(dto.getNodeReceiveDate());
        entity.setNodeCapacityDate(dto.getNodeCapacityDate());
        entity.setNodeStartDeliveryDate(dto.getNodeStartDeliveryDate());
        entity.setSupplierEdit(YesOrNoEnum.YES.getCode());
        this.tpmMaterialPurchasingOrderRepository.updateById(entity);
        this.tpmMaterialPurchasingOrderDetailRepository.updateDeliveryDate(dto.getNodeStartDeliveryDate(), entity.getCode());
        // 生成订单确认表
        TpmMaterialPurchasingOrderVo orderVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, TpmMaterialPurchasingOrderVo.class, HashSet.class, ArrayList.class);
        TpmMaterialPurchasingOrderFile file = this.pdfService.orderConfirmPdf(orderVo);
        // 判断文件是否已存在
        //List<TpmMaterialPurchasingOrderFile> fileEntityList = this.fileRepository.findByBusinessCode(entity.getCode());
        // 已存在就忽略，因为在生成pdf时已覆盖旧的文件
        //if (CollectionUtils.isEmpty(fileEntityList)) {
        // 删除旧的文件
        this.fileRepository.deleteByBusinessCode(entity.getCode());
        this.fileRepository.save(file);
        //}
    }

    /**
     * 新增明细验证
     * @param dto
     */
    private void addDetailValidate(TpmMaterialPurchasingOrderDto dto) {
        Validate.notNull(dto, "参数不能为空");
        Validate.notBlank(dto.getCacheKey(), "cacheKey不能为空");
        Validate.notEmpty(dto.getMaterialCode(), "物料编码不能为空");
        Validate.notEmpty(dto.getCode(), "物料采购单编码不能为空");
        List<TpmMaterialPurchasingOrderDetailDto> dtos = this.tpmMaterialPurchasingOrderDetailService.findCacheList(dto.getCacheKey());
        Validate.notEmpty(dtos, "明细不能为空");
        dtos.forEach(detail -> {
            Validate.notEmpty(detail.getActivityDetailItemCode(), "活动细案明细编码不能为空");
            Validate.notEmpty(detail.getSalesOrgCode(), "销售大区编码不能为空");
            Validate.notEmpty(detail.getMaterialCode(), "物料编码不能为空");
            Validate.notEmpty(detail.getCustomerCode(), "客户编码不能为空");
            Validate.notNull(detail.getPurchasingQuantity(), "采购数量不能为空");
        });
    }

    /**
     * 查询数据（后端用）
     * @param dto dto对象
     * @return
     */
    @Override
    public List<TpmMaterialPurchasingOrderVo> findByConditionsForJava(TpmMaterialPurchasingOrderDto dto) {
        if (Objects.isNull(dto)) {
            return Lists.newArrayList();
        }
        dto.setTenantCode(TenantUtils.getTenantCode());
        return this.tpmMaterialPurchasingOrderRepository.findByConditionsForJava(dto);
    }

    /**
     * 分页查询当前采购单所有客户
     * @param pageable 分页对象
     * @param dto      查询dto
     * @return 所有数据
     */
    @Override
    public Page<TpmMaterialPurchasingOrderFindCustomerVo> findCustomerByCode(Pageable pageable, TpmMaterialPurchasingOrderfindCustomerDto dto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(dto)) {
            dto = new TpmMaterialPurchasingOrderfindCustomerDto();
        }
        Validate.notNull(dto.getId(), "id不能为空，请刷新后重试！");
        TpmMaterialPurchasingOrder entity = tpmMaterialPurchasingOrderRepository.getById(dto.getId());
        Validate.notNull(entity, "数据不存在，请刷新后重试！");
        Validate.isTrue(DelFlagStatusEnum.NORMAL.getCode().equals(entity.getDelFlag()), "数据不存在，请刷新后重试！");
        String tenantCode = TenantUtils.getTenantCode();
        Validate.isTrue(tenantCode.equals(entity.getTenantCode()), "数据不存在，请刷新后重试！");
        dto.setCode(entity.getCode());
        dto.setTenantCode(TenantUtils.getTenantCode());
        return this.tpmMaterialPurchasingOrderDetailRepository.findCustomerByCode(pageable, dto);
    }

    /**
     * 修改客户地址
     * @param dto dto
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void editCustomerAddressById(TpmMaterialPurchasingOrderfindCustomerDto dto) {
        if (StringUtils.isBlank(dto.getId())) {
            return;
        }
        TpmMaterialPurchasingOrder entity = tpmMaterialPurchasingOrderRepository.getById(dto.getId());
        Validate.notNull(entity, "数据不存在，请刷新后重试！");
        Validate.isTrue(DelFlagStatusEnum.NORMAL.getCode().equals(entity.getDelFlag()), "数据不存在，请刷新后重试！");
        String tenantCode = TenantUtils.getTenantCode();
        Validate.isTrue(tenantCode.equals(entity.getTenantCode()), "数据不存在，请刷新后重试！");
        Validate.noNullElements(dto.getCustomerCodeList(), "请选择需要更改地址的客户！");
        dto.setCode(entity.getCode());
        List<TpmMaterialPurchasingOrderDetail> detailList = tpmMaterialPurchasingOrderDetailRepository.findByCodeAndCustomerCodeList(dto);
        String recipientAddress = dto.getRecipientAddress();
        detailList.forEach(e -> e.setRecipientAddress(recipientAddress));
        tpmMaterialPurchasingOrderDetailRepository.saveOrUpdateBatch(detailList);
    }

    /**
     * 完成出库
     * @param dto dto
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void finishOutStorage(TpmMaterialPurchasingOrderDto dto) {
        if (StringUtils.isBlank(dto.getId())) {
            return;
        }
        TpmMaterialPurchasingOrder entity = tpmMaterialPurchasingOrderRepository.getById(dto.getId());
        Validate.notNull(entity, "数据不存在，请刷新后重试！");
        Validate.isTrue(DelFlagStatusEnum.NORMAL.getCode().equals(entity.getDelFlag()), "数据不存在，请刷新后重试！");
        String tenantCode = TenantUtils.getTenantCode();
        Validate.isTrue(tenantCode.equals(entity.getTenantCode()), "数据不存在，请刷新后重试！");
        entity.setStatus(TpmMaterialPurchasingOrderEnums.OrderStatus.TO_SUBMIT_10.getValue());
        tpmMaterialPurchasingOrderRepository.saveOrUpdate(entity);
    }

    /**
     * 分页查询当前采购单所有可发货明细
     * @param pageable 分页对象
     * @param dto      查询dto
     * @return 所有数据
     */
    @Override
    public Page<TpmMaterialPurchasingOrderDetailVo> findOutStorageAbleDetailById(Pageable pageable, TpmMaterialPurchasingOrderDto dto) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));
        if (Objects.isNull(dto)) {
            dto = new TpmMaterialPurchasingOrderDto();
        }
        Validate.notNull(dto.getId(), "id不能为空，请刷新后重试！");
        TpmMaterialPurchasingOrder entity = tpmMaterialPurchasingOrderRepository.getById(dto.getId());
        Validate.notNull(entity, "数据不存在，请刷新后重试！");
        Validate.isTrue(DelFlagStatusEnum.NORMAL.getCode().equals(entity.getDelFlag()), "数据不存在，请刷新后重试！");
        String tenantCode = TenantUtils.getTenantCode();
        Validate.isTrue(tenantCode.equals(entity.getTenantCode()), "数据不存在，请刷新后重试！");
        Validate.isTrue(TpmMaterialPurchasingOrderEnums.OrderStatus.WAIT_CONFIRM_20.getValue().equals(entity.getStatus()), "只有待出库状态的采购单可以进行发货");
        dto.setCode(entity.getCode());
        dto.setTenantCode(TenantUtils.getTenantCode());
        return this.tpmMaterialPurchasingOrderDetailRepository.findOutStorageAbleDetailById(pageable, dto);
    }

    /**
     * 结案核销修改 已结案金额、是否完全结案状态
     * @param dtoList dtoList
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void audit(List<TpmMaterialPurchasingOrderDto> dtoList) {

    }

    /**
     * 通过主键查询单条数据
     * @param id 主键
     * @return 单条数据
     */
    @Override
    public TpmMaterialPurchasingOrderVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        TpmMaterialPurchasingOrder entity = tpmMaterialPurchasingOrderRepository.getById(id);
        Validate.notNull(entity, "数据不存在，请刷新后重试！");
        Validate.isTrue(DelFlagStatusEnum.NORMAL.getCode().equals(entity.getDelFlag()), "数据不存在，请刷新后重试！");
        String tenantCode = TenantUtils.getTenantCode();
        Validate.isTrue(tenantCode.equals(entity.getTenantCode()), "数据不存在，请刷新后重试！");
        TpmMaterialPurchasingOrderVo tpmMaterialPurchasingOrderVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, TpmMaterialPurchasingOrderVo.class, LinkedHashSet.class, ArrayList.class);
        List<TpmMaterialPurchasingOrderDetailVo> detailVos = tpmMaterialPurchasingOrderDetailRepository.findByCode(entity.getCode());
        tpmMaterialPurchasingOrderVo.setDetailList(detailVos);
        return tpmMaterialPurchasingOrderVo;
    }

    /**
     * 新增数据
     * @param dto dto
     * @return 新增结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Deprecated
    @Override
    public TpmMaterialPurchasingOrderVo create(TpmMaterialPurchasingOrderDto dto) {
        this.createValidate(dto);
        dto.setStatus(TpmMaterialPurchasingOrderEnums.OrderStatus.TO_SUBMIT_10.getValue());
        TpmMaterialPurchasingOrder entity = nebulaToolkitService.copyObjectByWhiteList(dto, TpmMaterialPurchasingOrder.class, HashSet.class, ArrayList.class);
        //租户编号赋值
        String tenantCode = TenantUtils.getTenantCode();
        entity.setTenantCode(tenantCode);
        String code = generateCodeService.generateCode(TpmMaterialPurchasingOrderConstant.TPM_MATERIAL_PURCHASING_ORDER_FORMULA_PREFIX, 1, 5, 2, TimeUnit.DAYS).get(0);
        AtomicInteger index = new AtomicInteger(1);
        Collection<TpmMaterialPurchasingOrderDetail> details = nebulaToolkitService.copyCollectionByWhiteList(dto.getDetailList(), TpmMaterialPurchasingOrderDetailDto.class,
                                                                                                              TpmMaterialPurchasingOrderDetail.class, HashSet.class, ArrayList.class);
        details.forEach(e -> {
            e.setTenantCode(tenantCode);
            e.setCode(code);
            e.setSubCode(code + "-" + index.getAndIncrement());
        });
        //保存
        this.tpmMaterialPurchasingOrderRepository.saveOrUpdate(entity);
        this.tpmMaterialPurchasingOrderDetailRepository.saveBatch(details);

        //日志
        TpmMaterialPurchasingOrderLogEventDto logEventDto = new TpmMaterialPurchasingOrderLogEventDto();
        logEventDto.setOriginal(null);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<TpmMaterialPurchasingOrderLogEventListener, TpmMaterialPurchasingOrderLogEventDto> onCreate =
                TpmMaterialPurchasingOrderLogEventListener::onCreate;
        this.nebulaNetEventClient.publish(logEventDto, TpmMaterialPurchasingOrderLogEventListener.class, onCreate);
        return this.nebulaToolkitService.copyObjectByWhiteList(entity, TpmMaterialPurchasingOrderVo.class, LinkedHashSet.class, ArrayList.class);
    }


    /**
     * 保存活动细案的物料采购单
     * @param tpmMaterialPurchasingOrderDetailDtoList tpmMaterialPurchasingOrderDetailDtoList
     */
    @Override
    public void updateBatch(List<TpmMaterialPurchasingOrderDetailDto> tpmMaterialPurchasingOrderDetailDtoList) {
        if (CollectionUtils.isEmpty(tpmMaterialPurchasingOrderDetailDtoList)) {
            return;
        }
        String tenantCode = TenantUtils.getTenantCode();
        List<String> detailPlanItemCodeList = tpmMaterialPurchasingOrderDetailDtoList.stream().map(TpmMaterialPurchasingOrderDetailDto::getActivityDetailItemCode).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        List<TpmMaterialPurchasingOrderDetail> detailEntityList = tpmMaterialPurchasingOrderDetailRepository.findByDetailPlanItemCodeList(detailPlanItemCodeList);
        Map<String, TpmMaterialPurchasingOrderDetailDto> detailPlanItemCodeToDetailDto = tpmMaterialPurchasingOrderDetailDtoList.stream().collect(Collectors.toMap(TpmMaterialPurchasingOrderDetailDto::getActivityDetailItemCode, Function.identity(), (e1, e2) -> e1));
        detailEntityList.forEach(entity -> {
            String activityDetailItemCode = entity.getActivityDetailItemCode();
            if (detailPlanItemCodeToDetailDto.containsKey(activityDetailItemCode)) {
                TpmMaterialPurchasingOrderDetailDto item = detailPlanItemCodeToDetailDto.get(activityDetailItemCode);
                entity.setTenantCode(tenantCode);
                //entity.setActivityDetailCode(item.getActivityDetailCode());
                //entity.setActivityDetailName(item.getActivityDetailName());
                entity.setActivityDetailItemCode(item.getActivityDetailItemCode());
//                BigDecimal quantity = item.getQuantity();
//                //entity.setActivityDetailItemQuantity(quantity);
//                BigDecimal amount = quantity.multiply(item.getPrice());
//                //entity.setActivityDetailItemAmount(amount);
//                entity.setCustomerCode(item.getCustomerCode());
//                entity.setCustomerName(item.getCustomerName());
//                entity.setMaterialCode(item.getMaterialCode());
//                entity.setMaterialName(item.getMaterialName());
                //entity.setPrice(item.getPrice());
            }
        });
        //保存
        this.tpmMaterialPurchasingOrderDetailRepository.saveOrUpdateBatch(detailEntityList);
    }

    /**
     * 修改新据
     * @param dto dto
     * @return 修改结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public TpmMaterialPurchasingOrderVo update(TpmMaterialPurchasingOrderDto dto) {
        this.updateValidate(dto);

        //取出旧的记录
        TpmMaterialPurchasingOrder oldEntity = tpmMaterialPurchasingOrderRepository.getById(dto.getId());
        Validate.notNull(oldEntity, "修改数据不存在或已被删除！");
        Validate.isTrue(DelFlagStatusEnum.NORMAL.getCode().equals(oldEntity.getDelFlag()), "修改数据不存在或已被删除！");
        //只修改待提交状态的物料采购单
        if (!TpmMaterialPurchasingOrderEnums.OrderStatus.TO_SUBMIT_10.getValue().equals(dto.getStatus())) {
            return new TpmMaterialPurchasingOrderVo();
        }
        Validate.notNull(oldEntity, "数据不存在，请刷新后重试！");
        Validate.isTrue(DelFlagStatusEnum.NORMAL.getCode().equals(oldEntity.getDelFlag()), "数据不存在，请刷新后重试！");
        TpmMaterialPurchasingOrderDto oldDto = this.nebulaToolkitService.copyObjectByWhiteList(oldEntity, TpmMaterialPurchasingOrderDto.class, LinkedHashSet.class, ArrayList.class);
        List<TpmMaterialPurchasingOrderDetailVo> detailVos = tpmMaterialPurchasingOrderDetailRepository.findByCode(oldEntity.getCode());
        oldDto.setDetailList(Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(detailVos, TpmMaterialPurchasingOrderDetailVo.class, TpmMaterialPurchasingOrderDetailDto.class, HashSet.class, ArrayList.class)));

        //新的实体,租户编号赋值
        String tenantCode = TenantUtils.getTenantCode();
        TpmMaterialPurchasingOrder newEntity = nebulaToolkitService.copyObjectByWhiteList(dto, TpmMaterialPurchasingOrder.class, HashSet.class, ArrayList.class);
        Collection<TpmMaterialPurchasingOrderDetail> newDetails = nebulaToolkitService.copyCollectionByWhiteList(
                dto.getDetailList(), TpmMaterialPurchasingOrderDetailDto.class, TpmMaterialPurchasingOrderDetail.class, HashSet.class, ArrayList.class);
        String code = newEntity.getCode();
        AtomicInteger index = new AtomicInteger(1);
        newDetails.forEach(e -> {
            e.setTenantCode(tenantCode);
            e.setCode(code);
            e.setSubCode(code + "-" + index.getAndIncrement());
        });

        //物理删除子表信息
        this.tpmMaterialPurchasingOrderDetailRepository.removeByCode(oldEntity.getCode());

        //保存
        this.tpmMaterialPurchasingOrderRepository.saveOrUpdate(newEntity);
        this.tpmMaterialPurchasingOrderDetailRepository.saveBatch(newDetails);

        //日志
        TpmMaterialPurchasingOrderLogEventDto logEventDto = new TpmMaterialPurchasingOrderLogEventDto();
        logEventDto.setOriginal(oldDto);
        logEventDto.setNewest(dto);
        SerializableBiConsumer<TpmMaterialPurchasingOrderLogEventListener, TpmMaterialPurchasingOrderLogEventDto> onUpdate =
                TpmMaterialPurchasingOrderLogEventListener::onUpdate;
        this.nebulaNetEventClient.publish(logEventDto, TpmMaterialPurchasingOrderLogEventListener.class, onUpdate);

        return this.nebulaToolkitService.copyObjectByWhiteList(newEntity, TpmMaterialPurchasingOrderVo.class, LinkedHashSet.class, ArrayList.class);

    }

    /**
     * 采购单确认
     * @param dto dto
     * @return 确认结果
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void confirm(TpmMaterialPurchasingOrderDto dto) {
        this.confirmOrOutstorageOrSignValidate(dto);
        List<TpmMaterialPurchasingOrder> entityList = this.tpmMaterialPurchasingOrderRepository.listByIds(dto.getIds());
        Validate.notEmpty(entityList, "未找到物料采购单信息");
        entityList.forEach(entity -> {
            Validate.isTrue(StringUtils.equalsAny(entity.getStatus(),
                                                  TpmMaterialPurchasingOrderEnums.OrderStatus.TO_SUBMIT_10.getValue(),
                                                  TpmMaterialPurchasingOrderEnums.OrderStatus.BACK_40.getValue(),
                                                  TpmMaterialPurchasingOrderEnums.OrderStatus.WAIT_EDIT.getValue()
            ), "只能确认状态为：【待提交】【已撤回】【待修改】的物料采购单");
            Validate.isTrue(!(TpmMaterialPurchasingOrderEnums.OrderChangeStatus.CHANGEING.getValue().equals(entity.getChangeStatus())
                                    && TpmMaterialPurchasingOrderEnums.OrderChangeStatus.PART_CHANGED.getValue().equals(entity.getChangeStatus()))
                    , "不能确认采购明细中状态有【调整中、部分调整】的物料采购单");
            Validate.isTrue(YesOrNoEnum.YES.getCode().equals(entity.getMaterialGroupEdit()),"物料组编辑必须为是");
        });
        //需要判断是否有调整中的明细
        /*List<TpmMaterialPurchasingOrderDetail> detailList = this.tpmMaterialPurchasingOrderDetailRepository.findByCodeList(entityList.stream().map(TpmMaterialPurchasingOrder::getCode).collect(Collectors.toList()));
        detailList.forEach(e -> Validate.isTrue(!TpmMaterialPurchasingOrderEnums.OrderChangeStatus.CHANGEING.getValue().equals(e.getChangeStatus())
                , "不能确认采购明细中状态有【调整中】的物料采购单"));*/
        List<String> codes = Lists.newArrayList();
        Date now = new Date();
        entityList.forEach(entity -> {
            entity.setConfirmDate(now);
            entity.setStatus(TpmMaterialPurchasingOrderEnums.OrderStatus.WAIT_CONFIRM_20.getValue());
//            entity.setSubmitTime(DateUtil.dateStrNowAll());
            codes.add(entity.getCode());
        });

        verifyDetailData(codes);
        ProcessBusinessDto processBusinessDto = dto.getProcessBusiness();
        processBusinessDto.setBusinessCode(TpmMaterialPurchasingOrderConstant.TPM_MATERIAL_PURCHASING_ORDER_PROCESS);
        processBusinessDto.setBusinessNoList(codes);
        processBusinessDto.setBusinessNo(new Snowflake().nextIdStr());
        log.info("采购单流程提交参数,{}", JSON.toJSONString(processBusinessDto));
        ProcessBusinessVo processBusinessVo = processBatchBusinessService.processStart(processBusinessDto);
        entityList.forEach(entity -> {
            entity.setProcessNo(processBusinessVo.getProcessNo());
        });
        this.tpmMaterialPurchasingOrderRepository.updateBatchById(entityList);
    }

    private void verifyDetailData(List<String> codes) {
        if (CollectionUtils.isEmpty(codes)){
            return;
        }
        List<TpmMaterialPurchasingOrderDetail> detailList = this.tpmMaterialPurchasingOrderDetailRepository.findByCodeList(codes);
        Validate.isTrue(CollectionUtils.isNotEmpty(detailList),"采购单都未维护明细数据");
        Map<String, List<TpmMaterialPurchasingOrderDetail>> detailMap = detailList.stream().collect(Collectors.groupingBy(TpmMaterialPurchasingOrderDetail::getCode));
        for (String code : codes) {
            List<TpmMaterialPurchasingOrderDetail> detailList1 = detailMap.get(code);
            Validate.isTrue(CollectionUtils.isNotEmpty(detailList1),"采购单编码【%s】明细数据不能为空",code);
        }
    }


    /**
     * 创建验证
     * @param dto dto
     */
    private void createValidate(TpmMaterialPurchasingOrderDto dto) {
        Validate.notNull(dto, "新增时，对象信息不能为空！");
        dto.setId(null);
        Validate.notBlank(dto.getBusinessFormatCode(), "新增数据时，业态不能为空！");
        Validate.notBlank(dto.getBusinessUnitCode(), "新增数据时，业务单元不能为空！");
        Validate.notBlank(dto.getStatus(), "新增数据时，状态不能为空！");
        Validate.notBlank(dto.getSupplierCode(), "新增数据时，供应商编码不能为空！");
        List<TpmMaterialPurchasingOrderDetailDto> detailList = dto.getDetailList();
        Validate.noNullElements(detailList, "新增数据时，明细不能为空！");
        detailList.forEach(detail -> {
            Validate.notBlank(detail.getActivityDetailItemCode(), "新增数据时，活动细案明细编码不能为空！");
            Validate.notBlank(detail.getCustomerCode(), "新增数据时，客户编码不能为空！");
            Validate.notBlank(detail.getMaterialCode(), "新增数据时，物料编码不能为空！");
        });
    }

    /**
     * 签收验证
     * @param dto dto
     */
    private void confirmOrOutstorageOrSignValidate(TpmMaterialPurchasingOrderDto dto) {
        Validate.notNull(dto, "对象信息不能为空！");
        Validate.notEmpty(dto.getIds(), "对象信息不能为空！");
    }

    /**
     * 供应商提交流程 回调
     * @param businessNoList
     * @param processStatus
     */
    @Override
    @Transactional
    public void supplierProcessBack(List<String> businessNoList, String processStatus) {
        if (CollectionUtils.isEmpty(businessNoList)) return;
        List<TpmMaterialPurchasingOrderDetailVo> detailVos = this.tpmMaterialPurchasingOrderDetailRepository.findByDetailCodes(businessNoList);
        log.info("物料采购单提交审批审批通过,detailVos:{}",JsonUtils.obj2JsonString(detailVos));
        this.tpmMaterialPurchasingOrderDetailRepository.supplierProcessBack(detailVos, processStatus);
        //修改活动明细上的字段采购单审批状态
        List<String> activityDetailItemCodes = detailVos.stream().map(TpmMaterialPurchasingOrderDetailVo::getActivityDetailItemCode).collect(Collectors.toList());
        this.activityDetailPlanItemSdkService.updateActivityDetailPurchasingOrderProcessStatus(activityDetailItemCodes,processStatus);
    }

    /**
     * 针对本次修改的明细对应的采购单做一个状态计算
     * @param materialPurchasingCodes
     */
    @Override
    @Transactional
    public void handleChangeStatus(List<String> materialPurchasingCodes) {
        //查询采购单下所有的明细行
        List<TpmMaterialPurchasingOrderDetail> detailList = this.tpmMaterialPurchasingOrderDetailRepository.findByCodeList(materialPurchasingCodes);
        if (CollectionUtils.isEmpty(detailList)) {
            return;
        }
        Map<String, List<TpmMaterialPurchasingOrderDetail>> groupBy = detailList.stream().collect(Collectors.groupingBy(TpmMaterialPurchasingOrderDetail::getCode));

        List<TpmMaterialPurchasingOrder> mainOrderList = new ArrayList<>();
        groupBy.forEach((k, v) -> {
            TpmMaterialPurchasingOrder main = new TpmMaterialPurchasingOrder();
            main.setCode(k);
            //1.明细行里，都为未调整的状态，采购单的状态为未调整
            //2.明细行里，有一个及以上是调整中的状态，其余的为未调整的状态，那么采购单的状态也变为调整中
            //3.明细行里，有一个及以上是已调整的状态，同时还有调整中的状态，其余为未调整，那么采购单的状态为部分已调整
            //4.明细行里，有一个及以上是已调整的状态，没有调整中的状态，可以有未调整的状态，那么采购单的状态为已调整
            if (v.stream().allMatch(e -> TpmMaterialPurchasingOrderEnums.OrderChangeStatus.NOT_CHANGED.getValue().equals(e.getActivityDetailChangeStatus()))) {
                main.setChangeStatus(TpmMaterialPurchasingOrderEnums.OrderChangeStatus.NOT_CHANGED.getValue());
            }
            if (v.stream().anyMatch(e -> TpmMaterialPurchasingOrderEnums.OrderChangeStatus.CHANGEING.getValue().equals(e.getActivityDetailChangeStatus()))
                    && v.stream().anyMatch(e -> TpmMaterialPurchasingOrderEnums.OrderChangeStatus.NOT_CHANGED.getValue().equals(e.getActivityDetailChangeStatus()))
                    && v.stream().noneMatch(e -> TpmMaterialPurchasingOrderEnums.OrderChangeStatus.CHANGED.getValue().equals(e.getActivityDetailChangeStatus()))
            ) {
                main.setChangeStatus(TpmMaterialPurchasingOrderEnums.OrderChangeStatus.CHANGEING.getValue());
            }
            if (v.stream().anyMatch(e -> TpmMaterialPurchasingOrderEnums.OrderChangeStatus.CHANGED.getValue().equals(e.getActivityDetailChangeStatus()))
                    && v.stream().anyMatch(e -> TpmMaterialPurchasingOrderEnums.OrderChangeStatus.CHANGEING.getValue().equals(e.getActivityDetailChangeStatus()))
                    && v.stream().anyMatch(e -> TpmMaterialPurchasingOrderEnums.OrderChangeStatus.NOT_CHANGED.getValue().equals(e.getActivityDetailChangeStatus()))
            ) {
                main.setChangeStatus(TpmMaterialPurchasingOrderEnums.OrderChangeStatus.PART_CHANGED.getValue());
            }
            if (v.stream().anyMatch(e -> TpmMaterialPurchasingOrderEnums.OrderChangeStatus.CHANGED.getValue().equals(e.getActivityDetailChangeStatus()))
                    && v.stream().noneMatch(e -> TpmMaterialPurchasingOrderEnums.OrderChangeStatus.CHANGEING.getValue().equals(e.getActivityDetailChangeStatus()))
            ) {
                main.setChangeStatus(TpmMaterialPurchasingOrderEnums.OrderChangeStatus.CHANGED.getValue());
            }
            mainOrderList.add(main);
        });

        this.tpmMaterialPurchasingOrderRepository.updateChangeStatusByCodes(mainOrderList);
    }

    @Override
    public List<TpmMaterialPurchasingOrderAuditFileVo> findAuditInfoByPurchaseCodes(Set<String> purchaseCodeList) {

        if(CollectionUtils.isEmpty(purchaseCodeList)){
            return Lists.newArrayList();
        }

        List<TpmMaterialPurchasingOrderAuditFile> auditFileList = this.auditFileRepository.findByBusinessCodes(new ArrayList<>(purchaseCodeList));
        if(CollectionUtils.isEmpty(auditFileList)){
            return Lists.newArrayList();
        }
        Collection<TpmMaterialPurchasingOrderAuditFileVo> tpmMaterialPurchasingOrderAuditFileVos = this.nebulaToolkitService.copyCollectionByWhiteList(auditFileList, TpmMaterialPurchasingOrderAuditFile.class, TpmMaterialPurchasingOrderAuditFileVo.class, LinkedHashSet.class, ArrayList.class);
        return (List<TpmMaterialPurchasingOrderAuditFileVo>)tpmMaterialPurchasingOrderAuditFileVos;
    }

    @Override
    public Page<TpmMaterialPurchasingOrderAuditFileVo> auditFindMaterialAuditInfo(Pageable pageable, List<String> materialPurchasingOrders) {
        if(CollectionUtils.isEmpty(materialPurchasingOrders)){
            return new Page<>(pageable.getPageNumber(),pageable.getPageSize());
        }
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(1, 50));

        return auditFileRepository.auditFindMaterialAuditInfo(pageable,materialPurchasingOrders);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void batchDeleteMaterialPurchaseOrder(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)){
            return;
        }

        List<TpmMaterialPurchasingOrderVo> materialPurchasingOrderVoList = this.tpmMaterialPurchasingOrderRepository.findByIds(ids);

        for (TpmMaterialPurchasingOrderVo tpmMaterialPurchasingOrderVo : materialPurchasingOrderVoList) {
            Validate.isTrue(TpmMaterialPurchasingOrderEnums.OrderStatus.TO_SUBMIT_10.getValue().equals(tpmMaterialPurchasingOrderVo.getStatus()),"采购单【%s】不为待提交状态，不能删除",tpmMaterialPurchasingOrderVo.getCode());
        }

        List<String> codeList = materialPurchasingOrderVoList.stream().map(TpmMaterialPurchasingOrderVo::getCode).collect(Collectors.toList());
        List<TpmMaterialPurchasingOrderDetail> detailList = this.tpmMaterialPurchasingOrderDetailRepository.findByCodeList(codeList);
        if(CollectionUtils.isNotEmpty(materialPurchasingOrderVoList)){
            List<String> codes = materialPurchasingOrderVoList.stream().map(TpmMaterialPurchasingOrderVo::getCode).collect(Collectors.toList());
            this.tpmMaterialPurchasingOrderDetailRepository.deleteByCodes(codes);
            this.tpmMaterialPurchasingOrderRepository.deleteByIds(ids);
        }

        //清除活动明细上的采购单号
        for (TpmMaterialPurchasingOrderVo orderVo : materialPurchasingOrderVoList) {
            Map<String, List<TpmMaterialPurchasingOrderDetail>> detailMap = detailList.stream().collect(Collectors.groupingBy(TpmMaterialPurchasingOrderDetail::getCode));
            List<TpmMaterialPurchasingOrderDetail> detailList1 = detailMap.get(orderVo.getCode());
            if(CollectionUtils.isEmpty(detailList1)){
                return;
            }
            List<String> activityDetailItemCodes = detailList1.stream().map(TpmMaterialPurchasingOrderDetail::getActivityDetailItemCode).collect(Collectors.toList());
            activityDetailPlanItemSdkService.updateActivityDetailPurchasingOrder(orderVo.getCode(),null,activityDetailItemCodes);
        }
    }
}

