package com.biz.crm.tpm.business.month.budget.local.calculate.strategy;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgAllParentVo;
import com.biz.crm.mdm.business.sales.org.sdk.vo.SalesOrgVo;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.budget.cal.config.sdk.eunm.ActualSalesAmountTypeEnum;
import com.biz.crm.tpm.business.budget.cal.config.sdk.eunm.CalDataFromEnum;
import com.biz.crm.tpm.business.budget.cal.config.sdk.eunm.SalesOrgAreaEnum;
import com.biz.crm.tpm.business.budget.cal.config.sdk.vo.*;
import com.biz.crm.tpm.business.budget.item.sdk.enums.BudgetControlTypeEnum;
import com.biz.crm.tpm.business.budget.item.sdk.enums.FeeBelongEnum;
import com.biz.crm.tpm.business.daily.sales.data.sdk.dto.TpmDailySalesDataTotalDto;
import com.biz.crm.tpm.business.daily.sales.data.sdk.service.TpmDailySalesDataService;
import com.biz.crm.tpm.business.daily.sales.data.sdk.vo.TpmDailySalesDataTotalVo;
import com.biz.crm.tpm.business.daily.sales.data.sdk.vo.TpmDailySalesDataVo;
import com.biz.crm.tpm.business.main.oneday.sale.data.sdk.dto.MainOnedaySalesDataDto;
import com.biz.crm.tpm.business.main.oneday.sale.data.sdk.service.MainOnedaySaleDataService;
import com.biz.crm.tpm.business.main.oneday.sale.data.sdk.vo.MainOnedaySalesDataVo;
import com.biz.crm.tpm.business.month.budget.local.calculate.BudgetCalculateStrategy;
import com.biz.crm.tpm.business.month.budget.local.calculate.dto.SalesDataDto;
import com.biz.crm.tpm.business.month.budget.local.calculate.vo.SalesDataVo;
import com.biz.crm.tpm.business.month.budget.local.helper.MonthBudgetCalculateHelper;
import com.biz.crm.tpm.business.month.budget.sdk.constant.MonthBudgetConstant;
import com.biz.crm.tpm.business.month.budget.sdk.dto.YearBudgetEventDto;
import com.biz.crm.tpm.business.month.budget.sdk.event.YearBudgetEventListener;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetActualSalesVo;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetVo;
import com.biz.crm.tpm.business.month.budget.sdk.vo.YearBudgetResponse;
import com.biz.crm.tpm.business.sales.plan.sdk.dto.SalesPlanDto;
import com.biz.crm.tpm.business.sales.plan.sdk.service.SalesPlanService;
import com.biz.crm.tpm.business.sales.plan.sdk.vo.SalesPlanVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.model.EventResponse;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import javax.validation.constraints.Min;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * @Description 计算主体
 * @Author YangWei
 * @Date 2023/2/18 上午12:23
 */
@Component
@Slf4j
public class BudgetCalculateHeadStrategy implements BudgetCalculateStrategy {

    @Autowired(required = false)
    private OrgVoService orgVoService;

    @Autowired(required = false)
    private CustomerVoService customerVoService;

    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private MainOnedaySaleDataService mainOnedaySaleDataService;

    @Resource
    private MonthBudgetCalculateHelper monthBudgetCalculateHelper;

    @Autowired(required = false)
    private SalesPlanService salesPlanService;

    @Autowired(required = false)
    private NebulaNetEventClient nebulaNetEventClient;

    @Autowired(required = false)
    private TpmDailySalesDataService tpmDailySalesDataService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    /**
     * 获取类型
     *
     * @param
     * @return {@link String}
     */
    @Override
    public String getBusinessUnit() {
        return BusinessUnitEnum.HEADQUARTERS.getCode();
    }

    /**
     * 构建查询销售数据参数
     *
     * @param monthBudgetVo
     * @param actualSalesVo
     * @return {@link SalesDataDto}
     */
    @Override
    public SalesDataDto buildDto(MonthBudgetVo monthBudgetVo, MonthBudgetActualSalesVo actualSalesVo) {
        log.info("实际销量计算对象,构建查询销售数据参数-主体");
        SalesDataDto dto = new SalesDataDto();
        dto.setYearMonthLy(monthBudgetVo.getYearMonthLy().replace("-", ""));
        dto.setBusinessFormatCode(monthBudgetVo.getBusinessFormatCode());
        dto.setBusinessUnitCode(monthBudgetVo.getBusinessUnitCode());
        //预算计算配置
        BudgetCalConfigVo budgetCalConfigVo = actualSalesVo.getBudgetCalConfigVo();
        if (ObjectUtils.isEmpty(budgetCalConfigVo)) {
            return null;
        }
        if (!CollectionUtils.isEmpty(budgetCalConfigVo.getCustomerList())) {
            List<String> customerCodeList = budgetCalConfigVo.getCustomerList().stream().map(BudgetCalConfigAreaVo::getDataCode).collect(Collectors.toList());
            dto.setExcludeCustomerCodeList(customerCodeList);
        }
        if (!CollectionUtils.isEmpty(budgetCalConfigVo.getTerminalList())) {
            List<String> terminalCodeList = budgetCalConfigVo.getTerminalList().stream().map(BudgetCalConfigAreaVo::getDataCode).collect(Collectors.toList());
            dto.setExcludeTerminalCodeList(terminalCodeList);
        }
        if (!CollectionUtils.isEmpty(budgetCalConfigVo.getProductList())) {
            List<String> productCodeList = budgetCalConfigVo.getProductList().stream().map(BudgetCalConfigAreaVo::getDataCode).collect(Collectors.toList());
            dto.setExcludeProductCodeList(productCodeList);
        }

        // 销售组织范围：全部、当前、自选；全部则不用查询客户过滤；当前需要按月度预算上的组织查询下级；自选则直接按所选销售组织查询
        if (SalesOrgAreaEnum.CUR_ORG.getCode().equals(budgetCalConfigVo.getOrgAreaCode())) {
            if (StringUtils.isEmpty(monthBudgetVo.getSalesOrgCode()) && StringUtils.isEmpty(monthBudgetVo.getOrgCode())) {
                return null;
            }
            String salesOrgCode = null;
            // 获取年度预算上当前组织
            if (!StringUtils.isEmpty(monthBudgetVo.getSalesOrgCode())) {
                salesOrgCode = monthBudgetVo.getSalesOrgCode();
            } else {
                if (!StringUtils.isEmpty(monthBudgetVo.getOrgCode())) {
                    OrgVo byOrgCode = orgVoService.findByOrgCode(monthBudgetVo.getOrgCode());
                    salesOrgCode = byOrgCode.getSalesOrgCode();
                }
            }
            if (StringUtils.isEmpty(salesOrgCode)) {
                return null;
            }
            List<SalesOrgVo> salesOrgVos = salesOrgVoService.findAllChildrenBySalesOrgCode(salesOrgCode);
            if (CollectionUtils.isEmpty(salesOrgVos)) {
                return null;
            }
            List<String> saleOrgCodeList = salesOrgVos.stream().map(SalesOrgVo::getSalesOrgCode).collect(Collectors.toList());
            dto.setCusSalesOrgCodeList(saleOrgCodeList);

            //集团卡客户用
            if (!CollectionUtils.isEmpty(saleOrgCodeList)) {
                List<SalesOrgAllParentVo> list = salesOrgVoService.findSalesOrgIncludeAllParentByCodes(saleOrgCodeList);
                if (!CollectionUtils.isEmpty(list)) {
                    List<String> salesInstitutionCodes = list.stream().filter(e -> saleOrgCodeList.contains(e.getCurrSalesOrgCode())).map(SalesOrgAllParentVo::getSalesInstitutionCode).distinct().collect(Collectors.toList());
                    dto.setSalesInstitutionCodes(salesInstitutionCodes);
                }
            }
//            log.error("预算销售组织={}，主数据销售组织={}",salesOrgCode,saleOrgCodeList);
            List<CustomerVo> customerVoList = customerVoService.findBySalesOrgCodes(saleOrgCodeList);
            if (CollectionUtils.isEmpty(customerVoList)) {
                return null;
            }
            List<String> customerCodeList = customerVoList.stream().map(CustomerVo::getCustomerCode).collect(Collectors.toList());
//            log.error("客户编码={}",customerCodeList);
            if (!CollectionUtils.isEmpty(dto.getExcludeCustomerCodeList())) {
                customerCodeList.removeAll(dto.getExcludeCustomerCodeList());
            }
            dto.setCustomerCodeList(customerCodeList);
        }
        if (SalesOrgAreaEnum.CUSTOMIZE_ORG.getCode().equals(budgetCalConfigVo.getOrgAreaCode())) {
            // 自定义的组织，自选组织
            if (CollectionUtils.isEmpty(budgetCalConfigVo.getSalesOrgList())) {
                return null;
            }
            List<String> saleOrgCodeList = budgetCalConfigVo.getSalesOrgList().stream()
                    .map(BudgetCalConfigSalesOrgVo::getSalesOrgCode).filter(Objects::nonNull)
                    .collect(Collectors.toList());
            dto.setCusSalesOrgCodeList(saleOrgCodeList);

            //集团卡客户用
            if (!CollectionUtils.isEmpty(saleOrgCodeList)) {
                List<SalesOrgAllParentVo> list = salesOrgVoService.findSalesOrgIncludeAllParentByCodes(saleOrgCodeList);
                if (!CollectionUtils.isEmpty(list)) {
                    List<String> salesInstitutionCodes = list.stream().filter(e -> saleOrgCodeList.contains(e.getCurrSalesOrgCode())).map(SalesOrgAllParentVo::getSalesInstitutionCode).distinct().collect(Collectors.toList());
                    dto.setSalesInstitutionCodes(salesInstitutionCodes);
                }
            }
            List<CustomerVo> customerVoList = customerVoService.findBySalesOrgCodes(saleOrgCodeList);
            if (CollectionUtils.isEmpty(customerVoList)) {
                return null;
            }
            List<String> customerCodeList = customerVoList.stream().map(CustomerVo::getCustomerCode).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(dto.getExcludeCustomerCodeList())) {
                customerCodeList.removeAll(dto.getExcludeCustomerCodeList());
            }
            dto.setCustomerCodeList(customerCodeList);
        }
        if (SalesOrgAreaEnum.CUR_CUSTOMER.getCode().equals(budgetCalConfigVo.getOrgAreaCode())) {
            List<String> customerCodes = new ArrayList<>();
            customerCodes.add(monthBudgetVo.getCustomerCode());
            dto.setCustomerCodeList(customerCodes);
        }
        //判断是否为按力度
        String controlType = budgetCalConfigVo.getControlType();
        if (BudgetControlTypeEnum.INTENSITY.getCode().equals(controlType) || BudgetControlTypeEnum.RATIO.getCode().equals(controlType)) {
            //根据预算范围的产品过滤
            List<BudgetCalConfigAreaVo> budgetItemList = budgetCalConfigVo.getBudgetItemList();
            if (!CollectionUtils.isEmpty(budgetItemList)) {
                Set<String> productCodes = Sets.newHashSet();
                //预算项目维护的产品
                budgetItemList.stream().filter(o -> o.getDataCode().equals(monthBudgetVo.getBudgetItemCode())).forEach(o -> {
                    List<BudgetCalConfigScopeProductVo> scopeProducts = o.getScopeProducts();
                    if (!CollectionUtils.isEmpty(scopeProducts)) {
                        scopeProducts.forEach(product -> {
                            productCodes.add(product.getProductCode());
                        });
                    }
                });
                dto.setIncludeProductCodeList(Lists.newArrayList(productCodes));
                //和产品非包含互斥
                if (!CollectionUtils.isEmpty(productCodes)) {
                    dto.setExcludeProductCodeList(null);
                }
            }
        }
        return dto;
    }

    /**
     * 查询销售数据
     *
     * @param dto
     * @return {@link Map}<{@link String}, {@link List}<{@link SalesDataVo}>>
     */
    @Override
    public Map<String, List<SalesDataVo>> findSalesData(SalesDataDto dto) {
        MainOnedaySalesDataDto salesDataDto = this.nebulaToolkitService
                .copyObjectByBlankList(dto, MainOnedaySalesDataDto.class, HashSet.class, ArrayList.class);
        salesDataDto.setExcludeCustomerCodeList(dto.getExcludeCustomerCodeList());
        salesDataDto.setExcludeTerminalCodeList(dto.getExcludeTerminalCodeList());
        salesDataDto.setExcludeProductCodeList(dto.getExcludeProductCodeList());
        salesDataDto.setCustomerCodeList(dto.getCustomerCodeList());
        salesDataDto.setProductCodes(dto.getIncludeProductCodeList());
        salesDataDto.setCusSalesOrgCodeList(dto.getCusSalesOrgCodeList());
        log.error("查询主体日销售数据入参{}", JSONObject.toJSONString(salesDataDto));
        List<MainOnedaySalesDataVo> mainOnedaySalesDataVos = mainOnedaySaleDataService.listMainOnedaySalesDataForMonthBudget(salesDataDto);
        if (CollectionUtils.isEmpty(mainOnedaySalesDataVos)) {
            return Maps.newHashMap();
        }
        log.error("主体日销售数据返回金额：{}", mainOnedaySalesDataVos.stream().map(MainOnedaySalesDataVo::getDiscountBehindSaleAmount).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add));
        log.error("主体日销售数据返回客户编码={},返回数据条数{}", JSONObject.toJSONString(mainOnedaySalesDataVos.stream().map(MainOnedaySalesDataVo::getCustomerCode).collect(Collectors.toSet())),
                mainOnedaySalesDataVos.size());
        List<SalesDataVo> salesDataVos = (List<SalesDataVo>) nebulaToolkitService.copyCollectionByWhiteList(
                mainOnedaySalesDataVos, MainOnedaySalesDataVo.class, SalesDataVo.class, HashSet.class, ArrayList.class
        );
        return salesDataVos.stream().collect(Collectors.groupingBy(i -> i.getCustomerCode() + i.getProductCode()));
    }

    /**
     * 汇总得到实际销量
     *
     * @param actualSalesVo 月度预算相关参数
     * @param salesDataMap  销售汇总数据
     * @param salesDataMap  销售数据筛选参数
     */
    @Override
    public void summarySalesData(MonthBudgetActualSalesVo actualSalesVo, Map<String, List<SalesDataVo>> salesDataMap, SalesDataDto salesDataDto, MonthBudgetVo monthBudgetVo) {
        log.info("实际销量计算对象,汇总");
        //预算计算配置
        BudgetCalConfigVo budgetCalConfigVo = actualSalesVo.getBudgetCalConfigVo();
        if (ObjectUtils.isEmpty(budgetCalConfigVo)) {
            return;
        }
        List<BudgetCalConfigDataVo> dataList = budgetCalConfigVo.getDataList();
        if (CollectionUtils.isEmpty(dataList)) {
            return;
        }
        //数据来源
        BudgetCalConfigDataVo budgetCalConfigDataVo = dataList.stream()
                .filter(o -> CalDataFromEnum.ACTUAL_SALES_AMOUNT.getCode().equals(o.getCalDataFromCode())).findFirst().orElse(null);
        if (ObjectUtils.isEmpty(budgetCalConfigDataVo)) {
            return;
        }

        //拿到配置的产品计费比例
        List<BudgetCalConfigProductRatioVo> productRatios = budgetCalConfigVo.getProductRatios();
        Map<String, BudgetCalConfigProductRatioVo> productRatioVoMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(productRatios)) {
            productRatioVoMap = productRatios.stream().collect(Collectors.toMap(BudgetCalConfigProductRatioVo::getProductCode, e -> e));
        }

        //获取年度预算总点数
        BigDecimal yearBudgetTotalPoint = null;
        YearBudgetEventDto eventDto = new YearBudgetEventDto();
        eventDto.setYearBudgetCode(actualSalesVo.getYearBudgetCode());
        eventDto.setSalesOrgCode(actualSalesVo.getSalesOrgCode());
        eventDto.setBusinessFormatCode(budgetCalConfigVo.getBusinessFormatCode());
        eventDto.setBusinessUnitCode(budgetCalConfigVo.getBusinessUnitCode());
        eventDto.setBudgetTypeCode(budgetCalConfigVo.getBudgetTypeCode());
        eventDto.setFeeBelongCode(budgetCalConfigVo.getFeeBelongCode());
        eventDto.setControlType(budgetCalConfigVo.getControlType());
        eventDto.setOrgAreaCode(budgetCalConfigVo.getOrgAreaCode());
        eventDto.setSalesOrgList(budgetCalConfigVo.getSalesOrgList());
        eventDto.setOrgCode(actualSalesVo.getOrgCode());
        eventDto.setCustomerCode(monthBudgetVo.getCustomerCode());
        eventDto.setGroupCode(monthBudgetVo.getGroupCode());
        eventDto.setYearLy(DateUtil.dateToStr(DateUtil.strToDate(actualSalesVo.getYearMonthLy(), DateUtil.date_yyyy_MM)
                , DateUtil.date_yyyy));
        log.info("月度预算计算计划量/回复量,eventDto==>:{}", eventDto);
        SerializableBiConsumer<YearBudgetEventListener, YearBudgetEventDto> getYearBudgetByCode =
                YearBudgetEventListener::getYearBudgetByCode;
        EventResponse eventResponse = this.nebulaNetEventClient.directPublish(eventDto, YearBudgetEventListener.class, getYearBudgetByCode);
        if (!Objects.isNull(eventResponse)) {
            YearBudgetResponse budgetResponse = (YearBudgetResponse) eventResponse;
            if (!Objects.isNull(budgetResponse.getBudgetTotalPoint())) {
                yearBudgetTotalPoint = budgetResponse.getBudgetTotalPoint().divide(new BigDecimal(100), 8, BigDecimal.ROUND_HALF_UP);
            }
        }

        //判断管控类型
        //需要除数逻辑(年度点数/年度力度)
        BigDecimal pointOrIntensity = null;
        //管控类型
        String controlTypeCode = budgetCalConfigVo.getControlType();
        if (BudgetControlTypeEnum.RATIO.getCode().equals(controlTypeCode)) {
            Assert.notNull(actualSalesVo.getBudgetTotalPoint(), "预算总点数不能为空");
            pointOrIntensity = actualSalesVo.getBudgetTotalPoint().divide(new BigDecimal(100), 8, BigDecimal.ROUND_HALF_UP);
        } else if (BudgetControlTypeEnum.INTENSITY.getCode().equals(controlTypeCode)) {
            log.info("summarySalesData 计算实销量回复差");
            //分子
            BigDecimal budgetIntensityNumerator = actualSalesVo.getBudgetIntensityNumerator();
            Assert.notNull(budgetIntensityNumerator, "年度力度分子不能为空");
            log.info("summarySalesData 计算实销量回复差：分子{}", budgetIntensityNumerator);
            //分母
            BigDecimal budgetIntensityDenominator = actualSalesVo.getBudgetIntensityDenominator();
            Assert.notNull(budgetIntensityDenominator, "年度力度分母不能为空");
            log.info("summarySalesData 计算实销量回复差：分母{}", budgetIntensityDenominator);
            pointOrIntensity = budgetIntensityNumerator.divide(budgetIntensityDenominator, 6, BigDecimal.ROUND_HALF_UP);
        }
        log.info("summarySalesData 计算实销量回复差：{}", pointOrIntensity);

        // 总部：计费产品费用合计值/单个预算项目点数=倒算计费产品对应的回复量
        //     大区：计费产品费用合计值/区域整体授权点数=倒算计费产品对应的回复量
        if (FeeBelongEnum.AREA.getCode().equals(budgetCalConfigVo.getFeeBelongCode())) {
            pointOrIntensity = yearBudgetTotalPoint;
        }

        if (null == pointOrIntensity) {
            return;
        }
        AtomicReference<BigDecimal> actualSales = new AtomicReference<>(BigDecimal.ZERO);
        //金额类型
        String amountTypeCode = budgetCalConfigDataVo.getAmountTypeCode();
        ActualSalesAmountTypeEnum amountTypeEnum = ActualSalesAmountTypeEnum.getByCode(amountTypeCode);
        Assert.notNull(amountTypeEnum, "金额类型[" + amountTypeCode + "]不合法!");
        List<TpmDailySalesDataVo> salesDataVos = new ArrayList<>();
        //集团卡客户
        Map<String, String> jituanCardMap = dictToolkitService.findMapByDictTypeCode(MonthBudgetConstant.MDM_CUSTOMER_JITUAN_CARD);
        List<String> jituanCardCustomerList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(salesDataDto.getSalesInstitutionCodes())) {
            jituanCardMap.forEach((k, v) -> {
                if (salesDataDto.getSalesInstitutionCodes().contains(k)) {
                    jituanCardCustomerList.add(v);
                }
            });
        }
        String yearMonthLy = actualSalesVo.getYearMonthLy().replace("-", "");
        switch (amountTypeEnum) {
            case DISCOUNT_BEHIND_SALE_AMOUNT:
                //折后销售额（不含奶卡）
                // 主体日销售033报表 折后总金额
                TpmDailySalesDataTotalDto totalDto3 = new TpmDailySalesDataTotalDto();
                //修改年月格式
                totalDto3.setYearMonthLy(yearMonthLy);
                totalDto3.setBusinessFormatCode(salesDataDto.getBusinessFormatCode());
                totalDto3.setBusinessUnitCode(salesDataDto.getBusinessUnitCode());
                totalDto3.setExcludeCustomerCodeList(salesDataDto.getExcludeCustomerCodeList());
                totalDto3.setExcludeTerminalCodeList(salesDataDto.getExcludeTerminalCodeList());
                totalDto3.setExcludeProductCodeList(salesDataDto.getExcludeProductCodeList());
                totalDto3.setCustomerCodeList(salesDataDto.getCustomerCodeList());
                totalDto3.setSalesProductCodeList(salesDataDto.getIncludeProductCodeList());
                salesDataVos = this.tpmDailySalesDataService.statisticsListByCondition(totalDto3);
                log.error("summarySalesData 033报表统计数据=DISCOUNT_BEHIND_SALE_AMOUNT={}", salesDataVos.stream().map(TpmDailySalesDataVo::getAfterDiscountAmt).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add));

                //集团卡客户
                if (!CollectionUtils.isEmpty(jituanCardCustomerList)) {
                    List<String> soldPartyCodeList = salesDataDto.getCustomerCodeList().stream().map(e -> e.substring(0, Math.min(e.length(), 10))).collect(Collectors.toList());
                    totalDto3.setCustomerCodeList(jituanCardCustomerList);
                    totalDto3.setServiceCodeList(soldPartyCodeList);
                    List<TpmDailySalesDataVo> vos = this.tpmDailySalesDataService.statisticsListByCondition(totalDto3);
                    log.error("summarySalesData 033报表统计数据=DISCOUNT_BEHIND_SALE_AMOUNT=集团卡={}", vos.stream().map(TpmDailySalesDataVo::getAfterDiscountAmt).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add));
                    salesDataVos.addAll(vos);
                }

                for (TpmDailySalesDataVo salesDataVo : salesDataVos) {
                    BigDecimal amount = Optional.ofNullable(salesDataVo.getAfterDiscountAmt()).orElse(BigDecimal.ZERO);
                    if (BigDecimal.ZERO.compareTo(amount) == 0) {
                        continue;
                    }
                    this.setReduce(amount, actualSales, productRatioVoMap, salesDataVo.getMaterialCode(), yearBudgetTotalPoint, pointOrIntensity);
                }

                for (String key : salesDataMap.keySet()) {
                    SalesDataVo dataVo = salesDataMap.get(key).get(0);
                    BigDecimal amount = Optional.ofNullable(dataVo.getDiscountBehindSaleAmount()).orElse(BigDecimal.ZERO);
                    if (BigDecimal.ZERO.compareTo(amount) == 0) {
                        continue;
                    }
                    log.error("summarySalesData setReduce amount={} ", amount);
                    this.setReduce(amount, actualSales, productRatioVoMap, dataVo.getProductCode(), yearBudgetTotalPoint, pointOrIntensity);
                }
                break;
            case ORDER_RULE_AMT_IN_REBATE:
                //折前销售额
                for (Map.Entry<String, List<SalesDataVo>> entry : salesDataMap.entrySet()) {
                    SalesDataVo dataVo = entry.getValue().get(0);
                    BigDecimal amount = Optional.ofNullable(dataVo.getDiscountFrontSaleAmount()).orElse(BigDecimal.ZERO);
                    if (BigDecimal.ZERO.compareTo(amount) == 0) {
                        continue;
                    }
                    this.setReduce(amount, actualSales, productRatioVoMap, dataVo.getProductCode(), yearBudgetTotalPoint, pointOrIntensity);
                }
                break;
            case WAREHS_OUT_DISCOUNT_AMT:
                //折后销售额
                for (Map.Entry<String, List<SalesDataVo>> entry : salesDataMap.entrySet()) {
                    SalesDataVo dataVo = entry.getValue().get(0);
                    BigDecimal amount = Optional.ofNullable(dataVo.getWarehsOutDiscountAmtIncM()).orElse(BigDecimal.ZERO);
                    if (BigDecimal.ZERO.compareTo(amount) == 0) {
                        continue;
                    }
                    this.setReduce(amount, actualSales, productRatioVoMap, dataVo.getProductCode(), yearBudgetTotalPoint, pointOrIntensity);
                }
                break;
            case DISCOUNT_BEHIND_TAX_SALE_AMOUNT:
                //折后销售额（含奶卡）
                for (Map.Entry<String, List<SalesDataVo>> entry : salesDataMap.entrySet()) {
                    SalesDataVo dataVo = entry.getValue().get(0);
                    BigDecimal amount = Optional.ofNullable(dataVo.getDiscountBehindTaxSaleAmount()).orElse(BigDecimal.ZERO);
                    if (BigDecimal.ZERO.compareTo(amount) == 0) {
                        continue;
                    }
                    this.setReduce(amount, actualSales, productRatioVoMap, dataVo.getProductCode(), yearBudgetTotalPoint, pointOrIntensity);
                }
                break;
            case SALE_QUANTITY:
                //销量
                //销量映射订单数量 主体日销售033报表
                TpmDailySalesDataTotalDto totalDto = new TpmDailySalesDataTotalDto();
                //修改年月格式
                totalDto.setYearMonthLy(yearMonthLy);
                totalDto.setBusinessFormatCode(salesDataDto.getBusinessFormatCode());
                totalDto.setBusinessUnitCode(salesDataDto.getBusinessUnitCode());
                totalDto.setExcludeCustomerCodeList(salesDataDto.getExcludeCustomerCodeList());
                totalDto.setExcludeTerminalCodeList(salesDataDto.getExcludeTerminalCodeList());
                totalDto.setExcludeProductCodeList(salesDataDto.getExcludeProductCodeList());
                totalDto.setCustomerCodeList(salesDataDto.getCustomerCodeList());
                totalDto.setSalesProductCodeList(salesDataDto.getIncludeProductCodeList());
                salesDataVos = this.tpmDailySalesDataService.statisticsListByCondition(totalDto);
                log.error("summarySalesData 033报表统计数据=SALE_QUANTITY={}", salesDataVos.stream().map(TpmDailySalesDataVo::getOrderNum).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add));

                //集团卡客户
                if (!CollectionUtils.isEmpty(jituanCardCustomerList)) {
                    List<String> soldPartyCodeList = salesDataDto.getCustomerCodeList().stream().map(e -> e.substring(0, Math.min(e.length(), 10))).collect(Collectors.toList());
                    totalDto.setCustomerCodeList(jituanCardCustomerList);
                    totalDto.setServiceCodeList(soldPartyCodeList);
                    List<TpmDailySalesDataVo> vos = this.tpmDailySalesDataService.statisticsListByCondition(totalDto);
                    log.error("summarySalesData 033报表统计数据=SALE_QUANTITY=集团卡客户={}", vos.stream().map(TpmDailySalesDataVo::getOrderNum).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add));
                    salesDataVos.addAll(vos);
                }

                for (TpmDailySalesDataVo salesDataVo : salesDataVos) {
                    BigDecimal amount = Optional.ofNullable(salesDataVo.getOrderNum()).orElse(BigDecimal.ZERO);
                    if (BigDecimal.ZERO.compareTo(amount) == 0) {
                        continue;
                    }
                    this.setReduce(amount, actualSales, productRatioVoMap, salesDataVo.getMaterialCode(), yearBudgetTotalPoint, pointOrIntensity);
                }

                for (Map.Entry<String, List<SalesDataVo>> entry : salesDataMap.entrySet()) {
                    SalesDataVo dataVo = entry.getValue().get(0);
                    BigDecimal amount = Optional.ofNullable(dataVo.getSaleQuantity()).orElse(BigDecimal.ZERO);
                    if (BigDecimal.ZERO.compareTo(amount) == 0) {
                        continue;
                    }
                    this.setReduce(amount, actualSales, productRatioVoMap, dataVo.getProductCode(), yearBudgetTotalPoint, pointOrIntensity);
                }
                break;
            case DELIVERY_MULTIPLY_PRICE:
                //出库件数*单品到岸价映射单价*订单数量 主体日销售033报表
                TpmDailySalesDataTotalDto totalDto2 = new TpmDailySalesDataTotalDto();
                //修改年月格式
                totalDto2.setYearMonthLy(yearMonthLy);
                totalDto2.setBusinessFormatCode(salesDataDto.getBusinessFormatCode());
                totalDto2.setBusinessUnitCode(salesDataDto.getBusinessUnitCode());
                totalDto2.setExcludeCustomerCodeList(salesDataDto.getExcludeCustomerCodeList());
                totalDto2.setExcludeTerminalCodeList(salesDataDto.getExcludeTerminalCodeList());
                totalDto2.setExcludeProductCodeList(salesDataDto.getExcludeProductCodeList());
                totalDto2.setCustomerCodeList(salesDataDto.getCustomerCodeList());
                totalDto2.setSalesProductCodeList(salesDataDto.getIncludeProductCodeList());

                salesDataVos = this.tpmDailySalesDataService.statisticsListByCondition(totalDto2);
                log.error("summarySalesData 033报表统计数据=DELIVERY_MULTIPLY_PRICE=集团卡客户={}", salesDataVos.stream().map(TpmDailySalesDataVo::getOrderAmount).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add));

                //集团卡客户
                if (!CollectionUtils.isEmpty(jituanCardCustomerList)) {
                    List<String> soldPartyCodeList = salesDataDto.getCustomerCodeList().stream().map(e -> e.substring(0, Math.min(e.length(), 10))).collect(Collectors.toList());
                    totalDto2.setCustomerCodeList(jituanCardCustomerList);
                    totalDto2.setServiceCodeList(soldPartyCodeList);
                    List<TpmDailySalesDataVo> vos = this.tpmDailySalesDataService.statisticsListByCondition(totalDto2);
                    log.error("summarySalesData 033报表统计数据=DELIVERY_MULTIPLY_PRICE=集团卡客户={}", vos.stream().map(TpmDailySalesDataVo::getOrderAmount).filter(Objects::nonNull).reduce(BigDecimal.ZERO, BigDecimal::add));
                    salesDataVos.addAll(vos);
                }

                for (TpmDailySalesDataVo salesDataVo : salesDataVos) {
                    BigDecimal amount = Optional.ofNullable(salesDataVo.getOrderAmount()).orElse(BigDecimal.ZERO);
                    if (BigDecimal.ZERO.compareTo(amount) == 0) {
                        continue;
                    }
                    this.setReduce(amount, actualSales, productRatioVoMap, salesDataVo.getMaterialCode(), yearBudgetTotalPoint, pointOrIntensity);
                }

                for (Map.Entry<String, List<SalesDataVo>> entry : salesDataMap.entrySet()) {
                    SalesDataVo dataVo = entry.getValue().get(0);
                    BigDecimal amount = Optional.ofNullable(dataVo.getDiscountFrontSaleAmount()).orElse(BigDecimal.ZERO);
                    if (BigDecimal.ZERO.compareTo(amount) == 0) {
                        continue;
                    }
                    this.setReduce(amount, actualSales, productRatioVoMap, dataVo.getProductCode(), yearBudgetTotalPoint, pointOrIntensity);
                }
                break;
            default:
                break;
        }

        actualSalesVo.setActualSales(actualSales.get());
        if (BudgetControlTypeEnum.INTENSITY.getCode().equals(controlTypeCode)) {
            log.error("summarySalesData 合计金额={}，点数={}", actualSales.get(), pointOrIntensity);
            actualSalesVo.setActualSalesAmount(actualSales.get().multiply(pointOrIntensity).setScale(6, BigDecimal.ROUND_HALF_UP));
            return;
        }

        BigDecimal budgetTotalPoint = monthBudgetVo.getBudgetTotalPoint();
        if (Objects.isNull(budgetTotalPoint)) {
            budgetTotalPoint = BigDecimal.ZERO;
        }

        BigDecimal actualSalesAmount = actualSales.get().multiply(budgetTotalPoint).divide(new BigDecimal(100), 6, BigDecimal.ROUND_HALF_UP);
        log.error("summarySalesData actualSalesAmount={}", actualSalesAmount);
        log.error("summarySalesData 合计金额={}，点数={}", actualSales.get(), budgetTotalPoint);

        actualSalesVo.setActualSalesAmount(actualSalesAmount);
    }

    /**
     * 计算结果
     *
     * @param amount               金额
     * @param actualSales          汇总金额
     * @param productRatioVoMap    产品计费比例
     * @param productCode          产品编码
     * @param yearBudgetTotalPoint 年度预算总点数
     */
    private void setReduce(BigDecimal amount, AtomicReference<BigDecimal> actualSales
            , Map<String, BudgetCalConfigProductRatioVo> productRatioVoMap, String productCode, BigDecimal yearBudgetTotalPoint,
                           BigDecimal pointOrIntensity) {
        if (productRatioVoMap.containsKey(productCode)) {
            BudgetCalConfigProductRatioVo ratioVo = productRatioVoMap.get(productCode);
            if (!Objects.isNull(ratioVo) && !Objects.isNull(ratioVo.getChargedRatio())) {
                if (StringUtils.isNotBlank(ratioVo.getReduceTag()) && BooleanEnum.TRUE.getCapital().equals(ratioVo.getReduceTag()) && !Objects.isNull(yearBudgetTotalPoint)) {
                    BigDecimal ratio = yearBudgetTotalPoint.subtract(ratioVo.getChargedRatio());
                    actualSales.set(actualSales.get().add(amount.multiply(ratio).divide(pointOrIntensity, 8, RoundingMode.HALF_UP)));
                } else {
                    actualSales.set(actualSales.get().add(amount.multiply(ratioVo.getChargedRatio()).divide(pointOrIntensity, 8, RoundingMode.HALF_UP)));
                }
            } else {
                actualSales.set(actualSales.get().add(amount));
            }
        } else {
            actualSales.set(actualSales.get().add(amount));
        }
    }

    /**
     * 计算数据
     *
     * @param actualSalesVo
     * @param currMonthBudget
     */
    @Override
    public void calculation(MonthBudgetActualSalesVo actualSalesVo, MonthBudgetVo currMonthBudget) {
        this.monthBudgetCalculateHelper.calculationActualReplyDiffHead(actualSalesVo, currMonthBudget);
    }
}