package com.biz.crm.tpm.business.month.budget.local.calculate.strategy;

import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.budget.cal.config.sdk.eunm.ActualSalesAmountTypeEnum;
import com.biz.crm.tpm.business.budget.cal.config.sdk.eunm.CalDataFromEnum;
import com.biz.crm.tpm.business.budget.cal.config.sdk.eunm.SalesPlanAmountTypeEnum;
import com.biz.crm.tpm.business.budget.cal.config.sdk.vo.BudgetCalConfigAreaVo;
import com.biz.crm.tpm.business.budget.cal.config.sdk.vo.BudgetCalConfigDataVo;
import com.biz.crm.tpm.business.budget.cal.config.sdk.vo.BudgetCalConfigProductRatioVo;
import com.biz.crm.tpm.business.budget.cal.config.sdk.vo.BudgetCalConfigVo;
import com.biz.crm.tpm.business.budget.item.sdk.enums.BudgetControlTypeEnum;
import com.biz.crm.tpm.business.main.oneday.sale.data.sdk.dto.MainOnedaySalesDataDto;
import com.biz.crm.tpm.business.main.oneday.sale.data.sdk.vo.MainOnedaySalesDataVo;
import com.biz.crm.tpm.business.month.budget.local.calculate.BudgetCalculateStrategy;
import com.biz.crm.tpm.business.month.budget.local.calculate.dto.SalesDataDto;
import com.biz.crm.tpm.business.month.budget.local.calculate.vo.SalesDataVo;
import com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetDetailEntity;
import com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity;
import com.biz.crm.tpm.business.month.budget.local.helper.MonthBudgetCalculateHelper;
import com.biz.crm.tpm.business.month.budget.local.repository.MonthBudgetDetailRepository;
import com.biz.crm.tpm.business.month.budget.local.repository.MonthBudgetRepository;
import com.biz.crm.tpm.business.month.budget.sdk.dto.MonthBudgetDetailDto;
import com.biz.crm.tpm.business.month.budget.sdk.eunm.BudgetOperationTypeEnum;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetActualSalesVo;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetDetailVo;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetVo;
import com.biz.crm.tpm.business.sales.goal.sdk.dto.SalesPerformanceDto;
import com.biz.crm.tpm.business.sales.goal.sdk.service.SalesPerformanceVoService;
import com.biz.crm.tpm.business.sales.goal.sdk.vo.SalesPerformanceSumVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.web.bind.annotation.RequestBody;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Description 计算主体
 * @Author YangWei
 * @Date 2023/2/18 上午12:23
 */
@Slf4j
@Component
public class BudgetCalculateVerticalStrategy implements BudgetCalculateStrategy {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private SalesPerformanceVoService salesPerformanceVoService;

    @Resource
    private MonthBudgetCalculateHelper monthBudgetCalculateHelper;


    /**
     * 获取类型
     * @param
     * @return {@link String}
     */
    @Override
    public String getBusinessUnit() {
        return BusinessUnitEnum.VERTICAL.getCode();
    }

    /**
     * 构建查询销售数据参数
     * @param monthBudgetVo
     * @param actualSalesVo
     * @return {@link SalesDataDto}
     */
    @Override
    public SalesDataDto buildDto(MonthBudgetVo monthBudgetVo, MonthBudgetActualSalesVo actualSalesVo) {
        log.info("实际销量计算对象,构建查询销售数据参数-垂直");
        BudgetCalConfigVo budgetCalConfigVo = actualSalesVo.getBudgetCalConfigVo();
        SalesDataDto dto = new SalesDataDto();
        dto.setSalesDate(monthBudgetVo.getYearMonthLy().split("-")[0]);
        dto.setSalesMonth(monthBudgetVo.getYearMonthLy().replace("-",""));
        dto.setYearMonthLy(monthBudgetVo.getYearMonthLy().replace("-",""));
        dto.setRegion(monthBudgetVo.getRegionName());
        dto.setSalesOrg(monthBudgetVo.getSalesOrgCode());
        dto.setRetailer(monthBudgetVo.getSystemName());
        dto.setShipToparty(monthBudgetVo.getTerminalCode());
        dto.setBrand(monthBudgetVo.getProductBrandCode());
        dto.setCategoryCode(monthBudgetVo.getProductItemCode());
        dto.setMaterialCode(monthBudgetVo.getProductCode());
        if(ObjectUtils.isEmpty(budgetCalConfigVo)){
            return null;
        }
        if (!CollectionUtils.isEmpty(budgetCalConfigVo.getTerminalList())) {
            List<String> terminalCodeList = budgetCalConfigVo.getTerminalList().stream().map(BudgetCalConfigAreaVo::getDataCode).collect(Collectors.toList());
            dto.setExcludeTerminalCodeList(terminalCodeList);
        }
        //当前年月
        String currYearMonth = DateUtil.dateToStr(new SimpleDateFormat("yyyy-MM"));
        if (StringUtils.isEmpty(dto.getMaterialCode())
            && !CollectionUtils.isEmpty(budgetCalConfigVo.getProductList())) {
            //非包含产品这里加一个年月过滤
            List<String> productCodeList = budgetCalConfigVo.getProductList().stream()
                .filter(o -> !org.springframework.util.StringUtils.hasText(o.getFilYearMonth())
                    || (org.springframework.util.StringUtils.hasText(o.getFilYearMonth()) && currYearMonth.equals(o.getFilYearMonth())))
                .map(BudgetCalConfigAreaVo::getDataCode).collect(Collectors.toList());
            dto.setExcludeProductCodeList(productCodeList);
        }
        //判断是否维护了产品计费比例
        List<BudgetCalConfigProductRatioVo> productRatios = budgetCalConfigVo.getProductRatios();
        if(!CollectionUtils.isEmpty(productRatios)){
            //包含产品
            dto.setIncludeProductCodeList(productRatios.stream()
                .map(BudgetCalConfigProductRatioVo::getProductCode).collect(Collectors.toList()));
        }
        return dto;
    }

    /**
     * 查询销售数据
     * @param dto
     * @return {@link Map}<{@link String}, {@link List}<{@link SalesDataVo}>>
     */
    @Override
    public Map<String, List<SalesDataVo>> findSalesData(SalesDataDto dto) {
        if(ObjectUtils.isEmpty(dto)){
            return Maps.newHashMap();
        }
        SalesPerformanceDto salesDataDto = this.nebulaToolkitService
            .copyObjectByWhiteList(dto, SalesPerformanceDto.class, HashSet.class, ArrayList.class);
        salesDataDto.setExcludeTerminalCodeList(dto.getExcludeTerminalCodeList());
        salesDataDto.setExcludeCustomerCodeList(dto.getExcludeCustomerCodeList());

        //查产品的逻辑，包含置空
        salesDataDto.setIncludeProductCodeList(null);
        //这里有两种情况，维护了产品比例的话，包含产品，    否则就是非包含
        if (!CollectionUtils.isEmpty(dto.getExcludeProductCodeList())){
            if (!CollectionUtils.isEmpty(dto.getIncludeProductCodeList())){
                //有非包含也有包含，去并集，从非包含里面吧包含的去掉
                List<String> newExcludeProductCodeList = Lists.newArrayList(dto.getExcludeProductCodeList());
                newExcludeProductCodeList.removeAll(dto.getIncludeProductCodeList());
                salesDataDto.setExcludeProductCodeList(newExcludeProductCodeList);
            }else{
                salesDataDto.setExcludeProductCodeList(dto.getExcludeProductCodeList());
            }
        }
        List<SalesPerformanceSumVo> salesPerformanceSumVos = this.salesPerformanceVoService.listYearByConditions(salesDataDto);
        if(CollectionUtils.isEmpty(salesPerformanceSumVos)){
            return Maps.newHashMap();
        }
        List<SalesDataVo> salesDataVos = (List<SalesDataVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
            salesPerformanceSumVos, SalesPerformanceSumVo.class, SalesDataVo.class, HashSet.class, ArrayList.class
        );
        salesDataVos.forEach(salesPerformanceSumVo -> {
            if (!StringUtils.isEmpty(salesPerformanceSumVo.getSalesMonth())) {
                Date yyyyMM = DateUtil.parse(salesPerformanceSumVo.getSalesMonth(), "yyyyMM");
                String format = DateUtil.format(yyyyMM, DateUtil.DEFAULT_YEAR_MONTH);
                salesPerformanceSumVo.setSalesMonth(format);
            }
        });
        return salesDataVos.stream().collect(Collectors.groupingBy(SalesDataVo::getSalesMonth));
    }

    /**
     * 汇总得到实际销量
     * @param actualSalesVo
     * @param salesDataMap
     */
    @Override
    public void summarySalesData(MonthBudgetActualSalesVo actualSalesVo, Map<String, List<SalesDataVo>> salesDataMap,SalesDataDto salesDataDto, MonthBudgetVo monthBudgetVo) {
        BudgetCalConfigVo budgetCalConfigVo = actualSalesVo.getBudgetCalConfigVo();
        if(ObjectUtils.isEmpty(budgetCalConfigVo)){
            return;
        }
        List<BudgetCalConfigDataVo> dataList = budgetCalConfigVo.getDataList();
        if(CollectionUtils.isEmpty(dataList)){
            return;
        }
        //数据来源
        BudgetCalConfigDataVo budgetCalConfigDataVo = dataList.stream()
            .filter(o -> CalDataFromEnum.ACTUAL_SALES_AMOUNT.getCode().equals(o.getCalDataFromCode())).findFirst().orElse(null);
        if(ObjectUtils.isEmpty(budgetCalConfigDataVo)){
            return;
        }
        //金额类型
        String amountTypeCode = budgetCalConfigDataVo.getAmountTypeCode();
        ActualSalesAmountTypeEnum amountTypeEnum = ActualSalesAmountTypeEnum.getByCode(amountTypeCode);
        if(ObjectUtils.isEmpty(amountTypeEnum)){
            return;
        }
        //实际销售数据
        List<SalesDataVo> salesDataVos = salesDataMap.get(actualSalesVo.getYearMonthLy());
        if(CollectionUtils.isEmpty(salesDataVos)){
            return;
        }
        //实际销售金额
        AtomicReference<BigDecimal> actualSalesAmount = new AtomicReference<>(BigDecimal.ZERO);
        //实际费用
        AtomicReference<BigDecimal> actualCostsAmount = new AtomicReference<>(BigDecimal.ZERO);
        //产品计费比例
        List<BudgetCalConfigProductRatioVo> budgetCalConfigProductRatioVos = budgetCalConfigVo.getProductRatios();
        Map<String, BudgetCalConfigProductRatioVo> productRatioVoMap = CollectionUtils.isEmpty(budgetCalConfigProductRatioVos)
            ? Maps.newHashMap() : budgetCalConfigProductRatioVos.stream()
            .collect(Collectors.toMap(BudgetCalConfigProductRatioVo::getProductCode, Function.identity()));
        //管控类型
        String controlTypeCode = actualSalesVo.getControlTypeCode();
        List<String> includeProductCodeList = salesDataDto.getIncludeProductCodeList();
        List<String> excludeProductCodeList = salesDataDto.getExcludeProductCodeList();
        salesDataVos.forEach(salesDataVo -> {
            //产品编码
            String productCode = salesDataVo.getMaterialCode();
            BudgetCalConfigProductRatioVo budgetCalConfigProductRatioVo = productRatioVoMap.get(productCode);
            BigDecimal productRatio = ObjectUtils.isEmpty(budgetCalConfigProductRatioVo) ? null
                : Optional.ofNullable(budgetCalConfigProductRatioVo.getChargedRatio()).orElse(BigDecimal.ONE);

            boolean addSales = false;
            if (!CollectionUtils.isEmpty(excludeProductCodeList)){
                //销售额要取非包含的产品
                if (!excludeProductCodeList.contains(productCode)){
                    //配置了非包含的话，算销售额指算非包含的
                    addSales = true;
                }
            }else{
                addSales = true;
            }

            //如果配置了产品比例，就只取包含的产品
            boolean addCost = false;
            if (!CollectionUtils.isEmpty(includeProductCodeList)){
                if (includeProductCodeList.contains(productCode)){
                    addCost = true;
                }
            }else{
                addCost = true;
            }

            switch (amountTypeEnum){
                case DISCOUNT_BEHIND_SALE_AMOUNT:
                    //折后销售额（不含奶卡）
                    break;
                case ORDER_RULE_AMT_IN_REBATE:
                    //折前销售额
                    BigDecimal orderRuleAmtInRebate = Optional.ofNullable(salesDataVo.getOrderRuleAmtInRebate()).orElse(BigDecimal.ZERO);
                    if (addSales){
                        actualSalesAmount.set(actualSalesAmount.get().add(orderRuleAmtInRebate));
                    }
                    if (addCost){
                        //按率
                        if(BudgetControlTypeEnum.RATIO.getCode().equals(controlTypeCode)){
                            //按率
                            if(!org.springframework.util.ObjectUtils.isEmpty(productRatio)){
                                actualCostsAmount.set(orderRuleAmtInRebate.multiply(productRatio)
                                        .setScale(6, BigDecimal.ROUND_HALF_UP)
                                        .add(actualCostsAmount.get()));
                            }else{
                                actualCostsAmount.set(orderRuleAmtInRebate.add(actualCostsAmount.get()));
                            }
                        }
                    }
                    break;
                case WAREHS_OUT_DISCOUNT_AMT:
                    //折后销售额
                    BigDecimal warehsOutDiscountAmt = Optional.ofNullable(salesDataVo.getWarehsOutDiscountAmt()).orElse(BigDecimal.ZERO);
                    if (addSales){
                        //判断按额 按率
                        actualSalesAmount.set(actualSalesAmount.get().add(warehsOutDiscountAmt));
                    }
                    if (addCost){
                        //按率
                        if(BudgetControlTypeEnum.RATIO.getCode().equals(controlTypeCode)){
                            //按率
                            if(!org.springframework.util.ObjectUtils.isEmpty(productRatio)){
                                actualCostsAmount.set(warehsOutDiscountAmt.multiply(productRatio)
                                        .setScale(6, BigDecimal.ROUND_HALF_UP)
                                        .add(actualCostsAmount.get()));
                            }else{
                                actualCostsAmount.set(warehsOutDiscountAmt.add(actualCostsAmount.get()));
                            }
                        }
                    }
                    break;
                case DISCOUNT_BEHIND_TAX_SALE_AMOUNT:
                    //折后销售额（含奶卡）

                    break;
                default:
                    break;
            }
        });
        //没有维护产品计算比例
        if(BudgetControlTypeEnum.RATIO.getCode().equals(controlTypeCode)){
            //按率
            if(CollectionUtils.isEmpty(budgetCalConfigProductRatioVos)){
                if(ObjectUtils.isEmpty(actualSalesVo.getBudgetTotalPoint())){
                    actualCostsAmount.set(null);
                }else{
                    //需要乘以预算点数
                    BigDecimal budgetTotalPoint = actualSalesVo.getBudgetTotalPoint().divide(new BigDecimal(100), 6, BigDecimal.ROUND_HALF_UP);
                    actualCostsAmount.set(actualCostsAmount.get().multiply(budgetTotalPoint).setScale(6, BigDecimal.ROUND_HALF_UP));
                }
            }
        }else if(BudgetControlTypeEnum.AMOUNT.getCode().equals(controlTypeCode)){
            //按额
            actualCostsAmount.set(actualSalesVo.getFirstReplyAmount());
        }
        actualSalesVo.setActualCostsAmount(actualCostsAmount.get());
        actualSalesVo.setActualSalesAmount(actualSalesAmount.get());
    }

    /**
     * 计算数据
     * @param actualSalesVo
     * @param currMonthBudget
     */
    @Override
    public void calculation(MonthBudgetActualSalesVo actualSalesVo, MonthBudgetVo currMonthBudget) {
        this.monthBudgetCalculateHelper.calculationActualReplyDiffVertical(actualSalesVo, currMonthBudget);
    }
}