package com.biz.crm.tpm.business.month.budget.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.month.budget.local.service.MonthBudgetActualReplayService;
import com.biz.crm.tpm.business.month.budget.local.service.MonthBudgetDetailService;
import com.biz.crm.tpm.business.month.budget.local.service.xxjob.VerticalActualSalesAutoSyncXxlJob;
import com.biz.crm.tpm.business.month.budget.local.service.xxjob.VerticalRollingAutoSyncXxlJob;
import com.biz.crm.tpm.business.month.budget.sdk.constant.BudgetLockConstant;
import com.biz.crm.tpm.business.month.budget.sdk.dto.*;
import com.biz.crm.tpm.business.month.budget.sdk.eunm.BudgetOperationTypeEnum;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetLockService;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetService;
import com.biz.crm.tpm.business.month.budget.sdk.vo.*;
import com.biz.crm.tpm.business.sales.plan.local.service.xxjob.AutoPullMonthPlanReplayXxJob;
import com.biz.crm.tpm.business.sales.plan.local.service.xxjob.AutoPullMonthPlanXxJob;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * @author huojia
 * @date 2022年10月27日 17:48
 */
@RestController
@RequestMapping("/v1/month/budget")
@Slf4j
@Api(tags = "月度预算")
public class MonthBudgetController {

    @Autowired(required = false)
    private MonthBudgetService monthBudgetService;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private MonthBudgetLockService monthBudgetLockService;

    @Autowired(required = false)
    private MonthBudgetDetailService monthBudgetDetailService;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private VerticalActualSalesAutoSyncXxlJob verticalActualSalesAutoSyncXxlJob;

    @Autowired(required = false)
    private AutoPullMonthPlanReplayXxJob autoPullMonthPlanReplayXxJob;

    @Autowired(required = false)
    private AutoPullMonthPlanXxJob autoPullMonthPlanXxJob;

    @Autowired(required = false)
    private VerticalRollingAutoSyncXxlJob verticalRollingAutoSyncXxlJob;

    @Autowired(required = false)
    private MonthBudgetActualReplayService monthBudgetActualReplayService;

    /**
     * 分页查询所有数据
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findByConditions")
    public Result<Page<MonthBudgetVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                        @ApiParam(name = "monthBudgetDto", value = "月度预算信息") MonthBudgetDto dto) {
        try {
            Page<MonthBudgetVo> page = this.monthBudgetService.findByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 分页查询所有数据
     *
     * @param codes
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @PostMapping("findByCodes")
    public Result<List<MonthBudgetVo>> findByCodes(@RequestBody List<String> codes) {
        try {
            List<MonthBudgetVo> page = this.monthBudgetService.findByCodes(codes, EnableStatusEnum.ENABLE.getCode());
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 分页查询所有数据（包含分子公司预算）
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findUnionByConditions")
    public Result<Page<MonthBudgetVo>> findUnionByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                             @ApiParam(name = "monthBudgetDto", value = "月度预算信息") MonthBudgetDto dto) {
        try {
            Page<MonthBudgetVo> page = this.monthBudgetService.findUnionByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 考核通报分页查询预算
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "考核通报分页查询预算")
    @GetMapping("findExamineCircularByConditions")
    public Result<Page<MonthBudgetExamineCircularQueryVo>> findExamineCircularByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                                           @ApiParam(name = "monthBudgetDto", value = "月度预算信息") MonthBudgetExamineCircularQueryDto dto) {
        try {
            Page<MonthBudgetExamineCircularQueryVo> page = this.monthBudgetService.findExamineCircularByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 批量查询所有数据
     *
     * @param codes 月度预算编码集合
     * @return 所有数据
     */
    @ApiOperation(value = "批量查询所有数据")
    @PostMapping("listByCodes")
    public Result<List<MonthBudgetVo>> listByCodes(@RequestBody List<String> codes) {
        try {
            List<MonthBudgetVo> result = this.monthBudgetService.listByCodes(codes);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 商务政策根据编码查询预算
     *
     * @param monthBudgetCodes 月度预算编码
     * @return 所有数据
     */
    @ApiOperation(value = "商务政策根据编码查询预算")
    @PostMapping("findBusinessPolicyByCodes")
    public Result<List<MonthBudgetBusinessPolicyQueryVo>> findBusinessPolicyByCodes(@RequestBody List<String> monthBudgetCodes) {
        try {
            List<MonthBudgetBusinessPolicyQueryVo> result = this.monthBudgetService.findBusinessPolicyByCodes(monthBudgetCodes);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 商务政策分页查询预算
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "商务政策分页查询预算")
    @GetMapping("findBusinessPolicyByConditions")
    public Result<Page<MonthBudgetBusinessPolicyQueryVo>> findBusinessPolicyByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                                                         @ApiParam(name = "MonthBudgetBusinessPolicyQueryDto", value = "月度预算信息") MonthBudgetBusinessPolicyQueryDto dto) {
        try {
            Page<MonthBudgetBusinessPolicyQueryVo> page = this.monthBudgetService.findBusinessPolicyByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 分页查询月度预算明细所有数据
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询月度预算明细所有数据")
    @GetMapping("/detail/findByConditions")
    public Result<Page<MonthBudgetDetailVo>> findByConditions(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                              @ApiParam(name = "monthBudgetDetailDto", value = "月度预算明细信息") MonthBudgetDetailDto dto) {
        try {
            Page<MonthBudgetDetailVo> page = this.monthBudgetDetailService.findByConditions(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过主键查询单条数据
     *
     * @param id 主键
     * @return 单条数据
     */
    @ApiOperation(value = "通过主键查询单条数据")
    @GetMapping("detail")
    public Result<MonthBudgetVo> detail(@RequestParam("id") @ApiParam(name = "id", value = "主键id") String id) {
        try {
            MonthBudgetVo monthBudgetVo = this.monthBudgetService.findById(id);
            return Result.ok(monthBudgetVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过主键查询多条数据
     *
     * @param ids 主键
     * @return 多条条数据
     */
    @ApiOperation(value = "通过主键查询多条数据")
    @GetMapping("details")
    public Result<List<MonthBudgetVo>> details(@RequestParam("ids") @ApiParam(name = "ids", value = "主键id") List<String> ids) {
        try {
            List<MonthBudgetVo> monthBudgetVoList = this.monthBudgetService.findByIds(ids);
            return Result.ok(monthBudgetVoList);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 通过预算项目编码查询数据
     *
     * @param budgetItemCodeList 预算项目编码
     * @return 单条数据
     */
    @ApiOperation(value = "通过预算项目编码查询数据")
    @PostMapping("listByBudgetItemCodeList")
    public Result<List<MonthBudgetVo>> listByBudgetItemCodeList(@RequestBody List<String> budgetItemCodeList) {
        try {
            List<MonthBudgetVo> result = this.monthBudgetService.listByBudgetItemCodeList(budgetItemCodeList);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 根据年度预算编码批量查询
     *
     * @param yearBudgetCode 年度预算编码
     * @return 单条数据
     */
    @ApiOperation(value = "根据年度预算编码批量查询")
    @GetMapping("listByYearBudgetCode")
    public Result<List<MonthBudgetVo>> listByYearBudgetCode(@RequestParam("yearBudgetCode") String yearBudgetCode) {
        try {
            List<MonthBudgetVo> result = this.monthBudgetService.listByYearBudgetCode(yearBudgetCode);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 操作月度预算
     *
     * @param monthBudgetCode
     * @param operationAmount
     * @param budgetType
     * @param operationType
     * @param businessCode
     * @return com.biz.crm.business.common.sdk.model.Result<java.util.List < com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetVo>>
     * @author huojia
     * @date 2022/12/19 19:58
     **/
    @ApiOperation(value = "操作月度预算")
    @GetMapping("operateBudget")
    public Result<?> operateBudget(String monthBudgetCode,
                                   BigDecimal operationAmount,
                                   String budgetType,
                                   String operationType,
                                   String businessCode) {
        try {
            this.monthBudgetService.operateBudget(monthBudgetCode, operationAmount, budgetType, operationType, businessCode);
            return Result.ok("操作成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 预算批量加锁
     *
     * @param monthBudgetCodeList
     * @param timeUnit
     * @param time
     * @return com.biz.crm.business.common.sdk.model.Result<boolean>
     * @author huojia
     * @date 2022/12/19 20:17
     **/
    @ApiOperation(value = "预算批量加锁")
    @GetMapping("lockBatch")
    public Result<Boolean> lockBatch(List<String> monthBudgetCodeList, TimeUnit timeUnit, int time) {
        try {
            boolean result = this.monthBudgetLockService.lock(monthBudgetCodeList, timeUnit, time);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 预算批量加锁
     *
     * @param dto
     * @return com.biz.crm.business.common.sdk.model.Result<boolean>
     * @author huojia
     * @date 2022/12/19 20:17
     **/
    @ApiOperation(value = "预算批量加锁")
    @PostMapping("lockBatchPost")
    public Result<Boolean> lockBatchPost(@RequestBody MonthBudgetBatchLockDto dto) {
        try {
            boolean result = this.monthBudgetLockService.lock(dto.getMonthBudgetCodeList(), dto.getTimeUnit(), dto.getLockTime(), dto.getWaiteTime());
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 批量解锁
     *
     * @param monthBudgetCodeList
     * @return com.biz.crm.business.common.sdk.model.Result<?>
     * @author huojia
     * @date 2022/12/19 20:22
     **/
    @ApiOperation(value = "预算批量解锁")
    @PostMapping("unLockBatch")
    public Result unLockBatch(@RequestBody List<String> monthBudgetCodeList) {
        try {
            this.monthBudgetLockService.unLock(monthBudgetCodeList);
            return Result.ok("预算批量解锁成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 预算加锁
     *
     * @param monthBudgetCode
     * @param timeUnit
     * @param time
     * @return com.biz.crm.business.common.sdk.model.Result<boolean>
     * @author huojia
     * @date 2022/12/19 20:17
     **/
    @ApiOperation(value = "预算加锁")
    @GetMapping("lock")
    public Result<Boolean> lock(String monthBudgetCode, TimeUnit timeUnit, int time) {
        try {
            boolean result = this.monthBudgetLockService.lock(monthBudgetCode, timeUnit, time);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 预算加锁
     *
     * @param dto
     * @return com.biz.crm.business.common.sdk.model.Result<boolean>
     * @author huojia
     * @date 2022/12/19 20:17
     **/
    @ApiOperation(value = "预算加锁")
    @PostMapping("lock")
    public Result<Boolean> lockPost(@RequestBody MonthBudgetLockDto dto) {
        try {
            boolean result = this.monthBudgetLockService.lock(dto.getMonthBudgetCode(), dto.getTimeUnit(), dto.getTime());
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 预算解锁
     *
     * @param monthBudgetCode
     * @return com.biz.crm.business.common.sdk.model.Result<?>
     * @author huojia
     * @date 2022/12/19 20:21
     **/
    @ApiOperation(value = "预算解锁")
    @GetMapping("unLock")
    public Result unLock(@RequestParam("monthBudgetCode") String monthBudgetCode) {
        try {
            this.monthBudgetLockService.unLock(monthBudgetCode);
            return Result.ok("解锁成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 修改数据
     *
     * @param dto 实体对象
     * @return 修改结果
     */
/*    @ApiOperation(value = "修改数据")
    @PatchMapping
    public Result<MonthBudgetVo> update(@ApiParam(name = "monthBudgetDto", value = "TPM-月度预算") @RequestBody MonthBudgetDto dto) {
        try {
            MonthBudgetVo result = this.monthBudgetService.update(dto);
            return Result.ok(result);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }*/

    /**
     * 调整
     *
     * @param dto 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "调整")
    @PostMapping("adjust")
    public Result adjust(@ApiParam(name = "monthBudgetAdjustDto", value = "TPM-月度预算") @RequestBody MonthBudgetAdjustDto dto) {
        try {
            this.monthBudgetService.adjust(dto);
            return Result.ok("提交成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 月度预算多选调整调整
     *
     * @param dto 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "月度预算多选调整调整")
    @PostMapping("monthBudgetAdjust")
    public Result monthBudgetAdjust(@ApiParam(name = "monthBudgetAdjustDto", value = "TPM-月度预算") @RequestBody MonthBudgetAdjustDto dto) {
        try {
            this.monthBudgetService.monthBudgetAdjust(dto);
            return Result.ok("提交成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 审批中查看 调整明细
     *
     * @param processNo 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "审批中查看调整明细（bpm使用）")
    @GetMapping("adjustQuery")
    public Result<MonthBudgetMainAdjustVo> adjustQuery(@RequestParam("processNo") @ApiParam(name = "processNo", value = "流程编码") String processNo) {
        try {
            MonthBudgetMainAdjustVo monthBudgetMainAdjustVo = this.monthBudgetService.adjustQuery(processNo);
            return Result.ok(monthBudgetMainAdjustVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 审批中查看 调整明细
     *
     * @param processNo 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "审批中查看调整明细-批量查询（bpm使用）")
    @GetMapping("adjustSumQuery")
    public Result<MonthBudgetMainAdjustVo> adjustSumQuery(@RequestParam("processNo") @ApiParam(name = "processNo", value = "流程编码") String processNo) {
        try {
            MonthBudgetMainAdjustVo monthBudgetMainAdjustVo = this.monthBudgetService.adjustSumQuery(processNo);
            return Result.ok(monthBudgetMainAdjustVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 变更
     *
     * @param dto 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "变更")
    @PostMapping("change")
    public Result change(@ApiParam(name = "monthBudgetOperateDto", value = "TPM-月度预算") @RequestBody MonthBudgetOperateDto dto) {
        try {
            this.monthBudgetService.change(dto);
            return Result.ok("提交成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 审批中查看 变更明细
     *
     * @param processNo 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "审批中查看变更明细（bpm使用）")
    @GetMapping("changeQuery")
    public Result<MonthBudgetChangeVo> changeQuery(@RequestParam("processNo") @ApiParam(name = "processNo", value = "流程编码") String processNo) {
        try {
            MonthBudgetChangeVo monthBudgetChangeVo = this.monthBudgetService.changeQuery(processNo);
            return Result.ok(monthBudgetChangeVo);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 冻结
     *
     * @param dto 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "冻结")
    @PostMapping("freeze")
    public Result freeze(@ApiParam(name = "monthBudgetOperateDto", value = "TPM-月度预算") @RequestBody MonthBudgetOperateDto dto) {
        try {
            this.monthBudgetService.unOrFreeze(dto, BudgetOperationTypeEnum.FREEZE.getCode());
            return Result.ok("冻结成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 解冻
     *
     * @param dto 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "解冻")
    @PostMapping("unfreeze")
    public Result unfreeze(@ApiParam(name = "monthBudgetOperateDto", value = "TPM-月度预算") @RequestBody MonthBudgetOperateDto dto) {
        try {
            this.monthBudgetService.unOrFreeze(dto, BudgetOperationTypeEnum.UNFREEZE.getCode());
            return Result.ok("解冻成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 划拨
     *
     * @param dto 实体对象
     * @return 修改结果
     */
    @ApiOperation(value = "划拨")
    @PostMapping("transfer")
    public Result transfer(@ApiParam(name = "monthBudgetOperateDto", value = "TPM-月度预算") @RequestBody MonthBudgetTransferDto dto) {
        try {
            this.monthBudgetService.transfer(dto);
            return Result.ok("划拨成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 手动执行滚动
     *
     * @param ids
     * @return 修改结果
     */
    @ApiOperation(value = "手动执行滚动")
    @PostMapping("manualRolling")
    public Result manualRolling(@ApiParam(name = "ids", value = "TPM-月度预算ids") @RequestBody List<String> ids) {
        try {
            this.monthBudgetService.manualRolling(ids);
            return Result.ok("手动执行滚动成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 手动执行计算回复量（计划量）
     *
     * @param ids
     * @return 修改结果
     */
    @ApiOperation(value = "手动执行计算回复量（计划量）")
    @PostMapping("manualReplay")
    public Result manualReplay(@ApiParam(name = "ids", value = "TPM-月度预算ids") @RequestBody List<String> ids) {
        try {
            this.monthBudgetService.manualReplay(ids, BooleanEnum.TRUE.getCapital());
            return Result.ok("手动执行计算回复量成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 按年月手动执行计算回复量（计划量）
     *
     * @param yearMonth
     * @return 修改结果
     */
    @ApiOperation(value = "按年月手动执行计算回复量（计划量）")
    @GetMapping("manualReplayYearMonth")
    public Result manualReplayYearMonth(@ApiParam(name = "yearMonth", value = "年月") @RequestParam("yearMonth") String yearMonth) {
        try {
            AbstractCrmUserIdentity crmUserIdentity = loginUserService.getAbstractLoginUser();
            if (redisLockService.isLock(BudgetLockConstant.MONTH_BUDGET_LOCK + yearMonth)) {
                throw new RuntimeException("计划量计算正在执行中，请稍后");
            } else {
                this.monthBudgetService.manualReplayYearMonth(yearMonth, BooleanEnum.TRUE.getCapital(), crmUserIdentity);
            }
            return Result.ok("开始手动计算计划量，请稍后查询结果");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 手动执行计算回复量（回复量）
     *
     * @param ids
     * @return 修改结果
     */
    @ApiOperation(value = "手动执行计算回复量（回复量）")
    @PostMapping("manualRestoreReplay")
    public Result manualRestoreReplay(@ApiParam(name = "ids", value = "TPM-月度预算ids") @RequestBody List<String> ids) {
        try {
            this.monthBudgetService.manualReplay(ids, BooleanEnum.FALSE.getCapital());
            return Result.ok("手动执行计算回复量成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 销管资金池预算项目计算回复量（回复量）
     *
     * @param ids
     * @return 修改结果
     */
    @ApiOperation(value = "销管资金池预算项目计算回复量（回复量）")
    @PostMapping("manualRestoreReplayReimburse")
    public Result manualRestoreReplayReimburse(@ApiParam(name = "ids", value = "TPM-月度预算ids") @RequestBody List<String> ids) {
        try {
            this.monthBudgetService.manualRestoreReplayReimburse(ids);
            return Result.ok("手动执行计算回复量成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 销管资金池预算项目计算回复差
     *
     * @param ids
     * @return 修改结果
     */
    @ApiOperation(value = "销管资金池预算项目计算回复差")
    @PostMapping("manualActualReplayDiffReimburse")
    public Result manualActualReplayDiffReimburse(@ApiParam(name = "ids", value = "TPM-月度预算ids") @RequestBody List<String> ids) {
        try {
            this.monthBudgetService.manualActualReplayDiffReimburse(ids);
            return Result.ok("手动执行计算回复量成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 其他预留项目预算项目计算计划量
     *
     * @param ids
     * @return
     */
    @ApiOperation(value = "其他预留项目预算项目计算计划量")
    @PostMapping("manualRestoreReplayPlanOthers")
    public Result manualRestoreReplayPlanOthers(@ApiParam(name = "ids", value = "TPM-月度预算ids") @RequestBody List<String> ids) {
        try {
            this.monthBudgetService.manualRestoreReplayOthers(ids, BooleanEnum.TRUE.getCapital());
            return Result.ok("手动执行其他预留项目预算项目计算计划量成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 其他预留项目预算项目计算回复量
     *
     * @param ids
     * @return
     */
    @ApiOperation(value = "其他预留项目预算项目计算回复量")
    @PostMapping("manualRestoreReplayRestoreOthers")
    public Result manualRestoreReplayRestoreOthers(@ApiParam(name = "ids", value = "TPM-月度预算ids") @RequestBody List<String> ids) {
        try {
            this.monthBudgetService.manualRestoreReplayOthers(ids, BooleanEnum.FALSE.getCapital());
            return Result.ok("手动执行其他预留项目预算项目计算回复量成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 其他预留项目预算项目计算回复差
     *
     * @param ids
     * @return 修改结果
     */
    @ApiOperation(value = "其他预留项目预算项目计算回复差")
    @PostMapping("manualActualReplayDiffOthers")
    public Result manualActualReplayDiffOthers(@ApiParam(name = "ids", value = "TPM-月度预算ids") @RequestBody List<String> ids) {
        try {
            this.monthBudgetService.manualActualReplayDiffOthers(ids);
            return Result.ok("手动执行计算回复量成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 特仑苏统筹和授权预算项目计算回复量
     *
     * @param ids
     * @return
     */
    @ApiOperation(value = "特仑苏统筹和授权预算项目计算回复量")
    @PostMapping("manualRestoreReplayRestoreDeluxu")
    public Result manualRestoreReplayRestoreDeluxu(@ApiParam(name = "ids", value = "TPM-月度预算ids") @RequestBody List<String> ids) {
        try {
            this.monthBudgetService.manualRestoreReplayRestoreDeluxu(ids);
            return Result.ok("手动执行特仑苏统筹和授权预算项目计算回复量成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 特仑苏统筹和授权预算项目计算月度实销回复差
     *
     * @param ids
     * @return 修改结果
     */
    @ApiOperation(value = "特仑苏统筹和授权预算项目计算月度实销回复差")
    @PostMapping("manualActualReplayDiffDeluxuMonth")
    public Result manualActualReplayDiffDeluxuMonth(@ApiParam(name = "ids", value = "TPM-月度预算ids") @RequestBody List<String> ids) {
        try {
            this.monthBudgetService.manualActualReplayDiffDeluxuMonth(ids);
            return Result.ok("手动执行计算回复差成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 特仑苏统筹和授权预算项目计算季度实销回复差
     *
     * @param ids
     * @return 修改结果
     */
    @ApiOperation(value = "特仑苏统筹和授权预算项目计算季度实销回复差")
    @PostMapping("manualActualReplayDiffDeluxuQuarter")
    public Result manualActualReplayDiffDeluxuQuarter(@ApiParam(name = "ids", value = "TPM-月度预算ids") @RequestBody List<String> ids) {
        try {
            this.monthBudgetService.manualActualReplayDiffDeluxuQuarter(ids);
            return Result.ok("手动执行计算回复差成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 手动执行实销量与回复量差额
     *
     * @param ids
     * @return 修改结果
     */
    @ApiOperation(value = "手动执行实销量与回复量差额")
    @PostMapping("manualActualReplayDiff")
    public Result manualActualReplayDiff(@ApiParam(name = "ids", value = "TPM-月度预算ids") @RequestBody List<String> ids) {
        try {
            this.monthBudgetActualReplayService.manualActualReplayDiffMq(ids);
            return Result.ok("异步执行实销量与回复量差额中,请稍后查询执行结果!");
        } catch (Exception e) {
            log.info("手动执行实销量与回复量差额失败");
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 查询管控金额
     *
     * @param monthBudgetCodeList 月度预算编码
     * @return 修改结果
     */
    @ApiOperation(value = "查询管控金额")
    @PostMapping("mapControlAmount")
    public Result<Map<String, MonthBudgetControlVo>> mapControlAmount(@RequestBody List<String> monthBudgetCodeList) {
        try {
            Map<String, MonthBudgetControlVo> map = this.monthBudgetService.mapControlAmount(monthBudgetCodeList);
            return Result.ok(map);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation("查询部门预算关联的客户预算")
    @PostMapping("findCustomerGroupMonthBudget")
    public Result<List<MonthBudgetVo>> findCustomerGroupMonthBudget(@RequestBody List<MonthBudgetDto> dtos) {
        try {
            return Result.ok(this.monthBudgetService.findCustomerGroupMonthBudget(dtos));
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "按年月手动执行计算计划量")
    @GetMapping("planAutoSyncXxlJob")
    public Result planAutoSyncXxlJob(@ApiParam(name = "yearMonth", value = "年月")
                                     @RequestParam(value = "yearMonth", required = false) String yearMonth) {
        try {
            log.info("=====>    垂直计算计划量[{}] start    <=====", DateUtil.dateStrNowAll());
            Date date = null;
            if (StringUtils.isEmpty(yearMonth)) {
                Calendar calendar = DateUtil.getCalendar();
                calendar.setTime(new Date());
                calendar.add(Calendar.MONTH, 1);
                date = calendar.getTime();
            } else {
                date = DateUtil.parseDate(yearMonth, DateUtil.DEFAULT_YEAR_MONTH);
            }
            Assert.notNull(date, "年月[" + yearMonth + "]不合法,格式必须为[" + DateUtil.DEFAULT_YEAR_MONTH + "]!");
            yearMonth = DateUtil.format(date, DateUtil.DEFAULT_YEAR_MONTH);
            List<String> businessUnitCodeList = new ArrayList<>();
            businessUnitCodeList.add(BusinessUnitEnum.VERTICAL.getCode());
            monthBudgetService.autoPlan(yearMonth, businessUnitCodeList);
            log.info("=====>    垂直计算计划量[{}] end    <=====", DateUtil.dateStrNowAll());
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @ApiOperation(value = "删除")
    @DeleteMapping("/deleteBatch")
    public Result<?> deleteBatch(@RequestParam("ids") List<String> ids) {
        try {
            this.monthBudgetService.deleteBatch(ids);
            return Result.ok("删除成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 手动执行主体计算年度累计可用金额
     *
     * @param yearMonth 年月(yyyy-MM)
     * @return 结果
     */
    @ApiOperation(value = "手动执行主体计算年度累计可用金额")
    @GetMapping("/manualcalHeadYearTotalAvailableBalance")
    public Result<?> manualcalHeadYearTotalAvailableBalance(@ApiParam(name = "yearMonth", value = "年月(yyyy-MM)")
                                                            @RequestParam(value = "yearMonth", required = true) String yearMonth) {
        try {
            this.monthBudgetService.calHeadYearTotalAvailableBalanceSyncXxlJob(yearMonth);
            return Result.ok("手动执行主体计算年度累计可用金额成功");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    @ApiOperation(value = "垂直计算实际销量定时任务手动执行")
    @PostMapping("manualVerticalActualSalesAutoSyncXxlJob")
    public Result manualVerticalActualSalesAutoSyncXxlJob(@RequestBody List<String> yearMonthList) {
        Validate.notEmpty(yearMonthList, "年月不能为空！");
        log.info("=====>    垂直计算实际销量定时任务手动执行[{}] start    <=====", DateUtil.dateStrNowAll());
        boolean lock = true;
        String lockKey = verticalActualSalesAutoSyncXxlJob.getLockKey() + DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        try {
            lock = this.redisLockService.tryLock(lockKey, TimeUnit.HOURS, 12);
            if (!lock) {
                return Result.error("执行中...");
            }
            for (String yearMonth : yearMonthList) {
                verticalActualSalesAutoSyncXxlJob.handle(yearMonth);
            }
        } catch (Exception e) {
            return Result.error("执行失败：" + e.getMessage());
        } finally {
            log.info("=====>    垂直计算实际销量定时任务手动执行[{}] end    <=====", DateUtil.dateStrNowAll());
            if (lock) {
                this.redisLockService.unlock(lockKey);
            }
        }
        return Result.ok("执行成功！");
    }

    @ApiOperation(value = "垂直计算回复量定时任务手动执行")
    @PostMapping("manualVerticalPlanReplayAutoSyncXxlJob")
    public Result manualVerticalPlanReplayAutoSyncXxlJob(@RequestBody List<String> yearMonthList) {
        Validate.notEmpty(yearMonthList, "年月不能为空！");
        log.info("=====>    垂直计算回复量定时任务手动执行[{}] start    <=====", DateUtil.dateStrNowAll());
        boolean lock = true;
        String lockKey = "manualVerticalPlanReplayAutoSyncXxlJob" + DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        try {
            lock = this.redisLockService.tryLock(lockKey, TimeUnit.HOURS, 12);
            if (!lock) {
                return Result.error("执行中...");
            }
            for (String yearMonth : yearMonthList) {
                autoPullMonthPlanReplayXxJob.handle(yearMonth);
            }
        } catch (Exception e) {
            return Result.error("执行失败：" + e.getMessage());
        } finally {
            log.info("=====>    垂直计算回复量定时任务手动执行[{}] end    <=====", DateUtil.dateStrNowAll());
            if (lock) {
                this.redisLockService.unlock(lockKey);
            }
        }
        return Result.ok("执行成功！");
    }

    @ApiOperation(value = "垂直计算销售计划定时任务手动执行")
    @PostMapping("manualVerticalPlanAutoSyncXxlJob")
    public Result manualVerticalPlanAutoSyncXxlJob(@RequestBody List<String> yearMonthList) {
        Validate.notEmpty(yearMonthList, "年月不能为空！");
        log.info("=====>    垂直计算销售计划定时任务手动执行[{}] start    <=====", DateUtil.dateStrNowAll());
        boolean lock = true;
        String lockKey = "manualVerticalPlanAutoSyncXxlJob" + DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        try {
            lock = this.redisLockService.tryLock(lockKey, TimeUnit.HOURS, 12);
            if (!lock) {
                return Result.error("执行中...");
            }
            for (String yearMonth : yearMonthList) {
                autoPullMonthPlanXxJob.handle(yearMonth);
            }
        } catch (Exception e) {
            return Result.error("执行失败：" + e.getMessage());
        } finally {
            log.info("=====>    垂直计算销售计划定时任务手动执行[{}] end    <=====", DateUtil.dateStrNowAll());
            if (lock) {
                this.redisLockService.unlock(lockKey);
            }
        }
        return Result.ok("执行成功！");
    }

    @ApiOperation(value = "垂直计算实销回复差")
    @PostMapping("manualVerticalActualDiffAutoSyncXxlJob")
    public Result manualVerticalActualDiffAutoSyncXxlJob(@RequestBody List<String> yearMonthList) {
        Validate.notEmpty(yearMonthList, "年月不能为空！");
        log.info("=====>    垂直计算实销回复差执行[{}] start    <=====", DateUtil.dateStrNowAll());
        boolean lock = true;
        String lockKey = "manualVerticalActualDiffAutoSyncXxlJob" + DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        try {
            lock = this.redisLockService.tryLock(lockKey, TimeUnit.HOURS, 12);
            if (!lock) {
                return Result.error("执行中...");
            }
            for (String yearMonth : yearMonthList) {
                log.info("manualVerticalActualDiffAutoSyncXxlJob，年月[{}}，计划量：{}", yearMonth, DateUtil.dateStrNowAll());
                autoPullMonthPlanXxJob.publish(yearMonth);
                log.info("manualVerticalActualDiffAutoSyncXxlJob，年月[{}}，回复量：{}", yearMonth, DateUtil.dateStrNowAll());
                autoPullMonthPlanReplayXxJob.publish(yearMonth);
                log.info("manualVerticalActualDiffAutoSyncXxlJob，年月[{}}，实销回复差：{}", yearMonth, DateUtil.dateStrNowAll());
                verticalActualSalesAutoSyncXxlJob.handle(yearMonth);
                log.info("manualVerticalActualDiffAutoSyncXxlJob，年月[{}}，结束：{}", yearMonth, DateUtil.dateStrNowAll());
            }
        } catch (Exception e) {
            return Result.error("执行失败：" + e.getMessage());
        } finally {
            log.info("=====>    垂直计算实销回复差执行[{}] end    <=====", DateUtil.dateStrNowAll());
            if (lock) {
                this.redisLockService.unlock(lockKey);
            }
        }
        return Result.ok("执行成功！");
    }

    @ApiOperation(value = "垂直预算滚动定时任务手动执行")
    @PostMapping("manualVerticalRollingAutoSyncXxlJob")
    public Result manualVerticalRollingAutoSyncXxlJob(@RequestBody List<String> yearMonthList) {
        Validate.notEmpty(yearMonthList, "年月不能为空！");
        log.info("=====>    垂直预算滚动定时任务手动执行[{}] start    <=====", DateUtil.dateStrNowAll());
        boolean lock = true;
        String lockKey = "manualVerticalRollingAutoSyncXxlJob" + DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        try {
            lock = this.redisLockService.tryLock(lockKey, TimeUnit.HOURS, 12);
            if (!lock) {
                return Result.error("执行中...");
            }
            for (String yearMonth : yearMonthList) {
                log.info("manualVerticalRollingAutoSyncXxlJob，年月[{}}，start ：{}", yearMonth, DateUtil.dateStrNowAll());
                verticalRollingAutoSyncXxlJob.handle(yearMonth);
                log.info("manualVerticalRollingAutoSyncXxlJob，年月[{}}，end ：{}", yearMonth, DateUtil.dateStrNowAll());
            }
        } catch (Exception e) {
            return Result.error("执行失败：" + e.getMessage());
        } finally {
            log.info("=====>    垂直预算滚动定时任务手动执行[{}] end    <=====", DateUtil.dateStrNowAll());
            if (lock) {
                this.redisLockService.unlock(lockKey);
            }
        }
        return Result.ok("执行成功！");
    }

    @ApiOperation("结案核销调整查询预算")
    @GetMapping("auditAdjustFindBudget")
    public Result<Page<MonthBudgetVo>> auditAdjustFindBudget(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable, String customerCode) {
        try {
            Page<MonthBudgetVo> page = monthBudgetService.auditAdjustFindBudget(pageable, customerCode);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 能力中心分页查询月度预算数据
     *
     * @param pageable 分页对象
     * @param dto      查询实体
     * @return 所有数据
     */
    @ApiOperation(value = "分页查询所有数据")
    @GetMapping("findPageForOut")
    public Result<Page<MonthBudgetVo>> findPageForOut(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                      MonthBudgetDto dto) {
        try {
            Page<MonthBudgetVo> page = this.monthBudgetService.findPageForOut(pageable, dto);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }
}
