package com.biz.crm.tpm.business.month.budget.local.helper;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.budget.controls.config.sdk.enums.RollingTypeEnum;
import com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetDetailEntity;
import com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity;
import com.biz.crm.tpm.business.month.budget.local.repository.MonthBudgetDetailRepository;
import com.biz.crm.tpm.business.month.budget.local.repository.MonthBudgetRepository;
import com.biz.crm.tpm.business.month.budget.sdk.dto.MonthBudgetDetailDto;
import com.biz.crm.tpm.business.month.budget.sdk.eunm.BudgetOperationTypeEnum;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetActualSalesVo;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetDetailVo;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.StopWatch;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.*;

/**
 * @Description 月度预算计算助手
 * @Author YangWei
 * @Date 2023/2/9 上午10:00
 */
@Slf4j
@Component
public class MonthBudgetCalculateHelper {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private MonthBudgetRepository monthBudgetRepository;

    @Autowired(required = false)
    private MonthBudgetDetailRepository monthBudgetDetailRepository;

    /**
     * 冻结可用 (垂直)
     * 冻结后可用金额=年初分解金额+月度分解金额与年初分解金额的差异+实销回复差+上月滚动金额+调整金额-冻结金额-本月滚动金额
     * @param entity
     * @return {@link BigDecimal}
     */
    public BigDecimal buildAfterFreezeAmount(MonthBudgetEntity entity, BusinessUnitEnum unitEnum) {
        return Optional.ofNullable(entity.getInitResolveAmount())
            .orElse(BigDecimal.ZERO)
            .add(Optional.ofNullable(entity.getFirstReplyResolveDiffAmount())
                .orElse(BigDecimal.ZERO))
            .add(Optional.ofNullable(entity.getActualReplyDiff())
                .orElse(BigDecimal.ZERO))
            .add(Optional.ofNullable(entity.getLastMonthRollingAmount())
                .orElse(BigDecimal.ZERO))
            .add(Optional.ofNullable(entity.getAdjustAmount())
                .orElse(BigDecimal.ZERO))
            .subtract(Optional.ofNullable(entity.getFreezeAmount())
                .orElse(BigDecimal.ZERO))
            .subtract(Optional.ofNullable(entity.getMonthRollingAmount())
                .orElse(BigDecimal.ZERO));
    }

    /**
     * 累计可用 (垂直)
     * 累计可用余额=年初分解金额+月度分解金额与年初分解金额的差异+实销回复差+上月滚动金额+调整金额-冻结金额-批复金额-预估超额+批复结案差-本月滚动金额
     * @param entity
     * @return {@link BigDecimal}
     */
    public BigDecimal buildAccumulatedAvailableBalance(MonthBudgetEntity entity, BusinessUnitEnum unitEnum) {
        return Optional.ofNullable(entity.getInitResolveAmount()).orElse(BigDecimal.ZERO)
            .add(Optional.ofNullable(entity.getFirstReplyResolveDiffAmount())
                .orElse(BigDecimal.ZERO))
            .add(Optional.ofNullable(entity.getActualReplyDiff())
                .orElse(BigDecimal.ZERO))
            .add(Optional.ofNullable(entity.getLastMonthRollingAmount())
                .orElse(BigDecimal.ZERO))
            .add(Optional.ofNullable(entity.getAdjustAmount())
                .orElse(BigDecimal.ZERO))
            .subtract(Optional.ofNullable(entity.getFreezeAmount())
                .orElse(BigDecimal.ZERO))
            .subtract(Optional.ofNullable(entity.getApprovedAmount())
                .orElse(BigDecimal.ZERO))
            .subtract(Optional.ofNullable(entity.getEstimatedExcessAmount())
                .orElse(BigDecimal.ZERO))
            .add(Optional.ofNullable(entity.getApprovedAuditDiff())
                .orElse(BigDecimal.ZERO))
            .subtract(Optional.ofNullable(entity.getMonthRollingAmount())
                .orElse(BigDecimal.ZERO));
    }

    /**
     * 累计可用 (垂直) 滚动前
     * 累计可用余额=年初分解金额+月度分解金额与年初分解金额的差异+实销回复差+上月滚动金额+调整金额-冻结金额-批复金额-预估超额+批复结案差
     * @param entity
     * @return {@link BigDecimal}
     */
    public BigDecimal buildAccumulatedAvailableBalanceBefore(MonthBudgetEntity entity, BusinessUnitEnum unitEnum) {
        return Optional.ofNullable(entity.getInitResolveAmount()).orElse(BigDecimal.ZERO)
                .add(Optional.ofNullable(entity.getFirstReplyResolveDiffAmount())
                        .orElse(BigDecimal.ZERO))
                .add(Optional.ofNullable(entity.getActualReplyDiff())
                        .orElse(BigDecimal.ZERO))
                .add(Optional.ofNullable(entity.getLastMonthRollingAmount())
                        .orElse(BigDecimal.ZERO))
                .add(Optional.ofNullable(entity.getAdjustAmount())
                        .orElse(BigDecimal.ZERO))
                .subtract(Optional.ofNullable(entity.getFreezeAmount())
                        .orElse(BigDecimal.ZERO))
                .subtract(Optional.ofNullable(entity.getApprovedAmount())
                        .orElse(BigDecimal.ZERO))
                .subtract(Optional.ofNullable(entity.getEstimatedExcessAmount())
                        .orElse(BigDecimal.ZERO))
                .add(Optional.ofNullable(entity.getApprovedAuditDiff())
                        .orElse(BigDecimal.ZERO));
    }



    /**
     * 冻结可用 (主体和电商)
     * 冻结后可用金额=年初分解金额+月度分解金额与年初分解金额的差异+上月滚动金额+调整金额-冻结金额
     * @param entity
     * @return {@link BigDecimal}
     */
    public BigDecimal buildAfterFreezeAmountHeadquartersAndOnline(MonthBudgetEntity entity, BusinessUnitEnum unitEnum) {
        BigDecimal result = Optional.ofNullable(entity.getInitResolveAmount())
                .orElse(BigDecimal.ZERO)
                .add(Optional.ofNullable(entity.getFirstReplyResolveDiffAmount())
                        .orElse(BigDecimal.ZERO))
                .add(Optional.ofNullable(entity.getLastMonthRollingAmount())
                        .orElse(BigDecimal.ZERO))
                .add(Optional.ofNullable(entity.getAdjustAmount())
                        .orElse(BigDecimal.ZERO))
                .subtract(Optional.ofNullable(entity.getFreezeAmount())
                        .orElse(BigDecimal.ZERO));
        log.info("预算["+entity.getMonthBudgetCode()+"]冻结后可用金额计算={}", result);
        return result;
    }

    /**
     * 累计可用 (主体和电商)
     * 累计可用余额=冻结后可用金额-批复金额-预估超额-本月滚动金额
     * @param entity
     * @return {@link BigDecimal}
     */
    public BigDecimal buildAccumulatedAvailableBalanceHeadquartersAndOnline(MonthBudgetEntity entity, BusinessUnitEnum unitEnum) {
        BigDecimal result = Optional.ofNullable(entity.getAfterFreezeAmount()).orElse(BigDecimal.ZERO)
                .subtract(Optional.ofNullable(entity.getApprovedAmount())
                        .orElse(BigDecimal.ZERO))
                .subtract(Optional.ofNullable(entity.getEstimatedExcessAmount())
                        .orElse(BigDecimal.ZERO))
                .subtract(Optional.ofNullable(entity.getMonthRollingAmount())
                        .orElse(BigDecimal.ZERO));
        log.info("预算["+entity.getMonthBudgetCode()+"]累计可用金额计算={}", result);
        return result;
    }



    /**
     * 计算实销回复差(垂直)
     * @param actualSalesVo
     * @param currMonthBudget
     */
    public void calculationActualReplyDiffVertical(MonthBudgetActualSalesVo actualSalesVo, MonthBudgetVo currMonthBudget) {
        //月度分解金额
        BigDecimal firstReplyAmount = actualSalesVo.getFirstReplyAmount();
        if(org.springframework.util.ObjectUtils.isEmpty(firstReplyAmount)){
            return;
        }
        //实销金额
        BigDecimal actualSalesAmount = actualSalesVo.getActualSalesAmount();
        //实际费用
        BigDecimal actualCostsAmount = actualSalesVo.getActualCostsAmount();
        if(org.springframework.util.ObjectUtils.isEmpty(actualCostsAmount)){
            return;
        }
        //计算数据(回复量/计划量)
        if(ObjectUtils.isEmpty(actualSalesVo.getCalAmount())){
            return;
        }

        //判断是否滚动
        //对应预算管控配置下“是否滚动”字段维护的“是”，则“实销回复差”展示在计算年月（当前年月）预算的调整金额下；
        //对应预算管控配置下“是否滚动”字段维护的“否”，则“实销回复差”展示在对应实销量年月预算的调整金额下；
        MonthBudgetEntity entity = null;
        if(BooleanEnum.TRUE.getCapital().equals(actualSalesVo.getRolling()) && !ObjectUtils.isEmpty(currMonthBudget)){
            //查询计算年月的月度计划
            entity = this.monthBudgetRepository.getById(currMonthBudget.getId());
        }else{
            //体现在当前计算的月度预算上面
            entity = this.monthBudgetRepository.getById(actualSalesVo.getId());
        }

        //实销回复差
        BigDecimal diffAmount = actualCostsAmount.subtract(firstReplyAmount).setScale(6, BigDecimal.ROUND_HALF_UP);
        log.error("实销回复差={}",diffAmount);
        //判断是否为定时任务执行
        if(org.springframework.util.StringUtils.hasText(actualSalesVo.getDAY())){
            //定时任务 非2号执行时 销售金额 小于 回复量 不计算实际销售回复差
            Calendar cl = Calendar.getInstance();
            String yearMonthDay = DateUtil.format(cl.getTime(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
            String currDay = yearMonthDay.split("-")[2];
            if(!currDay.equals(actualSalesVo.getDAY()) && actualSalesAmount.compareTo(actualSalesVo.getCalAmount()) <= 0){
                return;
            }
        }else if(BigDecimal.ZERO.compareTo(diffAmount) == 0){
            entity.setActualReplyDiff(diffAmount);
            log.error("第一种情况更新，数据={}",JSONObject.toJSONString(entity));
            this.monthBudgetRepository.updateById(entity);
            return;
        }
        //判断是否为第一次计算
        //查询调整明细
        String businessCode = actualSalesVo.getMonthBudgetCode() + DateUtil.getDate("yyyyMM");
        MonthBudgetDetailVo monthBudgetDetailVo = this.monthBudgetDetailRepository.findByBusinessCode(businessCode);
        if(org.springframework.util.ObjectUtils.isEmpty(monthBudgetDetailVo)){
            //第一次计算
            entity.setActualReplyDiff(diffAmount);
            //累计可用 = 年初分解金额 + 月度分解金额与年初分解金额的差异 + 上月滚动金额 + 调整金额 - 冻结金额 - 批复金额 - 预估超额
            entity.setAccumulatedAvailableBalance(this.buildAccumulatedAvailableBalance(entity, BusinessUnitEnum.VERTICAL));
            //冻结可用 = 累计可用 + 批复金额 + 预估超额
            entity.setAfterFreezeAmount(this.buildAfterFreezeAmount(entity, BusinessUnitEnum.VERTICAL));
            //调整明细
            MonthBudgetDetailEntity detailEntity = this.buildActualReplyDiffDetail(entity,
                diffAmount,
                BudgetOperationTypeEnum.ACTUAL_SALES.getCode(), businessCode);
            this.monthBudgetDetailRepository.save(detailEntity);
            log.error("第二种情况更新，数据={}",JSONObject.toJSONString(entity));
            this.monthBudgetRepository.updateById(entity);
        }else{
            //非第一次计算
            //判断是否为同一个月度预算
            if(monthBudgetDetailVo.getMonthBudgetCode().equals(entity.getMonthBudgetCode())){
                //同一个月度预算
                BigDecimal histOperationAmount = monthBudgetDetailVo.getCurOperationAmount();
                if(histOperationAmount.compareTo(diffAmount) != 0){
                    //金额计算变动
                    entity.setActualReplyDiff(entity.getActualReplyDiff().subtract(histOperationAmount));
                    //先将金额还回去
                    //累计可用 = 年初分解金额 + 月度分解金额与年初分解金额的差异 + 上月滚动金额 + 调整金额 - 冻结金额 - 批复金额 - 预估超额
                    entity.setAccumulatedAvailableBalance(this.buildAccumulatedAvailableBalance(entity, BusinessUnitEnum.VERTICAL));
                    //冻结可用 = 累计可用 + 批复金额 + 预估超额
                    entity.setAfterFreezeAmount(this.buildAfterFreezeAmount(entity, BusinessUnitEnum.VERTICAL));
                    //调整明细
                    MonthBudgetDetailEntity histDetailEntity = this.buildActualReplyDiffDetail(entity,
                        histOperationAmount,
                        BudgetOperationTypeEnum.ACTUAL_SALES.getCode(), BudgetOperationTypeEnum.ACTUAL_SALES.getCode());
                    //用新的差额计算
                    entity.setActualReplyDiff(diffAmount);
                    //累计可用 = 年初分解金额 + 月度分解金额与年初分解金额的差异 + 上月滚动金额 + 调整金额 - 冻结金额 - 批复金额 - 预估超额
                    entity.setAccumulatedAvailableBalance(this.buildAccumulatedAvailableBalance(entity, BusinessUnitEnum.VERTICAL));
                    //冻结可用 = 累计可用 + 批复金额 + 预估超额
                    entity.setAfterFreezeAmount(this.buildAfterFreezeAmount(entity, BusinessUnitEnum.VERTICAL));
                    //明细
                    MonthBudgetDetailEntity detailEntity = this.buildActualReplyDiffDetail(entity,
                        diffAmount,
                        BudgetOperationTypeEnum.ACTUAL_SALES.getCode(), businessCode);
                    this.monthBudgetDetailRepository.save(histDetailEntity);
                    this.monthBudgetDetailRepository.save(detailEntity);
                    this.monthBudgetRepository.updateById(entity);
                }
            }else{
                //不同月度预算
                //旧的归还
                String monthBudgetCode = monthBudgetDetailVo.getMonthBudgetCode();
                BigDecimal histOperationAmount = monthBudgetDetailVo.getCurOperationAmount();
                MonthBudgetEntity histEntity = this.monthBudgetRepository.getByMonthBudgetCode(monthBudgetCode, null);
                histEntity.setActualReplyDiff(histEntity.getActualReplyDiff().subtract(histOperationAmount));
                //累计可用 = 年初分解金额 + 月度分解金额与年初分解金额的差异 + 上月滚动金额 + 调整金额 - 冻结金额 - 批复金额 - 预估超额
                histEntity.setAccumulatedAvailableBalance(this.buildAccumulatedAvailableBalance(histEntity, BusinessUnitEnum.VERTICAL));
                //冻结可用 = 累计可用 + 批复金额 + 预估超额
                histEntity.setAfterFreezeAmount(this.buildAfterFreezeAmount(histEntity, BusinessUnitEnum.VERTICAL));
                //调整明细
                MonthBudgetDetailEntity histDetailEntity = this.buildActualReplyDiffDetail(histEntity,
                    histOperationAmount,
                    BudgetOperationTypeEnum.ACTUAL_SALES.getCode(), BudgetOperationTypeEnum.ACTUAL_SALES.getCode());
                this.monthBudgetDetailRepository.save(histDetailEntity);
                this.monthBudgetRepository.updateById(histEntity);
                //新的调整
                //第一次计算
                entity.setActualReplyDiff(diffAmount);
                //累计可用 = 年初分解金额 + 月度分解金额与年初分解金额的差异 + 上月滚动金额 + 调整金额 - 冻结金额 - 批复金额 - 预估超额
                entity.setAccumulatedAvailableBalance(this.buildAccumulatedAvailableBalance(entity, BusinessUnitEnum.VERTICAL));
                //冻结可用 = 累计可用 + 批复金额 + 预估超额
                entity.setAfterFreezeAmount(this.buildAfterFreezeAmount(entity, BusinessUnitEnum.VERTICAL));
                //调整明细
                MonthBudgetDetailEntity detailEntity = this.buildActualReplyDiffDetail(entity,
                    diffAmount,
                    BudgetOperationTypeEnum.ACTUAL_SALES.getCode(), businessCode);
                this.monthBudgetDetailRepository.save(detailEntity);
                log.error("第三种情况更新，数据={}",JSONObject.toJSONString(entity));
                this.monthBudgetRepository.updateById(entity);
            }
        }
    }

    /**
     * 计算实销回复差(主体)
     * @param actualSalesVo
     * @param currMonthBudget
     */
    public void calculationActualReplyDiffHead(MonthBudgetActualSalesVo actualSalesVo, MonthBudgetVo currMonthBudget) {
        //月度分解金额
        BigDecimal firstReplyAmount = actualSalesVo.getFirstReplyAmount();
        if (Objects.isNull(firstReplyAmount)) {
            return;
        }
        //实销金额
        BigDecimal actualSalesAmount = actualSalesVo.getActualSalesAmount();
        if (Objects.isNull(actualSalesAmount)) {
            return;
        }

        MonthBudgetEntity entity = this.monthBudgetRepository.getById(actualSalesVo.getId());
        if (Objects.isNull(entity)) {
            return;
        }
        //实销回复差
        BigDecimal diffAmount = actualSalesAmount.subtract(firstReplyAmount).setScale(6, BigDecimal.ROUND_HALF_UP);
        entity.setActualReplyDiff(diffAmount);
        log.error("实销回复差={}", diffAmount);
        if (BigDecimal.ZERO.compareTo(diffAmount) == 0) {
            this.monthBudgetRepository.updateById(entity);
            return;
        }
        Date date = DateUtil.strToDate(actualSalesVo.getYearMonthLy(), DateUtil.date_yyyy_MM);
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        Date currDate = DateUtil.getDate(DateUtil.date_yyyy_MM);
        Calendar calendar2 = Calendar.getInstance();
        calendar2.setTime(currDate);
        String curryearMonth = DateUtil.dateToStr(DateUtil.date_yyyy_MM);
        //必须属于同一年
        if (calendar.get(Calendar.YEAR) != calendar2.get(Calendar.YEAR)) {
            log.error("计算实销回复差-》必须属于同一年");
            return;
        }
        boolean rollTag = false;
        //判断滚动类型
        if(BooleanEnum.TRUE.getCapital().equals(actualSalesVo.getRolling()) && calendar.get(Calendar.MONTH) != 11){
            // a-全部滚动：不判断正负数；
            if (RollingTypeEnum.PLEASE_SPECIFY.getCode().equals(actualSalesVo.getRollingType())) {
                rollTag = true;
            }
            // b-结余滚动：为正数；
            if (RollingTypeEnum.SURPLUS_ROLL.getCode().equals(actualSalesVo.getRollingType())) {
                if (BigDecimal.ZERO.compareTo(diffAmount) < 0) {
                    rollTag = true;
                }
            }
            // c-超支滚动：为负数；
            if (RollingTypeEnum.OVERSPEND_ROLL.getCode().equals(actualSalesVo.getRollingType())) {
                if (BigDecimal.ZERO.compareTo(diffAmount) > 0) {
                    rollTag = true;
                }
            }
        }

        //判断是否滚动，月份12不滚动
        if (rollTag) {
            //当前年月必须大于选择的预算年月
            if (currDate.getTime() <= date.getTime()) {
                log.error("计算实销回复差-》当前年月必须大于选择的预算年月");
                return;
            }
            //对应预算管控配置下“是否滚动”字段维护的“是”，则“实销回复差”展示在计算年月（当前年月）预算的调整金额下；
            //获取当前年月的月度预算
            MonthBudgetEntity currEntity = this.monthBudgetRepository.getOneByYearBudgetCodeAndMonth(actualSalesVo.getYearBudgetCode(), curryearMonth);
            this.monthBudgetRepository.updateById(entity);

            String businessCode = entity.getMonthBudgetCode() + "-"+ DateUtil.getDate("yyyyMM");
            MonthBudgetDetailVo monthBudgetDetailVo = this.monthBudgetDetailRepository.findByBusinessCode(businessCode);
            if (Objects.isNull(monthBudgetDetailVo)) {
                BigDecimal beforeAmount = currEntity.getAccumulatedAvailableBalance();
                //第一次计算
                currEntity.setAdjustAmount(Optional.ofNullable(currEntity.getAdjustAmount()).orElse(BigDecimal.ZERO).add(diffAmount));
                //冻结可用
                currEntity.setAfterFreezeAmount(this.buildAfterFreezeAmountHeadquartersAndOnline(currEntity, BusinessUnitEnum.VERTICAL));
                //累计可用
                currEntity.setAccumulatedAvailableBalance(this.buildAccumulatedAvailableBalanceHeadquartersAndOnline(currEntity, BusinessUnitEnum.VERTICAL));
                //调整明细
                MonthBudgetDetailEntity detailEntity = this.buildActualReplyDiffDetailHead(currEntity,
                        diffAmount, BudgetOperationTypeEnum.ACTUAL_SALES.getCode(), businessCode, beforeAmount);
                this.monthBudgetDetailRepository.save(detailEntity);
                this.monthBudgetRepository.updateById(currEntity);
            } else {
                //非第一次计算
                BigDecimal histOperationAmount = monthBudgetDetailVo.getCurOperationAmount();
                if (histOperationAmount.compareTo(diffAmount) != 0) {
                    BigDecimal beforeAmount = currEntity.getAccumulatedAvailableBalance();
                    //先将金额还回去
                    //金额计算变动
                    currEntity.setAdjustAmount(Optional.ofNullable(currEntity.getAdjustAmount()).orElse(BigDecimal.ZERO).subtract(histOperationAmount));
                    //冻结可用
                    currEntity.setAfterFreezeAmount(this.buildAfterFreezeAmountHeadquartersAndOnline(currEntity, BusinessUnitEnum.VERTICAL));
                    //累计可用
                    currEntity.setAccumulatedAvailableBalance(this.buildAccumulatedAvailableBalanceHeadquartersAndOnline(currEntity, BusinessUnitEnum.VERTICAL));
                    //调整明细
                    MonthBudgetDetailEntity histDetailEntity = this.buildActualReplyDiffDetailHead(currEntity,
                            histOperationAmount,
                            BudgetOperationTypeEnum.ACTUAL_SALES.getCode(), businessCode,beforeAmount);
                    this.monthBudgetDetailRepository.save(histDetailEntity);

                    BigDecimal beforeAmount2 = currEntity.getAccumulatedAvailableBalance();
                    //用新的差额计算
                    currEntity.setAdjustAmount(Optional.ofNullable(entity.getAdjustAmount()).orElse(BigDecimal.ZERO).add(diffAmount));
                    //冻结可用
                    currEntity.setAfterFreezeAmount(this.buildAfterFreezeAmountHeadquartersAndOnline(currEntity, BusinessUnitEnum.VERTICAL));
                    //累计可用
                    currEntity.setAccumulatedAvailableBalance(this.buildAccumulatedAvailableBalanceHeadquartersAndOnline(currEntity, BusinessUnitEnum.VERTICAL));
                    //明细
                    MonthBudgetDetailEntity detailEntity = this.buildActualReplyDiffDetailHead(currEntity,
                            diffAmount, BudgetOperationTypeEnum.ACTUAL_SALES.getCode(), businessCode, beforeAmount2);
                    this.monthBudgetDetailRepository.save(detailEntity);
                    this.monthBudgetRepository.updateById(currEntity);
                }
            }
        } else {
            //对应预算管控配置下“是否滚动”字段维护的“否”，则“实销回复差”展示在对应实销量年月预算的调整金额下；
            //判断是否为第一次计算
            String businessCode = actualSalesVo.getMonthBudgetCode() + DateUtil.getDate("yyyyMM");
            MonthBudgetDetailVo monthBudgetDetailVo = this.monthBudgetDetailRepository.findByBusinessCode(businessCode);
            if (Objects.isNull(monthBudgetDetailVo)) {
                BigDecimal beforeAmount = entity.getAccumulatedAvailableBalance();
                //第一次计算
                entity.setAdjustAmount(Optional.ofNullable(entity.getAdjustAmount()).orElse(BigDecimal.ZERO).add(diffAmount));
                //冻结可用
                entity.setAfterFreezeAmount(this.buildAfterFreezeAmountHeadquartersAndOnline(entity, BusinessUnitEnum.VERTICAL));
                //累计可用
                entity.setAccumulatedAvailableBalance(this.buildAccumulatedAvailableBalanceHeadquartersAndOnline(entity, BusinessUnitEnum.VERTICAL));
                //调整明细
                MonthBudgetDetailEntity detailEntity = this.buildActualReplyDiffDetailHead(entity,
                        diffAmount, BudgetOperationTypeEnum.ACTUAL_SALES.getCode(), businessCode, beforeAmount);
                this.monthBudgetDetailRepository.save(detailEntity);
                this.monthBudgetRepository.updateById(entity);
            } else {
                //非第一次计算
                BigDecimal histOperationAmount = monthBudgetDetailVo.getCurOperationAmount();
                if (histOperationAmount.compareTo(diffAmount) != 0) {
                    BigDecimal beforeAmount = entity.getAccumulatedAvailableBalance();
                    //先将金额还回去
                    //金额计算变动
                    entity.setAdjustAmount(Optional.ofNullable(entity.getAdjustAmount()).orElse(BigDecimal.ZERO).subtract(histOperationAmount));
                    //冻结可用
                    entity.setAfterFreezeAmount(this.buildAfterFreezeAmountHeadquartersAndOnline(entity, BusinessUnitEnum.VERTICAL));
                    //累计可用
                    entity.setAccumulatedAvailableBalance(this.buildAccumulatedAvailableBalanceHeadquartersAndOnline(entity, BusinessUnitEnum.VERTICAL));
                    //调整明细
                    MonthBudgetDetailEntity histDetailEntity = this.buildActualReplyDiffDetailHead(entity,
                            histOperationAmount, BudgetOperationTypeEnum.ACTUAL_SALES.getCode(), businessCode, beforeAmount);
                    this.monthBudgetDetailRepository.save(histDetailEntity);

                    BigDecimal beforeAmount2 = entity.getAccumulatedAvailableBalance();
                    //用新的差额计算
                    entity.setAdjustAmount(Optional.ofNullable(entity.getAdjustAmount()).orElse(BigDecimal.ZERO).add(diffAmount));
                    //冻结可用
                    entity.setAfterFreezeAmount(this.buildAfterFreezeAmountHeadquartersAndOnline(entity, BusinessUnitEnum.VERTICAL));
                    //累计可用
                    entity.setAccumulatedAvailableBalance(this.buildAccumulatedAvailableBalanceHeadquartersAndOnline(entity, BusinessUnitEnum.VERTICAL));
                    //明细
                    MonthBudgetDetailEntity detailEntity = this.buildActualReplyDiffDetailHead(entity,
                            diffAmount, BudgetOperationTypeEnum.ACTUAL_SALES.getCode(), businessCode, beforeAmount2);
                    this.monthBudgetDetailRepository.save(detailEntity);
                    this.monthBudgetRepository.updateById(entity);
                }
            }
        }
    }

    /**
     * 构建实销差调整明细
     * @param entity
     * @param amount
     * @param operationType
     * @param businessCode
     * @return {@link MonthBudgetDetailDto}
     */
    private MonthBudgetDetailEntity buildActualReplyDiffDetail(MonthBudgetEntity entity, BigDecimal amount, String operationType, String businessCode) {
        MonthBudgetDetailEntity detailEntity = this.nebulaToolkitService.copyObjectByWhiteList(
            entity, MonthBudgetDetailEntity.class, LinkedHashSet.class, ArrayList.class
        );
        detailEntity.setId(null);
        detailEntity.setBusinessCode(businessCode);
        detailEntity.setOperationType(operationType);
        detailEntity.setInitialAmount(entity.getInitResolveAmount());
        //调整后
        detailEntity.setBalanceAmount(entity.getAccumulatedAvailableBalance());
        //判断是扣减还是增加
        if(amount.compareTo(BigDecimal.ZERO) == 1){
            //增加
            //调整前
            detailEntity.setBeforeAmount(entity.getAccumulatedAvailableBalance().subtract(amount));
        }else{
            //扣减
            //调整前
            detailEntity.setBeforeAmount(entity.getAccumulatedAvailableBalance().add(amount));
        }
        detailEntity.setCurOperationAmount(amount);
        return detailEntity;
    }

    /**
     * 构建实销差调整明细(主体)
     * @param entity
     * @param amount
     * @param operationType
     * @param beforeAmount
     * @return {@link MonthBudgetDetailDto}
     */
    public MonthBudgetDetailEntity buildActualReplyDiffDetailHead(MonthBudgetEntity entity, BigDecimal amount, String operationType, String businessCode,BigDecimal beforeAmount) {
        MonthBudgetDetailEntity detailEntity = this.nebulaToolkitService.copyObjectByWhiteList(
                entity, MonthBudgetDetailEntity.class, LinkedHashSet.class, ArrayList.class
        );
        detailEntity.setId(null);
        detailEntity.setBusinessCode(businessCode);
        detailEntity.setOperationType(operationType);
        detailEntity.setInitialAmount(entity.getInitResolveAmount());
        //调整后
        detailEntity.setBalanceAmount(entity.getAccumulatedAvailableBalance());
        detailEntity.setBeforeAmount(beforeAmount);
        detailEntity.setCurOperationAmount(amount);
        return detailEntity;
    }
}