package com.biz.crm.tpm.business.month.budget.local.helper;

import cn.hutool.core.collection.CollectionUtil;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.tpm.business.budget.cal.config.sdk.dto.BudgetCalConfigDto;
import com.biz.crm.tpm.business.budget.cal.config.sdk.eunm.BudgetTypeEnum;
import com.biz.crm.tpm.business.budget.cal.config.sdk.eunm.CalDataFromEnum;
import com.biz.crm.tpm.business.budget.cal.config.sdk.eunm.SalesPlanAmountTypeEnum;
import com.biz.crm.tpm.business.budget.cal.config.sdk.service.BudgetCalConfigService;
import com.biz.crm.tpm.business.budget.cal.config.sdk.vo.BudgetCalConfigDataVo;
import com.biz.crm.tpm.business.budget.cal.config.sdk.vo.BudgetCalConfigProductRatioVo;
import com.biz.crm.tpm.business.budget.cal.config.sdk.vo.BudgetCalConfigVo;
import com.biz.crm.tpm.business.budget.controls.config.sdk.dto.DimensionControlsDto;
import com.biz.crm.tpm.business.budget.controls.config.sdk.enums.DimensionControlsTypeEnum;
import com.biz.crm.tpm.business.budget.controls.config.sdk.enums.RollingTypeEnum;
import com.biz.crm.tpm.business.budget.controls.config.sdk.service.DimensionControlsService;
import com.biz.crm.tpm.business.budget.controls.config.sdk.vo.DimensionControlsVo;
import com.biz.crm.tpm.business.budget.item.sdk.enums.BudgetControlTypeEnum;
import com.biz.crm.tpm.business.budget.item.sdk.enums.FeeBelongEnum;
import com.biz.crm.tpm.business.budget.item.sdk.service.BudgetItemService;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemControlConditionVo;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo;
import com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetDetailEntity;
import com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity;
import com.biz.crm.tpm.business.month.budget.local.repository.MonthBudgetDetailRepository;
import com.biz.crm.tpm.business.month.budget.local.repository.MonthBudgetRepository;
import com.biz.crm.tpm.business.month.budget.sdk.eunm.BudgetOperationTypeEnum;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetVo;
import com.biz.crm.tpm.business.sales.plan.sdk.vo.SalesPlanVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Maps;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Description 月度预算助手
 * @Author YangWei
 * @Date 2023/2/9 上午10:00
 */
@Slf4j
@Component
public class MonthBudgetHelper {

    @Resource
    private MonthBudgetRepository monthBudgetRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Resource
    private BudgetItemService budgetItemService;

    @Autowired(required = false)
    private BudgetCalConfigService budgetCalConfigService;

    @Autowired(required = false)
    private DimensionControlsService dimensionControlsService;

    @Autowired(required = false)
    private MonthBudgetDetailRepository monthBudgetDetailRepository;

    @Resource
    private MonthBudgetCalculateHelper monthBudgetCalculateHelper;

    /**
     * 主体汇总预算计算配置表中金额类型的金额
     *
     * @param reduce
     * @param salesPlanVo
     * @param amountTypeCode
     * @param monthBudgetEntity
     * @param budgetTotalPoint
     */
    public void calculationReduce(AtomicReference<BigDecimal> reduce,
                                  SalesPlanVo salesPlanVo, String amountTypeCode,
                                  BudgetCalConfigVo budgetCalConfigVo,
                                  MonthBudgetEntity monthBudgetEntity,
                                  BigDecimal budgetTotalPoint) {
        //拿到配置的产品计费比例
        List<BudgetCalConfigProductRatioVo> productRatios = budgetCalConfigVo.getProductRatios();
        Map<String, BudgetCalConfigProductRatioVo> productRatioVoMap = new HashMap<>();
        if (!CollectionUtils.isEmpty(productRatios)) {
            productRatioVoMap.putAll(productRatios.stream().collect(Collectors.toMap(BudgetCalConfigProductRatioVo::getProductCode, e -> e)));
        }
        //商品编码
        String productCode = salesPlanVo.getProductCode();
        //获取计费比例
        BigDecimal chargedRatio = null;
        BudgetCalConfigProductRatioVo budgetCalConfigProductRatioVo = productRatioVoMap.get(productCode);
        if (!ObjectUtils.isEmpty(budgetCalConfigProductRatioVo)) {
            //判断是否扣减
            if (StringUtil.isNotBlank(budgetCalConfigProductRatioVo.getReduceTag()) && BooleanEnum.TRUE.getCapital().equals(budgetCalConfigProductRatioVo.getReduceTag())) {
                if (!Objects.isNull(budgetTotalPoint)) {
                    chargedRatio = budgetTotalPoint.subtract(budgetCalConfigProductRatioVo.getChargedRatio());
                } else {
                    chargedRatio = budgetCalConfigProductRatioVo.getChargedRatio();
                }
            } else {
                chargedRatio = budgetCalConfigProductRatioVo.getChargedRatio();
            }
        }


        //判断管控类型
        //需要除数逻辑(年度点数/年度力度)
        BigDecimal pointOrIntensity = null;
        //管控类型
        String controlTypeCode = budgetCalConfigVo.getControlType();
        if (BudgetControlTypeEnum.RATIO.getCode().equals(controlTypeCode)) {
            Assert.notNull(monthBudgetEntity.getBudgetTotalPoint(), "预算[" + monthBudgetEntity.getYearBudgetCode() + "]总点数不能为空");
            pointOrIntensity = monthBudgetEntity.getBudgetTotalPoint().divide(new BigDecimal(100), 2, BigDecimal.ROUND_HALF_UP);
        } else if (BudgetControlTypeEnum.INTENSITY.getCode().equals(controlTypeCode)) {
            //分子
            BigDecimal budgetIntensityNumerator = monthBudgetEntity.getBudgetIntensityNumerator();
            Assert.notNull(budgetIntensityNumerator, "预算[" + monthBudgetEntity.getYearBudgetCode() + "]年度力度分子不能为空");
            //分母
            BigDecimal budgetIntensityDenominator = monthBudgetEntity.getBudgetIntensityDenominator();
            Assert.notNull(budgetIntensityDenominator, "预算[" + monthBudgetEntity.getYearBudgetCode() + "]年度力度分母不能为空");
            pointOrIntensity = budgetIntensityNumerator.divide(budgetIntensityDenominator, 6, BigDecimal.ROUND_HALF_UP);
        }
        // 总部：计费产品费用合计值/单个预算项目点数=倒算计费产品对应的回复量
        //     大区：计费产品费用合计值/区域整体授权点数=倒算计费产品对应的回复量
        if (FeeBelongEnum.AREA.getCode().equals(budgetCalConfigVo.getFeeBelongCode())) {
            pointOrIntensity = budgetTotalPoint;
        }

        //取值逻辑
        SalesPlanAmountTypeEnum amountTypeEnum = SalesPlanAmountTypeEnum.getByCode(amountTypeCode);
        switch (amountTypeEnum) {
            case PLAN_AMOUNT:
                //折前计划量
                this.setReduce(salesPlanVo.getPlanAmount(), reduce, chargedRatio, pointOrIntensity);
                break;
            case DISCOUNT_PLAN_AMOUNT:
                //折后计划量
                this.setReduce(salesPlanVo.getDiscountPlanAmount(), reduce, chargedRatio, pointOrIntensity);
                break;
            case RESTORE_AMOUNT:
                //折前回复金额
                this.setReduce(salesPlanVo.getRestoreAmount(), reduce, chargedRatio, pointOrIntensity);
                break;
            case DISCOUNT_RESTORE_AMOUNT:
                //折后回复金额
                this.setReduce(salesPlanVo.getDiscountRestoreAmount(), reduce, chargedRatio, pointOrIntensity);
                break;
            case DELIVERY_MULTIPLY_PRICE:
                //出库件数*单品到岸价
                //this.setReduce(salesPlanVo.getDiscountPlanAmount(), reduce,chargedRatio, pointOrIntensity);
                break;
            case PLAN_QUANTITY:
                //计划数量
                this.setReduce(salesPlanVo.getPlanQuantity(), reduce, chargedRatio, pointOrIntensity);
                break;
            case RESTORE_QUANTITY:
                //回复数量
                this.setReduce(salesPlanVo.getRestoreQuantity(), reduce, chargedRatio, pointOrIntensity);
                break;
            default:
                break;
        }
    }

    /**
     * 计算结果
     *
     * @param salesAmount
     * @param reduce
     * @param chargedRatio
     */
    private void setReduce(BigDecimal salesAmount, AtomicReference<BigDecimal> reduce
            , BigDecimal chargedRatio, BigDecimal budgetTotalPoint) {
        //配置金额
        BigDecimal amount = Optional.ofNullable(salesAmount).orElse(BigDecimal.ZERO);
        if (!ObjectUtils.isEmpty(chargedRatio)) {
            Assert.notNull(budgetTotalPoint, "预算总点数不能为空");
            //金额计算
            BigDecimal amountResult = amount.multiply(chargedRatio).divide(budgetTotalPoint, 6, RoundingMode.HALF_UP);
            reduce.set(reduce.get().add(amountResult));
        } else {
            reduce.set(reduce.get().add(amount));
        }
    }


    /**
     * 获取管控条件
     *
     * @param monthBudgetEntity
     * @param finalBudgetItemVoMap
     * @return {@link BudgetItemControlConditionVo}
     */
    public BudgetItemControlConditionVo getBudgetItemControlCondition(MonthBudgetEntity monthBudgetEntity, Map<String, BudgetItemVo> finalBudgetItemVoMap) {
        BudgetItemVo budgetItemVo = finalBudgetItemVoMap.get(monthBudgetEntity.getBudgetItemCode());
        if (org.springframework.util.ObjectUtils.isEmpty(budgetItemVo)) {
            return null;
        } else {
            List<BudgetItemControlConditionVo> controlConditionDtoList = budgetItemVo.getControlConditionDtoList();
            if (CollectionUtils.isEmpty(controlConditionDtoList)) {
                return null;
            } else {
                List<BudgetItemControlConditionVo> conditionVos = finalBudgetItemVoMap.get(monthBudgetEntity.getBudgetItemCode()).getControlConditionDtoList().stream()
                        .filter(controlVo -> monthBudgetEntity.getBusinessFormatCode().equals(controlVo.getBusinessFormatCode())
                                && monthBudgetEntity.getBusinessUnitCode().equals(controlVo.getBusinessUnitCode()))
                        .collect(Collectors.toList());
                if (CollectionUtils.isEmpty(conditionVos)) {
                    return null;
                }
                return conditionVos.get(0);
            }
        }
    }

    /**
     * 根据id查询月度预算集合
     *
     * @param monthBudgetIds
     * @return {@link List}<{@link MonthBudgetVo}>
     */
    public List<MonthBudgetVo> findByIds(List<String> monthBudgetIds) {
        if (CollectionUtil.isEmpty(monthBudgetIds)){
           return Collections.emptyList();
        }
        List<MonthBudgetEntity> monthBudgetEntities = this.monthBudgetRepository.listByIds(monthBudgetIds);
        if (CollectionUtil.isEmpty(monthBudgetEntities)){
            return Collections.emptyList();
        }
        return (List<MonthBudgetVo>) nebulaToolkitService.copyCollectionByWhiteList(
                monthBudgetEntities, MonthBudgetEntity.class, MonthBudgetVo.class, HashSet.class, ArrayList.class);
    }

    /**
     * 获取预算项目配置
     *
     * @param budgetItemCodes
     * @return {@link Map}<{@link String}, {@link BudgetItemVo}>
     */
    public Map<String, BudgetItemVo> findBudgetItemByCodes(List<String> budgetItemCodes) {
        if (CollectionUtil.isEmpty(budgetItemCodes)){
            return Maps.newHashMap();
        }
        List<BudgetItemVo> budgetItemVos = this.budgetItemService.listByCodes(budgetItemCodes);
        if (CollectionUtil.isEmpty(budgetItemVos)){
            return Maps.newHashMap();
        }
        return budgetItemVos.stream().collect(Collectors.toMap(BudgetItemVo::getBudgetItemCode, Function.identity()));
    }

    /**
     * 查询预算计算配置
     *
     * @param monthBudgetVo
     * @param controlTypeCode
     * @return {@link BudgetCalConfigVo}
     */
    public BudgetCalConfigVo findMonthBudgetCalConfigByDto(MonthBudgetVo monthBudgetVo, String controlTypeCode) {
        BudgetCalConfigDto dto = new BudgetCalConfigDto();
        dto.setBudgetTypeCode(BudgetTypeEnum.MONTH_BUDGET.getCode());
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        dto.setBusinessFormatCode(monthBudgetVo.getBusinessFormatCode());
        dto.setBusinessUnitCode(monthBudgetVo.getBusinessUnitCode());
        dto.setFeeBelongCode(monthBudgetVo.getFeeBelongCode());

        //管控类型只有主体需要
        if (BusinessUnitEnum.isDefaultBusinessUnit(monthBudgetVo.getBusinessUnitCode())) {
            dto.setControlType(controlTypeCode);
            dto.setGroupCode(monthBudgetVo.getGroupCode());
        }
        List<BudgetCalConfigVo> budgetCalConfigVos = budgetCalConfigService.listByConditions(dto);
        if (CollectionUtils.isEmpty(budgetCalConfigVos)) {
            return null;
        }
        BudgetCalConfigVo budgetCalConfigVo = budgetCalConfigVos.stream().filter(o -> (!CollectionUtils.isEmpty(o.getBudgetItemList())
                && !ObjectUtils.isEmpty(o.getBudgetItemList().stream()
                .filter(d -> d.getDataCode().equals(monthBudgetVo.getBudgetItemCode())).findFirst().orElse(null)))).findFirst().orElse(null);
        if (!ObjectUtils.isEmpty(budgetCalConfigVo)) {
            return budgetCalConfigVo;
        }
        return budgetCalConfigVos.stream().filter(o -> CollectionUtils.isEmpty(o.getBudgetItemList()))
                .findFirst().orElse(null);
    }

    /**
     * 查询预算管控配置
     *
     * @param
     * @return {@link Map}<{@link String}, {@link List}<{@link DimensionControlsVo}>>
     */
    public Map<String, List<DimensionControlsVo>> findDimensionControl() {
        DimensionControlsDto dimensionControlsDto = new DimensionControlsDto();
        dimensionControlsDto.setControlType(DimensionControlsTypeEnum.DIMENSION_CONTROL.getCode());
        dimensionControlsDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        List<DimensionControlsVo> dimensionControlsVos = dimensionControlsService.listByConditions(dimensionControlsDto);
        if (CollectionUtils.isEmpty(dimensionControlsVos)) {
            return Maps.newHashMap();
        }
        return dimensionControlsVos.stream().collect(Collectors.groupingBy(vo ->
                vo.getBusinessFormatCode() + vo.getBusinessUnitCode() + vo.getMarketOrganization()));
    }

    /**
     * 查询预算管控配置（分组 ）
     *
     * @param
     * @return {@link Map}<{@link String}, {@link List}<{@link DimensionControlsVo}>>
     */
    public Map<String, List<DimensionControlsVo>> findDimensionControlNoSalesOrg() {
        DimensionControlsDto dimensionControlsDto = new DimensionControlsDto();
        dimensionControlsDto.setControlType(DimensionControlsTypeEnum.DIMENSION_CONTROL.getCode());
        dimensionControlsDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        List<DimensionControlsVo> dimensionControlsVos = dimensionControlsService.listByConditions(dimensionControlsDto);
        if (CollectionUtils.isEmpty(dimensionControlsVos)) {
            return Maps.newHashMap();
        }
        return dimensionControlsVos.stream().collect(Collectors.groupingBy(vo ->
                vo.getBusinessFormatCode() + vo.getBusinessUnitCode()));
    }

    /**
     * 滚动
     *
     * @param currId
     * @param lastId
     * @param rollingType
     */
    public void calRolling(String currId, String lastId, String rollingType) {
        MonthBudgetEntity currMonthBudget = this.monthBudgetRepository.getById(currId);
        MonthBudgetEntity lastMonthBudget = this.monthBudgetRepository.getById(lastId);
        /* a.全部滚动：等于同一维度下M-1月（不跨年）的月度累计可用金额（不判断正负数）；
         * b.结余滚动：判断M-1月累计可用金额是否正数，为正数才计入M月的上月滚动金额字段；
         * c.超支滚动：判断M-1月累计可用金额是否负数，为负数才计入M月的上月滚动金额字段；*/
        // 滚动金额
        BigDecimal rollAmount = BigDecimal.ZERO;
        //上月累计可用
        BigDecimal lastAccumulatedAvailableBalance = null;
        if (BusinessUnitEnum.isDefaultBusinessUnit(lastMonthBudget.getBusinessUnitCode())
                || BusinessUnitEnum.ONLINE.getCode().equals(lastMonthBudget.getBusinessUnitCode())) {
            lastAccumulatedAvailableBalance = Optional.ofNullable(lastMonthBudget
                    .getAccumulatedAvailableBalance()).orElse(BigDecimal.ZERO);
        } else {
            lastAccumulatedAvailableBalance = this.monthBudgetCalculateHelper
                    .buildAccumulatedAvailableBalanceBefore(lastMonthBudget, BusinessUnitEnum.VERTICAL);
        }

        // a-全部滚动：等于同一维度下M-1月（不跨年）的月度累计可用金额（不判断正负数）；
        if (RollingTypeEnum.PLEASE_SPECIFY.getCode().equals(rollingType)) {
            rollAmount = lastAccumulatedAvailableBalance;
        }
        // b-结余滚动：判断M-1月累计可用金额是否正数，为正数才计入M月的上月滚动金额字段；
        if (RollingTypeEnum.SURPLUS_ROLL.getCode().equals(rollingType)) {
            if (BigDecimal.ZERO.compareTo(lastAccumulatedAvailableBalance) < 0) {
                rollAmount = lastAccumulatedAvailableBalance;
            }
        }
        // c-超支滚动：判断M-1月累计可用金额是否负数，为负数才计入M月的上月滚动金额字段；
        if (RollingTypeEnum.OVERSPEND_ROLL.getCode().equals(rollingType)) {
            if (BigDecimal.ZERO.compareTo(lastAccumulatedAvailableBalance) > 0) {
                rollAmount = lastAccumulatedAvailableBalance;
            }
        }
        log.error("滚动进行中，上月预算编码：{}，本月预算编码：{}，滚动类型：{}，滚动金额：{}", lastMonthBudget.getMonthBudgetCode(), currMonthBudget.getMonthBudgetCode(), rollingType, rollAmount);
        //只要不为零就能一直滚动
        if (rollAmount.compareTo(BigDecimal.ZERO) == 0) {
            return;
        }
        //滚动
        //上月扣除

        BigDecimal beforeAmount = lastMonthBudget.getAccumulatedAvailableBalance();
        //这里区分业务单元
        // 主体电商
        if (BusinessUnitEnum.isDefaultBusinessUnit(lastMonthBudget.getBusinessUnitCode())
                || BusinessUnitEnum.ONLINE.getCode().equals(lastMonthBudget.getBusinessUnitCode())) {
            //调整金额
            lastMonthBudget.setMonthRollingAmount(Optional.ofNullable(lastMonthBudget.getMonthRollingAmount())
                    .orElse(BigDecimal.ZERO).add(rollAmount));
            //冻结后可用金额=年初分解金额+月度分解金额与年初分解金额的差异+上月滚动金额+调整金额-冻结金额
            lastMonthBudget.setAfterFreezeAmount(this.monthBudgetCalculateHelper
                    .buildAfterFreezeAmountHeadquartersAndOnline(lastMonthBudget, BusinessUnitEnum.HEADQUARTERS));
            //累计可用余额=冻结后可用金额-批复金额-预估超额-本月滚动金额
            lastMonthBudget.setAccumulatedAvailableBalance(this.monthBudgetCalculateHelper
                    .buildAccumulatedAvailableBalanceHeadquartersAndOnline(lastMonthBudget, BusinessUnitEnum.HEADQUARTERS));
        } else {
            //调整金额
            lastMonthBudget.setMonthRollingAmount(rollAmount);
            //累计可用 = 年初分解金额 + 月度分解金额与年初分解金额的差异 + 上月滚动金额 + 调整金额 - 冻结金额 - 批复金额 - 预估超额
            lastMonthBudget.setAccumulatedAvailableBalance(this.monthBudgetCalculateHelper
                    .buildAccumulatedAvailableBalance(lastMonthBudget, BusinessUnitEnum.VERTICAL));
            //冻结可用 = 累计可用 + 批复金额 + 预估超额
            lastMonthBudget.setAfterFreezeAmount(this.monthBudgetCalculateHelper
                    .buildAfterFreezeAmount(lastMonthBudget, BusinessUnitEnum.VERTICAL));
        }

        //调整明细
        MonthBudgetDetailEntity lastDetailEntity = this.buildDetail(lastMonthBudget,
                rollAmount, beforeAmount,
                BudgetOperationTypeEnum.ROLLING.getCode(), null);
        this.monthBudgetDetailRepository.save(lastDetailEntity);
        lastMonthBudget.setIfRolling(BooleanEnum.TRUE.getCapital());
        this.monthBudgetRepository.updateById(lastMonthBudget);
        //当前月份增加

        //这里区分业务单元
        // 主体电商
        BigDecimal currBeforeAmount = currMonthBudget.getAccumulatedAvailableBalance();
        if (BusinessUnitEnum.isDefaultBusinessUnit(lastMonthBudget.getBusinessUnitCode())
                || BusinessUnitEnum.ONLINE.getCode().equals(lastMonthBudget.getBusinessUnitCode())) {
            //上月滚动金额 叠加
            currMonthBudget.setLastMonthRollingAmount(Optional.ofNullable(currMonthBudget.getLastMonthRollingAmount())
                    .orElse(BigDecimal.ZERO).add(rollAmount));
            //冻结后可用金额=年初分解金额+月度分解金额与年初分解金额的差异+上月滚动金额+调整金额-冻结金额
            currMonthBudget.setAfterFreezeAmount(this.monthBudgetCalculateHelper
                    .buildAfterFreezeAmountHeadquartersAndOnline(currMonthBudget, BusinessUnitEnum.HEADQUARTERS));
            //累计可用余额=冻结后可用金额-批复金额-预估超额-本月滚动金额
            currMonthBudget.setAccumulatedAvailableBalance(this.monthBudgetCalculateHelper
                    .buildAccumulatedAvailableBalanceHeadquartersAndOnline(currMonthBudget, BusinessUnitEnum.HEADQUARTERS));
        } else {
            //上月滚动金额 覆盖
            currMonthBudget.setLastMonthRollingAmount(rollAmount);
            //累计可用 = 年初分解金额 + 月度分解金额与年初分解金额的差异 + 上月滚动金额 + 调整金额 - 冻结金额 - 批复金额 - 预估超额
            currMonthBudget.setAccumulatedAvailableBalance(this.monthBudgetCalculateHelper
                    .buildAccumulatedAvailableBalance(currMonthBudget, BusinessUnitEnum.VERTICAL));
            //冻结可用 = 累计可用 + 批复金额 + 预估超额
            currMonthBudget.setAfterFreezeAmount(this.monthBudgetCalculateHelper
                    .buildAfterFreezeAmount(currMonthBudget, BusinessUnitEnum.VERTICAL));
        }

        //调整明细
        MonthBudgetDetailEntity currDetailEntity = this.buildDetail(currMonthBudget,
                rollAmount, currBeforeAmount,
                BudgetOperationTypeEnum.ROLLING.getCode(), null);
        this.monthBudgetDetailRepository.save(currDetailEntity);
        this.monthBudgetRepository.updateById(currMonthBudget);
    }

    /**
     * 构建预算操作明细
     *
     * @param entity        月度预算
     * @param amount        操作金额
     * @param beforeAmount  操作前金额
     * @param operationType 操作类型
     * @param businessCode  业务编码
     * @return com.biz.crm.tpm.business.month.budget.sdk.dto.MonthBudgetDetailDto
     * @author huojia
     * @date 2022/11/1 11:33
     **/
    public MonthBudgetDetailEntity buildDetail(MonthBudgetEntity entity,
                                               BigDecimal amount,
                                               BigDecimal beforeAmount,
                                               String operationType,
                                               String businessCode) {
        MonthBudgetDetailEntity detailEntity = this.nebulaToolkitService.copyObjectByWhiteList(
                entity, MonthBudgetDetailEntity.class, LinkedHashSet.class, ArrayList.class
        );
        detailEntity.setId(null);
        detailEntity.setBusinessCode(businessCode);
        detailEntity.setOperationType(operationType);
        detailEntity.setInitialAmount(entity.getInitResolveAmount());
        //调整后
        detailEntity.setBalanceAmount(entity.getAccumulatedAvailableBalance());
        detailEntity.setBeforeAmount(beforeAmount);
        detailEntity.setCurOperationAmount(amount);
        return detailEntity;
    }

    /**
     * 判断是取计划量还是回复量
     *
     * @param budgetCalConfigVo
     * @param planFlag
     * @return {@link BudgetCalConfigDataVo}
     */
    public BudgetCalConfigDataVo findMonthBudgetCalConfigData(BudgetCalConfigVo budgetCalConfigVo, String planFlag) {
        List<BudgetCalConfigDataVo> calConfigDataVos = budgetCalConfigVo.getDataList().stream()
                .filter(budgetCalConfigDataVo -> CalDataFromEnum.SALES_PLAN.getCode().equals(budgetCalConfigDataVo.getCalDataFromCode()))
                .collect(Collectors.toList());
        if (BooleanEnum.TRUE.getCapital().equals(planFlag)) {
            log.info("获取计算配置数据配置,calConfigDataVos => :{}", calConfigDataVos);
            // 只取计划量
            return calConfigDataVos.stream()
                    .filter(data -> SalesPlanAmountTypeEnum.PLAN_AMOUNT.getCode().equals(data.getAmountTypeCode())
                            || SalesPlanAmountTypeEnum.DISCOUNT_PLAN_AMOUNT.getCode().equals(data.getAmountTypeCode())
                            || SalesPlanAmountTypeEnum.PLAN_QUANTITY.getCode().equals(data.getAmountTypeCode()))
                    .findFirst().orElse(null);
        } else {
            // 只取回复量
            return budgetCalConfigVo.getDataList().stream()
                    .filter(data -> SalesPlanAmountTypeEnum.RESTORE_AMOUNT.getCode().equals(data.getAmountTypeCode())
                            || SalesPlanAmountTypeEnum.DISCOUNT_RESTORE_AMOUNT.getCode().equals(data.getAmountTypeCode())
                            || SalesPlanAmountTypeEnum.RESTORE_QUANTITY.getCode().equals(data.getAmountTypeCode()))
                    .findFirst().orElse(null);
        }
    }

    /**
     * 根据预算项目匹配管控维度，未匹配在往上级预算项目匹配
     *
     * @param dimensionControlsVos 同纬度的预算管控配置
     * @param budgetItemCode       预算项目编码
     * @return {@link DimensionControlsVo}
     */
    public DimensionControlsVo matchDimensionControl(List<DimensionControlsVo> dimensionControlsVos, String budgetItemCode) {
        for (DimensionControlsVo vo : dimensionControlsVos) {
            if (!CollectionUtils.isEmpty(vo.getBudgetItemCodeList())
                    && vo.getBudgetItemCodeList().contains(budgetItemCode)) {
                return vo;
            }
        }
        //本级未查管控配置，继续找上级
        BudgetItemVo itemVo = budgetItemService.findByCode(budgetItemCode, EnableStatusEnum.ENABLE.getCode());
        if (null == itemVo || StringUtil.isBlank(itemVo.getParentBudgetItemCode())) {
            return null;
        }
        for (DimensionControlsVo vo : dimensionControlsVos) {
            if (!CollectionUtils.isEmpty(vo.getBudgetItemCodeList())
                    && vo.getBudgetItemCodeList().contains(itemVo.getParentBudgetItemCode())) {
                return vo;
            }
        }
        //本级未查管控配置，继续找上级
        BudgetItemVo itemVo2 = budgetItemService.findByCode(itemVo.getParentBudgetItemCode(), EnableStatusEnum.ENABLE.getCode());
        if (null == itemVo2 || StringUtil.isBlank(itemVo2.getParentBudgetItemCode())) {
            return null;
        }
        for (DimensionControlsVo vo : dimensionControlsVos) {
            if (!CollectionUtils.isEmpty(vo.getBudgetItemCodeList())
                    && vo.getBudgetItemCodeList().contains(itemVo2.getParentBudgetItemCode())) {
                return vo;
            }
        }
        return null;
    }
}