package com.biz.crm.tpm.business.month.budget.local.repository;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.local.entity.TenantFlagOpEntity;
import com.biz.crm.business.common.local.entity.UuidFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetDetailEntity;
import com.biz.crm.tpm.business.month.budget.local.mapper.MonthBudgetDetailMapper;
import com.biz.crm.tpm.business.month.budget.sdk.eunm.BudgetOperationTypeEnum;
import com.biz.crm.tpm.business.month.budget.sdk.eunm.YesOrNoEnum;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetDetailVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;

/**
 * @author huojia
 * @date 2022年10月27日 21:06
 */
@Component
public class MonthBudgetDetailRepository extends ServiceImpl<MonthBudgetDetailMapper, MonthBudgetDetailEntity> {

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 根据预算编码，查询预算明细
     *
     * @param monthBudgetCode
     * @return java.util.List<com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetDetailEntity>
     * @author huojia
     * @date 2022/10/27 21:13
     **/
    public List<MonthBudgetDetailEntity> listDetailByBudgetCode(String monthBudgetCode) {
        return this.lambdaQuery()
                .eq(MonthBudgetDetailEntity::getMonthBudgetCode, monthBudgetCode)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 根据月度预算对应年度预算编码，批量删除操作明细
     *
     * @param yearBudgetCodeList
     * @return java.util.List<com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetDetailEntity>
     * @author huojia
     * @date 2022/11/5 17:11
     **/
    public List<MonthBudgetDetailEntity> listDetailByYearBudgetCodeList(List<String> yearBudgetCodeList) {
        if (CollectionUtils.isEmpty(yearBudgetCodeList)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .in(MonthBudgetDetailEntity::getYearBudgetCode, yearBudgetCodeList)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 根据流程编码查询预算明细
     *
     * @param processNo
     * @return java.util.List<com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetDetailEntity>
     * @author huojia
     * @date 2022/11/14 9:59
     **/
    public List<MonthBudgetDetailEntity> listByProcessNo(String processNo) {
        if (StringUtils.isEmpty(processNo)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .eq(MonthBudgetDetailEntity::getProcessNo, processNo)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 根据业务编码查询调整明细（计算实销回复差使用）
     * @param businessCode
     * @return {@link MonthBudgetDetailVo}
     */
    public MonthBudgetDetailVo findByBusinessCode(String businessCode) {
        LambdaQueryWrapper<MonthBudgetDetailEntity> queryWrapper = new LambdaQueryWrapper<MonthBudgetDetailEntity>();
        queryWrapper.eq(MonthBudgetDetailEntity::getBusinessCode, businessCode)
            .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
            .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
            .orderByDesc(MonthBudgetDetailEntity::getCreateTime).last(" limit 1");
        MonthBudgetDetailEntity entity = baseMapper.selectOne(queryWrapper);
        if(ObjectUtils.isEmpty(entity)){
            return null;
        }
        return this.nebulaToolkitService.copyObjectByWhiteList(
            entity, MonthBudgetDetailVo.class, LinkedHashSet.class, ArrayList.class);
    }

    /**
     * 根据业务编码查询调整明细
     * @param businessCodes
     * @return
     */
    public List<MonthBudgetDetailVo> findByBusinessCodes(List<String> businessCodes) {
        LambdaQueryWrapper<MonthBudgetDetailEntity> queryWrapper = new LambdaQueryWrapper<MonthBudgetDetailEntity>();
        queryWrapper.in(MonthBudgetDetailEntity::getBusinessCode, businessCodes)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .orderByDesc(MonthBudgetDetailEntity::getCreateTime).last(" limit 1");
        List<MonthBudgetDetailEntity> entityList = baseMapper.selectList(queryWrapper);
        if(ObjectUtils.isEmpty(entityList)){
            return null;
        }
        return (List<MonthBudgetDetailVo>)this.nebulaToolkitService.copyCollectionByWhiteList(
                entityList, MonthBudgetDetailEntity.class, MonthBudgetDetailVo.class, LinkedHashSet.class, ArrayList.class
        );
    }

    /**
     * 根据月度预算Ids查询明细
     * @param ids
     * @return {@link List}<{@link MonthBudgetDetailVo}>
     */
    public List<MonthBudgetDetailVo> findByMainIds(List<String> ids) {
        return this.baseMapper.findByMainIds(ids, BudgetOperationTypeEnum.INIT.getCode());
    }

    public List<MonthBudgetDetailEntity> findListByBusiness(List<String> auditDetailCodes, String auditCode) {
        if(CollectionUtils.isEmpty(auditDetailCodes)){
            return Lists.newArrayList();
        }
        List<MonthBudgetDetailEntity> list = this.lambdaQuery().in(MonthBudgetDetailEntity::getBusinessCode, auditDetailCodes)
                .eq(MonthBudgetDetailEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(MonthBudgetDetailEntity::getTenantCode, TenantUtils.getTenantCode())
                .isNotNull(MonthBudgetDetailEntity::getPlanReturnBudgetYearAndMonth)
                .isNotNull(MonthBudgetDetailEntity::getAlreadyEndCaseAmount)
                .eq(MonthBudgetDetailEntity::getOperationType, BudgetOperationTypeEnum.AUDIT_USE.getCode())
                .and(wrapper->wrapper.ne(MonthBudgetDetailEntity::getReject, YesOrNoEnum.YES.getCode()).or().isNull(MonthBudgetDetailEntity::getReject))
                .list();
        if(CollectionUtils.isEmpty(list)){
            list = this.lambdaQuery().in(MonthBudgetDetailEntity::getAuditCode, auditCode)
                    .eq(MonthBudgetDetailEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                    .eq(MonthBudgetDetailEntity::getTenantCode, TenantUtils.getTenantCode())
                    .isNotNull(MonthBudgetDetailEntity::getPlanReturnBudgetYearAndMonth)
                    .isNotNull(MonthBudgetDetailEntity::getAlreadyEndCaseAmount)
                    .eq(MonthBudgetDetailEntity::getOperationType, BudgetOperationTypeEnum.AUDIT_USE.getCode())
                    .and(wrapper->wrapper.ne(MonthBudgetDetailEntity::getReject, YesOrNoEnum.YES.getCode()).or().isNull(MonthBudgetDetailEntity::getReject))
                    .list();
        }
        return list;
    }

    public void updateStatusByBusiness(List<MonthBudgetDetailVo> monthBudgetDetailVos) {
        if(CollectionUtils.isEmpty(monthBudgetDetailVos)){
            return;
        }
        this.getBaseMapper().updateStatusByBusiness(monthBudgetDetailVos);
    }

    public List<MonthBudgetDetailVo> findListByDetailPlanItemCodes(List<String> detailPlanItemCodes) {
        if(CollectionUtils.isEmpty(detailPlanItemCodes)){
            return Lists.newArrayList();
        }
        List<MonthBudgetDetailEntity> list = this.lambdaQuery().in(MonthBudgetDetailEntity::getBusinessCode, detailPlanItemCodes)
                .eq(MonthBudgetDetailEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(MonthBudgetDetailEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(MonthBudgetDetailEntity::getOperationType, BudgetOperationTypeEnum.FORECAST_OVER.getCode())
                .list();
        if(CollectionUtils.isEmpty(list)){
            return Lists.newArrayList();
        }

        Collection<MonthBudgetDetailVo> monthBudgetDetailVos = this.nebulaToolkitService.copyCollectionByWhiteList(list, MonthBudgetDetailEntity.class, MonthBudgetDetailVo.class, LinkedHashSet.class, ArrayList.class);
        return (List<MonthBudgetDetailVo>) monthBudgetDetailVos;
    }
}
