package com.biz.crm.tpm.business.month.budget.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.local.entity.TenantFlagOpEntity;
import com.biz.crm.business.common.local.entity.UuidEntity;
import com.biz.crm.business.common.local.entity.UuidFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity;
import com.biz.crm.tpm.business.month.budget.local.mapper.MonthBudgetMapper;
import com.biz.crm.tpm.business.month.budget.sdk.dto.MonthBudgetDto;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author huojia
 * @date 2022年10月27日 20:47
 */
@Component
public class MonthBudgetRepository extends ServiceImpl<MonthBudgetMapper, MonthBudgetEntity> {

    /**
     * 根据主键和删除标记查询月度预算
     *
     * @param id
     * @param delFlag
     * @return com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity
     * @author huojia
     * @date 2022/10/27 20:49
     **/
    public MonthBudgetEntity getById(String id, String delFlag) {
        if (StringUtils.isEmpty(id)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(UuidEntity::getId, id)
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(StringUtils.isNotEmpty(delFlag), UuidFlagOpEntity::getDelFlag, delFlag)
                .one();
    }

    /**
     * 根据月度预算编码、启禁用标记查询数据
     *
     * @param code
     * @param enableStatus
     * @return com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity
     * @author huojia
     * @date 2022/11/1 10:57
     **/
    public MonthBudgetEntity getByMonthBudgetCode(String code, String enableStatus) {
        if (StringUtils.isEmpty(code)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(MonthBudgetEntity::getMonthBudgetCode, code)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(StringUtils.isNotEmpty(enableStatus), UuidFlagOpEntity::getEnableStatus, enableStatus)
                .eq(MonthBudgetEntity::getTenantCode, TenantUtils.getTenantCode())
                .one();
    }

    public List<MonthBudgetEntity> getByMonthBudgetCodes(List<String> codes, String enableStatus) {
        if (CollectionUtils.isEmpty(codes)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .in(MonthBudgetEntity::getMonthBudgetCode, codes)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(StringUtils.isNotEmpty(enableStatus), UuidFlagOpEntity::getEnableStatus, enableStatus)
                .eq(MonthBudgetEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 根据预算项目编码批量查询预算
     *
     * @param budgetItemCodeList
     * @return java.util.List<com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity>
     * @author huojia
     * @date 2022/11/1 20:43
     **/
    public List<MonthBudgetEntity> listByBudgetItemCodeList(List<String> budgetItemCodeList) {
        if (CollectionUtils.isEmpty(budgetItemCodeList)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .in(MonthBudgetEntity::getBudgetItemCode, budgetItemCodeList)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(MonthBudgetEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 根据年度预算编码逻辑删除
     *
     * @param yearMonthCodeList
     * @author huojia
     * @date 2022/11/3 14:47
     **/
    public void delByYearBudgetCode(List<String> yearMonthCodeList) {
        List<MonthBudgetEntity> monthBudgetList = this.lambdaQuery()
                .in(MonthBudgetEntity::getYearBudgetCode, yearMonthCodeList)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(MonthBudgetEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
        if (!CollectionUtils.isEmpty(monthBudgetList)) {
            monthBudgetList.forEach(monthBudgetEntity -> monthBudgetEntity.setDelFlag(DelFlagStatusEnum.DELETE.getCode()));
            this.updateBatchById(monthBudgetList);
        }
    }

    /**
     * 查询
     *
     * @param yearBudgetCode
     * @return java.util.List<com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity>
     * @author huojia
     * @date 2022/11/3 21:14
     **/
    public List<MonthBudgetEntity> listByYearBudgetCode(String yearBudgetCode) {
        if (StringUtils.isEmpty(yearBudgetCode)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .eq(MonthBudgetEntity::getYearBudgetCode, yearBudgetCode)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(MonthBudgetEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 根据ids查询
     *
     * @param ids
     * @param delFlag
     * @return java.util.List<com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity>
     * @author huojia
     * @date 2022/11/5 20:14
     **/
    public List<MonthBudgetEntity> listByIds(List<String> ids, String delFlag) {
        if (CollectionUtils.isEmpty(ids)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .in(UuidEntity::getId, ids)
                .eq(StringUtils.isNotEmpty(delFlag), UuidFlagOpEntity::getDelFlag, delFlag)
                .eq(MonthBudgetEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 根据年月批量查询月度预算
     *
     * @param yearMonth
     * @return java.util.List<com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity>
     * @author huojia
     * @date 2022/11/16 17:30
     **/
    public List<MonthBudgetEntity> listByYearMonth(String yearMonth, List<String> businessUnitCodeList) {
        if (StringUtils.isEmpty(yearMonth)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .eq(MonthBudgetEntity::getYearMonthLy, yearMonth)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .in(!CollectionUtils.isEmpty(businessUnitCodeList), MonthBudgetEntity::getBusinessUnitCode, businessUnitCodeList)
                .list();
    }

    /**
     * 根据年度预算编码，加月份查询月度预算
     *
     * @param yearBudgetList
     * @param yearMonthList
     * @return java.util.List<com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity>
     * @author huojia
     * @date 2022/11/17 10:23
     **/
    public List<MonthBudgetEntity> getByYearMonthCodeAndMonth(List<String> yearBudgetList, List<String> yearMonthList) {
        if (CollectionUtils.isEmpty(yearBudgetList)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .in(MonthBudgetEntity::getYearBudgetCode, yearBudgetList)
                .in(!CollectionUtils.isEmpty(yearMonthList), MonthBudgetEntity::getYearMonthLy, yearMonthList)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(MonthBudgetEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 条件查询月度预算
     *
     * @param monthBudgetDto
     * @return java.util.List<com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity>
     * @author huojia
     * @date 2022/11/22 14:49
     **/
    public List<MonthBudgetEntity> listByConditions(MonthBudgetDto monthBudgetDto) {
        return this.lambdaQuery()
                .eq(!StringUtils.isEmpty(monthBudgetDto.getYearMonthLy()), MonthBudgetEntity::getYearMonthLy, monthBudgetDto.getYearMonthLy())
                .eq(!StringUtils.isEmpty(monthBudgetDto.getFeeBelongCode()), MonthBudgetEntity::getFeeBelongCode, monthBudgetDto.getFeeBelongCode())
                .eq(!StringUtils.isEmpty(monthBudgetDto.getBusinessFormatCode()), MonthBudgetEntity::getBusinessFormatCode, monthBudgetDto.getBusinessFormatCode())
                .eq(!StringUtils.isEmpty(monthBudgetDto.getBusinessUnitCode()), MonthBudgetEntity::getBusinessUnitCode, monthBudgetDto.getBusinessUnitCode())
                .in(!CollectionUtils.isEmpty(monthBudgetDto.getMonthBudgetCodeList()), MonthBudgetEntity::getMonthBudgetCode, monthBudgetDto.getMonthBudgetCodeList())
                .in(!CollectionUtils.isEmpty(monthBudgetDto.getBudgetItemCodeList()), MonthBudgetEntity::getBudgetItemCode, monthBudgetDto.getBudgetItemCodeList())
                .eq(MonthBudgetEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(MonthBudgetEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 批量查询
     *
     * @param codes
     * @return java.util.List<com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity>
     * @author huojia
     * @date 2022/12/20 16:42
     **/
    public List<MonthBudgetEntity> listByCodes(List<String> codes) {
        return this.listByCodes(codes, null);
    }

    /**
     * 批量查询
     *
     * @param codes
     * @return java.util.List<com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity>
     * @author huojia
     * @date 2022/12/20 16:42
     **/
    public List<MonthBudgetEntity> listByCodes(List<String> codes, String businessUnitCode) {
        if (CollectionUtils.isEmpty(codes)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .in(MonthBudgetEntity::getMonthBudgetCode, codes)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(StringUtils.isNotEmpty(businessUnitCode), MonthBudgetEntity::getBusinessUnitCode, businessUnitCode)
                .eq(MonthBudgetEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 根据年度预算编码批量查询
     *
     * @param yearBudgetList
     * @return java.util.List<com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity>
     * @author huojia
     * @date 2023/1/8 17:45
     **/
    public List<MonthBudgetEntity> listByYearBudgetCodes(List<String> yearBudgetList) {
        if (CollectionUtils.isEmpty(yearBudgetList)) {
            return null;
        }
        return this.lambdaQuery()
                .in(MonthBudgetEntity::getYearBudgetCode, yearBudgetList)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .list();
    }

    /**
     * 根据ids集合修改预算管控可用余额
     *
     * @param ids                  主键集合
     * @param controlBalanceAmount 预算管控可用余额
     * @date 2023/1/8 17:36
     **/
    public void updateControlBalanceAmountByIds(Set<String> ids, BigDecimal controlBalanceAmount) {
        if (CollectionUtils.isEmpty(ids) && Objects.nonNull(controlBalanceAmount)) {
            this.lambdaUpdate()
                    .in(MonthBudgetEntity::getId, ids)
                    .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                    .set(MonthBudgetEntity::getControlBalanceAmount, controlBalanceAmount)
                    .update();
        }
    }

    public List<MonthBudgetVo> findCustomerGroupMonthBudget(List<MonthBudgetDto> dtos) {

        return this.getBaseMapper().findCustomerGroupMonthBudget(dtos, TenantUtils.getTenantCode());
    }

    /**
     * 查询当前计算年月的月度计划
     *
     * @param dto
     * @return {@link MonthBudgetVo}
     */
    public MonthBudgetVo findCurrMonthBudgetById(MonthBudgetDto dto) {
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        dto.setTenantCode(TenantUtils.getTenantCode());
        return this.baseMapper.findCurrMonthBudgetById(dto);
    }

    public MonthBudgetEntity getOneByYearBudgetCodeAndMonth(String yearBudgetCode, String yearMonth) {
        if (StringUtils.isEmpty(yearBudgetCode)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(MonthBudgetEntity::getYearBudgetCode, yearBudgetCode)
                .eq(MonthBudgetEntity::getYearMonthLy, yearMonth)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .one();
    }

    public List<MonthBudgetEntity> getListByYearMonth(String yearBudgetCode, List<String> yearMonths) {
        if (StringUtils.isEmpty(yearBudgetCode)) {
            return Lists.newArrayList();
        }
        if (CollectionUtils.isEmpty(yearMonths)) {
            return Lists.newArrayList();
        }

        return this.lambdaQuery()
                .eq(MonthBudgetEntity::getYearBudgetCode, yearBudgetCode)
                .in(MonthBudgetEntity::getYearMonthLy, yearMonths)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 查询上月月度预算
     *
     * @param lastYearMonth
     * @param yearBudgetCode
     * @return {@link MonthBudgetEntity}
     */
    public MonthBudgetEntity findByYearMonthAndYearBudgetCode(String lastYearMonth, String yearBudgetCode) {
        List<MonthBudgetEntity> list = this.lambdaQuery().eq(MonthBudgetEntity::getYearBudgetCode, yearBudgetCode)
                .eq(MonthBudgetEntity::getYearMonthLy, lastYearMonth)
                .eq(MonthBudgetEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(MonthBudgetEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
        if (CollectionUtils.isEmpty(list)) {
            return null;
        }
        return list.get(0);
    }

    /**
     * 删除
     *
     * @param ids
     */
    public void deleteBatch(List<String> ids) {
        this.lambdaUpdate().in(MonthBudgetEntity::getId, ids)
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .set(MonthBudgetEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                .update();
    }

    public Page<MonthBudgetVo> findPageForOut(Pageable pageable, MonthBudgetDto dto) {
        Page<MonthBudgetVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.baseMapper.findPageForOut(page, dto);
    }

    public BigDecimal sumFirstReplyAmount(MonthBudgetDto dto) {
        return this.baseMapper.sumFirstReplyAmount(dto, TenantUtils.getTenantCode());
    }

    public BigDecimal sumFirstReplyAmountStudent(MonthBudgetDto dto) {
        return this.baseMapper.sumFirstReplyAmountStudent(dto, TenantUtils.getTenantCode());
    }

    public BigDecimal sumFirstReplyAmountOthers(MonthBudgetDto dto) {
        return this.baseMapper.sumFirstReplyAmountOthers(dto, TenantUtils.getTenantCode());
    }

    public BigDecimal sumActualSales(MonthBudgetDto dto) {
        return this.baseMapper.sumActualSales(dto, TenantUtils.getTenantCode());
    }

    public BigDecimal sumActualSalesStudent(MonthBudgetDto dto) {
        return this.baseMapper.sumActualSalesStudent(dto, TenantUtils.getTenantCode());
    }

    public BigDecimal sumActualSalesOthers(MonthBudgetDto dto) {
        return this.baseMapper.sumActualSalesOthers(dto, TenantUtils.getTenantCode());
    }

    public Page<MonthBudgetVo> findByConditions(Page<MonthBudgetVo> page, MonthBudgetDto dto) {

        return this.getBaseMapper().findByConditions(page, dto);
    }

    /**
     * 根据主键和删除标记查询月度预算
     *
     * @param ids
     * @param delFlag
     * @return com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity
     * @author cyj
     * @date 2023/07/19 20:49
     **/
    public List<MonthBudgetEntity> getByIds(List<String> ids, String delFlag) {
        if (CollectionUtils.isEmpty(ids)) {
            return null;
        }
        return this.lambdaQuery()
                .in(UuidEntity::getId, ids)
                .eq(TenantFlagOpEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(StringUtils.isNotEmpty(delFlag), UuidFlagOpEntity::getDelFlag, delFlag)
                .list();
    }

    public List<MonthBudgetVo> findMonthBudgetForVariable(MonthBudgetDto dto) {
        return this.baseMapper.findMonthBudgetForVariable(dto, TenantUtils.getTenantCode());
    }

    public List<MonthBudgetVo> findMonthBudgetConditionForVariable(MonthBudgetDto monthBudgetDto) {
        return this.baseMapper.findMonthBudgetConditionForVariable(monthBudgetDto);
    }

    public MonthBudgetVo findMonthBudgetSumAuditAmount(MonthBudgetDto monthBudgetDto) {
        monthBudgetDto.setTenantCode(TenantUtils.getTenantCode());
        return this.baseMapper.findMonthBudgetSumAuditAmount(monthBudgetDto);
    }

    public MonthBudgetVo findMonthBudgetSumInitResolveAmount(MonthBudgetDto monthBudgetDto) {
        monthBudgetDto.setTenantCode(TenantUtils.getTenantCode());
        return this.baseMapper.findMonthBudgetSumInitResolveAmount(monthBudgetDto);
    }

    /**
     * 预算总使用金额：按照年月+区域+系统汇总批复金额-批复结案差
     *
     * @param monthBudgetDtos
     * @return
     */
    public List<MonthBudgetVo> findListForFR(List<MonthBudgetDto> monthBudgetDtos) {
        if (CollectionUtils.isEmpty(monthBudgetDtos)) {
            return Collections.emptyList();
        }
        monthBudgetDtos = monthBudgetDtos.stream().filter(Objects::nonNull).distinct().collect(Collectors.toList());
        if (CollectionUtils.isEmpty(monthBudgetDtos)) {
            return Collections.emptyList();
        }
        return this.baseMapper.findListForFR(monthBudgetDtos, TenantUtils.getTenantCode());
    }
}
