package com.biz.crm.tpm.business.month.budget.local.repository;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.local.entity.UuidEntity;
import com.biz.crm.business.common.local.entity.UuidFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.tpm.business.month.budget.local.entity.SubComMonthBudgetEntity;
import com.biz.crm.tpm.business.month.budget.local.mapper.SubComMonthBudgetMapper;
import com.biz.crm.tpm.business.month.budget.sdk.dto.SubComMonthBudgetDto;
import com.biz.crm.tpm.business.month.budget.sdk.vo.SubComMonthBudgetVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @author huojia
 * @date 2022年10月31日 22:24
 */
@Component
public class SubComMonthBudgetRepository extends ServiceImpl<SubComMonthBudgetMapper, SubComMonthBudgetEntity> {

    /**
     * 根据id查询
     *
     * @param id
     * @param delFlag
     * @return com.biz.crm.tpm.business.month.budget.local.entity.SubComMonthBudgetEntity
     * @author huojia
     * @date 2022/10/31 22:27
     **/
    public SubComMonthBudgetEntity getById(String id, String delFlag) {
        if (StringUtils.isEmpty(id)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(UuidEntity::getId, id)
                .eq(StringUtils.isNotEmpty(delFlag), UuidFlagOpEntity::getDelFlag, delFlag)
                .one();
    }

    /**
     * 根据分子公司月度预算编码、启禁用查询分子公司月度预算
     *
     * @param monthBudgetCode
     * @param enableStatus
     * @return com.biz.crm.tpm.business.month.budget.local.entity.SubComMonthBudgetEntity
     * @author huojia
     * @date 2022/11/2 11:08
     **/
    public SubComMonthBudgetEntity getByMonthBudgetCode(String monthBudgetCode, String enableStatus) {
        if (StringUtils.isEmpty(monthBudgetCode)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(SubComMonthBudgetEntity::getMonthBudgetCode, monthBudgetCode)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(StringUtils.isNotEmpty(enableStatus), UuidFlagOpEntity::getEnableStatus, enableStatus)
                .one();
    }

    /**
     * 根据预算项目编码查询关联分子公司月度预算
     *
     * @param budgetItemCodeList
     * @return java.util.List<com.biz.crm.tpm.business.month.budget.local.entity.SubComMonthBudgetEntity>
     * @author huojia
     * @date 2022/11/12 11:57
     **/
    public List<SubComMonthBudgetEntity> listByBudgetItemCodeList(List<String> budgetItemCodeList) {
        if (CollectionUtils.isEmpty(budgetItemCodeList)) {
            return null;
        }
        return this.lambdaQuery()
                .in(SubComMonthBudgetEntity::getBudgetItemCode, budgetItemCodeList)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 批量查询
     *
     * @param ids
     * @param delFlag
     * @return java.util.List<com.biz.crm.tpm.business.month.budget.local.entity.SubComMonthBudgetEntity>
     * @author huojia
     * @date 2022/11/23 20:07
     **/
    public List<SubComMonthBudgetEntity> listByIds(List<String> ids, String delFlag) {
        if (CollectionUtils.isEmpty(ids)) {
            return null;
        }
        return this.lambdaQuery()
                .in(UuidEntity::getId, ids)
                .eq(UuidFlagOpEntity::getDelFlag, delFlag)
                .list();
    }

    public List<SubComMonthBudgetEntity> listByCodes(List<String> monthBudgetCodeList) {
        if (CollectionUtils.isEmpty(monthBudgetCodeList)) {
            return null;
        }
        return this.lambdaQuery()
                .in(SubComMonthBudgetEntity::getMonthBudgetCode, monthBudgetCodeList)
                .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 重构批量修改方法
     *
     * @param entitys    实体类集合
     * @param tenantCode 租户编码
     */
    public void updateBatchByIdAndTenantCode(Collection<SubComMonthBudgetEntity> entitys, String tenantCode) {
        if (CollectionUtils.isEmpty(entitys)) {
            return;
        }
        if (StringUtils.isEmpty(tenantCode)) {
            return;
        }
        entitys.forEach(item -> {
            if (StringUtils.isNotBlank(item.getId())) {
                LambdaUpdateWrapper<SubComMonthBudgetEntity> lambdaUpdateWrapper = Wrappers.lambdaUpdate();
                lambdaUpdateWrapper.eq(SubComMonthBudgetEntity::getTenantCode, tenantCode);
                lambdaUpdateWrapper.eq(SubComMonthBudgetEntity::getId, item.getId());
                this.baseMapper.update(item, lambdaUpdateWrapper);
            }
        });
    }

    /**
     * 删除数据
     *
     * @param ids
     */
    public void deleteById(List<String> ids) {
        if (CollectionUtil.isEmpty(ids)) {
            return;
        }
        this.lambdaUpdate()
                .set(SubComMonthBudgetEntity::getDelFlag,DelFlagStatusEnum.DELETE.getCode())
                .in(SubComMonthBudgetEntity::getId,ids)
                .eq(SubComMonthBudgetEntity::getTenantCode,TenantUtils.getTenantCode())
                .update();
    }

    /**
     *
     * @param dto
     * @return
     */
    public List<SubComMonthBudgetEntity> findListByCondition(SubComMonthBudgetDto dto) {
        if(Objects.isNull(dto)){
            return Lists.newArrayList();
        }
        List<SubComMonthBudgetEntity> list = this.lambdaQuery()
                .eq(StringUtils.isNotEmpty(dto.getBusinessFormatCode()), SubComMonthBudgetEntity::getBusinessFormatCode, dto.getBusinessFormatCode())
                .eq(StringUtils.isNotEmpty(dto.getBusinessUnitCode()),SubComMonthBudgetEntity::getBusinessUnitCode,dto.getBusinessUnitCode())
                .eq(StringUtils.isNotEmpty(dto.getFeeSourceCode()),SubComMonthBudgetEntity::getFeeSourceCode,dto.getFeeSourceCode())
                .eq(StringUtils.isNotEmpty(dto.getBudgetItemCode()),SubComMonthBudgetEntity::getBudgetItemCode,dto.getBudgetItemCode())
                .eq(StringUtils.isNotEmpty(dto.getYearMonthLy()),SubComMonthBudgetEntity::getYearMonthLy,dto.getYearMonthLy())
                .eq(StringUtils.isNotEmpty(dto.getSalesInstitutionCode()),SubComMonthBudgetEntity::getSalesInstitutionCode,dto.getSalesInstitutionCode())
                .eq(StringUtils.isNotEmpty(dto.getSalesRegionCode()),SubComMonthBudgetEntity::getSalesRegionCode,dto.getSalesRegionCode())
                .eq(StringUtils.isNotEmpty(dto.getSalesOrgCode()),SubComMonthBudgetEntity::getSalesOrgCode,dto.getSalesOrgCode())
                .eq(StringUtils.isNotEmpty(dto.getSalesOrgCode()),SubComMonthBudgetEntity::getSalesOrgCode,dto.getSalesOrgCode())
                .eq(StringUtils.isNotEmpty(dto.getOrgCode()),SubComMonthBudgetEntity::getOrgCode,dto.getOrgCode())
                .eq(StringUtils.isNotEmpty(dto.getChannelCode()),SubComMonthBudgetEntity::getChannelCode,dto.getChannelCode())
                .eq(StringUtils.isNotEmpty(dto.getCustomerCode()),SubComMonthBudgetEntity::getCustomerCode,dto.getCustomerCode())
                .eq(StringUtils.isNotEmpty(dto.getTerminalCode()),SubComMonthBudgetEntity::getTerminalCode,dto.getTerminalCode())
                .eq(StringUtils.isNotEmpty(dto.getProductBrandCode()),SubComMonthBudgetEntity::getProductBrandCode,dto.getProductBrandCode())
                .eq(StringUtils.isNotEmpty(dto.getProductCategoryCode()),SubComMonthBudgetEntity::getProductCategoryCode,dto.getProductCategoryCode())
                .eq(StringUtils.isNotEmpty(dto.getProductItemCode()),SubComMonthBudgetEntity::getProductItemCode,dto.getProductItemCode())
                .eq(StringUtils.isNotEmpty(dto.getProductCode()),SubComMonthBudgetEntity::getProductCode,dto.getProductCode())
                .eq(SubComMonthBudgetEntity::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
                .list();
        return list;
    }

    /**
     * 根据分子预算预测更新自动生成的分子月度预算
     *
     * @param onlyKeys 维度唯一拼接编码集合
     * @return List<SubComMonthBudgetEntity>
     */
    public List<SubComMonthBudgetEntity> listByOnlyKeys(List<String> onlyKeys) {
        if(CollectionUtils.isEmpty(onlyKeys)){
            return Lists.newArrayList();
        }
        return this.baseMapper.listByOnlyKeys(onlyKeys,TenantUtils.getTenantCode());
    }

    public List<SubComMonthBudgetEntity> findByIds(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return Lists.newArrayList();
        }
        return this.lambdaQuery()
                .eq(SubComMonthBudgetEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(SubComMonthBudgetEntity::getTenantCode, TenantUtils.getTenantCode())
                .in(SubComMonthBudgetEntity::getId, ids)
                .list();
    }

    public Page<SubComMonthBudgetVo> findBySonCompanyBudgetWarn(Pageable pageable, List<String> yearMonthList, String orgCode, String saleCompanyCode, String channelCode, String saleGroupCode, String customerCode, String budgetItemCode) {
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(1, 50));
        Page<SubComMonthBudgetVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        if(CollectionUtils.isEmpty(yearMonthList)){
            return page;
        }
        return this.baseMapper.findBySonCompanyBudgetWarn(page,yearMonthList,orgCode,saleCompanyCode,channelCode,saleGroupCode,customerCode,budgetItemCode);
    }
}
