package com.biz.crm.tpm.business.month.budget.local.service.imports;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.excel.util.Validate;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.retailer.sdk.vo.CustomerRetailerVo;
import com.biz.crm.mdm.business.product.brand.sdk.vo.ProductBrandVo;
import com.biz.crm.mdm.business.product.level.sdk.vo.ProductLevelVo;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.biz.crm.mn.common.base.eunm.BusinessFormatEnum;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo;
import com.biz.crm.tpm.business.month.budget.sdk.dto.MonthBudgetDto;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetService;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetImportsHeadVo;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetImportsVo;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import liquibase.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Description 月度预算导入
 * @Author YangWei
 * @Date 2023/2/2 下午4:17
 */
@Slf4j
@Component
public class MonthBudgetImportsHeadProcess implements ImportProcess<MonthBudgetImportsHeadVo> {

    @Autowired(required = false)
    @Qualifier("nebulaToolkitService")
    private NebulaToolkitService nebulaToolkitService;

    @Resource
    private MonthBudgetService monthBudgetService;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Map<Integer, String> execute(LinkedHashMap<Integer, MonthBudgetImportsHeadVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        try {
            Validate.notNull(data, "导入数据为空");
            Map<Integer, String> errorMap = new HashMap<>();
            Set<String> codeSet = new HashSet<>();
            //校验编码
            data.forEach((k, v) -> {
                if (StringUtils.isBlank(v.getMonthBudgetCode())) {
                    errorMap.put(k, "月度预算编码为空！");
                } else {
                    codeSet.add(v.getMonthBudgetCode());
                }
            });
            Map<String, MonthBudgetVo> voMap = new HashMap<>(codeSet.size());
            if (CollectionUtil.isNotEmpty(codeSet)) {
                //获取月度预算
                List<MonthBudgetVo> voList = monthBudgetService.findBudgetByMonthBudgetCodes(new ArrayList<>(codeSet));
                if (!CollectionUtils.isEmpty(voList)) {
                    voMap.putAll(voList.stream().collect(Collectors.toMap(MonthBudgetVo::getMonthBudgetCode, Function.identity(), (n, o) -> n)));
                }
            }
            String tenantCode = TenantUtils.getTenantCode();
            //校验月度预算数据
            List<MonthBudgetDto> dtoList = new ArrayList<>();
            data.forEach((k, v) -> {
                List<String> errorMsgList = new ArrayList<>();
                MonthBudgetDto dto = new MonthBudgetDto();
                if (StringUtil.isNotEmpty(v.getCalAmountStr())) {
                    try {
                        dto.setCalAmount(new BigDecimal(v.getCalAmountStr()));
                    } catch (Exception e) {
                        errorMsgList.add("计划量/回复量有误,应该为数字类型!");
                    }
                }
                if (StringUtil.isNotEmpty(v.getFirstReplyAmountStr())) {
                    try {
                        dto.setFirstReplyAmount(new BigDecimal(v.getFirstReplyAmountStr()));
                    } catch (Exception e) {
                        errorMsgList.add("月度分解金额有误,应该为数字类型!");
                    }
                }
                if (StringUtil.isNotEmpty(v.getFirstReplyResolveDiffAmountStr())) {
                    try {
                        dto.setFirstReplyResolveDiffAmount(new BigDecimal(v.getFirstReplyResolveDiffAmountStr()));
                    } catch (Exception e) {
                        errorMsgList.add("月月度分解金额与年初分解金额的差异有误,应该为数字类型!");
                    }
                }
                if (StringUtil.isNotEmpty(v.getLastMonthRollingAmountStr())) {
                    try {
                        dto.setLastMonthRollingAmount(new BigDecimal(v.getLastMonthRollingAmountStr()));
                    } catch (Exception e) {
                        errorMsgList.add("上月滚动金额有误,应该为数字类型!");
                    }
                }
                if (StringUtil.isNotEmpty(v.getActualSalesStr())) {
                    try {
                        dto.setActualSales(new BigDecimal(v.getActualSalesStr()));
                    } catch (Exception e) {
                        errorMsgList.add("实销金额有误,应该为数字类型!");
                    }
                }

                if (StringUtils.isNotEmpty(v.getMonthBudgetCode())) {
                    MonthBudgetVo vo = voMap.get(v.getMonthBudgetCode());
                    if (Objects.isNull(vo)) {
                        errorMsgList.add("月度预算[" + v.getMonthBudgetCode() + "]不存在!");
                    } else {
                        if (!BusinessUnitEnum.isDefaultBusinessUnit(vo.getBusinessUnitCode())) {
                            errorMsgList.add("月度预算业务单元不属于主体或小业务单元!");
                        }
                        if (!BusinessFormatEnum.NORMAL.getCode().equals(vo.getBusinessFormatCode())) {
                            errorMsgList.add("月度预算业态不属于常温！");
                        }
                        dto.setId(vo.getId());
                        dto.setMonthBudgetCode(vo.getMonthBudgetCode());
                    }
                }
                if (CollectionUtil.isNotEmpty(errorMsgList)) {
                    errorMap.put(k, String.join(" | ", errorMsgList));
                } else {
                    dto.setTenantCode(tenantCode);
                    dtoList.add(dto);
                }

            });
            //有错误信息直接返回
            if (!errorMap.isEmpty()) {
                return errorMap;
            }
            //更新并执行滚动
            if (CollectionUtil.isNotEmpty(dtoList)) {
                this.monthBudgetService.updateAndRoll(dtoList);
            }
        } catch (Exception e) {
            log.error("", e);
            throw new IllegalArgumentException(e);
        }
        return null;
    }

    @Override
    public Integer getBatchCount() {
        //一次处理1000条
        return 1000;
    }

    @Override
    public Class<MonthBudgetImportsHeadVo> findCrmExcelVoClass() {
        return MonthBudgetImportsHeadVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "TPM_MONTH_BUDGET_HEAD_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "TPM-月度预算-主体更新导入";
    }

    @Override
    public String getBusinessCode() {
        return "TPM_MONTH_BUDGET_IMPORT";
    }

    @Override
    public String getBusinessName() {
        return "TPM-月度预算导入";
    }

}