package com.biz.crm.tpm.business.month.budget.local.service.imports;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.retailer.sdk.service.CustomerRetailerVoService;
import com.biz.crm.mdm.business.customer.retailer.sdk.vo.CustomerRetailerVo;
import com.biz.crm.mdm.business.customer.sdk.dto.CustomerDto;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictDataVoService;
import com.biz.crm.mdm.business.dictionary.sdk.vo.DictDataVo;
import com.biz.crm.mdm.business.org.sdk.service.OrgRegionVoService;
import com.biz.crm.mdm.business.product.brand.sdk.service.ProductBrandService;
import com.biz.crm.mdm.business.product.brand.sdk.vo.ProductBrandVo;
import com.biz.crm.mdm.business.product.level.sdk.service.ProductLevelVoSdkService;
import com.biz.crm.mdm.business.product.level.sdk.vo.ProductLevelVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.ProductVo;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.util.ExceptionStackMsgUtil;
import com.biz.crm.tpm.business.budget.item.sdk.service.BudgetItemService;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo;
import com.biz.crm.tpm.business.month.budget.sdk.dto.MonthBudgetDto;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetService;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetImportsVo;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetVo;
import com.biz.crm.tpm.business.sales.goal.sdk.constant.DictTypeCodeConstant;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.annotations.Comment;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Description 月度预算导入
 * @Author YangWei
 * @Date 2023/2/2 下午4:17
 */
@Slf4j
@Component
public class MonthBudgetImportsProcess implements ImportProcess<MonthBudgetImportsVo> {

    @Autowired(required = false)
    @Qualifier("nebulaToolkitService")
    private NebulaToolkitService nebulaToolkitService;

    @Resource
    private DictDataVoService dictDataVoService;

    @Resource
    private MonthBudgetService monthBudgetService;

    @Resource
    private BudgetItemService budgetItemService;

    @Resource
    private CustomerRetailerVoService customerRetailerVoService;

    @Resource
    private TerminalVoService terminalVoService;

    @Resource
    private ProductBrandService productBrandService;

    @Resource
    private ProductLevelVoSdkService productLevelVoSdkService;

    @Resource
    private ProductVoService productVoService;

    /**
     * 数据字典
     */
    private Map<String, List<DictDataVo>> dictMap = Maps.newHashMap();

    /**
     * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
     * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
     *
     * @return
     */
    @Override
    public boolean importBeforeValidationFlag() {
        return Boolean.TRUE;
    }

    /**
     * 数据校验
     *
     * @param data     待处理的数据集合，k-流水号，v-excel解析后的对象
     * @param paramsVo 任务公共参数
     * @param params   导入任务自定义参数
     * @return k-对应data的k，v-对应data的k对应的v处理异常描述信息，会回写到错误文件
     */
    @Override
    public Map<Integer, String> tryVerify(LinkedHashMap<Integer, MonthBudgetImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();
        for (Map.Entry<Integer, MonthBudgetImportsVo> row : data.entrySet()) {
            int rowNum = row.getKey();
            MonthBudgetImportsVo vo = row.getValue();
            this.validateIsTrue(StringUtils.hasText(vo.getBusinessFormatCode()), "业态不能为空");
            this.validateIsTrue(StringUtils.hasText(vo.getBusinessUnitCode()),"业务单元不能为空");
            this.validateIsTrue(StringUtils.hasText(vo.getYearMonthLy()),"年月不能为空");
            this.validateIsTrue(StringUtils.hasText(vo.getBudgetItemCode()),"预算项目编码不能为空");
            this.validateIsTrue(StringUtils.hasText(vo.getFeeBelongCode()),"费用归口不能为空");
            this.validateIsTrue(vo.getAmount() != null,"费用金额不能为空");
            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(rowNum, errInfo);
            }
        }
        return errMap;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Map<Integer, String> execute(LinkedHashMap<Integer, MonthBudgetImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        final Optional<MonthBudgetImportsVo> first = data.values().stream().findFirst();
        if (!first.isPresent()) {
            return null;
        }
        Map<Integer, String> errMap = new HashMap<>();
        try {
            //初始数据字典
            this.initDict();
            List<MonthBudgetDto> list = new ArrayList<>();
            //校验
            data.forEach((k, v) -> {
                list.add(validate(v));
                String errInfo = this.validateGetErrorInfo();
                if (errInfo != null) {
                    errMap.put(k, errInfo);
                }
            });
            if (errMap.isEmpty()) {
                //保存
                list.forEach(e -> this.monthBudgetService.importSave(e));
            }
        } catch (IllegalArgumentException e) {
            throw e;
        } catch (Exception e) {
            String error = ExceptionStackMsgUtil.stackMsg(e);
            log.error("月度预算导入异常:{}", error);
            throw new IllegalArgumentException("导入异常[" + error + "]");
        }
        return errMap;
    }

    /**
     * 初始数据字典
     * @param
     */
    private void initDict() {
        this.dictMap = this.dictDataVoService.findByDictTypeCodeList(
            Lists.newArrayList(DictTypeCodeConstant.MDM_BUSINESS_FORMAT
                , DictTypeCodeConstant.MDM_BUSINESS_UNIT, DictTypeCodeConstant.TPM_FEE_BELONG_VERTICAL
                , DictTypeCodeConstant.MDM_CUSTOMIZE_ORG));
    }

    /**
     * 校验
     * @param vo
     * @return {@link MonthBudgetDto}
     */
    private MonthBudgetDto validate(MonthBudgetImportsVo vo) {
        //校验数据字典
        this.verifyDictCode(vo);

        //年月格式
        SimpleDateFormat yearMonthSF = new SimpleDateFormat("yyyy-MM");
        yearMonthSF.setLenient(false);
        try {
            yearMonthSF.parse(vo.getYearMonthLy());
        } catch (ParseException e) {
            this.validateIsTrue(false, "年月格式错误");
        }
        //预算项目编码校验
        try {
            BudgetItemVo budgetItem = this.budgetItemService.findByCode(vo.getBudgetItemCode(), EnableStatusEnum.ENABLE.getCode());
            Assert.notNull(budgetItem, "未查询到预算项目");
            vo.setBudgetItemName(budgetItem.getBudgetItemName());
            vo.setBudgetItemLevelCode(budgetItem.getBudgetItemLevelCode());
        }catch (Exception e){
            this.validateIsTrue(false, "查询预算项目错误[" + e.getMessage() + "]");
        }
        //零售商编码
        if (!org.apache.commons.lang3.StringUtils.isEmpty(vo.getSystemCode())) {
            CustomerRetailerVo customerRetailer = this.customerRetailerVoService.findByCode(vo.getSystemCode());
            if (customerRetailer != null) {
                vo.setSystemName(customerRetailer.getCustomerRetailerName());
            } else {
                this.validateIsTrue(false, "未查询到零售商信息");
            }
        }
        //门店
        String terminalCode = vo.getTerminalCode();
        if(StringUtils.hasText(terminalCode)){
            List<TerminalVo> terminalList = this.terminalVoService.findDetailsByIdsOrTerminalCodes(null, Lists.newArrayList(terminalCode));
            if (!CollectionUtils.isEmpty(terminalList)) {
                TerminalVo terminalVo = terminalList.get(0);
                vo.setTerminalName(terminalVo.getTerminalName());
            } else {
                this.validateIsTrue(false, "未查询到门店信息");
            }
        }
        //品牌编码
        String productBrandCode = vo.getProductBrandCode();
        if(StringUtils.hasText(productBrandCode)){
            List<ProductBrandVo> productBrandList = this.productBrandService.listByCodes(Lists.newArrayList(productBrandCode));
            if (!CollectionUtils.isEmpty(productBrandList)) {
                ProductBrandVo productBrandVo = productBrandList.get(0);
                vo.setProductBrandName(productBrandVo.getProductBrandName());
            } else {
                this.validateIsTrue(false, "为查询到品牌信息");
            }
        }
        //品类
        String productCategoryCode = vo.getProductCategoryCode();
        if(StringUtils.hasText(productCategoryCode)){
            List<ProductLevelVo> productCategoryList = this.productLevelVoSdkService.findListByCodes(Lists.newArrayList(productCategoryCode));
            if (!CollectionUtils.isEmpty(productCategoryList)) {
                ProductLevelVo productCategoryVo = productCategoryList.get(0);
                vo.setProductCategoryName(productCategoryVo.getProductLevelName());
            } else {
                this.validateIsTrue(false, "未查询到品类信息");
            }
        }
        //品项
        String productItemCode = vo.getProductItemCode();
        if(StringUtils.hasText(productItemCode)){
            List<ProductLevelVo> productItemList = this.productLevelVoSdkService.findListByCodes(Lists.newArrayList(productItemCode));
            if (!CollectionUtils.isEmpty(productItemList)) {
                ProductLevelVo productItemVo = productItemList.get(0);
                vo.setProductItemName(productItemVo.getProductLevelName());
            } else {
                this.validateIsTrue(false, "未查询到品项信息");
            }
        }
        //产品
        String productCode = vo.getProductCode();
        if(StringUtils.hasText(productCode)){
            List<ProductVo> productList = this.productVoService.findByCodes(Lists.newArrayList(productCode));
            if (!CollectionUtils.isEmpty(productList)) {
                ProductVo productVo = productList.get(0);
                vo.setProductName(productVo.getProductName());
            } else {
                this.validateIsTrue(false, "未查询到产品信息");
            }
        }
        //金额
        this.validateIsTrue(vo.getAmount().compareTo(BigDecimal.ZERO) == 1, "费用金额必须大于零");
        //唯一校验
        try {
            this.verifyUniqueness(vo);
        } catch (Exception e) {
            this.validateIsTrue(false, "唯一校验异常[" + e.getMessage() + "]");
        }

        //数据拷贝
        MonthBudgetDto dto =
            this.nebulaToolkitService.copyObjectByBlankList(
                vo, MonthBudgetDto.class, HashSet.class, ArrayList.class);
        dto.setAccumulatedAvailableBalance(vo.getAmount());
        dto.setInitResolveAmount(vo.getAmount());
        dto.setFirstReplyAmount(vo.getAmount());
        dto.setAfterFreezeAmount(vo.getAmount());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        dto.setTenantCode(TenantUtils.getTenantCode());
        return dto;
    }

    /**
     * 唯一校验
     * @param vo
     */
    private void verifyUniqueness(MonthBudgetImportsVo vo) {
        /**
         * * 业态	* 业务单元	* 年月	* 预算项目编码	预算项目名称	* 费用归口	* 零售商编码	零售商名称
         * * 区域编码	区域名称	门店编码	门店名称	品牌编码	品牌名称	品类编码	品类名称	品项编码	品项名称	产品编码	产品名称	* 费用金额
         */
        //根据必填项查询出已有的月度预算
        MonthBudgetDto dto = new MonthBudgetDto();
        dto.setBusinessFormatCode(vo.getBusinessFormatCode());
        dto.setBusinessUnitCode(vo.getBusinessUnitCode());
        dto.setYearMonthLy(vo.getYearMonthLy());
        dto.setBudgetItemCode(vo.getBudgetItemCode());
        dto.setFeeBelongCode(vo.getFeeBelongCode());
        dto.setSystemCode(vo.getSystemCode());
        List<MonthBudgetVo> monthBudgetVoList = this.monthBudgetService.findListByConditions(dto);
        if(CollectionUtils.isEmpty(monthBudgetVoList)){
            return;
        }
        //唯一key
        String uniqKey = this.buildKey(vo.getRegionCode(),vo.getTerminalCode()
            ,vo.getProductBrandCode(),vo.getProductCategoryCode(),vo.getProductItemCode(),vo.getProductCode());
        //分组
        Map<String, MonthBudgetVo> monthBudgetMap = monthBudgetVoList.stream()
            .collect(Collectors.toMap(o -> this.buildKey(o.getRegionCode(),
                o.getTerminalCode(), o.getProductBrandCode(), o.getProductCategoryCode(),
                o.getProductItemCode(), o.getProductCode()), Function.identity()));
        MonthBudgetVo monthBudgetVo = monthBudgetMap.get(uniqKey);
        Assert.isNull(monthBudgetVo, "已经存在相同月度预算");
    }

    /**
     * 唯一key
     * @param regionCode
     * @param terminalCode
     * @param productBrandCode
     * @param productCategoryCode
     * @param productItemCode
     * @param productCode
     * @return {@link String}
     */
    private String buildKey(String regionCode, String terminalCode, String productBrandCode
        , String productCategoryCode, String productItemCode, String productCode) {
        StringBuilder sb = new StringBuilder(this.handleStr(regionCode));
        sb.append(this.handleStr(terminalCode));
        sb.append(this.handleStr(productBrandCode));
        sb.append(this.handleStr(productCategoryCode));
        sb.append(this.handleStr(productItemCode));
        sb.append(this.handleStr(productCode));
        return sb.toString();
    }

    private String handleStr(String regionCode) {
        return StringUtils.hasText(regionCode) ? regionCode : "0";
    }

    /**
     * 校验数据字典
     * @param vo
     */
    private void verifyDictCode(MonthBudgetImportsVo vo) {
        //业态
        vo.setBusinessFormatCode(this.findDictValue(vo.getBusinessFormatCode(), DictTypeCodeConstant.MDM_BUSINESS_FORMAT));
        this.validateIsTrue(StringUtils.hasText(vo.getBusinessFormatCode()), "未配置该业态");
        //业务单元
        vo.setBusinessUnitCode(this.findDictValue(vo.getBusinessUnitCode(), DictTypeCodeConstant.MDM_BUSINESS_UNIT));
        this.validateIsTrue(StringUtils.hasText(vo.getBusinessUnitCode()), "未配置该业务单元");
        //费用归口
        vo.setFeeBelongCode(this.findDictValue(vo.getFeeBelongCode(), DictTypeCodeConstant.TPM_FEE_BELONG_VERTICAL));
        this.validateIsTrue(StringUtils.hasText(vo.getFeeBelongCode()), "未配置该费用归口");
        //区域
        if (!org.apache.commons.lang3.StringUtils.isEmpty(vo.getRegionCode())) {
            vo.setRegionCode(this.findDictCode(vo.getRegionCode(), DictTypeCodeConstant.MDM_CUSTOMIZE_ORG));
            this.validateIsTrue(StringUtils.hasText(vo.getRegionCode()), "未配置该区域");
        }
    }

    /**
     * 获取字典对应的value值
     *
     * @param dictValue
     * @param typeCode
     * @return
     */
    private String findDictValue(String dictValue, String typeCode) {
        DictDataVo dictDataVo = this.dictMap.get(typeCode).stream()
            .filter(a -> a.getDictValue().equals(dictValue)).findFirst().orElse(null);
        if (!ObjectUtils.isEmpty(dictDataVo)) {
            return dictDataVo.getDictCode();
        }
        return null;
    }

    /**
     * 获取字典对应的code值
     *
     * @param dictCode
     * @param typeCode
     * @return
     */
    private String findDictCode(String dictCode, String typeCode) {
        DictDataVo dictDataVo = this.dictMap.get(typeCode).stream()
            .filter(a -> a.getDictCode().equals(dictCode)).findFirst().orElse(null);
        if (!ObjectUtils.isEmpty(dictDataVo)) {
            return dictDataVo.getDictCode();
        }
        return null;
    }

    @Override
    public Integer getBatchCount() {
        return Integer.MAX_VALUE;
    }

    @Override
    public Class<MonthBudgetImportsVo> findCrmExcelVoClass() {
        return MonthBudgetImportsVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "TPM_MONTH_BUDGET_VERTICAL_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "TPM-月度预算-垂直导入";
    }

    @Override
    public String getBusinessCode() {
        return "TPM_MONTH_BUDGET_IMPORT";
    }

    @Override
    public String getBusinessName() {
        return "TPM-月度预算导入";
    }

}