package com.biz.crm.tpm.business.month.budget.local.service.imports;

import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.vo.TaskGlobalParamsVo;
import com.biz.crm.mdm.business.customer.channel.sdk.service.CustomerChannelVoService;
import com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo;
import com.biz.crm.mdm.business.customer.retailer.sdk.service.CustomerRetailerVoService;
import com.biz.crm.mdm.business.customer.retailer.sdk.vo.CustomerRetailerVo;
import com.biz.crm.mdm.business.dictionary.sdk.service.DictToolkitService;
import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import com.biz.crm.mdm.business.product.brand.sdk.service.ProductBrandService;
import com.biz.crm.mdm.business.product.level.sdk.service.ProductLevelVoSdkService;
import com.biz.crm.mdm.business.product.level.sdk.vo.ProductLevelVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.sales.org.sdk.service.SalesOrgVoService;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.biz.crm.mn.common.base.eunm.BusinessUnitEnum;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.tpm.business.budget.item.sdk.service.BudgetItemService;
import com.biz.crm.tpm.business.budget.item.sdk.vo.BudgetItemVo;
import com.biz.crm.tpm.business.month.budget.sdk.constant.SubComMonthBudgetConstant;
import com.biz.crm.tpm.business.month.budget.sdk.dto.SubComMonthBudgetDto;
import com.biz.crm.tpm.business.month.budget.sdk.service.SubComMonthBudgetService;
import com.biz.crm.tpm.business.month.budget.sdk.vo.SubComMonthBudgetImportsVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Description 月度预算导入
 * @Author YangWei
 * @Date 2023/2/2 下午4:17
 */
@Slf4j
@Component
public class SubComMonthBudgetImportsProcess implements ImportProcess<SubComMonthBudgetImportsVo> {

    @Autowired(required = false)
    @Qualifier("nebulaToolkitService")
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private DictToolkitService dictToolkitService;

    @Autowired(required = false)
    private OrgVoService orgVoService;
    @Autowired(required = false)
    private SalesOrgVoService salesOrgVoService;
    @Autowired(required = false)
    private CustomerChannelVoService customerChannelVoService;
    @Autowired(required = false)
    private BudgetItemService budgetItemService;
    @Autowired(required = false)
    private CustomerRetailerVoService customerRetailerVoService;
    @Autowired(required = false)
    private TerminalVoService terminalVoService;
    @Autowired(required = false)
    private ProductBrandService productBrandService;
    @Autowired(required = false)
    private ProductLevelVoSdkService productLevelVoSdkService;
    @Autowired(required = false)
    private ProductVoService productVoService;
    @Autowired(required = false)
    private SubComMonthBudgetService subComMonthBudgetService;

    /**
     * 数据分片
     *
     * @return 分片长度
     */
    @Override
    public Integer getBatchCount() {
        return 50001;
    }

    /**
     * 是否开启先校验后导入的模式 默认false: false执行旧逻辑 true执行新逻辑(先校验再保存) <br/>
     * 新逻辑需同时实现接口 tryVerify tryConfirm（默认方法调用execute）
     *
     * @return
     */
    @Override
    public boolean importBeforeValidationFlag() {
        return Boolean.TRUE;
    }

    @Override
    public Map<Integer, String> execute(LinkedHashMap<Integer, SubComMonthBudgetImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Validate.notNull(data, "导入数据不能为空！");
        Validate.isTrue(data.values().size() <= 50000, "单次导入数据不能超过50000条");
        Map<Integer, String> errMap = new HashMap<>();
        //大批量保存
        try {
            log.info("开始数据校验-----------------");
            List<SubComMonthBudgetDto> importList = this.validate(data, errMap);
            if (errMap.isEmpty()) {
                log.info("开始执行大批量保存-----------------");
                subComMonthBudgetService.bulkImportSave(importList);
            }
        } catch (Exception e) {
            log.error("导入执行失败,异常{}", e.getMessage());
            throw new IllegalArgumentException(e);
        }
        log.info("导入逻辑处理完毕-----------------");
        return errMap;
    }


    @Override
    public Map<Integer, String> tryVerify(LinkedHashMap<Integer, SubComMonthBudgetImportsVo> data, TaskGlobalParamsVo paramsVo, Map<String, Object> params) {
        Map<Integer, String> errMap = new HashMap<>();

        for (Map.Entry<Integer, SubComMonthBudgetImportsVo> row : data.entrySet()) {
            int rowNum = row.getKey();
            this.validateIsTrue(StringUtils.isNotBlank(row.getValue().getBusinessFormatCode()), "业态不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(row.getValue().getBusinessUnitCode()), "业务单元不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(row.getValue().getYearMonthLy()), "年月不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(row.getValue().getGroupCode()), "分组不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(row.getValue().getOrgCode()), "组织编码不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(row.getValue().getSalesInstitutionCode()), "销售机构编码不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(row.getValue().getChannelCode()), "渠道编码不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(row.getValue().getBudgetItemCode()), "预算项目编码不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(row.getValue().getFeeSourceCode()), "费用来源不能为空");
            this.validateIsTrue(StringUtils.isNotBlank(row.getValue().getBudgetAmountStr()), "预算金额不能为空");
            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(rowNum, errInfo);
            }
        }

        return errMap;
    }

    /**
     * 校验数据
     *
     * @param data data
     **/
    private List<SubComMonthBudgetDto> validate(LinkedHashMap<Integer, SubComMonthBudgetImportsVo> data, Map<Integer, String> errMap) {
        List<SubComMonthBudgetDto> list = new ArrayList<>();
        //获取业态字典
        Map<String, String> formatMap = dictToolkitService.findConvertMapByDictTypeCode(SubComMonthBudgetConstant.DICT_MDM_BUSINESS_FORMAT);
        //获取分组
        Map<String, String> groupMap = dictToolkitService.findConvertMapByDictTypeCode(SubComMonthBudgetConstant.DICT_TPM_GROUP_CODE);
        //获取费用来源
        Map<String, String> feeSourceMap = dictToolkitService.findConvertMapByDictTypeCode(SubComMonthBudgetConstant.DICT_TPM_FEE_SOURCE);

        data.forEach((k, v) -> list.add(this.validateData(k, v, formatMap, feeSourceMap, groupMap, errMap)));

        if (!errMap.isEmpty()) {
            return list;
        }

        //组织编码
        Set<String> orgSet = new HashSet<>();
        //销售组织编码
        Set<String> salesOrgSet = new HashSet<>();
        //渠道编码
        Set<String> channelSet = new HashSet<>();
        //预算项目编码
        Set<String> budgetSet = new HashSet<>();
        //系统编码
        Set<String> sysSet = new HashSet<>();
        //门店编码
        Set<String> terSet = new HashSet<>();
        //品牌编码
        Set<String> brandSet = new HashSet<>();
        //品类、品项编码
        Set<String> itemSet = new HashSet<>();
        //产品编码
        Set<String> proSet = new HashSet<>();
        for (SubComMonthBudgetDto dto : list) {
            orgSet.add(dto.getOrgCode());
            String insCode = dto.getChannelCode() + dto.getBusinessFormatCode() + dto.getSalesInstitutionCode();
            salesOrgSet.add(insCode);
            if (StringUtils.isNotBlank(dto.getSalesRegionCode())) {
                String regionCode = insCode + dto.getSalesRegionCode();
                salesOrgSet.add(regionCode);
                if (StringUtils.isNotBlank(dto.getSalesOrgCode())) {
                    String salesOrgCode = regionCode + dto.getSalesOrgCode();
                    salesOrgSet.add(salesOrgCode);
                }
            }

            if (StringUtils.isNotBlank(dto.getChannelCode())) {
                channelSet.add(dto.getChannelCode());
            }
            budgetSet.add(dto.getBudgetItemCode());
            if (StringUtils.isNotBlank(dto.getSystemCode())) {
                sysSet.add(dto.getSystemCode());
            }
            if (StringUtils.isNotBlank(dto.getCustomerCode())) {
                terSet.add(dto.getCustomerCode());
            }
            if (StringUtils.isNotBlank(dto.getTerminalCode())) {
                terSet.add(dto.getTerminalCode());
            }
            if (StringUtils.isNotBlank(dto.getProductBrandCode())) {
                brandSet.add(dto.getProductBrandCode());
            }
            if (StringUtils.isNotBlank(dto.getProductCategoryCode())) {
                itemSet.add(dto.getProductCategoryCode());
            }
            if (StringUtils.isNotBlank(dto.getProductItemCode())) {
                itemSet.add(dto.getProductItemCode());
            }
            if (StringUtils.isNotBlank(dto.getProductCode())) {
                proSet.add(dto.getProductCode());
            }
        }
        //获取组织数据
        Map<String, OrgVo> orgMap = new HashMap<>();
        List<OrgVo> orgLsit = orgVoService.findByOrgCodes(new ArrayList<>(orgSet));
        if (!CollectionUtils.isEmpty(orgLsit)) {
            orgMap = orgLsit.stream().collect(Collectors.toMap(OrgVo::getOrgCode, Function.identity()));
        }
        //获取销售组织数据
        Map<String, String> salesOrgMap = new HashMap<>();
        if (!salesOrgSet.isEmpty()) {
            List<List<String>> sales = Lists.partition(new ArrayList<>(salesOrgSet), 500);
            salesOrgMap = salesOrgVoService.findNameBySalesOrgCodes(sales);
        }
        //获取渠道数据
        Map<String, String> channelMap = new HashMap<>();
        if (!channelSet.isEmpty()) {
            List<CustomerChannelVo> channels = customerChannelVoService.findByCodes(new ArrayList<>(channelSet));
            if (!CollectionUtils.isEmpty(channels)) {
                channelMap = channels.stream().collect(Collectors.toMap(CustomerChannelVo::getCustomerChannelCode, CustomerChannelVo::getCustomerChannelName));
            }
        }
        //获取预算项目数据
        Map<String, BudgetItemVo> budgetMap = new HashMap<>();
        if (!budgetSet.isEmpty()) {
            List<BudgetItemVo> budgets = budgetItemService.listByCodes(new ArrayList<>(budgetSet));
            if (!CollectionUtils.isEmpty(budgets)) {
                budgetMap = budgets.stream().collect(Collectors.toMap(BudgetItemVo::getBudgetItemCode, Function.identity()));
            }
        }
        //获取系统数据
        Map<String, String> sysMap = new HashMap<>();
        if (!sysSet.isEmpty()) {
            List<CustomerRetailerVo> syss = customerRetailerVoService.findByCodes(new ArrayList<>(sysSet));
            if (!CollectionUtils.isEmpty(syss)) {
                sysMap = syss.stream().collect(Collectors.toMap(CustomerRetailerVo::getCustomerRetailerCode, CustomerRetailerVo::getCustomerRetailerName));
            }
        }
        //获取门店数据
        Map<String, String> terMap = new HashMap<>();
        if (!terSet.isEmpty()) {
            List<TerminalVo> ters = terminalVoService.findBaseByTerminalCodes(new ArrayList<>(terSet));
            if (!CollectionUtils.isEmpty(ters)) {
                terMap = ters.stream().collect(Collectors.toMap(TerminalVo::getTerminalCode, TerminalVo::getTerminalName));
            }
        }
        //获取品牌数据
        Map<String, String> brandMap = new HashMap<>();
        if (!brandSet.isEmpty()) {
            brandMap = productBrandService.findNameByCodeList(new ArrayList<>(brandSet));
        }
        //获取品类、品项数据
        Map<String, String> itemMap = new HashMap<>();
        if (!itemSet.isEmpty()) {
            List<ProductLevelVo> levels = productLevelVoSdkService.findListByCodes(new ArrayList<>(itemSet));
            if (!CollectionUtils.isEmpty(levels)) {
                itemMap = levels.stream().collect(Collectors.toMap(ProductLevelVo::getProductLevelCode, ProductLevelVo::getProductLevelName, (oldValue, newValue) -> newValue));
            }
        }
        //获取产品数据
        Map<String, String> proMap = new HashMap<>();
        if (!proSet.isEmpty()) {
            List<List<String>> pros = Lists.partition(new ArrayList<>(proSet), 500);
            proMap = productVoService.getProductByCodes(pros);
        }
        //组装数据
        for (SubComMonthBudgetDto dto : list) {
            if (orgMap.containsKey(dto.getOrgCode())) {
                OrgVo o = orgMap.get(dto.getOrgCode());
                dto.setOrgName(o.getOrgName());
                dto.setOrgLevelCode(o.getOrgLevel());
            } else {
                this.validateIsTrue(false, "通过组织编码为查找到组织名称");
            }
            String insCode = dto.getChannelCode() + dto.getBusinessFormatCode() + dto.getSalesInstitutionCode();
            if (salesOrgMap.containsKey(insCode)) {
                dto.setSalesInstitutionName(salesOrgMap.get(insCode));
                dto.setSalesInstitutionErpCode(dto.getSalesInstitutionCode());
                dto.setSalesInstitutionCode(insCode);
            } else {
                this.validateIsTrue(false, "通过销售机构编码为查找到销售机构名称");
            }
            if (StringUtils.isNotBlank(dto.getSalesRegionCode())) {
                String regionCode = insCode + dto.getSalesRegionCode();
                if (salesOrgMap.containsKey(regionCode)) {
                    dto.setSalesRegionName(salesOrgMap.get(regionCode));
                    dto.setSalesRegionErpCode(dto.getSalesRegionCode());
                    dto.setSalesRegionCode(regionCode);
                    if (StringUtils.isNotBlank(dto.getSalesOrgCode())) {
                        String salesOrgCode = regionCode + dto.getSalesOrgCode();
                        if (salesOrgMap.containsKey(salesOrgCode)) {
                            dto.setSalesOrgName(salesOrgMap.get(salesOrgCode));
                            dto.setSalesOrgErpCode(dto.getSalesOrgCode());
                            dto.setSalesOrgCode(salesOrgCode);
                        } else {
                            this.validateIsTrue(false, "通过销售组编码为查找到销售组名称");
                        }
                    }
                } else {
                    this.validateIsTrue(false, "通过销售部门编码为查找到销售部门名称");
                }
            }
            if (StringUtils.isNotBlank(dto.getChannelCode())) {
                if (channelMap.containsKey(dto.getChannelCode())) {
                    dto.setChannelName(channelMap.get(dto.getChannelCode()));
                } else {
                    this.validateIsTrue(false, "通过渠道编码为查找到渠道名称");
                }
            }
            if (budgetMap.containsKey(dto.getBudgetItemCode())) {
                BudgetItemVo itemVo = budgetMap.get(dto.getBudgetItemCode());
                dto.setBudgetItemName(itemVo.getBudgetItemName());
                dto.setBudgetItemLevelCode(itemVo.getBudgetItemLevelCode());
            } else {
                this.validateIsTrue(false, "通过预算项目编码为查找到预算项目名称");
            }
            if (StringUtils.isNotBlank(dto.getSystemCode())) {
                if (sysMap.containsKey(dto.getSystemCode())) {
                    dto.setSystemName(sysMap.get(dto.getSystemCode()));
                } else {
                    this.validateIsTrue(false, "通过系统编码为查找到系统名称");
                }
            }
            //注：此处查询的客户也是门店
            if (StringUtils.isNotBlank(dto.getCustomerCode())) {
                if (terMap.containsKey(dto.getCustomerCode())) {
                    dto.setCustomerName(terMap.get(dto.getCustomerCode()));
                } else {
                    this.validateIsTrue(false, "通过客户编码为查找到客户名称");
                }
            }
            if (StringUtils.isNotBlank(dto.getTerminalCode())) {
                if (terMap.containsKey(dto.getTerminalCode())) {
                    dto.setTerminalName(terMap.get(dto.getTerminalCode()));
                } else {
                    this.validateIsTrue(false, "通过门店编码为查找到门店名称");
                }
            }
            if (StringUtils.isNotBlank(dto.getProductBrandCode())) {
                if (brandMap.containsKey(dto.getProductBrandCode())) {
                    dto.setProductBrandName(brandMap.get(dto.getProductBrandCode()));
                } else {
                    this.validateIsTrue(false, "通过品牌编码为查找到品牌名称");
                }
            }
            if (StringUtils.isNotBlank(dto.getProductCategoryCode())) {
                if (itemMap.containsKey(dto.getProductCategoryCode())) {
                    dto.setProductCategoryName(itemMap.get(dto.getProductCategoryCode()));
                } else {
                    this.validateIsTrue(false, "通过品类编码为查找到品类名称");
                }
            }
            if (StringUtils.isNotBlank(dto.getProductItemCode())) {
                if (itemMap.containsKey(dto.getProductItemCode())) {
                    dto.setProductItemName(itemMap.get(dto.getProductItemCode()));
                } else {
                    this.validateIsTrue(false, "通过品项编码为查找到品项名称");
                }
            }
            if (StringUtils.isNotBlank(dto.getProductCode())) {
                if (proMap.containsKey(dto.getProductCode())) {
                    dto.setProductName(proMap.get(dto.getProductCode()));
                } else {
                    this.validateIsTrue(false, "通过产品编码为查找到产品名称");
                }
            }
            String errInfo = this.validateGetErrorInfo();
            if (errInfo != null) {
                errMap.put(dto.getLineIndex(), errInfo);
            }
        }
        return new ArrayList<>(list);
    }


    /**
     * 校验数据
     *
     * @param index 行号
     * @param dto   行数据
     **/
    private SubComMonthBudgetDto validateData(Integer index, SubComMonthBudgetImportsVo dto, Map<String, String> formatMap,
                                              Map<String, String> feeSourceMap, Map<String, String> groupMap, Map<Integer, String> errMap) {
        SubComMonthBudgetDto budget = new SubComMonthBudgetDto();

        budget = this.nebulaToolkitService.copyObjectByBlankList(dto, SubComMonthBudgetDto.class, null, null);
        if (formatMap.containsKey(dto.getBusinessFormatCode())) {
            budget.setBusinessFormatCode(formatMap.get(dto.getBusinessFormatCode()));
        } else {
            this.validateIsTrue(false, "业态未能识别");
        }
        if (BusinessUnitEnum.SON_COMPANY.getDesc().equals(dto.getBusinessUnitCode())) {
            budget.setBusinessUnitCode(BusinessUnitEnum.SON_COMPANY.getCode());
        } else {
            this.validateIsTrue(false, "业务单元只能填写分子公司");
        }
        if (groupMap.containsKey(dto.getGroupCode())) {
            budget.setGroupCode(feeSourceMap.get(dto.getGroupCode()));
        } else {
            this.validateIsTrue(false, "分组未能识别");
        }
        if (feeSourceMap.containsKey(dto.getFeeSourceCode())) {
            budget.setFeeSourceCode(feeSourceMap.get(dto.getFeeSourceCode()));
        } else {
            this.validateIsTrue(false, "费用来源未能识别");
        }
        try {
            budget.setYearMonthLy(
                    DateUtil.dateToStr(DateUtil.date_yyyy_MM.parse(dto.getYearMonthLy()), DateUtil.date_yyyy_MM));
        } catch (Exception e) {
            this.validateIsTrue(false, "年月格式错误【yyyy-MM】");
        }
        try {
            budget.setBudgetAmount(new BigDecimal(dto.getBudgetAmountStr()));
        } catch (Exception e) {
            this.validateIsTrue(false, "预算金额格式错误【100000.111】");
        }
        budget.setLineIndex(index);

        String errInfo = this.validateGetErrorInfo();
        if (errInfo != null) {
            errMap.put(index, errInfo);
        }
        return budget;
    }

    @Override
    public Class<SubComMonthBudgetImportsVo> findCrmExcelVoClass() {
        return SubComMonthBudgetImportsVo.class;
    }

    @Override
    public String getTemplateCode() {
        return "TPM_SUB_COM_MONTH_BUDGET_IMPORT";
    }

    @Override
    public String getTemplateName() {
        return "TPM-分子月度预算导入";
    }

//    @Override
//    public String getBusinessCode() {
//        return "TPM_SUB_COM_MONTH_BUDGET_IMPORT";
//    }
//
//    @Override
//    public String getBusinessName() {
//        return "TPM-分子月度预算导入";
//    }

}