package com.biz.crm.tpm.business.month.budget.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mn.common.base.service.RedisLockService;
import com.biz.crm.mn.common.rocketmq.service.RocketMqProducer;
import com.biz.crm.mn.common.rocketmq.vo.MqMessageVo;
import com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity;
import com.biz.crm.tpm.business.month.budget.local.repository.MonthBudgetRepository;
import com.biz.crm.tpm.business.month.budget.local.service.MonthBudgetActualReplayService;
import com.biz.crm.tpm.business.month.budget.sdk.constant.MonthBudgetConstant;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetLockService;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetService;
import com.biz.crm.tpm.business.month.budget.sdk.vo.MonthBudgetActualSalesVo;
import com.google.common.collect.Lists;
import liquibase.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.StopWatch;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 执行实销量与回复量差额
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2023-12-13 23:32
 */
@Slf4j
@Service
public class MonthBudgetActualReplayServiceImpl implements MonthBudgetActualReplayService {


    @Autowired(required = false)
    private RedisLockService redisLockService;

    @Autowired(required = false)
    private RocketMqProducer rocketMqProducer;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Autowired(required = false)
    private MonthBudgetRepository monthBudgetRepository;

    @Autowired(required = false)
    private MonthBudgetLockService monthBudgetLockService;

    @Autowired(required = false)
    private MonthBudgetService monthBudgetService;

    /**
     * 执行实销量与回复量差额
     *
     * @param ids
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-12-13 23:43
     */
    @Override
    public void manualActualReplayDiffMq(List<String> ids) {
        Assert.notEmpty(ids, "请选择预算!");
        List<MonthBudgetEntity> monthBudgetEntityList = monthBudgetRepository.listByIds(ids);
        Assert.notEmpty(monthBudgetEntityList, "选择的预算不存在,请刷新重试!");
        Assert.isTrue(ids.size() == monthBudgetEntityList.size(), "选择的预算不存在,请刷新重试!");
        List<String> monthBudgetCodeList = monthBudgetEntityList.stream()
                .filter(k -> StringUtil.isNotEmpty(k.getMonthBudgetCode()))
                .map(MonthBudgetEntity::getMonthBudgetCode).distinct().collect(Collectors.toList());
        boolean isLock = monthBudgetLockService.isLock(monthBudgetCodeList);
        Assert.isTrue(!isLock, "其他人正在操作数据,加锁失败,请稍后重试!");
        List<List<String>> entityGroupList = Lists.partition(ids, 2);
        entityGroupList.forEach(list -> {
            MqMessageVo mqMessageVo = new MqMessageVo();
            mqMessageVo.setMsgBody(JSON.toJSONString(list));
            mqMessageVo.setTag(MonthBudgetConstant.CAL_MONTH_BUDGET_ACTUAL_REPLAY_TAG);
            rocketMqProducer.sendMqMsg(mqMessageVo);
            try {
                //单位：毫秒 防止MQ消息发送过于频繁
                Thread.sleep(50);
            } catch (Exception e) {
                log.error("", e);
            }
        });

    }


    /**
     * 执行实销量与回复量差额
     *
     * @param ids
     * @return void
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2023-12-14 00:02
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void manualActualReplayDiff(List<String> ids) {
        StopWatch stopWatch = new StopWatch();
        stopWatch.start();
        Map<String, MonthBudgetActualSalesVo> actualSalesMap = this.monthBudgetService.calculateActualSales(ids);
        stopWatch.stop();
        log.info("=====>    执行实销量与回复量差额 actualSalesMap 耗时[{}]    <=====", stopWatch.formatTime());
        List<String> monthBudgetCodeList = actualSalesMap.values().stream().filter(k -> StringUtil.isNotEmpty(k.getMonthBudgetCode()))
                .map(MonthBudgetActualSalesVo::getMonthBudgetCode).distinct().collect(Collectors.toList());
        boolean lockSuccess = monthBudgetLockService.lock(monthBudgetCodeList, TimeUnit.MINUTES, 20);
        Assert.isTrue(lockSuccess, "其他人正在操作数据,加锁失败,请稍后重试!");
        //计算实销量与回复量差额
        try {
            actualSalesMap.forEach((k, v) -> {
                stopWatch.reset();
                stopWatch.start();
                this.monthBudgetService.calculateActualReplyDiff(v, null);
                stopWatch.stop();
                log.info("=====>    执行实销量与回复量差额 calculateActualReplyDiff [{}] 耗时[{}]    <=====", v.getYearMonthLy(), stopWatch.formatTime());
            });
        } finally {
            monthBudgetLockService.unLock(monthBudgetCodeList);
        }
    }
}
