package com.biz.crm.tpm.business.month.budget.local.service.internal;

import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetDetailEntity;
import com.biz.crm.tpm.business.month.budget.local.entity.MonthBudgetEntity;
import com.biz.crm.tpm.business.month.budget.local.repository.MonthBudgetDetailRepository;
import com.biz.crm.tpm.business.month.budget.local.repository.MonthBudgetRepository;
import com.biz.crm.tpm.business.month.budget.local.service.MonthBudgetCallBackService;
import com.biz.crm.tpm.business.month.budget.local.service.MonthBudgetDetailService;
import com.biz.crm.tpm.business.month.budget.sdk.constant.BudgetLockConstant;
import com.biz.crm.tpm.business.month.budget.sdk.dto.MonthBudgetDetailDto;
import com.biz.crm.tpm.business.month.budget.sdk.eunm.BudgetOperationTypeEnum;
import com.biz.crm.tpm.business.month.budget.sdk.service.MonthBudgetLockService;
import com.biz.crm.workflow.sdk.dto.ProcessStatusDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 月度预算回调
 *
 * @author huojia
 * @date 2022年11月12日 17:50
 */
@Slf4j
@Service
public class MonthBudgetCallBackServiceImpl implements MonthBudgetCallBackService {

    @Resource
    private NebulaToolkitService nebulaToolkitService;

    @Resource
    private MonthBudgetRepository monthBudgetRepository;

    @Resource
    private MonthBudgetLockService monthBudgetLockService;

    @Resource
    private MonthBudgetDetailRepository monthBudgetDetailRepository;

    @Resource
    private MonthBudgetDetailService monthBudgetDetailService;

    /**
     * 月度预算调整审批通过
     *
     * @param dto
     * @author huojia
     * @date 2022/11/14 9:54
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void adjustPass(ProcessStatusDto dto) {
        Validate.notNull(dto, "审批通过回调失败，请求参数不能为空！");
        Validate.notNull(dto.getProcessNo(), "审批通过回调失败，流程编码不能为空！");
        List<MonthBudgetDetailEntity> monthBudgetDetailEntityList = monthBudgetDetailRepository.listByProcessNo(dto.getProcessNo());
        Validate.notEmpty(monthBudgetDetailEntityList, "审批通过回调失败，对应月度预算审批数据不存在！");
        List<String> monthBudgetCodeList = monthBudgetDetailEntityList.stream().map(MonthBudgetDetailEntity::getMonthBudgetCode).collect(Collectors.toList());
        boolean lock = true;
        try {
            lock = monthBudgetLockService.lock(monthBudgetCodeList, TimeUnit.SECONDS, BudgetLockConstant.LOCK_TIME_FIVE);
            Validate.isTrue(lock, "预算调整失败，当前调入、调出预算正在操作中，请稍后！");
            // 修改审批状态，调整预算
            monthBudgetDetailEntityList.forEach(monthBudgetDetailEntity -> {
                monthBudgetDetailEntity.setProcessStatus(ProcessStatusEnum.PASS.getDictCode());
                MonthBudgetEntity monthBudgetEntity = monthBudgetRepository.getByMonthBudgetCode(monthBudgetDetailEntity.getMonthBudgetCode(), null);
                // 调入方对应金额要改变
                if (BudgetOperationTypeEnum.ADJUST_IN.getCode().equals(monthBudgetDetailEntity.getOperationType())) {
                    // 显隐状态改变
                    monthBudgetDetailEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                    // 调整金额增加
                    monthBudgetEntity.setAdjustAmount(
                            Optional.ofNullable(monthBudgetEntity.getAdjustAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 审批中金额减少
                    monthBudgetEntity.setApprovingAmount(
                            Optional.ofNullable(monthBudgetEntity.getApprovingAmount()).orElse(BigDecimal.ZERO)
                                    .subtract(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 冻结后可用金额增加
                    monthBudgetEntity.setAfterFreezeAmount(
                            Optional.ofNullable(monthBudgetEntity.getAfterFreezeAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 余额增加
                    monthBudgetEntity.setAccumulatedAvailableBalance(
                            Optional.ofNullable(monthBudgetEntity.getAccumulatedAvailableBalance()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    monthBudgetDetailEntity.setBeforeAmount(
                            Optional.ofNullable(monthBudgetEntity.getAccumulatedAvailableBalance()).orElse(BigDecimal.ZERO)
                                    .subtract(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    monthBudgetDetailEntity.setBalanceAmount(monthBudgetEntity.getAccumulatedAvailableBalance());
                }
                // 调出方只用更新审批中金额
                if (BudgetOperationTypeEnum.ADJUST_OUT.getCode().equals(monthBudgetDetailEntity.getOperationType())) {
                    // 审批中金额减少
                    monthBudgetEntity.setApprovingAmount(
                            Optional.ofNullable(monthBudgetEntity.getApprovingAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO)));
                }
                monthBudgetRepository.updateById(monthBudgetEntity);
            });
            monthBudgetDetailRepository.updateBatchById(monthBudgetDetailEntityList);
        } finally {
            if (lock) {
                monthBudgetLockService.unLock(monthBudgetCodeList);
            }
        }
    }

    /**
     * 月度预算调整驳回、追回
     *
     * @param dto
     * @author huojia
     * @date 2022/11/14 10:31
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void adjustRejectAndRecover(ProcessStatusDto dto) {
        Validate.notNull(dto, "回调失败，请求参数不能为空！");
        Validate.notNull(dto.getProcessNo(), "回调失败，流程编码不能为空！");
        List<MonthBudgetDetailEntity> monthBudgetDetailEntityList = monthBudgetDetailRepository.listByProcessNo(dto.getProcessNo());
        Validate.notEmpty(monthBudgetDetailEntityList, "回调失败，对应月度预算审批数据不存在！");
        List<String> monthBudgetCodeList = monthBudgetDetailEntityList.stream().map(MonthBudgetDetailEntity::getMonthBudgetCode).collect(Collectors.toList());
        boolean lock = true;
        try {
            lock = monthBudgetLockService.lock(monthBudgetCodeList, TimeUnit.SECONDS, BudgetLockConstant.LOCK_TIME_FIVE);
            Validate.isTrue(lock, "回调失败，预算调整失败，当前调入、调出预算正在操作中，请稍后！");
            // 修改审批状态，调整预算
            monthBudgetDetailEntityList.forEach(monthBudgetDetailEntity -> {
                monthBudgetDetailEntity.setProcessStatus(dto.getProcessStatus());
                MonthBudgetEntity monthBudgetEntity = monthBudgetRepository.getByMonthBudgetCode(monthBudgetDetailEntity.getMonthBudgetCode(), null);
                // 调入方
                if (BudgetOperationTypeEnum.ADJUST_IN.getCode().equals(monthBudgetDetailEntity.getOperationType())) {
                    // 审批中金额减少
                    monthBudgetEntity.setApprovingAmount(
                            Optional.ofNullable(monthBudgetEntity.getApprovingAmount()).orElse(BigDecimal.ZERO)
                                    .subtract(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO)));
                    monthBudgetDetailEntity.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
                }
                // 调出方预算恢复
                if (BudgetOperationTypeEnum.ADJUST_OUT.getCode().equals(monthBudgetDetailEntity.getOperationType())) {
                    // 调整金额恢复
                    monthBudgetEntity.setAdjustAmount(
                            Optional.ofNullable(monthBudgetEntity.getAdjustAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 审批中金额增加
                    monthBudgetEntity.setApprovingAmount(
                            Optional.ofNullable(monthBudgetEntity.getApprovingAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 冻结后可用金额恢复
                    monthBudgetEntity.setAfterFreezeAmount(
                            Optional.ofNullable(monthBudgetEntity.getAfterFreezeAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 余额恢复
                    monthBudgetEntity.setAccumulatedAvailableBalance(
                            Optional.ofNullable(monthBudgetEntity.getAccumulatedAvailableBalance()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                }
                monthBudgetRepository.updateById(monthBudgetEntity);
                if (BudgetOperationTypeEnum.ADJUST_OUT.getCode().equals(monthBudgetDetailEntity.getOperationType())) {
                    MonthBudgetDetailDto monthBudgetDetailDto = this.buildDetail(monthBudgetEntity, monthBudgetDetailEntity.getCurOperationAmount(), BudgetOperationTypeEnum.RELEASE.getCode(), null);
                    this.monthBudgetDetailService.create(monthBudgetDetailDto);
                }
                /*if (BudgetOperationTypeEnum.ADJUST_IN.getCode().equals(monthBudgetDetailEntity.getOperationType())) {
                    MonthBudgetDetailDto monthBudgetDetailDto = this.buildDetail(monthBudgetEntity, monthBudgetDetailEntity.getCurOperationAmount(), BudgetOperationTypeEnum.RELEASE.getCode(), null);
                    this.monthBudgetDetailService.create(monthBudgetDetailDto);
                }*/
            });
            monthBudgetDetailRepository.updateBatchById(monthBudgetDetailEntityList);
        } finally {
            if (lock) {
                monthBudgetLockService.unLock(monthBudgetCodeList);
            }
        }
    }

    /**
     * 预算变更审批通过
     *
     * @param dto
     * @author huojia
     * @date 2022/11/14 11:10
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changePass(ProcessStatusDto dto) {
        Validate.notNull(dto, "审批通过回调失败，请求参数不能为空！");
        Validate.notNull(dto.getProcessNo(), "审批通过回调失败，流程编码不能为空！");
        List<MonthBudgetDetailEntity> monthBudgetDetailEntityList = monthBudgetDetailRepository.listByProcessNo(dto.getProcessNo());
        Validate.notEmpty(monthBudgetDetailEntityList, "审批通过回调失败，对应月度预算审批数据不存在！");
        List<String> monthBudgetCodeList = monthBudgetDetailEntityList.stream().map(MonthBudgetDetailEntity::getMonthBudgetCode).collect(Collectors.toList());
        boolean lock = true;
        try {
            lock = monthBudgetLockService.lock(monthBudgetCodeList, TimeUnit.SECONDS, BudgetLockConstant.LOCK_TIME_FIVE);
            Validate.isTrue(lock, "回调失败，预算变更失败，当前变更预算正在操作中，请稍后！");
            monthBudgetDetailEntityList.forEach(monthBudgetDetailEntity -> {
                monthBudgetDetailEntity.setProcessStatus(dto.getProcessStatus());
                MonthBudgetEntity monthBudgetEntity = monthBudgetRepository.getByMonthBudgetCode(monthBudgetDetailEntity.getMonthBudgetCode(), null);
                // 追加
                if (BudgetOperationTypeEnum.ADD.getCode().equals(monthBudgetDetailEntity.getOperationType())) {
                    monthBudgetDetailEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                    // 调整金额追加
                    monthBudgetEntity.setAdjustAmount(
                            Optional.ofNullable(monthBudgetEntity.getAdjustAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 审批中金额减少
                    monthBudgetEntity.setApprovingAmount(
                            Optional.ofNullable(monthBudgetEntity.getApprovingAmount()).orElse(BigDecimal.ZERO)
                                    .subtract(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 冻结后可用金额
                    monthBudgetEntity.setAfterFreezeAmount(
                            Optional.ofNullable(monthBudgetEntity.getAfterFreezeAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 可用余额追加
                    monthBudgetEntity.setAccumulatedAvailableBalance(
                            Optional.ofNullable(monthBudgetEntity.getAccumulatedAvailableBalance()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    monthBudgetDetailEntity.setBeforeAmount(
                            Optional.ofNullable(monthBudgetEntity.getAccumulatedAvailableBalance()).orElse(BigDecimal.ZERO)
                                    .subtract(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    monthBudgetDetailEntity.setBalanceAmount(monthBudgetEntity.getAccumulatedAvailableBalance());
                }
                // 削减
                if (BudgetOperationTypeEnum.SUBTRACT.getCode().equals(monthBudgetDetailEntity.getOperationType())) {
                    // 审批中金额增加
                    monthBudgetEntity.setApprovingAmount(
                            Optional.ofNullable(monthBudgetEntity.getApprovingAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                }
                monthBudgetRepository.updateById(monthBudgetEntity);
            });
            monthBudgetDetailRepository.updateBatchById(monthBudgetDetailEntityList);
        } finally {
            if (lock) {
                monthBudgetLockService.unLock(monthBudgetCodeList);
            }
        }
    }

    /**
     * 预算变更审批驳回
     *
     * @param dto
     * @author huojia
     * @date 2022/11/14 11:10
     **/
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changeRejectAndRecover(ProcessStatusDto dto) {
        Validate.notNull(dto, "回调失败，请求参数不能为空！");
        Validate.notNull(dto.getProcessNo(), "回调失败，流程编码不能为空！");
        List<MonthBudgetDetailEntity> monthBudgetDetailEntityList = monthBudgetDetailRepository.listByProcessNo(dto.getProcessNo());
        Validate.notEmpty(monthBudgetDetailEntityList, "回调失败，对应月度预算审批数据不存在！");
        List<String> monthBudgetCodeList = monthBudgetDetailEntityList.stream().map(MonthBudgetDetailEntity::getMonthBudgetCode).collect(Collectors.toList());
        boolean lock = true;
        try {
            lock = monthBudgetLockService.lock(monthBudgetCodeList, TimeUnit.SECONDS, BudgetLockConstant.LOCK_TIME_FIVE);
            Validate.isTrue(lock, "预算调整失败，当前调入、调出预算正在操作中，请稍后！");
            // 修改审批状态，调整预算
            monthBudgetDetailEntityList.forEach(monthBudgetDetailEntity -> {
                monthBudgetDetailEntity.setProcessStatus(dto.getProcessStatus());
                MonthBudgetEntity monthBudgetEntity = monthBudgetRepository.getByMonthBudgetCode(monthBudgetDetailEntity.getMonthBudgetCode(), null);
                // 追加
                if (BudgetOperationTypeEnum.ADD.getCode().equals(monthBudgetDetailEntity.getOperationType())) {
                    // 审批中金额减少
                    monthBudgetEntity.setApprovingAmount(
                            Optional.ofNullable(monthBudgetEntity.getApprovingAmount()).orElse(BigDecimal.ZERO)
                                    .subtract(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    monthBudgetDetailEntity.setDelFlag(DelFlagStatusEnum.DELETE.getCode());
                }
                // 削减
                if (BudgetOperationTypeEnum.SUBTRACT.getCode().equals(monthBudgetDetailEntity.getOperationType())) {
                    // 调整金额追加
                    monthBudgetEntity.setAdjustAmount(
                            Optional.ofNullable(monthBudgetEntity.getAdjustAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 审批中金额增加
                    monthBudgetEntity.setApprovingAmount(
                            Optional.ofNullable(monthBudgetEntity.getApprovingAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    monthBudgetEntity.setAfterFreezeAmount(
                            Optional.ofNullable(monthBudgetEntity.getAfterFreezeAmount()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                    // 可用余额追加
                    monthBudgetEntity.setAccumulatedAvailableBalance(
                            Optional.ofNullable(monthBudgetEntity.getAccumulatedAvailableBalance()).orElse(BigDecimal.ZERO)
                                    .add(Optional.ofNullable(monthBudgetDetailEntity.getCurOperationAmount()).orElse(BigDecimal.ZERO))
                    );
                }
                monthBudgetRepository.updateById(monthBudgetEntity);
                if (BudgetOperationTypeEnum.SUBTRACT.getCode().equals(monthBudgetDetailEntity.getOperationType())) {
                    MonthBudgetDetailDto monthBudgetDetailDto = this.buildDetail(monthBudgetEntity, monthBudgetDetailEntity.getCurOperationAmount(), BudgetOperationTypeEnum.RELEASE.getCode(), null);
                    monthBudgetDetailService.create(monthBudgetDetailDto);
                }
                /*if (BudgetOperationTypeEnum.ADD.getCode().equals(monthBudgetDetailEntity.getOperationType())) {
                    MonthBudgetDetailDto monthBudgetDetailDto = this.buildDetail(monthBudgetEntity, monthBudgetDetailEntity.getCurOperationAmount(), BudgetOperationTypeEnum.RELEASE.getCode(), null);
                    monthBudgetDetailService.create(monthBudgetDetailDto);
                }*/
            });
            monthBudgetDetailRepository.updateBatchById(monthBudgetDetailEntityList);

        } finally {
            if (lock) {
                monthBudgetLockService.unLock(monthBudgetCodeList);
            }
        }
    }

    /**
     * 生成预算明细
     *
     * @param monthBudgetEntity
     * @param operationAmount
     * @param operationType
     * @param businessCode
     * @return com.biz.crm.tpm.business.month.budget.sdk.dto.MonthBudgetDetailDto
     * @author huojia
     * @date 2023/1/3 1:20
     **/
    private MonthBudgetDetailDto buildDetail(MonthBudgetEntity monthBudgetEntity, BigDecimal operationAmount, String operationType, String businessCode) {
        MonthBudgetDetailDto monthBudgetDetailDto = nebulaToolkitService.copyObjectByWhiteList(
                monthBudgetEntity, MonthBudgetDetailDto.class, LinkedHashSet.class, ArrayList.class
        );
        monthBudgetDetailDto.setId(null);
        monthBudgetDetailDto.setBusinessCode(businessCode);
        monthBudgetDetailDto.setOperationType(operationType);
        monthBudgetDetailDto.setInitialAmount(monthBudgetEntity.getInitResolveAmount());
        // 根据不同操作类型，设置不同操作前金额
        if (BudgetOperationTypeEnum.UNFREEZE.getCode().equals(operationType)
                || BudgetOperationTypeEnum.RETURN.getCode().equals(operationType)
                || BudgetOperationTypeEnum.REPLAY.getCode().equals(operationType)
                || BudgetOperationTypeEnum.ACTUAL_SALES.getCode().equals(operationType)) {
            monthBudgetDetailDto.setBeforeAmount(Optional.ofNullable(monthBudgetEntity.getAccumulatedAvailableBalance()).orElse(BigDecimal.ZERO).subtract(operationAmount));
        } else if (BudgetOperationTypeEnum.ADJUST_OUT.getCode().equals(operationType)
                || BudgetOperationTypeEnum.SUBTRACT.getCode().equals(operationType)
                || BudgetOperationTypeEnum.FREEZE.getCode().equals(operationType)
                || BudgetOperationTypeEnum.USE.getCode().equals(operationType)
                || BudgetOperationTypeEnum.FORECAST_OVER.getCode().equals(operationType)) {
            monthBudgetDetailDto.setBeforeAmount(Optional.ofNullable(monthBudgetEntity.getAccumulatedAvailableBalance()).orElse(BigDecimal.ZERO).add(operationAmount));
        }
        monthBudgetDetailDto.setCurOperationAmount(operationAmount);
        monthBudgetDetailDto.setBeforeAmount(Optional.ofNullable(monthBudgetEntity.getAccumulatedAvailableBalance()).orElse(BigDecimal.ZERO).subtract(operationAmount));
        monthBudgetDetailDto.setBalanceAmount(monthBudgetEntity.getAccumulatedAvailableBalance());
        return monthBudgetDetailDto;
    }

    /**
     * 预算划拨审批通过
     *
     * @param dto
     * @author huojia
     * @date 2022/11/14 11:10
     **/
    @Override
    public void transferPass(ProcessStatusDto dto) {

    }

    /**
     * 预算划拨驳回
     *
     * @param dto
     * @author huojia
     * @date 2022/11/14 11:11
     **/
    @Override
    public void transferRejectAndRecover(ProcessStatusDto dto) {

    }
}
